/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny <tnowotny@ucsd.edu>
 *
 * License: GPL-2+
 *
 * Initial version: 2010-02-24
 *
 */


#include <cmath>
#include <sys/time.h>
#include <iostream>
#include <fstream>
#include <limits>

#include "sources.hh"

#include "config.h"

using namespace std;

const char * const CNRun::__SourceTypes[] = {
	"Null",
	"Tape",
	"Periodic",
	"Function",
	"Noise",
};


CNRun::CSourceTape::
CSourceTape( const char *id, const char *infname, bool inis_looping)
      : C_BaseSource (id, SRC_TAPE), is_looping (inis_looping)
{
	ifstream ins( infname);
	if ( !ins.good() ) {
		name = "";
		return;
	}
	skipws(ins);

	while ( !ins.eof() && ins.good() ) {
		while ( ins.peek() == '#' || ins.peek() == '\n' )
			ins.ignore( numeric_limits<streamsize>::max(), '\n');
		double timestamp, datum;
		ins >> timestamp >> datum;
		values.push_back( pair<double,double>(timestamp, datum));
	}

	if ( values.size() == 0 ) {
		fprintf( stderr, "No usable values in \"%s\"\n", infname);
		return;
	}

	fname = infname;
	I = values.begin();
}

double
CNRun::CSourceTape::
operator() ( double t)
{
      // position I such that *I < t < *(I+1)
	while ( I+1 != values.end() && (I+1)->first < t )
		++I;

	if ( I+1 == values.end() && is_looping )
		I = values.begin();

	return I->second;
}






CNRun::CSourcePeriodic::
CSourcePeriodic( const char *id, const char *infname, bool inis_looping, double inperiod)
      : C_BaseSource (id, SRC_PERIODIC), is_looping (inis_looping)
{
	ifstream ins( infname);
	if ( !ins.good() ) {
		name = "";
		return;
	}
	skipws(ins);

	while ( ins.peek() == '#' || ins.peek() == '\n' )
		ins.ignore( numeric_limits<streamsize>::max(), '\n');

	if ( !isfinite(inperiod) || inperiod <= 0. ) {
		ins >> inperiod;
		if ( !isfinite(inperiod) || inperiod <= 0. ) {
			fprintf( stderr, "Bad interval for \"%s\"\n", infname);
			name = "";
			return;
		}
	}
	period = inperiod;

	while ( true ) {
		while ( ins.peek() == '#' || ins.peek() == '\n' )
			ins.ignore( numeric_limits<streamsize>::max(), '\n');
		double datum;
		ins >> datum;
		if ( ins.eof() || !ins.good() )
			break;
		values.push_back( datum);
	}

	if ( values.size() < 2 ) {
		fprintf( stderr, "Need at least 2 scheduled values in \"%s\"\n", infname);
		name = "";
		return;
	}

	fname = infname;
}



const char * const CNRun::distribution_names[] = { "uniform", "gaussian" };

CNRun::CSourceNoise::
CSourceNoise( const char *id,
	      double in_min, double in_max,
	      TDistribution indist_type,
	      int seed)
      : C_BaseSource (id, SRC_NOISE), _min (in_min), _max (in_max), _sigma (in_max - in_min), dist_type (indist_type)
{
	const gsl_rng_type *T;
	gsl_rng_env_setup();
	T = gsl_rng_default;
	if ( gsl_rng_default_seed == 0 ) {
		struct timeval tp = { 0L, 0L };
		gettimeofday( &tp, nullptr);
		gsl_rng_default_seed = tp.tv_usec;
	}
	rng = gsl_rng_alloc( T);
}


CNRun::CSourceNoise::
~CSourceNoise()
{
	gsl_rng_free( rng);
}


// eof
