/* 
   Unix SMB/Netbios implementation.
   Version 1.7.
   Copyright (C) Andrew Tridgell 1992,1993,1994
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "loadparm.h"
#include "pcap.h"

/* default to not LANMAN1 */
#ifndef LANMAN1
#define LANMAN1 0
#endif

pstring OriginalDir ="/";
pstring servicesf = SERVICES;
pstring debugf = DEBUGFILE;

/* users from session setup */
pstring session_users="";

BOOL append_log = True;

char *InBuffer = NULL;
char *OutBuffer = NULL;

int initial_uid = 0;
int initial_gid = 0;

/* have I done a become_user? */
int done_become_user = -1;

/* the last message the was processed */
int last_message = -1;

/* a useful macro to debug the last message processed */
#define LAST_MESSAGE() smb_fn_name(last_message)

/* this is how errors are generated */
#define ERROR(class,x) error_packet(inbuf,outbuf,class,x,__LINE__)
#define UNIXERROR(defclass,deferror) unix_error_packet(inbuf,outbuf,defclass,deferror,__LINE__)

extern BOOL NeedSwap;
extern pstring scope;
extern int DEBUGLEVEL;

connection_struct Connections[MAX_CONNECTIONS];
files_struct Files[MAX_OPEN_FILES];

/* local prototype */
static char *build_print_command(int snum, char *command, char *syscmd, char *filename);

int Protocol = PROT_CORE;

int maxxmit = BUFFER_SIZE;

int chain_size = 0;

/* a fnum to use when chaining */
int chain_fnum = -1;

/* this macro should always be used to extract an fnum (smb_fid) from
a packet to ensure chaining works correctly */
#define GETFNUM(buf,where) (chain_fnum!=-1?chain_fnum:SVAL(buf,where))

/* total number of open files */
int num_files_open = 0;

/* this structure is used to hold information about the machine that 
   the program is running on */
extern machine_struct machine_info;


/* stuff for profiling */
#if PROFILING


#endif

/****************************************************************************
  change a dos mode to a unix mode
****************************************************************************/
mode_t unix_mode(int cnum,int dosmode)
{
  mode_t result = 0444;
  
  if ((dosmode & aRONLY) == 0)
    result |= (S_IWUSR | S_IWGRP | S_IWOTH);

  if ((dosmode & aDIR) != 0)
    result |= (S_IFDIR | S_IXUSR | S_IXGRP | S_IXOTH);

  if ((dosmode & aARCH) != 0)
    result |= S_IXUSR;

  if (MAP_SYSTEM(cnum) && ((dosmode & aSYSTEM) != 0))
    result |= S_IXGRP;

  if (MAP_HIDDEN(cnum) && ((dosmode & aHIDDEN) != 0))
    result |= S_IXOTH;  

  result &= CREATE_MODE(cnum);
  return(result);
}


/****************************************************************************
  change a unix mode to a dos mode
****************************************************************************/
int dos_mode(int cnum,char *path,struct stat *sbuf)
{
  int result = 0;

  if (!CAN_WRITE(cnum))
    result |= aRONLY;

  if (!((sbuf->st_mode & S_IWOTH) ||
      ((sbuf->st_mode & S_IWUSR) && Connections[cnum].uid == sbuf->st_uid) ||
      ((sbuf->st_mode & S_IWGRP) && in_group(sbuf->st_gid,Connections[cnum].gid))))
    result |= aRONLY;

  if ((sbuf->st_mode & S_IXUSR) != 0)
    result |= aARCH;

  if (MAP_SYSTEM(cnum) && ((sbuf->st_mode & S_IXGRP) != 0))
    result |= aSYSTEM;

  if (MAP_HIDDEN(cnum) && ((sbuf->st_mode & S_IXOTH) != 0))
    result |= aHIDDEN;   
  
  if (S_ISDIR(sbuf->st_mode))
    result = aDIR;

  /* hide files with a name starting with a . */
  {
    char *p = strrchr(path,'/');
    if (p)
      p++;
    else
      p = path;

    if (p[0] == '.' && p[1] != '.' && p[1] != 0)
      result |= aHIDDEN;
  }

  return(result);
}


struct dptr_struct
{
  int key;
  int pid;
  void *ptr;
  BOOL valid;
  BOOL expect_close;
  pstring path;
}
dirptrs[MAXDIR];

BOOL dptrs_init=False;
int next_key = 1;

/****************************************************************************
add a name to the session users list
****************************************************************************/
void add_session_user(char *user)
{
  if (user && *user && !in_list(user,session_users,False))
    {
      if (strlen(user) + strlen(session_users) + 2 >= sizeof(pstring))
	DEBUG(1,("Too many session users??\n"));
      else
	{
	  strcat(session_users," ");
	  strcat(session_users,user);
	}
    }
}

/****************************************************************************
convert a filename to a unix filename from dos
****************************************************************************/
void unix_convert(char *name,int cnum)
{
  unix_convert_83(name,HOME(cnum),lp_manglednames());
}

/****************************************************************************
check a filename - possibly caling reducename
****************************************************************************/
BOOL check_name(char *name,int cnum)
{
  BOOL ret;
  ret = reduce_name(name,HOME(cnum),lp_widelinks(SNUM(cnum)));
  if (!ret)
    DEBUG(5,("check_name on %s failed\n",name));

  return(ret);
}

/****************************************************************************
initialise the dir array
****************************************************************************/
void init_dptrs(void)
{
  int i;
  if (dptrs_init) return;
  for (i=0;i<MAXDIR;i++)    
    dirptrs[i].valid = False;
  dptrs_init = True;
}

/****************************************************************************
get the dir ptr for a dir index
****************************************************************************/
void *dptr_get(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(dirptrs[i].ptr);
  return(NULL);
}

/****************************************************************************
get the dir path for a dir index
****************************************************************************/
char *dptr_path(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(dirptrs[i].path);
  return(NULL);
}

/****************************************************************************
check a key
****************************************************************************/
BOOL key_ok(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(True);
  return(False);
}

/****************************************************************************
prompte a dptr (to make it recently used)
****************************************************************************/
void dptr_promote(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	struct dptr_struct d;
	int j;
	d = dirptrs[i];
	for (j=i;j>0;j--)
	  dirptrs[j] = dirptrs[j-1];
	dirptrs[0] = d;
	return;
      }
}


/****************************************************************************
demote a dptr (to make it likely to get closed)
****************************************************************************/
void dptr_demote(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	struct dptr_struct d;
	int j;
	d = dirptrs[i];
	for (j=i;j<(MAXDIR-1) && dirptrs[j].valid;j++)
	  dirptrs[j] = dirptrs[j+1];
	dirptrs[j] = d;
	return;
      }
}


/****************************************************************************
find a free key
****************************************************************************/
int find_key(int start)
{
  int key;

  for (key=start;key<256;key++)
    if (!key_ok(key))
      return(key);

  for (key=1;key<start;key++)
    if (!key_ok(key))
      return(key);

  DEBUG(0,("ERROR: Out of dirptr keys!\n"));
  return 1;
}       
  

/****************************************************************************
close a dptr
****************************************************************************/
void dptr_close(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	if (dirptrs[i].ptr)
	  closedir(dirptrs[i].ptr);
	dirptrs[i].valid = False;
	next_key = key+1;
	if (next_key > 255) next_key = 1;
	return;
      }
}

/****************************************************************************
close a dptr that matches a given path, only if it matches the pid also
****************************************************************************/
void dptr_closepath(char *path,int pid)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && pid == dirptrs[i].pid)
      {      
	DEBUG(5,("closepath [%s] [%s]\n",path,dirptrs[i].path));
	if (strequal(dirptrs[i].path,path))
	  {
	    if (dirptrs[i].ptr)
	      closedir(dirptrs[i].ptr);
	    dirptrs[i].valid = False;
	    next_key = dirptrs[i].key+1;
	    if (next_key > 255) next_key = 1;
	  }
      }
}

/****************************************************************************
create a new dir ptr
****************************************************************************/
int dptr_create(void *p,char *path, BOOL expect_close,int pid)
{
  int i,key;

  for (i=0;i<MAXDIR;i++)
    if (!dirptrs[i].valid)
      break;

  /* as a second option, grab one not marked for expect_close */
  if (i == MAXDIR)
    for (i=MAXDIR-1;i>=0;i--)
      if (!dirptrs[i].expect_close)
	{
	  dptr_close(dirptrs[i].key);	
	  break;
	}

  if (i < 0)
    {
      /* All dirptrs are used by uncloseable SMBffirsts */
      DEBUG(0,("Error - all dirptrs in use by SMBffirsts.\n"));
      return(-1);
    }

  dirptrs[i].ptr = p;
  strcpy(dirptrs[i].path,path);
  key = find_key(next_key);
  dirptrs[i].key = key;
  dirptrs[i].valid = True;
  dirptrs[i].pid = pid;
  dirptrs[i].expect_close = expect_close;

  dptr_promote(key);

  DEBUG(3,("creating new dirptr %d (0x%x) for path %s, expect_close = %d\n",key,p,path,
	expect_close));  

  return(key);
}

/****************************************************************************
fill the 5 byte server reserved dptr field
****************************************************************************/
BOOL dptr_fill(char *buf,unsigned int key)
{
  void *p = dptr_get(key);
  int offset;
  if (!p)
    {
      DEBUG(3,("filling null dirptr %d\n",key));
      return(False);
    }
  offset = telldir(p);
  DEBUG(3,("fill on dirptr 0x%x now at %d\n",p,offset));
  buf[0] = key;
  memcpy(buf+1,(char *)&offset,4);
  DEBUG(3,("filled dirptr %d at offset %d\n",key,offset));
  return(True);
}

/****************************************************************************
return True is the offset is at zero
****************************************************************************/
BOOL dptr_zero(char *buf)
{
  int offset;
  memcpy((char *)&offset,buf+1,4);
  return (offset == 0);
}

/****************************************************************************
fetch the dir ptr and seek it given the 5 byte server field
****************************************************************************/
void *dptr_fetch(char *buf,int *num)
{
  unsigned int key = *(unsigned char *)buf;
  void *p = dptr_get(key);
  int offset;
  if (!p)
    {
      DEBUG(3,("fetched null dirptr %d\n",key));
      return(NULL);
    }
  *num = key;
  memcpy((char *)&offset,buf+1,4);
  seekdir(p,offset);
  dptr_promote(key);
  DEBUG(3,("fetching dirptr %d for path %s\n",key,dptr_path(key)));
  return(p);
}


/****************************************************************************
  start a directory listing
****************************************************************************/
BOOL start_dir(int cnum,char *directory)
{
  DEBUG(5,("start_dir cnum=%d dir=%s\n",cnum,directory));

  if (!check_name(directory,cnum))
    return(False);
  
  Connections[cnum].dirptr = (void *)opendir(directory);

  strncpy(Connections[cnum].dirpath,directory,sizeof(pstring));
  
  return(Connections[cnum].dirptr != NULL);
}

/****************************************************************************
  get a directory entry
****************************************************************************/
BOOL get_dir_entry(int cnum,char *mask,int dirtype,char *fname,int *size,int *mode,time_t *date,BOOL check_descend)
{
  struct DIRECT *dptr;
  BOOL do_mangling;
  BOOL found = False;
  struct stat sbuf;
  pstring path="";
  pstring pathreal="";
  BOOL isrootdir;
  pstring filename="";
  BOOL matched;
  BOOL has_wild = (strchr(mask,'?') || strchr(mask,'*'));
  
  isrootdir = (strequal(Connections[cnum].dirpath,"./") ||
	       strequal(Connections[cnum].dirpath,"."));
  
  if (!Connections[cnum].dirptr)
    return(False);
  
  while (!found)
    {
      dptr = readdir(Connections[cnum].dirptr);

      DEBUG(3,("readdir on dirptr 0x%x now at offset %d\n",
	    Connections[cnum].dirptr,telldir(Connections[cnum].dirptr)));
      
      if (dptr == NULL) 
	return(False);
      
     
#ifdef NEXT2
      DEBUG(3,("name is: %s\n", dptr->d_name));
      if (telldir(Connections[cnum].dirptr)<0) {
        DEBUG(3,("get_dir_entry encountered null - returning false.\n"));
        return(False);
      }
#endif

      matched = False;

      do_mangling = ((has_wild || strchr(mask,MAGIC_CHAR)) && 
		     lp_manglednames());

      strcpy(filename,dptr->d_name);      

      if ((strcmp(filename,mask) == 0) ||
	  (name_convert(filename,dptr->d_name,do_mangling) &&
	   mask_match(filename,mask,!isrootdir,False,False)))
	{
	  strcpy(fname,filename);
	  strcpy(path,Connections[cnum].dirpath);
	  strcat(path,"/");
	  strcpy(pathreal,path);
	  strcat(path,fname);
	  strcat(pathreal,dptr->d_name);
	  if (stat(pathreal,&sbuf) != 0) 
	    {
	      DEBUG(5,("Couldn't stat 1 [%s]\n",path));
	      continue;
	    }

	  if (check_descend &&
	      !strequal(fname,".") && !strequal(fname,".."))
	    continue;
	  
	  *mode = dos_mode(cnum,pathreal,&sbuf);

	  if (((*mode & ~dirtype) & (aHIDDEN | aSYSTEM | aDIR)) != 0)
	    {	      
	      DEBUG(5,("[%s] attribs didn't match %x\n",filename,dirtype));
	      continue;
	    }
	  *size = sbuf.st_size;
	  *date = sbuf.st_mtime;

	  DEBUG(5,("get_dir_entry found %s fname=%s\n",pathreal,fname));
	  
	  found = True;
	}
    }

  return(found);
}

static int old_umask = 0755;

/****************************************************************************
  become the user of a connection number
****************************************************************************/
BOOL become_user(int cnum)
{
  if (!OPEN_CNUM(cnum))
    {
      DEBUG(0,("ERROR: Connection %d not open\n",cnum));
      return(False);
    }

  if (done_become_user == cnum)
    {
      DEBUG(3,("Skipping become_user - already user\n"));
      return(True);
    }

  if (done_become_user != -1)
    unbecome_user();
  
  if (initial_uid == 0)
    {
#ifdef USE_SETRES 
      if (setresgid(-1,Connections[cnum].gid,-1) != 0)
#else
	if (setgid(Connections[cnum].gid) != 0)
#endif
	  {
	    DEBUG(0,("Couldn't set gid %d currently set to (%d,%d)\n",
		  Connections[cnum].gid,getgid(),getegid()));
	    return(False);
	  }


      /* groups stuff added by ih/wreu */
      if (Connections[cnum].ngroups > 0)
	if (setgroups(Connections[cnum].ngroups,Connections[cnum].groups))
	  DEBUG(0,("setgroups call failed!\n"));
  
#ifdef USE_SETRES
      if (setresuid(-1,Connections[cnum].uid,-1) != 0)
#else  
      if ((seteuid(Connections[cnum].uid) != 0) && 
	  (setuid(Connections[cnum].uid) != 0))
#endif
	{
	  DEBUG(0,("Couldn't set uid %d currently set to (%d,%d)\n",
		Connections[cnum].uid,getuid(), geteuid()));
	  return(False);
	}
    }
  
  if (ChDir(Connections[cnum].connectpath) != 0)
    {
      DEBUG(0,("%s chdir (%s) failed cnum=%d\n",timestring(),
	    Connections[cnum].connectpath,cnum));     
      return(False);
    }
  
  old_umask = umask(0777 & ~lp_create_mode(SNUM(cnum)));

  DEBUG(5,("become_user now uid=(%d,%d) gid=(%d,%d)\n",
	getuid(),geteuid(),getgid(),getegid()));

  done_become_user = cnum;

  return(True);
}

/****************************************************************************
  unbecome the user of a connection number
****************************************************************************/
BOOL unbecome_user(void )
{
  if (done_become_user == -1)
    DEBUG(0,("ERROR: unbecome_user() out of sequence?? (pid %d)\n",getpid()));

  umask(old_umask);

  if (initial_uid == 0)
    {
#ifdef USE_SETRES
      setresuid(-1,getuid(),-1);
      setresgid(-1,getgid(),-1);
#else
      if (seteuid(initial_uid) != 0) 
	setuid(initial_uid);
      setgid(initial_gid);
#endif
    }
#ifdef NO_EID
  DEBUG(2,("Running with no EID\n"));
  initial_uid = getuid();
  initial_gid = getgid();
#else
  if (geteuid() != initial_uid)
    {
      DEBUG(0,("Warning: You appear to have a trapdoor uid system\n"));
      initial_uid = geteuid();
    }
  if (getegid() != initial_gid)
    {
      DEBUG(0,("Warning: You appear to have a trapdoor gid system\n"));
      initial_gid = getegid();
    }
#endif

  if (ChDir(OriginalDir) != 0)
    DEBUG(0,("%s chdir(%s) failed\n",timestring(),OriginalDir));  
  
  DEBUG(5,("unbecome_user now uid=(%d,%d) gid=(%d,%d)\n",
	getuid(),geteuid(),getgid(),getegid()));

  done_become_user = -1;

  return(True);
}

/****************************************************************************
close a file - possibly invalidating the read prediction
****************************************************************************/
int close_file(int fd)
{
  invalidate_read_prediction(fd);
  return(close(fd));
}

/****************************************************************************
  find a service entry
****************************************************************************/
int find_service(char *service)
{
   int iService;

   if (strequal(service,"IPC$"))
     return(-1);

   iService = lp_servicenumber(service);

   /* now handle the special case of a home directory */
   if (iService < 0)
   {
      char *phome_dir = get_home_dir(service);
      DEBUG(3,("checking for home directory %s gave %s\n",service,
	    phome_dir?phome_dir:"(NULL)"));
      if (phome_dir)
      {   
	 int iHomeService;
	 if ((iHomeService = lp_servicenumber(HOMES_NAME)) >= 0)
	 {
	    lp_add_home(service,iHomeService,phome_dir);
	    iService = lp_servicenumber(service);
	 }
      }
   }

   /* If we still don't have a service, attempt to add it as a printer. */
   if (iService < 0)
   {
      int iPrinterService;

      if ((iPrinterService = lp_servicenumber(PRINTERS_NAME)) >= 0)
      {
         char *pszTemp;

         DEBUG(3,("checking whether %s is a valid printer name...\n", service));
         pszTemp = PRINTCAP;
         if ((pszTemp != NULL) && pcap_printername_ok(service, pszTemp))
         {
            DEBUG(3,("%s is a valid printer name\n", service));
            DEBUG(3,("adding %s as a printer service\n", service));
            lp_add_printer(service,iPrinterService);
            iService = lp_servicenumber(service);
            if (iService < 0)
               DEBUG(0,("failed to add %s as a printer service!\n", service));
         }
         else
            DEBUG(3,("%s is not a valid printer name\n", service));
      }
   }

   /* just possibly it's a default service? */
   if (iService < 0) 
     {
       char *defservice = lp_defaultservice();
       if (defservice && *defservice && !strequal(defservice,service))
	 return(find_service(defservice));
     }

   if (iService >= 0)
      if (!VALID_SNUM(iService))
      {
         DEBUG(0,("Invalid snum %d for %s\n",iService,service));
	 iService = -1;
      }

   if (iService < 0)
      DEBUG(3,("find_service() failed to find service %s\n", service));

   return (iService);
}

/* this holds info on user ids that are already validated for this VC */
user_struct *validated_users = NULL;
int num_validated_users = 0;

/****************************************************************************
check if a uid has been validated, and return an index if it has. -1 if not
****************************************************************************/
int valid_uid(int uid)
{
  int i;
  for (i=0;i<num_validated_users;i++)
    if (validated_users[i].uid == uid)
      {
	DEBUG(3,("valid uid %d mapped to vuid %d\n",uid,i));
	return(i);
      }
  return(-1);
}

/****************************************************************************
register a uid/name pair as being valid and that a valid password
has been given.
****************************************************************************/
void register_uid(int uid,char *name,BOOL guest)
{
  if (valid_uid(uid) >= 0)
    return;
  if (!validated_users)
    validated_users = (user_struct *)malloc(sizeof(user_struct));
  else
    validated_users = (user_struct *)realloc(validated_users,
					     sizeof(user_struct)*
					     (num_validated_users+1));

  if (!validated_users)
    {
      DEBUG(0,("Failed to realloc users struct!\n"));
      return;
    }

  validated_users[num_validated_users].uid = uid;
  validated_users[num_validated_users].guest = guest;
  validated_users[num_validated_users].name = strdup(name);    

  DEBUG(3,("uid %d registered to name %s\n",uid,name));
  
  num_validated_users++;
}


/****************************************************************************
  find first available connection slot, starting from a random position.

The randomisation stops problems with the server dieing and clients
thinking the server is still available.
****************************************************************************/
int find_free_connection(void )
{
  int rand_start = (getpid() + (time(NULL)%1000)) % MAX_CONNECTIONS;
  int i;

  if (rand_start < 1) rand_start = 1;

  for (i=rand_start;i<MAX_CONNECTIONS;i++)
    if (!Connections[i].open) 
      {
	DEBUG(3,("found free connection number %d\n",i));
	return(i);
      }

  for (i=1;i<rand_start;i++)
    if (!Connections[i].open) 
      {
	DEBUG(3,("found free connection number %d\n",i));
	return(i);
      }

  return(-1);
}

/****************************************************************************
  find first available file slot
****************************************************************************/
int find_free_file(void )
{
  int i;
  for (i=1;i<MAX_OPEN_FILES;i++)
    if (!Files[i].open)
      return(i);
  return(-1);
}

/****************************************************************************
validate a group username entry. Return the username or NULL
****************************************************************************/
char *validate_group(char *group,char *password)
{
#if HAVE_GETGRNAM 
  struct group *gptr = getgrnam(group);
  char **member;
  if (gptr)
    {
      member = gptr->gr_mem;
      while (member && *member)
	{
	  if (password_ok(*member,password,lp_passwordlevel()))
	    return(*member);
	  member++;
	}
    }		      
#endif
  return(NULL);
}



/****************************************************************************
  make a connection to a service
****************************************************************************/
int make_connection(char *service,char *user,char *password,char *dev,int vuid)
{
  int cnum;
  int snum;
  struct passwd *pass = NULL;
  BOOL guest = False;
  BOOL ok = False;

  strlower(service);

  snum = find_service(service);
  if (snum < 0)
    {
      if (strequal(service,"IPC$"))
	{	  
	  DEBUG(3,("%s refusing IPC connection\n",timestring()));
	  return(-3);
	}

      DEBUG(0,("%s couldn't find service %s\n",timestring(),service));      
      return(-2);
    }

  if (!check_access(snum))
    return(-1);

  /* if the request is as a printer and you can't print then refuse */
  strupper(dev);
  if (!lp_print_ok(snum) && (strncmp(dev,"LPT",3) == 0))
    return(-2);

  /* lowercase the user name */
  strlower(user);

  /* now there are 5 possabilities:
     1) login as given user with given password
     2) login as the "user =" user with given password
     3) login as a previously validated user/password pair
     4) login as the "user =" user with no password (guest connection)
     5) login as guest user with no password

     if the service is guest_only then steps 1,2 and 3 are skipped
  */

  if (!(GUEST_ONLY(snum) && GUEST_OK(snum)))
    {

      /* check the given username and password */
      if (!lp_onlyuser(snum) && (*password) && (*user))
	{
	  ok = password_ok(user,password,lp_passwordlevel());
	  if (ok) DEBUG(3,("ACCEPTED: given username password ok\n"));
	}

      /* check the user= fields and the given password */
      if (!ok && *password)
	{
	  pstring user_list;
	  char *auser;
	  if (USER(snum))
	    {
	      strcpy(user_list,USER(snum));
	      auser = strtok(user_list," \t,");
	      while (!ok && auser)
		{
		  if (*auser == '@')
		    {
		      auser = validate_group(auser+1,password);
		      if (auser)
			{
			  ok = True;
			  strcpy(user,auser);
			  DEBUG(3,("ACCEPTED: group username and given password ok\n"));
			}
		    }
		  else
		    {
		      ok = password_ok(auser,password,lp_passwordlevel());
		      if (ok) 
			{
			  strcpy(user,auser);
			  DEBUG(3,("ACCEPTED: user list username and given password ok\n"));
			}
		    }
		  if (!ok)
		    auser = strtok(NULL," \t,");
		}
	    }

	  if (!ok && !lp_onlyuser(snum))
	    {
	      /* now check the list of session users */
	      strcpy(user_list,session_users);
	      auser = strtok(user_list," \t,");
	      while (!ok && auser)
		{
		  ok = password_ok(auser,password,lp_passwordlevel());
		  if (ok) 
		    {
		      strcpy(user,auser);
		      DEBUG(3,("ACCEPTED: session list username and given password ok\n"));
		    }
		  else
		    auser = strtok(NULL," \t,");
		}
	    }
	}
      
      /* check for a previously validated username/password pair */
      if (!lp_onlyuser(snum) && !ok && 
	  (vuid >= 0) && !validated_users[vuid].guest)
	{
	  strcpy(user,validated_users[vuid].name);
	  guest = validated_users[vuid].guest;
	  if (guest)
	    DEBUG(3,("ACCEPTED: validated uid ok as guest\n"));
	  else
	    DEBUG(3,("ACCEPTED: validated uid ok as non-guest\n"));
	  ok = True;
	}

    } /* not guest only */

  /* check for a user= with no password (guest connection) */
  if (!ok && USER(snum) && (*USER(snum)) && GUEST_OK(snum))
    {
      pstring user_list;
      char *auser;
      strcpy(user_list,USER(snum));
      auser = strtok(user_list," \t,");
      if (*auser == '@')
	DEBUG(2,("can't use group as guest user\n"));
      else
	{
	  if (auser && Get_Pwnam(auser))
	    {
	      ok = True;
	      strcpy(user,auser);
	      DEBUG(3,("ACCEPTED: user list username and guest ok\n"));
	    }
	  else
	    DEBUG(0,("Invalid user account %s??\n",auser?auser:"NULL"));
	}
      guest = True;
    }


  /* check for a normal guest connection */
  if (!ok && GUEST && (*GUEST) && GUEST_OK(snum))
    {
      if (Get_Pwnam(GUEST))
	{
	  strcpy(user,GUEST);
	  ok = True;
	  DEBUG(3,("ACCEPTED: guest account and guest ok\n"));
	}
      else
	DEBUG(0,("Invalid guest account %s??\n",GUEST?GUEST:"NULL"));
      guest = True;
    }

  /* don't let them in */
  if (!ok)
    {
      DEBUG(1,("%s invalid username/password for %s\n",timestring(),service));
      return(-1);
    }
  
  cnum = find_free_connection();
  if (cnum < 0)
    {
      DEBUG(0,("%s couldn't find free connection\n",timestring()));      
      return(-1);
    }

  /* find out some info about the user */
  pass = Get_Pwnam(user);

  if (pass == NULL)
    {
      DEBUG(0,("%s couldn't find account %s\n",timestring(),user)); 
      return(-1);
    }

  /* check number of connections */
  if (!claim_connection(lp_lockdir(),
			lp_servicename(snum),
			lp_max_connections(snum)))
    {
      DEBUG(1,("too many connections - rejected\n"));
      return(-4);
    }
  

  Connections[cnum].uid = pass->pw_uid;
  Connections[cnum].gid = pass->pw_gid;
  Connections[cnum].connect_num = cnum;
  Connections[cnum].service = snum;
  Connections[cnum].printer = (strncmp(dev,"LPT",3) == 0);
  Connections[cnum].dirptr = NULL;
  Connections[cnum].dirpath[0] = 0;
  strcpy(Connections[cnum].connectpath, PATH(snum));

  /* groups stuff added by ih */
  Connections[cnum].ngroups = 0 ;
  if (-1 == initgroups(pass->pw_name,pass->pw_gid))
    DEBUG(0,("Unable to initgroups!\n")) ;
  else
    Connections[cnum].ngroups =
      getgroups(NGROUPS,Connections[cnum].groups) ;

  Connections[cnum].open = True;

  add_session_user(user);

  {
    extern struct from_host Client_info;
    DEBUG(1,("%s %s (%s) connect to service %s as user %s (pid %d)\n",
	     timestring(),
	     Client_info.name,Client_info.addr,
	     lp_servicename(SNUM(cnum)),user,getpid()));
  }

  return(cnum);
}


/****************************************************************************
this prevents zombie child processes
****************************************************************************/
int sig_cld()
{
  static int depth = 0;
  if (depth != 0)
    {
      DEBUG(0,("ERROR: Recursion in sig_cld? Perhaps you need `#define USE_WAITPID'?\n"));
      return(0);
    }
  depth++;

  DEBUG(5,("got SIGCLD\n"));

#ifdef USE_WAITPID
  while (waitpid((pid_t)-1,(int *)NULL, WNOHANG) > 0);
#endif

  /* Stop zombies */
  /* Stevens, Adv. Unix Prog. says that on system V you must call
     wait before reinstalling the signal handler, because the kernel
     calls the handler from within the signal-call when there is a
     child that has exited. This would lead to an infinite recursion
     if done vice versa. */
        
  signal(SIGCLD, SIGNAL_CAST sig_cld);

#ifndef USE_WAITPID
  while (wait3((int *)NULL, WNOHANG, WAIT3_CAST NULL) > 0);
#endif
  depth--;
  return 0;
}

/****************************************************************************
  open the socket communication
****************************************************************************/
BOOL open_sockets(BOOL is_daemon,int port)
{
  extern int Client;
  int type, optlen;

  /* Check if we need to open a new socket by doing
     a harmless socket option on fd 0. If this fails
     we need to create one. */
  
  if (!is_daemon && 
      (getsockopt(0, SOL_SOCKET, SO_TYPE, (char *)&type, &optlen)==-1) &&
      (errno == ENOTSOCK))
    {
      DEBUG(0,("standard input is not a socket, will create a socket\n"));
      is_daemon = True;
    }

  if (is_daemon)
    {
      int s;
      int sockopt = 1;
      struct sockaddr addr;
      int in_addrlen = sizeof(addr);
       
      /* Stop zombies */
      signal(SIGCLD, SIGNAL_CAST sig_cld);

      /* open an incoming socket */
      s = open_socket_in(SOCK_STREAM, port);
      if (s == -1)
	return(False);

      if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&sockopt, sizeof(int)))
	{
	  DEBUG(0,("Error in setting socket option: %s",strerror(errno)));
	  close(s);
	  return False;
	}
  
      /* ready to listen */
      if (listen(s, 5) == -1) 
	{
	  DEBUG(0,("listen: %s",strerror(errno)));
	  close(s);
	  return False;
	}
  
  
      /* now accept incoming connections - forking a new process
	 for each incoming connection */
      DEBUG(2,("waiting for a connection\n"));
      while ((Client = accept(s,&addr,&in_addrlen)))
	{
	  if (Client == -1 && errno == EINTR)
	    continue;

	  if (Client == -1)
	    {
	      DEBUG(0,("accept: %s",strerror(errno)));
	      return False;
	    }

#ifdef NO_FORK_DEBUG
	  return True;
#else
	  if (Client != -1 && fork()==0)
	    {
	      signal(SIGPIPE, SIGNAL_CAST Abort);
	      signal(SIGCLD, SIGNAL_CAST SIG_DFL);

	      /* now set appropriate socket options */
	      {
		int one=1;
		setsockopt(Client,SOL_SOCKET,SO_KEEPALIVE,(char *)&one,sizeof(one));
	      }
	      return True; 
	    }
          close(Client); /* The parent doesn't need this socket */
#endif
	}
      close(s);
      return False;
    }
  else
    {
      /* We will abort gracefully when the client or remote system 
	 goes away */
      signal(SIGPIPE, SIGNAL_CAST Abort);
      Client = 0;
    }

  /* now set appropriate socket options */
  {
    int one=1;
    setsockopt(Client,SOL_SOCKET,SO_KEEPALIVE,(char *)&one,sizeof(one));
  }

  return True;
}

/****************************************************************************
  utility function called by unlocking requests
****************************************************************************/
BOOL do_unlock(int fnum,int cnum,uint32 count,uint32 offset,int *eclass,uint32 *ecode)
{
  BOOL ok = False;

  /* mask out possible negative numbers ?? */
  offset &= 0x7FFFFFFF;

  if (!lp_locking(SNUM(cnum)))
    return(True);

  if (Files[fnum].can_lock && OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    ok = fcntl_lock(Files[fnum].fd,F_SETLK,offset,count,F_UNLCK);
   
  if (!ok) {
    *eclass = ERRDOS;
    *ecode = ERRlock;
    return False;
  }
  return True; /* Did unlock */
}

/****************************************************************************
  utility function called by locking requests
****************************************************************************/
BOOL do_lock(int fnum,int cnum,uint32 count,uint32 offset,int *eclass,uint32 *ecode)
{
  BOOL ok = False;

  /* mask out possible negative numbers ?? */
  offset &= 0x7FFFFFFF;

  if (!lp_locking(SNUM(cnum)))
    return(True);

  if (count == 0) {
    *eclass = ERRDOS;
    *ecode = ERRnoaccess;
    return False;
  }

  if (Files[fnum].can_lock && OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    ok = fcntl_lock(Files[fnum].fd,F_SETLK,offset,count,
		    (Files[fnum].read_only?F_RDLCK:F_WRLCK));

  if (!ok) {
    *eclass = ERRDOS;
    *ecode = ERRlock;
    return False;
  }
  return True; /* Got lock */
}


/****************************************************************************
  utility function called to see if a file region is locked
****************************************************************************/
BOOL is_locked(int fnum,int cnum,uint32 count,uint32 offset)
{
  if (count == 0)
    return(False);

  /* mask out possible negative numbers ?? */
  offset &= 0x7FFFFFFF;

  if (!lp_locking(SNUM(cnum)))
    return(False);

  return(fcntl_lock(Files[fnum].fd,F_GETLK,offset,count,
		    (Files[fnum].read_only?F_RDLCK:F_WRLCK)));

  return(False);
}


/****************************************************************************
  reply to an special message 
****************************************************************************/
int reply_special(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = 4;
  int msg_type = CVAL(inbuf,0);
  int msg_flags = CVAL(inbuf,1);
  pstring name1="";
  pstring name2="";

  switch (msg_type)
    {
    case 0x81: /* session request */
      CVAL(outbuf,0) = 0x82;
      CVAL(outbuf,3) = 0;
      name_interpret(inbuf + 4,name1);
      name_interpret(inbuf + 4 + name_len(inbuf + 4),name2);
      DEBUG(2,("netbios connect: name1=%s name2=%s\n",name1,name2));      
      add_session_user(name2);
      break;
    case 0x85: /* session keepalive */
    default:
      return(0);
    }
  
  DEBUG(5,("%s init msg_type=0x%x msg_flags=0x%x\n",timestring(),msg_type,msg_flags));
  
  smb_setlen(outbuf,0);
  return(outsize);
}


/****************************************************************************
  create an error packet. Normally called using the ERROR() macro
****************************************************************************/
int error_packet(char *inbuf,char *outbuf,int error_class,uint32 error_code,int line)
{
  int outsize = set_message(outbuf,0,0,True);
  int cmd;
  cmd = CVAL(inbuf,smb_com);
  
  CVAL(outbuf,smb_rcls) = error_class;
  SSVAL(outbuf,smb_err,error_code);  
  
  DEBUG(3,("%s error packet at line %d cmd=%d (%s) eclass=%d ecode=%d\n",
	   timestring(),
	   line,
	   (int)CVAL(inbuf,smb_com),
	   smb_fn_name(CVAL(inbuf,smb_com)),
	   error_class,
	   error_code));

  if (errno != 0)
    DEBUG(3,("error string = %s\n",strerror(errno)));
  
  return(outsize);
}

struct
{
  int unixerror;
  int smbclass;
  int smbcode;
} unix_smb_errmap[] =
{
  {EPERM,ERRDOS,ERRnoaccess},
  {EACCES,ERRDOS,ERRnoaccess},
  {ENOENT,ERRDOS,ERRbadfile},
  {EIO,ERRHRD,ERRgeneral},
  {EBADF,ERRSRV,ERRsrverror},
  {EINVAL,ERRSRV,ERRsrverror},
  {EEXIST,ERRDOS,ERRfilexists},
  {ENFILE,ERRDOS,ERRnofids},
  {EMFILE,ERRDOS,ERRnofids},
  {ENOSPC,ERRHRD,ERRdiskfull},
#ifdef EDQUOT
  {EDQUOT,ERRHRD,ERRdiskfull},
#endif
  {EROFS,ERRHRD,ERRnowrite},
  {0,0,0}
};

/****************************************************************************
  create an error packet from errno
****************************************************************************/
int unix_error_packet(char *inbuf,char *outbuf,int def_class,uint32 def_code,int line)
{
  int eclass=def_class;
  int ecode=def_code;
  int i=0;

  while (unix_smb_errmap[i].smbclass != 0)
    {
      if (unix_smb_errmap[i].unixerror == errno)
	{
	  eclass = unix_smb_errmap[i].smbclass;
	  ecode = unix_smb_errmap[i].smbcode;
	  break;
	}
      i++;
    }

  return(error_packet(inbuf,outbuf,eclass,ecode,line));
}


/****************************************************************************
  create an error packet from a cached error.
****************************************************************************/
int cached_error_packet(char *inbuf,char *outbuf,int fnum,int line)
{
  write_bmpx_struct *wbmpx = Files[fnum].wbmpx_ptr;

  int32 eclass = wbmpx->wr_errclass;
  int32 err = wbmpx->wr_error;

  /* We can now delete the auxiliary struct */
  free((char *)wbmpx);
  Files[fnum].wbmpx_ptr = NULL;
  return error_packet(inbuf,outbuf,eclass,err,line);
}

/****************************************************************************
reply for the core protocol
****************************************************************************/
int reply_corep(char *outbuf, int choice)
{
  int outsize = set_message(outbuf,1,0,True);

  Protocol = PROT_CORE;

  return outsize;
}

/****************************************************************************
reply for the coreplus protocol
****************************************************************************/
int reply_coreplus(char *outbuf, int choice)
{
  int raw = (lp_readraw()?1:0) | (lp_writeraw()?2:0);
  int outsize = set_message(outbuf,13,0,True);
  SSVAL(outbuf,smb_vwv5,raw); /* tell redirector we support
				 readbraw and writebraw (possibly) */
  CVAL(outbuf,smb_flg) = 0x81; /* Reply, SMBlockread, SMBwritelock supported */
#ifdef TESTING
  SSVAL(outbuf,smb_vwv1,0x1); /* user level security, don't encrypt */	
#endif

  Protocol = PROT_COREPLUS;

  return outsize;
}


/****************************************************************************
reply for the lanman 1.0 protocol
****************************************************************************/
int reply_lanman1(char *outbuf, int choice)
{
  int raw = (lp_readraw()?1:0) | (lp_writeraw()?2:0);
  int outsize = set_message(outbuf,13,0,True);
  CVAL(outbuf,smb_flg) = 0x81; /* Reply, SMBlockread, SMBwritelock supported */
  SSVAL(outbuf,smb_vwv1,(lp_security()==SEC_USER)?1:0); /* user level security? don't encrypt */	
  SSVAL(outbuf,smb_vwv2,lp_maxxmit());
  SSVAL(outbuf,smb_vwv3,1); /* maxmux */
  SSVAL(outbuf,smb_vwv4,1);
  SSVAL(outbuf,smb_vwv5,raw); /* tell redirector we support
				 readbraw writebraw (possibly) */
  SIVAL(outbuf,smb_vwv6,getpid());
  SSVAL(outbuf,smb_vwv10, (TimeDiff() / 60) );

  put_dos_date(outbuf,smb_vwv8,time(NULL));

  Protocol = PROT_LANMAN1;

  return outsize;
}

/* List of supported protocols, most desired first */
struct {
  char *proto_name;
  int (*proto_reply_fn)(char *, int);
  int protocol_level;
} supported_protocols[] = {
#if TESTING
  {"LM1.2X002", reply_lanman1 , PROT_LANMAN1},
#endif
  {"LANMAN1.0", reply_lanman1 , PROT_LANMAN1},
  {"MICROSOFT NETWORKS 3.0", reply_lanman1 , PROT_LANMAN1},
  {"MICROSOFT NETWORKS 1.03",reply_coreplus,PROT_COREPLUS},/* core+ protocol */
  {"PC NETWORK PROGRAM 1.0", reply_corep, PROT_CORE}, /* core protocol */
  {NULL,NULL},
};

/****************************************************************************
  reply to a negprot
****************************************************************************/
int reply_negprot(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = set_message(outbuf,1,0,True);
  int index;
  int choice=-1;
  int protocol;
  char *p;
  int bcc = SVAL(smb_buf(inbuf),-2);

  /* Check for protocols, most desirable first */
  for (protocol = 0; supported_protocols[protocol].proto_name; protocol++)
    {
      p = smb_buf(inbuf)+1;
      index = 0;
      if (lp_maxprotocol() >= supported_protocols[protocol].protocol_level)
	while (p < (smb_buf(inbuf) + bcc))
	  { 
	    DEBUG(3,("protocol [%s]\n",p));
	    if (strequal(p,supported_protocols[protocol].proto_name))
	      choice = index;
	    index++;
	    p += strlen(p) + 2;
	  }
      if(choice != -1)
	break;
    }
  
  SSVAL(outbuf,smb_vwv0,choice);
  if(choice != -1)
    {
      outsize = supported_protocols[protocol].proto_reply_fn(outbuf,choice);
      DEBUG(2,("Chose protocol %s\n",supported_protocols[protocol].proto_name));
    }
  else {
    DEBUG(0,("No protocol supported !\n"));
  }
  
  DEBUG(5,("%s negprot index=%d\n",timestring(),choice));

  return(outsize);
}

/****************************************************************************
  parse a connect packet
****************************************************************************/
void parse_connect(char *buf,char *service,char *user,char *password,char *dev)
{
  char *p = smb_buf(buf) + 1;
  char *p2;
  
  p2 = strrchr(p,'\\');
  if (p2 == NULL)
    strcpy(service,p);
  else
    strcpy(service,p2+1);
  
  p += strlen(p) + 2;
  
  strcpy(password,p);

  p += strlen(p) + 2;

  strcpy(dev,p);
  
  *user = 0;
  p = strchr(service,'%');
  if (p != NULL)
    {
      *p = 0;
      strcpy(user,p+1);
    }
}

/****************************************************************************
  reply to a tcon
****************************************************************************/
int reply_tcon(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring service="";
  pstring user="";
  pstring password="";
  pstring dev="";
  int connection_num;
  int outsize = 0;
  int uid = SVAL(inbuf,smb_uid);
  int vuid;

  vuid = valid_uid(uid);
  
  parse_connect(inbuf,service,user,password,dev);

  connection_num = make_connection(service,user,password,dev,vuid);
  
  if (connection_num < 0)
    switch (connection_num)
      {
      case -4:
	return(ERROR(ERRSRV,ERRerror));
      case -3:
	return(ERROR(ERRDOS,ERRnoipc));
      case -2:
	return(ERROR(ERRSRV,ERRinvnetname));
      default:
	return(ERROR(ERRSRV,ERRbadpw));
      }
  
  outsize = set_message(outbuf,2,0,True);
  SSVAL(outbuf,smb_vwv0,MIN(lp_maxxmit(),BUFFER_SIZE)-4);
  SSVAL(outbuf,smb_vwv1,connection_num);
  SSVAL(outbuf,smb_tid,connection_num);
  
  DEBUG(3,("%s tcon service=%s user=%s cnum=%d\n",timestring(),service,user,connection_num));
  
  return(outsize);
}


/****************************************************************************
  reply to a tcon and X
****************************************************************************/
int reply_tcon_and_X(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring service="";
  pstring user="";
  pstring password="";
  pstring devicename="";
  int connection_num;
  int outsize = 0;
  int uid = SVAL(inbuf,smb_uid);
  int vuid;
  int smb_com2 = SVAL(inbuf,smb_vwv0);
  int smb_off2 = SVAL(inbuf,smb_vwv1);
  
  /* we might have to close an old one */
  if ((SVAL(inbuf,smb_vwv2) & 0x1) != 0)
    close_cnum(SVAL(inbuf,smb_tid));
  
  vuid = valid_uid(uid);
  
  {
    int passlen = SVAL(inbuf,smb_vwv3);
    char *path;
    char *p;
    memcpy(password,smb_buf(inbuf),passlen);
    password[passlen]=0;
    path = smb_buf(inbuf) + passlen;
    DEBUG(4,("parsing net-path %s, passlen=%d\n",path,passlen));
    strcpy(service,path+2);
    p = strchr(service,'\\');
    if (!p)
      return(ERROR(ERRSRV,ERRinvnetname));
    *p = 0;
    strcpy(service,p+1);
    p = strchr(service,'%');
    if (p)
      {
	*p++ = 0;
	strcpy(user,p);
      }
    strcpy(devicename,path + strlen(path) + 1);  
    DEBUG(4,("Got device type %s\n",devicename));
  }

  connection_num = make_connection(service,user,password,devicename,vuid);
  
  if (connection_num < 0)
    switch (connection_num)
      {
      case -4:
	return(ERROR(ERRSRV,ERRerror));
      case -3:
	return(ERROR(ERRDOS,ERRnoipc));
      case -2:
	return(ERROR(ERRSRV,ERRinvnetname));
      default:
	return(ERROR(ERRSRV,ERRbadpw));
      }
  
  outsize = set_message(outbuf,2,strlen(devicename)+1,True);
  
  DEBUG(3,("%s tconX service=%s user=%s cnum=%d\n",timestring(),service,user,connection_num));
  
  /* set the incoming and outgoing tid to the just created one */
  SSVAL(inbuf,smb_tid,connection_num);
  SSVAL(outbuf,smb_tid,connection_num);

  CVAL(outbuf,smb_vwv0) = smb_com2;
  SSVAL(outbuf,smb_vwv1,(chain_size + outsize)-4);

  strcpy(smb_buf(outbuf),devicename);

  if (smb_com2 != 0xFF)
    outsize += chain_reply(smb_com2,inbuf,inbuf+smb_off2+4,
			   outbuf,outbuf+outsize,
			   length,bufsize);

  return(outsize);
}

/****************************************************************************
  reply to an unknown type
****************************************************************************/
int reply_unknown(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int type;
  cnum = SVAL(inbuf,smb_tid);
  type = CVAL(inbuf,smb_com);
  
  DEBUG(0,("%s unknown command type (%s): cnum=%d type=%d (0x%X)\n",
	timestring(),
	smb_fn_name(type),
	cnum,type,type));
  
  return(ERROR(ERRSRV,ERRnosupport));
}


/****************************************************************************
reply to a session setup command
****************************************************************************/
int reply_sesssetup_and_X(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = 0;
  int sess_uid;
  int   smb_com2;
  int   smb_off2;       
  int   smb_bufsize;    
  int   smb_mpxmax;     
  int   smb_vc_num;     
  uint32   smb_sesskey;    
  int   smb_apasslen;   
  pstring smb_apasswd="";
  pstring smb_aname="";    
  BOOL guest=False;
  
  sess_uid = SVAL(inbuf,smb_uid);
  smb_com2 = CVAL(inbuf,smb_vwv0);
  smb_off2 = SVAL(inbuf,smb_vwv1);
  smb_bufsize = SVAL(inbuf,smb_vwv2);
  smb_mpxmax = SVAL(inbuf,smb_vwv3);
  smb_vc_num = SVAL(inbuf,smb_vwv4);
  smb_sesskey = IVAL(inbuf,smb_vwv5);
  smb_apasslen = SVAL(inbuf,smb_vwv7);

  strncpy(smb_apasswd,smb_buf(inbuf),smb_apasslen);
  strncpy(smb_aname,smb_buf(inbuf)+smb_apasslen,sizeof(smb_aname));

  DEBUG(3,("sesssetupX:name=[%s]\n",smb_aname));

  add_session_user(smb_aname);

  if (strequal(smb_aname,lp_guestaccount()) && (*smb_apasswd == 0))
    guest = True;
  
  /* now check if it's a valid username/password */
  if (!guest && !password_ok(smb_aname,smb_apasswd,lp_passwordlevel()))
#if GUEST_SESSION_SETUP
    {
      DEBUG(3,("Invalid username/password for %s - only guest services will be available\n",
	    smb_aname));
      strcpy(smb_aname,lp_guestaccount());
      guest = True;
    }
#else
    return(ERROR(ERRSRV,ERRbadpw));
#endif

  /* it's ok - setup a reply */
  outsize = set_message(outbuf,3,0,True);

  CVAL(outbuf,smb_vwv0) = smb_com2;
  SSVAL(outbuf,smb_vwv1,(chain_size+outsize)-4);

  if (guest)
    SSVAL(outbuf,smb_vwv2,1);

  /* register the name and uid as being validated, so further connections
     to a uid can get through without a password, on the same VC */
  register_uid(SVAL(inbuf,smb_uid),smb_aname,guest);
  
  if (smb_com2 != 0xFF)
    outsize += chain_reply(smb_com2,inbuf,inbuf+smb_off2+4,
			   outbuf,outbuf+outsize,
			   length,bufsize);

  maxxmit = MIN(lp_maxxmit(),smb_bufsize);

  return(outsize);
}

/****************************************************************************
  reply to a chkpth
****************************************************************************/
int reply_chkpth(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = 0;
  int cnum,mode;
  pstring name="";
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  
  strcpy(name,smb_buf(inbuf) + 1);
  unix_convert(name,cnum);

  mode = SVAL(inbuf,smb_vwv0);

  if (check_name(name,cnum))
    ok = directory_exist(name);

  if (!ok)
    return(ERROR(ERRDOS,ERRbadpath));
  
  outsize = set_message(outbuf,0,0,True);
  
  DEBUG(3,("%s chkpth %s cnum=%d mode=%d\n",timestring(),name,cnum,mode));
  
  return(outsize);
}


/****************************************************************************
  reply to a getatr
****************************************************************************/
int reply_getatr(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  int cnum;
  int outsize = 0;
  struct stat sbuf;
  BOOL ok = False;
  int mode=0;
  uint32 size=0;
  time_t mtime=0;
  
  cnum = SVAL(inbuf,smb_tid);

  strcpy(fname,smb_buf(inbuf) + 1);
  unix_convert(fname,cnum);

  if (check_name(fname,cnum))
    {
      if (stat(fname,&sbuf) == 0)
	{
	  mode = dos_mode(cnum,fname,&sbuf);
	  size = sbuf.st_size;
	  mtime = sbuf.st_mtime;
	  if (mode & aDIR)
	    size = 0;
	  ok = True;
	}
      else
	DEBUG(3,("stat of %s failed (%s)\n",fname,strerror(errno)));
    }
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRbadfile));
  
  outsize = set_message(outbuf,10,0,True);

  SSVAL(outbuf,smb_vwv0,mode);
  SIVAL(outbuf,smb_vwv1,mtime + GMT_TO_LOCAL * TimeDiff());
  SIVAL(outbuf,smb_vwv3,size);
  
  DEBUG(3,("%s getatr name=%s mode=%d size=%d\n",timestring(),fname,mode,size));
  
  return(outsize);
}


/****************************************************************************
  reply to a setatr
****************************************************************************/
int reply_setatr(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  int cnum;
  int outsize = 0;
  BOOL ok=False;
  int mode;
  time_t mtime;
  
  cnum = SVAL(inbuf,smb_tid);
  
  strcpy(fname,smb_buf(inbuf) + 1);
  unix_convert(fname,cnum);

  mode = SVAL(inbuf,smb_vwv0);
  mtime = IVAL(inbuf,smb_vwv1);
  
  if (directory_exist(fname))
    mode |= aDIR;
  if (check_name(fname,cnum))
    ok =  (chmod(fname,unix_mode(cnum,mode)) == 0);
  if (ok && mtime != 0)
    ok = set_filetime(fname,mtime + LOCAL_TO_GMT * TimeDiff());
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,0,0,True);
  
  DEBUG(3,("%s setatr name=%s mode=%d\n",timestring(),fname,mode));
  
  return(outsize);
}


/****************************************************************************
  return number of 1K blocks available on a path and total number 
****************************************************************************/
void disk_free(char *path,int *bsize,int *dfree,int *dsize)
{
  char *df_command = lp_dfree_command();
#ifdef USE_STATVFS
  struct statvfs fs;
#else
#ifdef ULTRIX
  struct fs_data fs;
#else
  struct statfs fs;
#endif
#endif

  /* possibly use system() to get the result */
  if (df_command && *df_command)
    {
      int ret;
      pstring syscmd;
      pstring outfile;

      sprintf(outfile,"/tmp/dfree.smb.%d",getpid());
      sprintf(syscmd,"%s %s > %s",df_command,path,outfile);
      ret = smbrun(syscmd);
      DEBUG(3,("Running the command `%s' gave %d\n",syscmd,ret));

      {
	FILE *f = fopen(outfile,"r");	
	*dsize = 0;
	*dfree = 0;
	*bsize = 1024;
	if (f)
	  {
	    fscanf(f,"%d %d %d",dsize,dfree,bsize);
	    fclose(f);
	  }
	else
	  DEBUG(0,("Can't open %s\n",outfile));
      }

      unlink(outfile);
    }
  else
    {
#ifdef STATFS4
      if (statfs(path,&fs,sizeof(fs),0) != 0)
#else
#ifdef USE_STATVFS
	if (statvfs(path, &fs))
#else
#ifdef STATFS3
	  if (statfs(path,&fs,sizeof(fs)) == -1)	 
#else
	  if (statfs(path,&fs) == -1)
#endif /* STATFS3 */
#endif /* USE_STATVFS */
#endif /* STATFS4 */
	    {
	      DEBUG(3,("dfree call failed code errno=%d\n",errno));
	      *bsize = 0;
	      *dfree = 0;
	      *dsize = 0;
	    }
	  else
	    {
#ifdef ULTRIX
	      *bsize = 1024;
	      *dfree = fs.fd_req.bfree;
	      *dsize = fs.fd_req.btot;
#else
#ifdef USE_STATVFS
	      *bsize = fs.f_frsize;
#else
#if defined(STATFS3)
	      /* eg: osf1 has f_fsize = fundamental filesystem block size, 
		 f_bsize = optimal transfer block size (MX: 94-04-19) */
              *bsize = fs.f_fsize;
#else
/* sun appears to be broken - (noticed by jra@vantive.com) */
#ifdef SUN
	      *bsize = 1024;
#else
	      *bsize = fs.f_bsize;
#endif
#endif /* STATFS3 */
#endif /* USE_STATVFS */
#ifdef STATFS4
	      *dfree = fs.f_bfree;
#else
	      *dfree = fs.f_bavail;
#endif /* STATFS4 */
	      *dsize = fs.f_blocks;
#endif /* ULTRIX */
#ifdef SCO
	      /* Workaround bug in SCO ODT 2.0 and 3.0.
	       * (It returns the filesystem block size in bsize,
	       * but returns bfree and blocks in 512-byte units!)
	       */
	      *bsize = 512;
#endif
    }
    }
  
  /* normalise for DOS usage */
  while (*dfree > WORDMAX || *dsize > WORDMAX) {
    *dfree /= 2;
    *dsize /= 2;
    *bsize *= 2;
  }
}

/****************************************************************************
  reply to a dskattr
****************************************************************************/
int reply_dskattr(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int outsize = 0;
  int dfree,dsize,bsize;
  
  cnum = SVAL(inbuf,smb_tid);
  
  disk_free(".",&bsize,&dfree,&dsize);
  
  outsize = set_message(outbuf,5,0,True);
  
  SSVAL(outbuf,smb_vwv0,dsize);
  SSVAL(outbuf,smb_vwv1,1);
  SSVAL(outbuf,smb_vwv2,bsize);
  SSVAL(outbuf,smb_vwv3,dfree);
  
  DEBUG(3,("%s dskattr cnum=%d dfree=%d\n",timestring(),cnum,dfree));
  
  return(outsize);
}


/****************************************************************************
  reply to a search
  Can be called from SMBsearch, SMBffirst or SMBfunique.
****************************************************************************/
int reply_search(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring mask="";
  pstring directory="";
  pstring fname="";
  int size,mode;
  time_t date;
  int dirtype;
  int cnum;
  int outsize = 0;
  int numentries = 0;
  BOOL finished = False;
  int maxentries;
  int i;
  char *p;
  BOOL ok = False;
  int status_len;
  char *path;
  char status[21];
  int dptr_num=-1;
  BOOL check_descend = False;
  BOOL expect_close = False;

  /* If we were called as SMBffirst then we must expect close. */
  if(CVAL(inbuf,smb_com) == SMBffirst)
    expect_close = True;
  
  cnum = SVAL(inbuf,smb_tid);

  outsize = set_message(outbuf,1,3,True);
  maxentries = SVAL(inbuf,smb_vwv0); 
  dirtype = SVAL(inbuf,smb_vwv1);
  path = smb_buf(inbuf) + 1;
  status_len = SVAL(smb_buf(inbuf),3 + strlen(path));

  
  /* dirtype &= ~aDIR; */
  
  DEBUG(5,("path=%s status_len=%d\n",path,status_len));

  
  if (status_len == 0)
    {
      pstring dir2;

      strcpy(directory,smb_buf(inbuf)+1);
      strcpy(dir2,smb_buf(inbuf)+1);
      unix_convert(directory,cnum);
      unix_format(dir2);

      if (!check_name(directory,cnum))
	return(ERROR(ERRDOS,ERRbadpath));

      p = strrchr(dir2,'/');
      if (p == NULL) 
	{strcpy(mask,dir2);*dir2 = 0;}
      else
	{*p = 0;strcpy(mask,p+1);}

      p = strrchr(directory,'/');
      if (!p) 
	*directory = 0;
      else
	*p = 0;

      if (strlen(directory) == 0)
	strcpy(directory,"./");
      memset(status,0,21);
      CVAL(status,0) = dirtype;
    }
  else
    {
      memcpy(status,smb_buf(inbuf) + 1 + strlen(path) + 4,21);
      memcpy(mask,status+1,11);
      mask[11] = 0;
      Connections[cnum].dirptr = dptr_fetch(status+12,&dptr_num);      
      dirtype = CVAL(status,0) & 0x1F;
      if (!Connections[cnum].dirptr)
	return(ERROR(ERRDOS,ERRnofiles));
      strncpy(Connections[cnum].dirpath,dptr_path(dptr_num),sizeof(pstring));
      strlower(mask);
    }
  

  DEBUG(5,("mask=%s directory=%s\n",mask,directory));
  CVAL(smb_buf(outbuf),0) = 5;
  
    {
      char *p = smb_buf(outbuf) + 3;
      
      ok = True;
      
      if (status_len == 0)
	{
	  if (!start_dir(cnum,directory))
	    ok = False;
	  else
	    {
	      /* close any old instances of this directory being open */
#if 0
	      dptr_closepath(directory,SVAL(inbuf,smb_pid));
#endif

	      dptr_num = dptr_create(Connections[cnum].dirptr,directory,
				     expect_close,SVAL(inbuf,smb_pid));
	      if (dptr_num < 0)
		return(ERROR(ERRDOS,ERRnofids));
	    }
	}

      DEBUG(3,("dptr_num is %d\n",dptr_num));

      if (ok)
	{
	  if ((dirtype & aVOLID) != 0)
	    {	  
	      memcpy(p,status,21);
	      make_dir_struct(p,"???????????",SERVICE(SNUM(cnum)),0,aVOLID,0);
	      dptr_fill(p+12,dptr_num);
	      if (dptr_zero(p+12))
		numentries = 1;
	      else
		numentries = 0;
	    }
	  else
	    {
	      DEBUG(8,("dirpath=<%s> dontdescend=<%s>\n",Connections[cnum].dirpath,lp_dontdescend(SNUM(cnum))));
	      if (in_list(Connections[cnum].dirpath,
			  lp_dontdescend(SNUM(cnum)),True))
		check_descend = True;

	      for (i=0;(i<maxentries) && !finished;i++)
		{
		  finished = 
		    !get_dir_entry(cnum,mask,dirtype,fname,&size,&mode,&date,check_descend);
		  if (!finished)
		    {
		      memcpy(p,status,21);
		      make_dir_struct(p,mask,fname,size,mode,date);
		      dptr_fill(p+12,dptr_num);
		      numentries++;
		    }
		  p += DIR_STRUCT_SIZE;
		}
	    }
	}
    }
#if 1
  if ((numentries == 0 && status_len != 0) || !ok)
#else
    if (finished || !ok)
#endif
      {
	CVAL(outbuf,smb_rcls) = ERRDOS;
	SSVAL(outbuf,smb_err,ERRnofiles);
	if (dptr_num >= 0)
	  dptr_demote(dptr_num);
      }

  /* If we were called as SMBffirst with smb_search_id == NULL
     and no entries were found then return error and close dirptr 
     (X/Open spec) */

  if(ok && expect_close && numentries == 0 && status_len == 0)
    {
      CVAL(outbuf,smb_rcls) = ERRDOS;
      SSVAL(outbuf,smb_err,ERRnofiles);
      /* Also close the dptr - we know it's gone */
      dptr_demote(dptr_num);
      dptr_close(dptr_num);
    }

  /* If we were called as SMBfunique, then we can close the dirptr now ! */
  if(CVAL(inbuf,smb_com) == SMBfunique)
    {
      dptr_demote(dptr_num);
      dptr_close(dptr_num);
    }

  SSVAL(outbuf,smb_vwv0,numentries);
  SSVAL(outbuf,smb_vwv1,3 + numentries * DIR_STRUCT_SIZE);
  CVAL(smb_buf(outbuf),0) = 5;
  SSVAL(smb_buf(outbuf),1,numentries*DIR_STRUCT_SIZE);
  
  outsize += DIR_STRUCT_SIZE*numentries;
  smb_setlen(outbuf,outsize - 4);
  
  if ((! *directory) && dptr_path(dptr_num))
    sprintf(directory,"(%s)",dptr_path(dptr_num));

  DEBUG(4,("%s %s mask=%s directory=%s cnum=%d dirtype=%d numentries=%d\n",
	timestring(),
	smb_fn_name(CVAL(inbuf,smb_com)), 
	mask,directory,cnum,dirtype,numentries));

  return(outsize);
}

/****************************************************************************
  reply to a fclose (stop directory search)
****************************************************************************/
int reply_fclose(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int outsize = 0;
  int status_len;
  char *path;
  char status[21];
  int dptr_num=-1;

  cnum = SVAL(inbuf,smb_tid);

  outsize = set_message(outbuf,1,0,True);
  path = smb_buf(inbuf) + 1;
  status_len = SVAL(smb_buf(inbuf),3 + strlen(path));

  
  if (status_len == 0)
    return(ERROR(ERRSRV,ERRsrverror));

  memcpy(status,smb_buf(inbuf) + 1 + strlen(path) + 4,21);

  if(!dptr_fetch(status+12,&dptr_num))
    return(ERROR(ERRSRV,ERRsrverror));
  
  /*  Close the dptr - we know it's gone */
  dptr_demote(dptr_num);
  dptr_close(dptr_num);

  SSVAL(outbuf,smb_vwv0,0);

  DEBUG(3,("%s search close cnum=%d\n",timestring(),cnum));

  return(outsize);
}

/****************************************************************************
print a file - called on closing the file
****************************************************************************/
void print_file(int fnum)
{
  pstring syscmd="";
  int cnum = Files[fnum].cnum;
  char *tempstr;

  tempstr = build_print_command(SNUM(cnum), PRINTCOMMAND(SNUM(cnum)), syscmd, Files[fnum].name);
  if (tempstr != NULL)
    {
      int ret = smbrun(syscmd);
      DEBUG(3,("Running the command `%s' gave %d\n",syscmd,ret));
    }
  else
    DEBUG(0,("Null print command?\n"));
}

/****************************************************************************
  reply to an open
****************************************************************************/
int reply_open(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  int cnum;
  int fnum = -1;
  int outsize = 0;
  int mode,share,fmode=0,attribute;
  int openmode = 0;
  BOOL ok = False;
  int size = 0;
  time_t mtime=0;
  int unixmode;
  int rmode;
  char *tmpname;
  int eclass = ERRDOS;
  int ecode = ERRbadfile;

  cnum = SVAL(inbuf,smb_tid);

  share = SVAL(inbuf,smb_vwv0);
  mode = share & 0xF;
#if 0
  attribute = SVAL(inbuf,smb_vwv1);
#else
  /* X/Open says to ignore this */
  attribute = aARCH;
#endif
  strcpy(fname,smb_buf(inbuf)+1);
  unix_convert(fname,cnum);
  
  rmode = mode;

  switch (mode)
    {
    case 0: 
      openmode = O_RDONLY; 
      break;
    case 1: 
      openmode = O_WRONLY; 
      break;
    case 0xF: 
      mode = 2;      
    case 2: 
      openmode = O_RDWR; 
      break;
    default:
      rmode = 0;
      openmode = O_RDONLY;
      break;
    }

  if ((openmode != O_RDONLY) && !CAN_WRITE(cnum))
    return(ERROR(ERRSRV,ERRaccess));
  
  ok = True;
  fnum = find_free_file();
  
  if (!check_name(fname,cnum))
    ok = False;      
  
  if (fnum >= 0 && ok)
    {
#if 0
      /* X/Open says fail if doesn't exist ?? even for WRONLY ?? */
      if (openmode != O_RDONLY) openmode |= O_CREAT;
#endif
      unixmode = unix_mode(cnum,attribute);
      
      Files[fnum].fd = open(fname,openmode,unixmode);	  
      
      /* Fix for files ending in '.' */
      if((Files[fnum].fd == -1) && (errno == ENOENT) && (strchr(fname,'.')==NULL))
	{
	  if ((tmpname = malloc(strlen(fname) + 2)))
	    {
	      strcpy(tmpname,fname);
	      strcat(tmpname,".");
	      Files[fnum].fd = open(tmpname,openmode,unixmode);
	      free(tmpname);
	    }
	}
      
      ok = (Files[fnum].fd >= 0);
      if (!ok)
	{
	  if (errno == ENOENT)
	    {
	      eclass = ERRDOS;
	      ecode = ERRbadfile;
	    }
	  else
	    {
	      eclass = ERRDOS;
	      ecode = ERRnoaccess;
	    }
	}
      
      if (Files[fnum].fd >= 0)
	{
	  struct stat sbuf;
	  if (fstat(Files[fnum].fd,&sbuf) == 0)
	    {
	      size = sbuf.st_size;
	      fmode = dos_mode(cnum,fname,&sbuf);
	      mtime = sbuf.st_mtime;
	      if (fmode & aDIR)
		{
		  DEBUG(3,("attempt to open a directory %s\n",fname));
		  close_file(Files[fnum].fd);
		  ok = False;
		}
	    }	      
	  if (ok)
	    {
	      num_files_open++;
	      Files[fnum].open = True;
	      Files[fnum].can_lock = True;
	      Files[fnum].read_only = (openmode == O_RDONLY);
	      Files[fnum].print_file = False;
	      Files[fnum].cnum = cnum;
	      strcpy(Files[fnum].name,fname);
	      Files[fnum].wbmpx_ptr = NULL;
	    }
	}
      else
	fnum = -1;
    }
  
  if ((fnum < 0) || !ok)
    {
      DEBUG(3,("Error opening file %s (%s)\n",fname,strerror(errno)));
      return(ERROR(eclass,ecode));
    }

  outsize = set_message(outbuf,7,0,True);
  SSVAL(outbuf,smb_vwv0,fnum);
  SSVAL(outbuf,smb_vwv1,fmode);
  SIVAL(outbuf,smb_vwv2,mtime + GMT_TO_LOCAL * TimeDiff());
  SIVAL(outbuf,smb_vwv4,size);
  SSVAL(outbuf,smb_vwv6,rmode);
  
  DEBUG(2,("opened file %s for %s\n",fname,openmode==O_RDONLY?"reading":"writing"));
  DEBUG(3,("%s open %s fd=%d fnum=%d cnum=%d mode=%d omode=%d (numopen=%d)\n",timestring(),fname,Files[fnum].fd,fnum,cnum,mode,openmode,num_files_open));
  
  return(outsize);
}


/****************************************************************************
  reply to an open and X
****************************************************************************/
int reply_open_and_X(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  int cnum = SVAL(inbuf,smb_tid);
  int fnum = -1;
  int outsize = 0;
  int openmode = 0;
  BOOL ok = False;
  int smb_com2 = CVAL(inbuf,smb_vwv0);
  int smb_off2 = SVAL(inbuf,smb_vwv1);
  int smb_mode = SVAL(inbuf,smb_vwv3);
  int smb_attr = SVAL(inbuf,smb_vwv5);
#if 0
  int smb_flags = SVAL(inbuf,smb_vwv2);
  int smb_sattr = SVAL(inbuf,smb_vwv4); 
  uint32 smb_time = IVAL(inbuf,smb_vwv6);
#endif
  int smb_ofun = SVAL(inbuf,smb_vwv8);
  BOOL file_existed = False;
  int unixmode;
  int size=0,fmode=0,mtime=0,rmode;
  char *tmpname;
  int eclass = ERRDOS;
  int ecode = ERRbadfile;

  DEBUG(3,("open inbuf = %d\n",(int)inbuf));

  /* XXXX we need to handle passed times, sattr and flags */


  strcpy(fname,smb_buf(inbuf));
  unix_convert(fname,cnum);
  
  rmode = smb_mode & 0x7;

  switch (smb_mode & 0x7)
    {
    case 0:       
      openmode = O_RDONLY; 
      break;
    case 1: 
      openmode = O_WRONLY; 
      break;
    case 0x7: 
    case 2: 
      rmode = 2;
      openmode = O_RDWR; 
      break;
    case 3:  /* map execute to read */
      openmode = O_RDONLY; 
      break;
    default:
      rmode = 0;
      openmode = O_RDONLY;
      break;
    }
  
  if ((openmode != O_RDONLY) && !CAN_WRITE(cnum))
    return(ERROR(ERRSRV,ERRaccess));

  /* now add create and trunc bits */
  if (smb_ofun & 0x10)
    openmode |= O_CREAT;
  if ((smb_ofun & 0x3) == 2)
    openmode |= O_TRUNC;
  
    {
      ok = True;
      fnum = find_free_file();

      if (!check_name(fname,cnum))
	  ok = False;      

      if (fnum >= 0 && ok)
	{
	  unixmode = unix_mode(cnum,smb_attr);

	  Files[fnum].fd = -1;

	  file_existed = file_exist(fname);

	  if ((smb_ofun & 0x3) == 0 && file_existed)
	    return(ERROR(ERRDOS,ERRfilexists));

	  Files[fnum].fd = open(fname,openmode,unixmode);


	  /* Fix for files ending in '.' */
	  if((Files[fnum].fd == -1) && (errno == ENOENT) && (strchr(fname,'.')==NULL))
	    {
	      if ((tmpname = malloc(strlen(fname) + 2)))
		{
		  strcpy(tmpname,fname);
		  strcat(tmpname,".");
		  Files[fnum].fd = open(tmpname,openmode,unixmode);
		  free(tmpname);
		}
	    }

	  ok = (Files[fnum].fd >= 0);
	  if (!ok)
	    {
	      if (errno == ENOENT)
		{
		  eclass = ERRDOS;
		  ecode = ERRbadfile;
		}
	      else
		{
		  eclass = ERRDOS;
		  ecode = ERRnoaccess;
		}
	    }

	  if (Files[fnum].fd >= 0)
	    {
	      struct stat sbuf;
	      if (fstat(Files[fnum].fd,&sbuf) == 0)
		{
		  size = sbuf.st_size;
		  fmode = dos_mode(cnum,fname,&sbuf);
		  mtime = sbuf.st_mtime;
		  if (fmode & aDIR)
		    {
		      close_file(Files[fnum].fd);
		      ok = False;
		    }
		}	 
	      if (ok)
		{
		  num_files_open++;
		  Files[fnum].open = True;
		  Files[fnum].can_lock = True;
		  Files[fnum].read_only = (openmode == O_RDONLY);
		  Files[fnum].print_file = False;
		  Files[fnum].cnum = cnum;
		  strcpy(Files[fnum].name,fname);
		  Files[fnum].wbmpx_ptr = NULL;
		}
	    }
	  else
	    fnum = -1;
	}
    }
  
  if ((fnum < 0) || !ok)
    return(ERROR(eclass,ecode));
  
  outsize = set_message(outbuf,15,0,True);
  CVAL(outbuf,smb_vwv0) = smb_com2;
  SSVAL(outbuf,smb_vwv1,(chain_size+outsize)-4);
  SSVAL(outbuf,smb_vwv2,fnum);
  SSVAL(outbuf,smb_vwv3,fmode);
  SIVAL(outbuf,smb_vwv4,mtime + GMT_TO_LOCAL * TimeDiff());
  SIVAL(outbuf,smb_vwv6,size);
  SSVAL(outbuf,smb_vwv8,rmode);

  {
    int smb_action = 0;
    
    if (file_existed && !(openmode & O_TRUNC)) smb_action = 1;
    if (!file_existed) smb_action = 2;
    if (file_existed && (openmode & O_TRUNC)) smb_action = 3;

    /* if they requested a opportunistic lock and they have it read-only
       then tell them they have it - this is wrong but it might work */
    if ((CVAL(inbuf,smb_flg) & 0x20) && openmode == O_RDONLY)
      smb_action |= 0x8000;
    
    SSVAL(outbuf,smb_vwv11,smb_action);
  }


  DEBUG(2,("opened file %s for %s\n",fname,openmode==O_RDONLY?"reading":"writing"));
  DEBUG(2,("%s openX %s fd=%d fnum=%d cnum=%d mode=%d omode=%d (numopen=%d)\n",
	timestring(),fname,Files[fnum].fd,fnum,cnum,
	smb_mode,openmode,num_files_open));

  chain_fnum = fnum;

  if (smb_com2 != 0xFF)
    outsize += chain_reply(smb_com2,inbuf,inbuf+smb_off2+4,
			   outbuf,outbuf+outsize,
			   length,bufsize);

  chain_fnum = -1;
  
  return(outsize);
}



/****************************************************************************
  reply to a mknew
****************************************************************************/
int reply_mknew(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  int cnum,com;
  int fnum = -1;
  int outsize = 0;
  int createmode;
  
  mode_t unixmode;
  BOOL ok = False;
  
  com = SVAL(inbuf,smb_com);
  cnum = SVAL(inbuf,smb_tid);

  createmode = SVAL(inbuf,smb_vwv0);
  strcpy(fname,smb_buf(inbuf)+1);
  unix_convert(fname,cnum);

  if (createmode & aVOLID)
    {
      DEBUG(0,("Attempt to create file (%s) with volid set - please report this\n",fname));
    }
  
  unixmode = unix_mode(cnum,createmode);
  
  if (com == SMBmknew && file_exist(fname))
    return(ERROR(ERRDOS,ERRfilexists));
    
  ok = True;
  fnum = find_free_file();

  if (!check_name(fname,cnum))
    ok = False;      

  if (fnum >= 0 && ok)
    {
      Files[fnum].fd = open(fname,O_RDWR | O_CREAT | O_TRUNC,unixmode);
      if (Files[fnum].fd >= 0)
	{
	  num_files_open++;
	  Files[fnum].open = True;
	  Files[fnum].can_lock = True;
	  Files[fnum].read_only = False;
	  Files[fnum].print_file = False;
	  Files[fnum].cnum = cnum;
	  strcpy(Files[fnum].name,fname);
	  /* Clear SMBwritebmpx flags */
	  Files[fnum].wbmpx_ptr = NULL;
	}
      else
	fnum = -1;
    }
  
  if (!ok || (fnum < 0))
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,1,0,True);
  SSVAL(outbuf,smb_vwv0,fnum);
  
  DEBUG(2,("new file %s\n",fname));
  DEBUG(3,("%s mknew %s fd=%d fnum=%d cnum=%d dmode=%d umode=%o\n",timestring(),fname,Files[fnum].fd,fnum,cnum,createmode,unixmode));
  
  return(outsize);
}

/****************************************************************************
  reply to a create temporary file
****************************************************************************/
int reply_ctemp(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  pstring fname2="";
  int cnum;
  int fnum = -1;
  int outsize = 0;
  int createmode;
  
  mode_t unixmode;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  createmode = SVAL(inbuf,smb_vwv0);
  sprintf(fname,"%s/TMXXXXXX",smb_buf(inbuf)+1);
  unix_convert(fname,cnum);
  
  unixmode = unix_mode(cnum,createmode);
  
  ok = True;
  fnum = find_free_file();

  if (!check_name(fname,cnum))
    ok = False;      

  if (fnum >= 0 && ok)
    {
      strcpy(fname2,mktemp(fname));
      Files[fnum].fd = open(fname2,O_RDWR | O_CREAT | O_TRUNC,unixmode);
      if (Files[fnum].fd >= 0)
	{
	  num_files_open++;
	  Files[fnum].open = True;
	  Files[fnum].cnum = cnum;
	  Files[fnum].can_lock = True;
	  Files[fnum].read_only = False;
	  Files[fnum].print_file = False;
	  strcpy(Files[fnum].name,fname2);
	  /* Clear SMBwritebmpx flags */
	  Files[fnum].wbmpx_ptr = NULL;
	}
      else
	fnum = -1;
    }
  
  if (!ok || (fnum < 0))
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,1,2 + strlen(fname2),True);
  SSVAL(outbuf,smb_vwv0,fnum);
  CVAL(smb_buf(outbuf),0) = 4;
  strcpy(smb_buf(outbuf) + 1,fname2);
  
  DEBUG(2,("created temp file %s\n",fname2));
  DEBUG(3,("%s ctemp %s fd=%d fnum=%d cnum=%d dmode=%d umode=%o\n",timestring(),fname2,Files[fnum].fd,fnum,cnum,createmode,unixmode));
  
  return(outsize);
}


/****************************************************************************
  reply to a unlink
****************************************************************************/
int reply_unlink(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = 0;
  pstring name="";
  BOOL isrootdir;
  int cnum;
  int dirtype;
  pstring directory="";
  pstring mask="";
  char *p;
  int count=0;
  int error = ERRnoaccess;
  
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);

  dirtype = SVAL(inbuf,smb_vwv0);
  
  strcpy(name,smb_buf(inbuf) + 1);
   
  DEBUG(3,("reply_unlink : %s\n",name));
   
  unix_convert(name,cnum);

  p = strrchr(name,'/');
  if (!p)
    {
      strcpy(directory,"./");
      strcpy(mask,name);
    }
  else
    {
      *p = 0;
      strcpy(directory,name);
      strcpy(mask,p+1);
    }

  isrootdir = (strequal(directory,"./") ||
	       strequal(directory,"."));
  
  
  {
    void *dirptr = NULL;
    struct DIRECT *dptr;

    if (check_name(directory,cnum))
      dirptr = (void *)opendir(directory);

    if (dirptr)
      {
	BOOL has_wild = (strchr(mask,'?') || strchr(mask,'*'));
	BOOL do_mangling;

	ok = True;
	error = ERRbadfile;
	
	while ((dptr = readdir(dirptr)))
	  {
	    pstring fname="";
	    int fmode;
	    
	    strcpy(fname,dptr->d_name);

	    do_mangling = ((has_wild || strchr(mask,MAGIC_CHAR)) && 
			   lp_manglednames());
	    
	    if ((strcmp(fname,mask) == 0) ||
		(name_convert(fname,dptr->d_name,do_mangling) &&
		 mask_match(fname,mask,!isrootdir,False,False)))
	      {
		struct stat sbuf;
		error = ERRnoaccess;
		sprintf(fname,"%s/%s",directory,dptr->d_name);
		if (stat(fname,&sbuf) != 0) continue;
		fmode = dos_mode(cnum,fname,&sbuf);
		if ((fmode & aDIR) != 0) continue;
		if ((fmode & aRONLY) != 0) continue;
		if (((fmode & ~dirtype) & (aHIDDEN | aSYSTEM)) != 0)
		  continue;		  
		unlink(fname);
		DEBUG(3,("reply_unlink : doing unlink on %s\n",fname));
		count++;
	      }		
	  }
	closedir(dirptr);
      }
  }
  
  if (!ok || count == 0)
    return(ERROR(ERRDOS,error));
  
  outsize = set_message(outbuf,0,0,True);
  
  return(outsize);
}

/****************************************************************************
   reply to a readbraw (core+ protocol)
****************************************************************************/
int reply_readbraw(char *inbuf, char *outbuf, int length, int bufsize)
{
  extern int Client;
  int cnum,maxcount,mincount,timeout,fnum;
  BOOL nonblocking = False;
  int nread = 0;
  int ret=0, nwritten;
  int startpos;
  static char *read_buf=NULL;
  static int read_buf_size=0;
 
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  startpos = IVAL(inbuf,smb_vwv1);
  maxcount = SVAL(inbuf,smb_vwv3);
  mincount = SVAL(inbuf,smb_vwv4);
  timeout = IVALS(inbuf,smb_vwv5);
  if(timeout == 0)
    nonblocking = True;

  /* ensure we don't overrun the packet size */
  maxcount = MIN(65535,maxcount);
  maxcount = MAX(mincount,maxcount);

  if (!read_buf || read_buf_size < maxcount+8)    
    {
      DEBUG(5,("expanding read_buf\n"));
      read_buf = (char *)Realloc(read_buf,maxcount+8);
      read_buf_size = maxcount+8;
    }

  if (!read_buf)
    DEBUG(0,("Out of memory in readbraw!\n"));
  
    {
      int toread = maxcount;

      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	{
	  if (!is_locked(fnum,cnum,toread,startpos))
	    {
	      /* try to get some from the read-prediction */
	      nread = 0;
	      if (Files[fnum].read_only)
		nread = read_predict(Files[fnum].fd,
				     startpos,
				     &read_buf[4],
				     toread);

	      toread -= nread;

	      if (toread > 0)
		{
		  /* NOTE: We are ignoring mincount! */
		  if (lseek(Files[fnum].fd,startpos+nread,SEEK_SET) != 
		      startpos+nread)
		    DEBUG(3,("Failed to seek to %d\n",startpos+nread));
		  else
		    nread += read_with_timeout(Files[fnum].fd,
					       &read_buf[4+nread],
					       toread, toread,
					       (long)timeout,
					       True);
		}
	    }
	}
    }
  
  DEBUG(3,("%s readbraw fnum=%d cnum=%d start=%d max=%d min=%d timeout=%d nread=%d\n",
	timestring(),
	fnum,cnum,startpos,
	maxcount,mincount,timeout,nread));
  
  if (nread < 0)
    nread = 0; /* X/Open says not to return an error - just send a 0 
		  buffer back */
  
  /* Set up a return message and send it directly to the SMB redirector.
	 Return -1 to signal no reply should be sent */
  /* Set up the Netbios message header */
  CVAL(read_buf,0) = 0;
  CVAL(read_buf,1) = 0;

  SSVAL(read_buf,2,nread);
  BSWP(read_buf+2,2); 

  nread += 4; /* Include header */
  nwritten = 0;

  log_out(read_buf,nread);

  while (nwritten < nread)
    {
      ret = write_socket(Client,read_buf+nwritten,nread - nwritten);
      if (ret <= 0)
	{
	  DEBUG(0,("Error writing %d bytes to client. %d. Exiting\n",nread-nwritten,ret));
	  close_sockets();
	  exit(1);
	}
      nwritten += ret;
    }

  DEBUG(5,("readbraw finished\n"));
  return -1;
}

/****************************************************************************
  reply to a lockread (core+ protocol)
****************************************************************************/
int reply_lockread(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,fnum;
  int nread = -1;
  char *data;
  int outsize = 0;
  uint32 startpos, numtoread;
  int eclass;
  uint32 ecode;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }
  numtoread = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  
  outsize = set_message(outbuf,5,3,True);
  numtoread = MIN(bufsize-outsize,numtoread);
  data = smb_buf(outbuf) + 3;
  
    {
      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	{
	  if(!do_lock( fnum, cnum, numtoread, startpos, &eclass, &ecode))
	    {
	      return (ERROR(eclass,ecode));
	    }
	  lseek(Files[fnum].fd,startpos,SEEK_SET);
	  nread = read(Files[fnum].fd,data,numtoread);
	  ok = True;	  
	}
    }
  
  if ((nread < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize += nread;
  SSVAL(outbuf,smb_vwv0,nread);
  SSVAL(outbuf,smb_vwv5,nread+3);
  SSVAL(smb_buf(outbuf),1,nread);
  
  DEBUG(3,("%s lockread fnum=%d cnum=%d num=%d nread=%d\n",timestring(),fnum,cnum,numtoread,nread));
  
  return(outsize);
}

/****************************************************************************
  reply to a read
****************************************************************************/
int reply_read(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtoread,fnum;
  int nread = -1;
  char *data;
  int startpos;
  int outsize = 0;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }
  numtoread = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  
  outsize = set_message(outbuf,5,3,True);
  numtoread = MIN(bufsize-outsize,numtoread);
  data = smb_buf(outbuf) + 3;
  
  {
    if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
      {
	if (is_locked(fnum,cnum,numtoread,startpos))
	  return(ERROR(ERRDOS,ERRlock));
	
	nread = 0;

	if (Files[fnum].read_only)
	  nread = read_predict(Files[fnum].fd,startpos,data,numtoread);
	
	if (nread < numtoread)
	  {
	    if (lseek(Files[fnum].fd,startpos+nread,SEEK_SET) !=
		startpos+nread)
	      DEBUG(3,("Failed to seek to %d\n",startpos+nread));
	    else
	      nread += read(Files[fnum].fd,data+nread,numtoread-nread);
	  }
	ok = (nread >= 0);
      }
  }
  
  
  if ((nread < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize += nread;
  SSVAL(outbuf,smb_vwv0,nread);
  SSVAL(outbuf,smb_vwv5,nread+3);
  SSVAL(smb_buf(outbuf),1,nread);
  
  DEBUG(3,("%s read fnum=%d cnum=%d num=%d nread=%d\n",timestring(),fnum,cnum,numtoread,nread));
  
  return(outsize);
}


/****************************************************************************
  reply to a read and X
****************************************************************************/
int reply_read_and_X(char *inbuf,char *outbuf,int length,int bufsize)
{
  int smb_com2 = CVAL(inbuf,smb_vwv0);
  int smb_off2 = SVAL(inbuf,smb_vwv1);
  int smb_fid = GETFNUM(inbuf,smb_vwv2);
  uint32 smb_offset = IVAL(inbuf,smb_vwv3);
  int smb_maxcnt = SVAL(inbuf,smb_vwv5);
  int smb_mincnt = SVAL(inbuf,smb_vwv6);
#if 0
  uint32 smb_timeout = SVAL(inbuf,smb_vwv7);
#endif
  int cnum;
  int nread = -1;
  char *data;
  int outsize = 0;
  BOOL ok = False;

  cnum = SVAL(inbuf,smb_tid);

  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(smb_fid))
	return (CACHED_ERROR(smb_fid));
    }
  outsize = set_message(outbuf,12,0,True);
  data = smb_buf(outbuf);
  
    {
      if (OPEN_FNUM(smb_fid) && (Files[smb_fid].cnum == cnum))
	{
	  if (is_locked(smb_fid,cnum,smb_maxcnt,smb_offset))
	    return(ERROR(ERRDOS,ERRlock));
	  lseek(Files[smb_fid].fd,smb_offset,SEEK_SET);
	  nread = read(Files[smb_fid].fd,data,smb_maxcnt);
	  ok = True;
	}
    }
  
  
  if ((nread < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize += nread;
  CVAL(outbuf,smb_vwv0) = smb_com2;
  SSVAL(outbuf,smb_vwv1,(outsize+chain_size)-4);
  SSVAL(outbuf,smb_vwv5,nread);
  SSVAL(outbuf,smb_vwv6,(int)(data - ((outbuf-chain_size)+4)));
  SSVAL(smb_buf(outbuf),-2,nread);
  
  DEBUG(3,("%s readX fnum=%d cnum=%d min=%d max=%d nread=%d com2=%d off2=%d\n",
	timestring(),smb_fid,cnum,
	smb_mincnt,smb_maxcnt,nread,smb_com2,smb_off2));

  chain_fnum = smb_fid;

  if (smb_com2 != 0xFF)
    outsize += chain_reply(smb_com2,inbuf,inbuf+smb_off2+4,
			   outbuf,outbuf+outsize,
			   length,bufsize);
  
  chain_fnum = -1;
  
  return(outsize);
}


/****************************************************************************
  reply to a writebraw (core+ or LANMAN1.0 protocol)
****************************************************************************/
int reply_writebraw(char *inbuf,char *outbuf,int length,int bufsize)
{
  int nwritten=0;
  int total_written=0;
  int numtowrite=0;
  int cnum,fnum;
  int outsize = 0;
  long startpos, timeout;
  char *read_buf=NULL;
  char *data=NULL;
  BOOL ok = False;
  BOOL write_through;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (!CAN_WRITE(cnum) && !Files[fnum].print_file)
    return(ERROR(ERRSRV,ERRaccess));
  
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return(CACHED_ERROR(fnum));
    }

  startpos = IVAL(inbuf,smb_vwv3);
  timeout = IVALS(inbuf,smb_vwv5);
  write_through = SVAL(inbuf,smb_vwv7) & 1;

  /* We have to deal with slightly different formats depending
     on whether we are using the core+ or lanman1.0 protocol */
  if(Protocol == PROT_COREPLUS) {
    numtowrite = SVAL(smb_buf(inbuf),-2);
    /* NB there is no length field here */
    data = smb_buf(inbuf);
  } else { /* LANMAN1.0 Protocol */
    numtowrite = SVAL(inbuf,smb_vwv10);
    data = inbuf + 4 + SVAL(inbuf, smb_vwv11);
  }
  
    {
      extern int Client;
      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	{
	  if (is_locked(fnum,cnum,numtowrite,startpos))
	    return(ERROR(ERRDOS,ERRlock));

	  if (lseek(Files[fnum].fd,startpos,SEEK_SET) != startpos)
	    DEBUG(0,("couldn't seek to %d in writebraw\n",startpos));

	  if (numtowrite>0)
	    nwritten = write_with_timeout(Files[fnum].fd,
					  data,numtowrite,
					  timeout);
#ifndef NO_FSYNC
	  if(write_through)
	    fsync(Files[fnum].fd);
#endif

	  ok = True;
  
	  DEBUG(3,("%s writebraw message 1 fnum=%d cnum=%d start=%d num=%d wrote=%d\n",
		timestring(),fnum,cnum,startpos,numtowrite,nwritten));
	  DEBUG(5,("writethrough=%d\n",write_through));

	  if ((nwritten < numtowrite) || !ok) 
	    return(UNIXERROR(ERRDOS,ERRnoaccess));

	  total_written = nwritten;

	  /* Return a message to the redirector to tell it
	     to send more bytes */
	  CVAL(outbuf,smb_com) = SMBwritebraw;

	  if(Protocol <= PROT_COREPLUS) {
	    outsize = set_message(outbuf,0,0,True);
	  } else { /* LANMAN1.0 Protocol */
	    outsize = set_message(outbuf,1,0,True);
	    SSVAL(outbuf,smb_vwv0,-1);  /* We don't support smb_remaining */
	  }
	  smb_setlen(outbuf, outsize-4);
	  send_smb(outbuf);

	  /* Now read the raw data into the buffer and write it */
	  if(read_smb_length(Client,inbuf,0) == -1) {
	    DEBUG(0,("Failed to read length of secondary writebraw\n"));
	    close_sockets();
	    exit(1);
	  }

	  /* Even though this is not an smb message, smb_len
	     returns the generic length of an smb message */
	  numtowrite = smb_len(inbuf);
	  if (numtowrite > 0)
	    {
	      read_buf = (char *)malloc(numtowrite);
	      if (!read_buf)
		DEBUG(0,("Out of memory in writebraw!\n"));

	      if(!read_data(Client,read_buf,numtowrite)) {
		DEBUG(0,("Failed to read data in secondary writebraw\n"));
		close_sockets();
		exit(1);
	      }


	      if (is_locked(fnum,cnum,numtowrite,startpos + total_written))
		return(ERROR(ERRDOS,ERRlock));

	      nwritten = write_with_timeout(Files[fnum].fd,
					    read_buf,numtowrite,
					    timeout);
	      if (read_buf) free(read_buf);

	      total_written += nwritten;
	    }

#ifndef NO_FSYNC
	  if(write_through)
	    fsync(Files[fnum].fd);
#endif
	  DEBUG(5,("writethrough=%d\n",write_through));

	  ok = True;

	  /* Set up outbuf to return the correct return */
	  outsize = set_message(outbuf,1,0,True);
	  CVAL(outbuf,smb_com) = SMBwritec;
	  SSVAL(outbuf,smb_vwv0,total_written);
	}
    }

  DEBUG(3,("%s writebraw extra fnum=%d cnum=%d start=%d num=%d wrote=%d\n",
		timestring(),fnum,cnum,startpos,numtowrite,total_written));

  return(outsize);
}

/****************************************************************************
  reply to a writeunlock (core+)
****************************************************************************/
int reply_writeunlock(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,fnum;
  int nwritten = -1;
  int outsize = 0;
  int fd;
  char *data;
  uint32 numtowrite,startpos;
  int eclass;
  uint32 ecode;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }  
  numtowrite = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  data = smb_buf(inbuf) + 3;
  
  fd = Files[fnum].fd;

  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    {
      if (is_locked(fnum,cnum,numtowrite,startpos))
	return(ERROR(ERRDOS,ERRlock));

      lseek(fd,startpos,SEEK_SET);

      /* The special X/Open SMB protocol handling of
	 zero length writes is *NOT* done for
	 this call */
      if(numtowrite == 0)
	nwritten = 0;
      else
	nwritten = write(fd,data,numtowrite);
      ok = True;
    }
  
  if(((nwritten == 0) && (numtowrite != 0))||(nwritten < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  if(!do_unlock(fnum, cnum, numtowrite, startpos, &eclass, &ecode))
    return(ERROR(eclass,ecode));

  outsize = set_message(outbuf,1,0,True);
  
  SSVAL(outbuf,smb_vwv0,nwritten);
  
  DEBUG(3,("%s writeunlock fnum=%d cnum=%d num=%d wrote=%d\n",timestring(),fnum,cnum,numtowrite,nwritten));
  
  return(outsize);
}

/****************************************************************************
  reply to a write
****************************************************************************/
int reply_write(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtowrite,fnum;
  int nwritten = -1;
  int outsize = 0;
  int startpos;
  int fd;
  char *data;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (!CAN_WRITE(cnum) && !Files[fnum].print_file)
    return(ERROR(ERRSRV,ERRaccess));

  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }  
  numtowrite = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  data = smb_buf(inbuf) + 3;
  
  fd = Files[fnum].fd;

  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    {
      if (is_locked(fnum,cnum,numtowrite,startpos))
	return(ERROR(ERRDOS,ERRlock));

      lseek(fd,startpos,SEEK_SET);

      /* X/Open SMB protocol says that if smb_vwv1 is
	 zero then the file size should be extended or
	 truncated to the size given in smb_vwv[2-3] */
      if(numtowrite == 0)
	nwritten = set_filelen(fd, startpos);
      else
	nwritten = write(fd,data,numtowrite);
      ok = True;
    }
  
  if(((nwritten == 0) && (numtowrite != 0))||(nwritten < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  outsize = set_message(outbuf,1,0,True);
  
  SSVAL(outbuf,smb_vwv0,nwritten);
  
  DEBUG(3,("%s write fnum=%d cnum=%d num=%d wrote=%d\n",timestring(),fnum,cnum,numtowrite,nwritten));
  
  return(outsize);
}


/****************************************************************************
  reply to a write and X
****************************************************************************/
int reply_write_and_X(char *inbuf,char *outbuf,int length,int bufsize)
{
  int smb_com2 = CVAL(inbuf,smb_vwv0);
  int smb_off2 = SVAL(inbuf,smb_vwv1);
  int fnum = GETFNUM(inbuf,smb_vwv2);
  uint32 smb_offset = IVAL(inbuf,smb_vwv3);
  int smb_dsize = SVAL(inbuf,smb_vwv10);
  int smb_doff = SVAL(inbuf,smb_vwv11);

  int cnum;
  int nwritten = -1;
  int outsize = 0;
  int fd;
  char *data;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (!CAN_WRITE(cnum) && !Files[fnum].print_file)
    return(ERROR(ERRSRV,ERRaccess));

  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  data = inbuf + 4 + smb_doff;
  
  fd = Files[fnum].fd;

  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    {
      if (is_locked(fnum,cnum,smb_dsize,smb_offset))
	return(ERROR(ERRDOS,ERRlock));

      lseek(fd,smb_offset,SEEK_SET);

      /* X/Open SMB protocol says that, unlike SMBwrite
	 if the length is zero then NO truncation is
	 done, just a write of zero. To truncate a file,
	 use SMBwrite. */
      if(smb_dsize == 0)
	nwritten = 0;
      else
	nwritten = write(fd,data,smb_dsize);
      ok = True;
    }
  
  if(((nwritten == 0) && (smb_dsize != 0))||(nwritten < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  outsize = set_message(outbuf,6,0,True);
  
  CVAL(outbuf,smb_vwv0) = smb_com2;
  SSVAL(outbuf,smb_vwv1,(outsize+chain_size)-4);
  SSVAL(outbuf,smb_vwv2,nwritten);
  
  DEBUG(3,("%s writeX fnum=%d cnum=%d num=%d wrote=%d\n",timestring(),fnum,cnum,smb_dsize,nwritten));

  chain_fnum = fnum;

  if (smb_com2 != 0xFF)
    outsize += chain_reply(smb_com2,inbuf,inbuf+smb_off2+4,
			   outbuf,outbuf+outsize,
			   length,bufsize);
  
  chain_fnum = -1;
  
  return(outsize);
}



/****************************************************************************
  reply to a lseek
****************************************************************************/
int reply_lseek(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,fnum;
  uint32 startpos;
  int32 res=-1;
  int mode;
  int outsize = 0;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }
  mode = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  
    {
      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	res = lseek(Files[fnum].fd,startpos,mode);
      ok = True;
    }
  
  
  if ((res < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,2,0,True);
  SIVAL(outbuf,smb_vwv0,res);
  
  DEBUG(3,("%s lseek fnum=%d cnum=%d pos=%d\n",timestring(),fnum,cnum,startpos));
  
  return(outsize);
}


/****************************************************************************
  reply to a flush
****************************************************************************/
int reply_flush(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum, fnum;
  int outsize = set_message(outbuf,0,0,True);

  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (Protocol >= PROT_LANMAN1 && OPEN_FNUM(fnum))
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

#ifndef NO_FSYNC
  if (fnum == 0xFFFF)
    {
      int i;
      for (i=0;i<MAX_OPEN_FILES;i++)
	if (OPEN_FNUM(i))
	  fsync(Files[i].fd);
    }
  else
    {
      if (OPEN_FNUM(fnum))
	fsync(Files[fnum].fd);
    }
#endif

  DEBUG(3,("%s flush fnum=%d\n",timestring(),fnum));
  return(outsize);
}

/****************************************************************************
  reply to a exit
****************************************************************************/
int reply_exit(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = set_message(outbuf,0,0,True);
  DEBUG(3,("%s exit\n",timestring()));
  
  return(outsize);
}


/****************************************************************************
run a file if it is a magic script
****************************************************************************/
void check_magic(int fnum,int cnum)
{
  DEBUG(5,("checking magic for %s\n",Files[fnum].name));

  if (!lp_magicscript(SNUM(cnum)) || !(*lp_magicscript(SNUM(cnum))))
    return;

  {
    char *p;
    if (!(p = strrchr(Files[fnum].name,'/')))
      p = Files[fnum].name;
    else
      p++;

    if (!strequal(lp_magicscript(SNUM(cnum)),p))
      return;
  }

  {
    int ret;
    pstring command;
    pstring magic_output;
    pstring fname;
    strcpy(fname,Files[fnum].name);

    if (lp_magicoutput(SNUM(cnum)))
      strcpy(magic_output,lp_magicoutput(SNUM(cnum)));
    else
      sprintf(magic_output,"%s.out",fname);

    sprintf(command,"%s %s > %s",
	    SHELL_PATH,
	    fname,
	    magic_output);

    ret = smbrun(command);
    DEBUG(3,("Invoking magic command %s gave %d\n",command,ret));
    unlink(fname);
  }
}



/****************************************************************************
  reply to a close
****************************************************************************/
int reply_close(char *inbuf,char *outbuf,int length,int bufsize)
{
  int fnum,cnum;
  int ret=0;
  int outsize = 0;
  BOOL ok = False;  
  time_t mtime;
  int32 eclass = 0, err = 0;

  outsize = set_message(outbuf,0,0,True);

  cnum = SVAL(inbuf,smb_tid);

  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (Protocol >= PROT_LANMAN1)
    {
      /* Make sure we have done the close first */
      if(HAS_CACHED_ERROR(fnum))
	{
	  eclass = Files[fnum].wbmpx_ptr->wr_errclass;
	  err = Files[fnum].wbmpx_ptr->wr_error;
	}
    }
  mtime = IVAL(inbuf,smb_vwv1);

    {
      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	{
	  ret = close_file(Files[fnum].fd);

	  /* try and set the date */
	  if (mtime == 0 || mtime == -1)
	    DEBUG(5,("not setting null date\n"));
	  else
	    if (!Files[fnum].read_only)
	      set_filetime(Files[fnum].name,mtime + LOCAL_TO_GMT * TimeDiff());

	  num_files_open--;
	  Files[fnum].open = False;
	  if(Files[fnum].wbmpx_ptr) 
	    {
	      free((char *)Files[fnum].wbmpx_ptr);
	      Files[fnum].wbmpx_ptr = NULL;
	    }
	  ok = True;

	  /* NT uses smbclose to start a print - weird */
	  if (Files[fnum].print_file)
	    print_file(fnum);

	  /* check for magic scripts */
	  check_magic(fnum,cnum);
	}
    }

  /* We have a cached error */
  if(eclass || err)
    return(ERROR(eclass,err));

  if ((ret < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  

  DEBUG(3,("%s close fd=%d fnum=%d cnum=%d ret=%d (numopen=%d)\n",
	timestring(),Files[fnum].fd,fnum,cnum,ret,num_files_open));
  
  return(outsize);
}


/****************************************************************************
  reply to a writeclose (Core+ protocol)
****************************************************************************/
int reply_writeclose(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtowrite,fnum;
  int nwritten = -1;
  int outsize = 0;
  int startpos;
  char *data;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (!CAN_WRITE(cnum) && !Files[fnum].print_file)
    return(ERROR(ERRSRV,ERRaccess));
  
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }
  numtowrite = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  data = smb_buf(inbuf) + 1;
  
  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    {
      if (is_locked(fnum,cnum,numtowrite,startpos))
	return(ERROR(ERRDOS,ERRlock));
      
      lseek(Files[fnum].fd,startpos,SEEK_SET);
      
      nwritten = write(Files[fnum].fd,data,numtowrite);
      ok = True;
      
      close_file(Files[fnum].fd);

      num_files_open--;
      Files[fnum].open = False;
      if(Files[fnum].wbmpx_ptr) 
	{
	  free((char *)Files[fnum].wbmpx_ptr);
	  Files[fnum].wbmpx_ptr = NULL;
	}
      
      if (Files[fnum].print_file)
	print_file(fnum);

      /* check for magic scripts */
      check_magic(fnum,cnum);
      
    }
  
  DEBUG(3,("%s writeclose fnum=%d cnum=%d num=%d wrote=%d (numopen=%d)\n",
	timestring(),fnum,cnum,numtowrite,nwritten,num_files_open));
  
  if ((nwritten <= 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,1,0,True);
  
  SSVAL(outbuf,smb_vwv0,nwritten);
  return(outsize);
}

/****************************************************************************
  reply to a lock
****************************************************************************/
int reply_lock(char *inbuf,char *outbuf,int length,int bufsize)
{
  int fnum,cnum;
  int outsize = set_message(outbuf,0,0,True);
  uint32 count,offset;
  int eclass;
  uint32 ecode;

  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  count = IVAL(inbuf,smb_vwv1);
  offset = IVAL(inbuf,smb_vwv3);

  if(!do_lock( fnum, cnum, count, offset, &eclass, &ecode))
    return (ERROR(eclass,ecode));

  DEBUG(3,("%s lock fd=%d fnum=%d cnum=%d ofs=%d cnt=%d\n",timestring(),Files[fnum].fd,fnum,cnum,offset,count));
  
  return(outsize);
}

/****************************************************************************
  reply to a unlock
****************************************************************************/
int reply_unlock(char *inbuf,char *outbuf,int length,int bufsize)
{
  int fnum,cnum;
  int outsize = set_message(outbuf,0,0,True);
  uint32 count,offset;
  int eclass;
  uint32 ecode;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  count = IVAL(inbuf,smb_vwv1);
  offset = IVAL(inbuf,smb_vwv3);

  if(!do_unlock(fnum, cnum, count, offset, &eclass, &ecode))
    return (ERROR(eclass,ecode));

  DEBUG(3,("%s unlock fd=%d fnum=%d cnum=%d ofs=%d cnt=%d\n",timestring(),Files[fnum].fd,fnum,cnum,offset,count));
  
  return(outsize);
}

/****************************************************************************
close all open files for a connection
****************************************************************************/
void close_open_files(int cnum)
{
  int i;
  for (i=0;i<MAX_OPEN_FILES;i++)
    if( Files[i].cnum == cnum && Files[i].open == True) {
      close_file(Files[i].fd);
      num_files_open--;
      Files[i].open = False;
      Files[i].cnum = -1;
    }
}



/****************************************************************************
close a cnum
****************************************************************************/
void close_cnum(int cnum)
{
  extern struct from_host Client_info;

  if (!OPEN_CNUM(cnum))
    {
      DEBUG(0,("Can't close cnum %d\n",cnum));
      return;
    }

  DEBUG(1,("%s %s (%s) closed connection to service %s\n",
	timestring(),
	Client_info.name,Client_info.addr,lp_servicename(SNUM(cnum))));

  yield_connection(lp_lockdir(),
		   lp_servicename(SNUM(cnum)),
		   lp_max_connections(SNUM(cnum)));

  close_open_files(cnum);
  Connections[cnum].open = False;
}

/****************************************************************************
  reply to a tdis
****************************************************************************/
int reply_tdis(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int outsize = set_message(outbuf,0,0,True);
  
  cnum = SVAL(inbuf,smb_tid);

  close_cnum(cnum);
  
  DEBUG(3,("%s tdis cnum=%d\n",timestring(),cnum));

  return outsize;
}


/****************************************************************************
  reply to a echo
****************************************************************************/
int reply_echo(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int smb_reverb = SVAL(inbuf,smb_vwv0);
  int seq_num;
  int data_len = smb_buflen(inbuf);
  int outsize = set_message(outbuf,1,data_len,True);

  cnum = SVAL(inbuf,smb_tid);

  if (cnum != 0xFFFF && !OPEN_CNUM(cnum))
    return(ERROR(ERRSRV,ERRinvnid));

  /* copy any incoming data back out */
  if (data_len > 0)
    memcpy(smb_buf(outbuf),smb_buf(inbuf),data_len);

  if (smb_reverb > 100)
    {
      DEBUG(0,("large reverb (%d)?? Setting to 100\n",smb_reverb));
      smb_reverb = 100;
    }

  for (seq_num =1 ; seq_num <= smb_reverb ; seq_num++)
    {
      SSVAL(outbuf,smb_vwv0,seq_num);

      smb_setlen(outbuf,outsize - 4);

      send_smb(outbuf);
    }

  DEBUG(3,("%s echo %d times cnum=%d\n",timestring(),smb_reverb,cnum));

  return -1;
}

/****************************************************************************
  reply to a printopen
****************************************************************************/
int reply_printopen(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring fname="";
  pstring fname2="";
  int cnum;
  int fnum = -1;
  int outsize = 0;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);

  if (!CAN_PRINT(cnum))
    return(ERROR(ERRDOS,ERRnoaccess));

  {
    pstring s;
    char *p;
    strncpy(s,smb_buf(inbuf)+1,sizeof(pstring));
    p = s;
    while (*p)
      {
	if (!(isalnum(*p) || strchr("._-",*p)))
	  *p = 'X';
	p++;
      }

    if (strlen(s) > 6) s[6] = 0;

    sprintf(fname,"%s.XXXXXX",s);  
  }

    {
      fnum = find_free_file();
      if (fnum >= 0)
	{
	  strcpy(fname2,mktemp(fname));
	  Files[fnum].fd = 
	    open(fname2,O_WRONLY | O_CREAT | O_TRUNC,unix_mode(cnum,0));
	  if (Files[fnum].fd >= 0)
	    {
	      strcpy(Files[fnum].name,fname2);
	      num_files_open++;
	      Files[fnum].open = True;
	      Files[fnum].cnum = cnum;
	      Files[fnum].read_only = False;
	      Files[fnum].can_lock = False;
	      Files[fnum].print_file = True;
	      /* Clear SMBwritebmpx flags */
	      Files[fnum].wbmpx_ptr = NULL;
	    }
	  else
	    fnum = -1;
	}
      ok = True;
    }
  
  if (!ok || (fnum < 0))
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,1,0,True);
  SSVAL(outbuf,smb_vwv0,fnum);
  
  DEBUG(3,("%s openprint %s fd=%d fnum=%d cnum=%d\n",timestring(),fname2,Files[fnum].fd,fnum,cnum));
  
  return(outsize);
}

/****************************************************************************
Build the print command in the supplied buffer. This means getting the
print command for the service and inserting the printer name and the
print file name. Return NULL on error, else the passed buffer pointer.
****************************************************************************/
static char *build_print_command(int snum, char *command, char *syscmd, char *filename)
{
   char *tstr;
   int iOffset;

   /* get the print command for the service. */
   tstr = command;
   if (tstr == NULL)
   {
      DEBUG(0,("No print command for service `%s'\n", SERVICE(snum)));
      return (NULL);
   }

   /* copy the command into the buffer for extensive meddling. */
   strcpy(syscmd, tstr);

   /* look for "%s" in the string. If there is no %s, we cannot print. */   
   if (!strstr(syscmd, "%s"))
   {
      DEBUG(0,("No placeholder for the filename in the print command "
              "for service %s!\n", SERVICE(snum)));
      return (NULL);
   }
   iOffset = strstr(syscmd, "%s") - syscmd;

   /* Now, some would say that we should recursively replace all %s with*/
   /* the filename and all %p with the printer name. However, it's too  */
   /* much hard work to avoid infinite recursion (filename = "%s"!), so */
   /* we just do it once. If you have anything complicated to do, you'll*/
   /* just have to write a script to do it. :-) */
   if (filename)
     {
       if (strlen(syscmd) + strlen(filename) - 2 > sizeof(pstring))
	 {
	   DEBUG(0,("Unable to print file \"%s\"!\n", filename));
	   DEBUG(0,("Constructed print command is too long for service %s.\n",
		 SERVICE(snum)));
	   return (NULL);
	 }
       else
	 replacestr(syscmd, filename, iOffset, 2);
     }

   /* Does the service have a printername? If not, make a fake and empty    */
   /* printer name. That way a %p is treated sanely (removed) if no printer */
   /* name was specified to replace it. This eventuality is logged.         */
   tstr = PRINTERNAME(snum);
   if (tstr == NULL || tstr[0] == '\0')
   {
      DEBUG(3,( "No printer name - using \"\" in service %s.\n", SERVICE(snum)));
      tstr = "";
   }

   if (!strstr(syscmd, "%p"))
   {
      /* Not having a placeholder when you DO have a printer name is    */
      /* treated as a possible problem and logged, but is not an error. */
      if (strlen(tstr) > 0)
         DEBUG(3,("No placeholder for the printer name in the print command "
                 "for service %s\n", SERVICE(snum)));
   }
   else
   {
     iOffset = strstr(syscmd, "%p") - syscmd;
      /* Not having a printer name when you DO have a placeholder is also */
      /* worth a warning... It is conceivable that the %p was in the      */
      /* filename. */
      if (tstr[0] == '\0')
         DEBUG(3,("Empty printer name given for service %s\n", SERVICE(snum)));

      if (strlen(syscmd) + strlen(tstr) - 2 > sizeof(pstring))
      {
         DEBUG(0,("Unable to print to printer \"%s\"!\n", tstr));
         DEBUG(0,("Constructed print command is too long for service %s.\n",
                 SERVICE(snum)));
         return (NULL);
      }
      else
         replacestr(syscmd, tstr, iOffset, 2);
   }

   return (syscmd);
}


/****************************************************************************
  reply to a printclose
****************************************************************************/
int reply_printclose(char *inbuf,char *outbuf,int length,int bufsize)
{
  int fnum,cnum;
  int ret=0;
  int outsize = set_message(outbuf,0,0,True);
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  if (!CAN_PRINT(cnum))
    return(ERROR(ERRDOS,ERRnoaccess));
  
    {
      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
  	{
  	  ret = close_file(Files[fnum].fd);
	  print_file(fnum);
  	}
      num_files_open--;
      Files[fnum].open = False;
      if(Files[fnum].wbmpx_ptr) 
	{
	  free((char *)Files[fnum].wbmpx_ptr);
	  Files[fnum].wbmpx_ptr = NULL;
        }
      ok = True;
    }

#ifdef IGNORE_PRINT_RETURN
  ret = 0;
#endif

  if ((ret != 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  
  DEBUG(3,("%s printclose fd=%d fnum=%d cnum=%d ret=%d\n",timestring(),Files[fnum].fd,fnum,cnum,ret));
  
  return(outsize);
}



/****************************************************************************
parse a lpq line

here is an example of lpq output under sunos

Only lines after the Rank get to this routine

Warning: no daemon present
Rank   Owner      Job  Files                                 Total Size
1st    tridge     148  README                                8096 bytes
****************************************************************************/
BOOL parse_lpq_entry(char *line,char *buf,BOOL printing)
{
  char owner[50]="";
  char fname[50]="";
  int jobid = 0;
  int size = 0;
  pstring dummy;
  
  if (sscanf(line,"%s%s%d%s%d",dummy,owner,&jobid,fname,&size) < 5) 
    return(False);

  put_dos_date2(buf,0,time(NULL));

  CVAL(buf,4) = (printing?2:3); /* status */
  SSVAL(buf,5,jobid); /* job number */
  SIVAL(buf,7,size); /* file size */
  CVAL(buf,11) = 0;
  strncpy(buf+12,owner,16); /* owner */

  return(True);
}



/****************************************************************************
  reply to a printqueue
****************************************************************************/
int reply_printqueue(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int outsize = set_message(outbuf,2,3,True);
  int max_count = SVAL(inbuf,smb_vwv0);
  int start_index = SVAL(inbuf,smb_vwv1);

  cnum = SVAL(inbuf,smb_tid);

/* allow checking the queue for anyone */
#if 0
  if (!CAN_PRINT(cnum))
    return(ERROR(ERRDOS,ERRnoaccess));
#endif

  SSVAL(outbuf,smb_vwv0,0);
  SSVAL(outbuf,smb_vwv1,0);
  CVAL(smb_buf(outbuf),0) = 1;
  SSVAL(smb_buf(outbuf),1,0);
  
  DEBUG(3,("%s printqueue cnum=%d start_index=%d max_count=%d\n",
	timestring(),cnum,start_index,max_count));

  if (!OPEN_CNUM(cnum) || !Connections[cnum].printer)
    {
      int i;
      cnum = -1;

      for (i=0;i<MAX_CONNECTIONS;i++)
	if (CAN_PRINT(i) && Connections[i].printer)
	  cnum = i;

      if (cnum == -1)
	for (i=0;i<MAX_CONNECTIONS;i++)
	  if (OPEN_CNUM(i))
	    cnum = i;

      if (!OPEN_CNUM(cnum))
	return(ERROR(ERRSRV,ERRinvnid));

      DEBUG(5,("connection not open or not a printer, using cnum %d\n",cnum));
    }

  if (!become_user(cnum))
    return(ERROR(ERRSRV,ERRinvnid));

  {
    char *lpq_command = lp_lpqcommand(SNUM(cnum));
    char *printername = PRINTERNAME(SNUM(cnum));

    if (printername && !(*printername))
      printername = NULL;
    
    if (!lpq_command || !(*lpq_command))
      {
	DEBUG(5,("No lpq command\n"));
	return(outsize);
      }
    
    {
      int ret;
      BOOL foundrank=False;
      pstring syscmd;
      pstring outfile;
      pstring line="";
      int lpq_index = 0;
      
      FILE *f;
      char *p = smb_buf(outbuf) + 3;
      
      sprintf(outfile,"/tmp/lpq.smb.%d",getpid());
      sprintf(syscmd,"%s %s > %s",
	      lpq_command,printername?printername:"lp",outfile);
      ret = smbrun(syscmd);
      DEBUG(3,("Running the command `%s' gave %d\n",syscmd,ret));
      
      f = fopen(outfile,"r");
      if (!f)
	return(outsize);
      
      /* read lines until we get one starting with "Rank" */
      while (!foundrank && !feof(f) && fgets(line,sizeof(pstring),f))
	if (strncmp(line,"Rank",strlen("Rank")) == 0)
	  foundrank = True;
      
      if (foundrank)
	{
	  int count=0;
	  int num_to_get = ABS(max_count);
	  int first = (max_count>0?start_index:start_index+max_count+1);
	  lpq_index = -1; /* assumes 0 based start_index */
	  
	  while (fgets(line,sizeof(pstring),f))
	    {
	      lpq_index++;
	      if (lpq_index < first)
		continue;
	      if (lpq_index >= first + num_to_get)
		break;
	      
	      /* parse it */
	      if (!parse_lpq_entry(line,p,(lpq_index == first)))
		break;
	      
	      p+= 28;
	      count++;
	    }	      
	  if (count > 0)
	    {
	      outsize = set_message(outbuf,2,28*count+3,False);	  
	      SSVAL(outbuf,smb_vwv0,count);
	      SSVAL(outbuf,smb_vwv1,(max_count>0?first+count:first-1));
	      CVAL(smb_buf(outbuf),0) = 1;
	      SSVAL(smb_buf(outbuf),1,28*count);
	    }
	  
	  DEBUG(3,("%d entries returned in queue\n",count));
	}
      else
	DEBUG(5,("Didn't find keyword Rank\n"));
      
      fclose(f);
      unlink(outfile);
    }
  }
  
  return(outsize);
}

/****************************************************************************
  reply to a printwrite
****************************************************************************/
int reply_printwrite(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtowrite,fnum;
  int outsize = set_message(outbuf,0,0,True);
  char *data;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);

  if (!CAN_PRINT(cnum))
    return(ERROR(ERRDOS,ERRnoaccess));

  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  numtowrite = SVAL(smb_buf(inbuf),1);
  data = smb_buf(inbuf) + 3;
  
  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    ok = (write(Files[fnum].fd,data,numtowrite) == numtowrite);
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  DEBUG(3,("%s printwrite fnum=%d cnum=%d num=%d\n",timestring(),fnum,cnum,numtowrite));
  
  return(outsize);
}


/****************************************************************************
  reply to a mkdir
****************************************************************************/
int reply_mkdir(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring directory="";
  int cnum;
  int outsize,ret=-1;
  
  strcpy(directory,smb_buf(inbuf) + 1);
  cnum = SVAL(inbuf,smb_tid);
  unix_convert(directory,cnum);
  
  if (check_name(directory,cnum))
    ret = mkdir(directory,unix_mode(cnum,aDIR));
  
  if (ret < 0)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,0,0,True);
  
  DEBUG(3,("%s mkdir %s cnum=%d ret=%d\n",timestring(),directory,cnum,ret));
  
  return(outsize);
}

/****************************************************************************
  reply to a rmdir
****************************************************************************/
int reply_rmdir(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring directory="";
  int cnum;
  int outsize = 0;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  strcpy(directory,smb_buf(inbuf) + 1);
  unix_convert(directory,cnum);
  
  if (check_name(directory,cnum))
    {
      dptr_closepath(directory,SVAL(inbuf,smb_pid));
      ok = (rmdir(directory) == 0);
      if (!ok)
	DEBUG(3,("couldn't remove directory %s : %s\n",
		 directory,strerror(errno)));
    }
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRbadpath));
  
  outsize = set_message(outbuf,0,0,True);
  
  DEBUG(3,("%s rmdir %s\n",timestring(),directory));
  
  return(outsize);
}

/****************************************************************************
  reply to a mv
****************************************************************************/
int reply_mv(char *inbuf,char *outbuf,int length,int bufsize)
{
  pstring oldname="";
  pstring newname="";
  int cnum;
  int outsize = 0;
  BOOL ok = False;
  
  strcpy(oldname,smb_buf(inbuf) + 1);
  strcpy(newname,smb_buf(inbuf) + 3 + strlen(oldname));
  
  cnum = SVAL(inbuf,smb_tid);
  unix_convert(oldname,cnum);
  unix_convert(newname,cnum);
  
  if (check_name(oldname,cnum) && check_name(newname,cnum))
    ok = (rename(oldname,newname) == 0);
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));
  
  outsize = set_message(outbuf,0,0,True);
  
  DEBUG(3,("%s mv %s to %s cnum=%d\n",timestring(),oldname,newname,cnum));
  
  return(outsize);
}

/****************************************************************************
  reply to a setdir
****************************************************************************/
int reply_setdir(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,snum;
  int outsize = 0;
  BOOL ok = False;
  pstring newdir="";
  
  cnum = SVAL(inbuf,smb_tid);
  
  snum = Connections[cnum].service;
  if (!CAN_SETDIR(snum))
    return(ERROR(ERRDOS,ERRnoaccess));
  
  strcpy(newdir,smb_buf(inbuf) + 1);
  strlower(newdir);
  
  if (strlen(newdir) == 0)
    ok = True;
  else
    {
      ok = directory_exist(newdir);
      if (ok)
	strcpy(Connections[cnum].connectpath,newdir);
    }
  
  if (!ok)
    return(ERROR(ERRDOS,ERRbadpath));
  
  outsize = set_message(outbuf,0,0,True);
  CVAL(outbuf,smb_reh) = CVAL(inbuf,smb_reh);
  
  DEBUG(3,("%s setdir %s cnum=%d\n",timestring(),newdir,cnum));
  
  return(outsize);
}

/****************************************************************************
  reply to a lockingX request
****************************************************************************/
int reply_lockingX(char *inbuf,char *outbuf,int length,int bufsize)
{
  int smb_com2 = CVAL(inbuf,smb_vwv0);
  int smb_off2 = SVAL(inbuf,smb_vwv1);
  int fnum = GETFNUM(inbuf,smb_vwv2);
  uint16 locktype = SVAL(inbuf,smb_vwv3);
#if 0
  uint32 timeout = IVAL(inbuf,smb_vwv4);
#endif
  uint16 num_locks = SVAL(inbuf,smb_vwv6);
  uint16 num_ulocks = SVAL(inbuf,smb_vwv7);
  uint32 count, offset;

  int cnum;
  int i;
  char *data;
  uint32 ecode, dummy2;
  int outsize, eclass, dummy1;

  cnum = SVAL(inbuf,smb_tid);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  data = smb_buf(inbuf);
  /* Data now points at the beginning of the list
     of smb_unlkrng structs */
  for(i = 0; i < (int)num_ulocks; i++) {
    count = IVAL(data,SMB_LKLEN_OFFSET(i));
    offset = IVAL(data,SMB_LKOFF_OFFSET(i));
    if(!do_unlock(fnum,cnum,count,offset,&eclass, &ecode))
      return ERROR(eclass,ecode);
  }

  /* Now do any requested locks */
  data += 10*num_ulocks;
  /* Data now points at the beginning of the list
     of smb_lkrng structs */
  for(i = 0; i < (int)num_locks; i++) {
    count = IVAL(data,SMB_LKLEN_OFFSET(i)); 
    offset = IVAL(data,SMB_LKOFF_OFFSET(i)); 
    if(!do_lock(fnum,cnum,count,offset, &eclass, &ecode))
      break;
  }

  /* If any of the above locks failed, then we must unlock
     all of the previous locks (X/Open spec). */
  if(i != num_locks) {
    for(; i >= 0; i--) {
      count = IVAL(data,SMB_LKLEN_OFFSET(i));  
      offset = IVAL(data,SMB_LKOFF_OFFSET(i)); 
      do_unlock(fnum,cnum,count,offset,&dummy1,&dummy2);
    }
    return ERROR(eclass,ecode);
  }

  outsize = set_message(outbuf,2,0,True);
  
  CVAL(outbuf,smb_vwv0) = smb_com2;
  SSVAL(outbuf,smb_vwv1,(outsize+chain_size)-4);
  
  DEBUG(3,("%s lockingX fnum=%d cnum=%d type=%d num_locks=%d num_ulocks=%d\n",
	timestring(),fnum,cnum,locktype,num_locks,num_ulocks));

  chain_fnum = fnum;

  if (smb_com2 != 0xFF)
    outsize += chain_reply(smb_com2,inbuf,inbuf+smb_off2+4,
			   outbuf,outbuf+outsize,
			   length,bufsize);
  
  chain_fnum = -1;
  
  return(outsize);
}

/****************************************************************************
  reply to a SMBreadbmpx (read block multiplex) request
****************************************************************************/
int reply_readbmpx(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtoread,fnum;
  int nread = -1;
  int total_read;
  char *data;
  int32 startpos;
  int32 timeout;
  int outsize, mincount, maxcount;
  BOOL ok = False, nonblocking = False;
  
  outsize = set_message(outbuf,8,0,True);

  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  startpos = IVAL(inbuf,smb_vwv1);
  maxcount = SVAL(inbuf,smb_vwv3);
  mincount = SVAL(inbuf,smb_vwv4);
  timeout = IVALS(inbuf,smb_vwv5);
  if(timeout == 0)
    nonblocking = True;

  numtoread = MIN(bufsize-outsize,maxcount);

  if(numtoread < maxcount) 
    {

    /* We don't have enough space to read all the
       requested bytes into the buffer, so, while
       this is true we must send our own constructed
       SMB messages back, leaving one to be sent
       back as the return from this function.
       We *should* do this asynchronously so
       other SMB requests get a look in, but I
       want to finish LANMAN1.0 support before the end
       of the century, so..... */

    int new_bufsize = MIN(lp_maxxmit(),BUFFER_SIZE);
    char *new_outbuf = (char *)malloc(new_bufsize);
    int num_thisread=0;
    int new_maxcount = maxcount - numtoread; /* Max we should read
						in these temp reads */
    int new_outsize, new_numtoread;
    int32 new_startpos = startpos;

    if(!new_outbuf) 
      {
	DEBUG(0,("Out of memory in reply_readmpx\n"));
	return(ERROR(ERRSRV,ERRnoresource));
      }

    /* Set up the smb header for this intermediate packet */

    memset(new_outbuf,0,smb_size);
    new_outsize = set_message(new_outbuf,8,0,True);
    CVAL(new_outbuf,smb_com) = CVAL(inbuf,smb_com);
    memcpy(new_outbuf+4,inbuf+4,4);
    CVAL(new_outbuf,smb_rcls) = SUCCESS;
    CVAL(new_outbuf,smb_reh) = 0;
    CVAL(new_outbuf,smb_flg) = 0x80; /* bit 7 set means a reply */
    SSVAL(new_outbuf,smb_err,SUCCESS);
    SSVAL(new_outbuf,smb_tid,SVAL(inbuf,smb_tid));
    SSVAL(new_outbuf,smb_pid,SVAL(inbuf,smb_pid));
    SSVAL(new_outbuf,smb_uid,SVAL(inbuf,smb_uid));
    SSVAL(new_outbuf,smb_mid,SVAL(inbuf,smb_mid));

    new_numtoread = MIN(new_bufsize-new_outsize,new_maxcount);

    /* Repeat until we have one read remaining */
    while( nread < new_maxcount)
      {
      ok = False;
      new_outsize = set_message(new_outbuf,8,0,True);
      data = smb_buf(new_outbuf);
  
	{
	  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	    {
	      if (is_locked(fnum,cnum,new_numtoread,new_startpos))
		{
		  free(new_outbuf);
		  return(ERROR(ERRDOS,ERRlock));
		}


	      lseek(Files[fnum].fd,new_startpos,SEEK_SET);
	      num_thisread = read_with_timeout(Files[fnum].fd,
					       data,mincount, new_numtoread,
					       timeout,
					       True);
	      ok = True;
	    }
	}
  
      if ((num_thisread < 0) || !ok) 
	{
	  free(new_outbuf);
	  return(UNIXERROR(ERRDOS,ERRnoaccess));
        }

      /* Update all the counts */
      nread += num_thisread;
      new_startpos += num_thisread;

      /* Now set up the buffer and send the message */
      new_outsize += num_thisread;
      SIVAL(new_outbuf,smb_vwv0, new_startpos - num_thisread);
      SSVAL(new_outbuf,smb_vwv2, maxcount);
      SSVAL(new_outbuf,smb_vwv3, -1); /* Don't support smb_remaining yet */
      SSVAL(new_outbuf,smb_vwv6, num_thisread);
      SSVAL(new_outbuf,smb_vwv7, smb_vwv9 - 4); /* Offset of data */
      SSVAL(new_outbuf,smb_vwv8, num_thisread); /* smb_bcc */
      smb_setlen(new_outbuf,new_outsize - 4);
      send_smb(new_outbuf);

      DEBUG(3,("%s intermediate readmpx fnum=%d cnum=%d num=%d start=%d nread=%d\n",
	    timestring(),fnum,cnum,new_numtoread,new_startpos - num_thisread,
	    num_thisread));

      }

    free(new_outbuf);
  }

  /* Here we know we have read nread bytes, so we have
     maxcount - nread bytes left to read. We know this
     will fit in outbuf because of the calculation we 
     did earlier.*/

  numtoread = maxcount - nread;
  startpos += nread;
  total_read = nread;
  nread = -1;
  ok = False;

  data = smb_buf(outbuf);
  
    {
      if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	{
	  if (is_locked(fnum,cnum,numtoread,startpos))
	    return(ERROR(ERRDOS,ERRlock));

	  lseek(Files[fnum].fd,startpos,SEEK_SET);
	  nread = read_with_timeout(Files[fnum].fd,
				    data,mincount, numtoread,
				    timeout,
				    True);
	  ok = True;
	}
    }
  
  
  if ((nread < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  /* Update totals */
  outsize += nread;
  total_read += nread;

  /* Setup the final packet */
  SIVAL(outbuf,smb_vwv0, startpos);
  SSVAL(outbuf,smb_vwv2, total_read);
  SSVAL(outbuf,smb_vwv3, -1); /* Don't support smb_remaining yet */
  SSVAL(outbuf,smb_vwv6, nread);
  SSVAL(outbuf,smb_vwv7, smb_vwv9 - 4); /* Offset of data */
  SSVAL(outbuf,smb_vwv8, nread); /* smb_bcc */
  
  DEBUG(3,("%s readmpx fnum=%d cnum=%d num=%d nread=%d\n",timestring(),fnum,cnum,numtoread,nread));
  
  return(outsize);
}

/****************************************************************************
  reply to a SMBwritebmpx (write block multiplex primary) request
****************************************************************************/
int reply_writebmpx(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtowrite,fnum;
  int nwritten = -1;
  int outsize = 0;
  int32 startpos, timeout;
  int tcount, write_through, smb_doff;
  int fd;
  char *data;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (!CAN_WRITE(cnum) && !Files[fnum].print_file)
    return(ERROR(ERRSRV,ERRaccess));

  if (Protocol >= PROT_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  tcount = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv3);
  timeout = IVALS(inbuf,smb_vwv5);
  write_through = SVAL(inbuf,smb_vwv7) & 1;
  numtowrite = SVAL(inbuf,smb_vwv10);
  smb_doff = SVAL(inbuf,smb_vwv11);

  data = inbuf + 4 + smb_doff;

  /* If this fails we need to send an SMBwriteC response,
     not an SMBwritebmpx - set this up now so we don't forget */
  CVAL(outbuf,smb_com) = SMBwritec;

  fd = Files[fnum].fd;

  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    {
      if (is_locked(fnum,cnum,numtowrite,startpos))
	return(ERROR(ERRDOS,ERRlock));

      lseek(fd,startpos,SEEK_SET);
      nwritten = write(fd,data,numtowrite);
#ifndef NO_FSYNC
      if(write_through)
	fsync(Files[fnum].fd);
#endif

      ok = True;
    }
  
  if(((nwritten == 0) && (numtowrite != 0))||(nwritten < 0) || !ok)
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  /* If the maximum to be written to this file
     is greater than what we just wrote then set
     up a secondary struct to be attached to this
     fd, we will use this to cache error messages etc. */
  if(tcount > nwritten) 
    {
    write_bmpx_struct *wbms;
    if(Files[fnum].wbmpx_ptr != NULL)
      wbms = Files[fnum].wbmpx_ptr; /* Use an existing struct */
    else
      wbms = (write_bmpx_struct *)malloc(sizeof(write_bmpx_struct));
    if(!wbms)
      {
	DEBUG(0,("Out of memory in reply_readmpx\n"));
	return(ERROR(ERRSRV,ERRnoresource));
      }
    wbms->wr_mode = write_through;
    wbms->wr_discard = False; /* No errors yet */
    wbms->wr_timeout = timeout;
    wbms->wr_total_written = nwritten;
    wbms->wr_errclass = 0;
    wbms->wr_error = 0;
    Files[fnum].wbmpx_ptr = wbms;
    }

  /* We are returning successfully, set the message type back to
     SMBwritebmpx */
  CVAL(outbuf,smb_com) = SMBwritec;
  
  outsize = set_message(outbuf,1,0,True);
  
  SSVAL(outbuf,smb_vwv0,-1); /* We don't support smb_remaining */
  
  DEBUG(3,("%s writebmpx fnum=%d cnum=%d num=%d wrote=%d\n",
	timestring(),fnum,cnum,numtowrite,nwritten));
  
  return(outsize);
}


/****************************************************************************
  reply to a SMBwritebs (write block multiplex secondary) request
****************************************************************************/
int reply_writebs(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,numtowrite,fnum;
  int nwritten = -1;
  int outsize = 0;
  int32 startpos;
  int tcount, write_through, smb_doff;
  int fd;
  char *data;
  write_bmpx_struct *wbms;
  BOOL ok = False;
  
  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  tcount = SVAL(inbuf,smb_vwv1);
  startpos = IVAL(inbuf,smb_vwv2);
  numtowrite = SVAL(inbuf,smb_vwv6);
  smb_doff = SVAL(inbuf,smb_vwv7);

  data = inbuf + 4 + smb_doff;

  /* We need to send an SMBwriteC response, not an SMBwritebs */
  CVAL(outbuf,smb_com) = SMBwritec;

  /* This fd should have an auxiliary struct attached,
     check that it does */
  wbms = Files[fnum].wbmpx_ptr;
  if(!wbms) /* smbd is confused - it can't do this ! */
    return(ERROR(ERRDOS,ERRnoaccess));

  /* If write through is set we can return errors, else we must
     cache them */
  write_through = wbms->wr_mode;

  /* Check for an earlier error */
  if(wbms->wr_discard)
    return -1; /* Just discard the packet */

  if (!CAN_WRITE(cnum) && !Files[fnum].print_file)
    {
      if(write_through)
	return(ERROR(ERRSRV,ERRaccess));
      /* Cache the error - it must be returned by another
	 file request */
      return(CACHE_ERROR(wbms,ERRSRV,ERRaccess));
    }
    
  fd = Files[fnum].fd;

  if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
    {
      if (is_locked(fnum,cnum,numtowrite,startpos))
	{
	  if(write_through)
	    {
	      /* We are returning an error - delete the aux struct */
	      if (wbms) free((char *)wbms);
	      Files[fnum].wbmpx_ptr = NULL;
	      return(ERROR(ERRDOS,ERRlock));
	    }
	  return CACHE_ERROR(wbms,ERRDOS,ERRlock);
	}

      lseek(fd,startpos,SEEK_SET);
      nwritten = write(fd,data,numtowrite);
#ifndef NO_FSYNC
      if(write_through)
	fsync(Files[fnum].fd);
#endif
      ok = True;
    }
  
  if(((nwritten == 0) && (numtowrite != 0))||(nwritten < 0) || !ok)
    {
      if(write_through)
	{
	  /* We are returning an error - we can delete the aux struct */
	  if (wbms) free((char *)wbms);
	  Files[fnum].wbmpx_ptr = NULL;
	  return(ERROR(ERRDOS,ERRnoaccess));
	}
      return(CACHE_ERROR(wbms,ERRDOS,ERRnoaccess));
    }

  /* Increment the total written, if this matches tcount
     we can discard the auxiliary struct (hurrah !) */
  wbms->wr_total_written += nwritten;
  if(wbms->wr_total_written >= tcount)
    {
    free((char *)wbms);
    Files[fnum].wbmpx_ptr = NULL;
    }
     
  outsize = set_message(outbuf,1,0,True);
  
  SSVAL(outbuf,smb_vwv0,nwritten);
  
  DEBUG(3,("%s writebmpx secondary fnum=%d cnum=%d num=%d wrote=%d\n",timestring(),fnum,cnum,numtowrite,nwritten));

  if(!write_through)
    return -1; /* Don't bother with a response */

  return(outsize);
}

/****************************************************************************
  reply to a SMBsetattrE
****************************************************************************/

int reply_setattrE(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,fnum;
  struct utimbuf unix_times;
  int outsize = 0;

  outsize = set_message(outbuf,0,0,True);

  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if(HAS_CACHED_ERROR(fnum))
    return (CACHED_ERROR(fnum));

    {
      if(!(OPEN_FNUM(fnum)) || (Files[fnum].cnum != cnum))
	{
	  return(ERROR(ERRDOS,ERRbadfid));
	}

      /* Convert the DOS times into dos times. Ignore create
	 time as UNIX can't set this.
      */

      unix_times.actime = make_unix_date(IVAL(inbuf,smb_vwv3));
      unix_times.modtime = make_unix_date(IVAL(inbuf,smb_vwv5));

      /* Set the date on this file */
      if(utime(Files[fnum].name, &unix_times))
	return(ERROR(ERRDOS,ERRnoaccess));

    }

  DEBUG(3,("%s reply_setattrE fnum=%d cnum=%d\n",timestring(),fnum,cnum));

  return(outsize);
}

/****************************************************************************
  reply to a SMBgetattrE
****************************************************************************/

int reply_getattrE(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum,fnum;
  struct stat sbuf;
  int outsize = 0;

  outsize = set_message(outbuf,11,0,True);

  cnum = SVAL(inbuf,smb_tid);
  fnum = GETFNUM(inbuf,smb_vwv0);
  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if(HAS_CACHED_ERROR(fnum))
    return (CACHED_ERROR(fnum));

    {
      int mode;

      if(!(OPEN_FNUM(fnum)) || (Files[fnum].cnum != cnum))
	return(ERROR(ERRDOS,ERRbadfid));

      /* Do an fstat on this file */
      if(fstat( Files[fnum].fd, &sbuf))
	return(UNIXERROR(ERRDOS,ERRnoaccess));

      mode = dos_mode(cnum,Files[fnum].name,&sbuf);

      /* Convert the times into dos times. Set create
	 date to be last modify date as UNIX doesn't save
	 this */
      put_dos_date2(outbuf,smb_vwv0,sbuf.st_mtime);
      put_dos_date2(outbuf,smb_vwv2,sbuf.st_atime);
      put_dos_date2(outbuf,smb_vwv4,sbuf.st_mtime);
      if (mode & aDIR)
	{
	  SIVAL(outbuf,smb_vwv6,0);
	  SIVAL(outbuf,smb_vwv8,0);
	}
      else
	{
	  SIVAL(outbuf,smb_vwv6,sbuf.st_size);
	  SIVAL(outbuf,smb_vwv8,sbuf.st_size);
	}
      SSVAL(outbuf,smb_vwv10, mode);

    }

  DEBUG(3,("%s reply_getattrE fnum=%d cnum=%d\n",timestring(),fnum,cnum));

  return(outsize);
}

/*
These flags determine some of the permissions required to do an operation 

Note that I don't set NEED_WRITE on some write operations because they
are used by some brain-dead clients when printing, and I don't want to
force write permissions on print services.
*/
#define AS_USER (1<<0)
#define NEED_WRITE (1<<1)
#define TIME_INIT (1<<2)

/* 
   define a list of possible SMB messages and their corresponding
   functions. Any message that has a NULL function is unimplemented -
   please feel free to contribute implementations!
*/
struct smb_message_struct
{
  int code;
  char *name;
  int (*fn)();
  int flags;
#if PROFILING
  unsigned long time;
#endif
}
 smb_messages[] = {

    /* CORE PROTOCOL */

   {SMBnegprot,"SMBnegprot",reply_negprot,0},
   {SMBgetatr,"SMBgetatr",reply_getatr,AS_USER},
   {SMBsetatr,"SMBsetatr",reply_setatr,AS_USER | NEED_WRITE},
   {SMBtcon,"SMBtcon",reply_tcon,0},
   {SMBchkpth,"SMBchkpth",reply_chkpth,AS_USER},
   {SMBsearch,"SMBsearch",reply_search,AS_USER},
   {SMBopen,"SMBopen",reply_open,AS_USER},

   /* note that SMBmknew and SMBcreate are deliberately overloaded */   
   {SMBcreate,"SMBcreate",reply_mknew,AS_USER | NEED_WRITE},
   {SMBmknew,"SMBmknew",reply_mknew,AS_USER | NEED_WRITE}, 

   {SMBunlink,"SMBunlink",reply_unlink,AS_USER | NEED_WRITE},
   {SMBread,"SMBread",reply_read,AS_USER},
   {SMBwrite,"SMBwrite",reply_write,AS_USER},
   {SMBclose,"SMBclose",reply_close,AS_USER},
   {SMBtdis,"SMBtdis",reply_tdis,AS_USER},
   {SMBmkdir,"SMBmkdir",reply_mkdir,AS_USER | NEED_WRITE},
   {SMBrmdir,"SMBrmdir",reply_rmdir,AS_USER | NEED_WRITE},
   {SMBdskattr,"SMBdskattr",reply_dskattr,AS_USER},
   {SMBmv,"SMBmv",reply_mv,AS_USER | NEED_WRITE},

   /* this is a Pathworks specific call, allowing the 
      changing of the root path */
   {pSETDIR,"pSETDIR",reply_setdir,AS_USER}, 

   {SMBlseek,"SMBlseek",reply_lseek,AS_USER},
   {SMBflush,"SMBflush",reply_flush,AS_USER},
   {SMBctemp,"SMBctemp",reply_ctemp,AS_USER | NEED_WRITE},
   {SMBexit,"SMBexit",reply_exit,0},
   {SMBsplopen,"SMBsplopen",reply_printopen,AS_USER},
   {SMBsplclose,"SMBsplclose",reply_printclose,AS_USER},
   {SMBsplretq,"SMBsplretq",reply_printqueue,0},
   {SMBsplwr,"SMBsplwr",reply_printwrite,AS_USER},
   {SMBlock,"SMBlock",reply_lock,AS_USER},
   {SMBunlock,"SMBunlock",reply_unlock,AS_USER},
   
   /* CORE+ PROTOCOL FOLLOWS */
   
   {SMBreadbraw,"SMBreadbraw",reply_readbraw,AS_USER},
   {SMBwritebraw,"SMBwritebraw",reply_writebraw,AS_USER},
   {SMBwriteclose,"SMBwriteclose",reply_writeclose,AS_USER},
   {SMBlockread,"SMBlockread",reply_lockread,AS_USER},
   {SMBwriteunlock,"SMBwriteunlock",reply_writeunlock,AS_USER | NEED_WRITE},
   
   /* LANMAN1.0 PROTOCOL FOLLOWS */
   
   {SMBreadBmpx,"SMBreadBmpx",reply_readbmpx,AS_USER},
   {SMBreadBs,"SMBreadBs",NULL,AS_USER},
   {SMBwriteBmpx,"SMBwriteBmpx",reply_writebmpx,AS_USER | NEED_WRITE},
   {SMBwriteBs,"SMBwriteBs",NULL,AS_USER | NEED_WRITE},
   {SMBwritec,"SMBwritec",NULL,AS_USER | NEED_WRITE},
   {SMBsetattrE,"SMBsetattrE",reply_setattrE,AS_USER | NEED_WRITE},
   {SMBgetattrE,"SMBgetattrE",reply_getattrE,AS_USER},
   {SMBtrans,"SMBtrans",NULL,AS_USER},
   {SMBtranss,"SMBtranss",NULL,AS_USER},
   {SMBioctl,"SMBioctl",NULL,AS_USER},
   {SMBioctls,"SMBioctls",NULL,AS_USER},
   {SMBcopy,"SMBcopy",NULL,AS_USER},
   {SMBmove,"SMBmove",NULL,AS_USER | NEED_WRITE},
   
   {SMBecho,"SMBecho",reply_echo,0},
   {SMBopenX,"SMBopenX",reply_open_and_X,AS_USER},
   {SMBreadX,"SMBreadX",reply_read_and_X,AS_USER},
   {SMBwriteX,"SMBwriteX",reply_write_and_X,AS_USER},
   {SMBsesssetupX,"SMBsesssetupX",reply_sesssetup_and_X,0},
   {SMBtconX,"SMBtconX",reply_tcon_and_X,0},
   {SMBlockingX,"SMBlockingX",reply_lockingX,AS_USER},
   
   {SMBffirst,"SMBffirst",reply_search,AS_USER},
   {SMBfunique,"SMBfunique",reply_search,AS_USER},
   {SMBfclose,"SMBfclose",reply_fclose,AS_USER},

   /* messaging routines */
#if 0
   {SMBsends,"SMBsends",test_sends,AS_USER},
   {SMBsendstrt,"SMBsendstrt",string_start,AS_USER},
   {SMBsendend,"SMBsendend",string_stop,AS_USER},
   {SMBsendtxt,"SMBsendtxt",string_text,AS_USER},
#else
   {SMBsends,"SMBsends",NULL,AS_USER},
   {SMBsendstrt,"SMBsendstrt",NULL,AS_USER},
   {SMBsendend,"SMBsendend",NULL,AS_USER},
   {SMBsendtxt,"SMBsendtxt",NULL,AS_USER},
#endif   
   /* NON-IMPLEMENTED PARTS OF THE CORE PROTOCOL */
   
   {SMBsendb,"SMBsendb",NULL,AS_USER},
   {SMBfwdname,"SMBfwdname",NULL,AS_USER},
   {SMBcancelf,"SMBcancelf",NULL,AS_USER},
   {SMBgetmac,"SMBgetmac",NULL,AS_USER}
 };

/****************************************************************************
return a string containing the function name of a SMB command
****************************************************************************/
char *smb_fn_name(int type)
{
  static char *unknown_name = "SMBunknown";
  static int num_smb_messages = 
    sizeof(smb_messages) / sizeof(struct smb_message_struct);
  int match;

  for (match=0;match<num_smb_messages;match++)
    if (smb_messages[match].code == type)
      break;

  if (match == num_smb_messages)
    return(unknown_name);

  return(smb_messages[match].name);
}


/****************************************************************************
do a switch on the message type, and return the response size
****************************************************************************/
int switch_message(int type,char *inbuf,char *outbuf,int size,int bufsize)
{
  int outsize = 0;
  static int num_smb_messages = 
    sizeof(smb_messages) / sizeof(struct smb_message_struct);
  int match;
#if PROFILING
  struct timeval msg_start_time;
  struct timeval msg_end_time;
  static unsigned long total_time = 0;

  gettimeofday(&msg_start_time,NULL);
#endif

  errno = 0;
  last_message = type;

  for (match=0;match<num_smb_messages;match++)
    if (smb_messages[match].code == type)
      break;

  if (match == num_smb_messages)
    {
      DEBUG(0,("Unknown message type %d!\n",type));
      outsize = reply_unknown(inbuf,outbuf,size,bufsize);
    }
  else
    {
      DEBUG(3,("switch message %s\n",smb_messages[match].name));
      if (smb_messages[match].fn)
	{
	  int cnum = SVAL(inbuf,smb_tid);
	  int flags = smb_messages[match].flags;

	  /* does this protocol need to be run as root? */
	  if (!(flags & AS_USER) && (done_become_user != -1))
	    unbecome_user();

	  /* does this protocl need to be run as the connected user? */
	  if ((flags & AS_USER) && !become_user(cnum))
	    return(ERROR(ERRSRV,ERRinvnid));

	  /* does it need write permission? */
	  if ((flags & NEED_WRITE) && !CAN_WRITE(cnum))
	    return(ERROR(ERRSRV,ERRaccess));

	  outsize = smb_messages[match].fn(inbuf,outbuf,size,bufsize);
	}
      else
	outsize = reply_unknown(inbuf,outbuf,size,bufsize);
    }

#if PROFILING
  gettimeofday(&msg_end_time,NULL);
  if (!(smb_messages[match].flags & TIME_INIT))
    {
      smb_messages[match].time = 0;
      smb_messages[match].flags |= TIME_INIT;
    }
  {
    unsigned long this_time =     
      (msg_end_time.tv_sec - msg_start_time.tv_sec)*1e6 +
	(msg_end_time.tv_usec - msg_start_time.tv_usec);
    smb_messages[match].time += this_time;
    total_time += this_time;
  }
  DEBUG(2,("TIME %s  %d usecs   %g pct\n",
	   smb_fn_name(type),smb_messages[match].time,
	(100.0*smb_messages[match].time) / total_time));
#endif

  return(outsize);
}


/****************************************************************************
construct a chained reply and add it to the already made reply

inbuf points to the original message start.
inbuf2 points to the smb_wct part of the secondary message
type is the type of the secondary message
outbuf points to the original outbuffer
outbuf2 points to the smb_wct field of the new outbuffer
size is the total length of the incoming message (from inbuf1)
bufsize is the total buffer size

return how many bytes were added to the response
****************************************************************************/
int chain_reply(int type,char *inbuf,char *inbuf2,char *outbuf,char *outbuf2,int size,int bufsize)
{
  int outsize = 0;
  char *ibuf,*obuf;
  static int chain_size_in = 0;
  static BOOL in_chain = False;
  static char *last_inbuf=NULL,*last_outbuf=NULL;
  BOOL was_inchain = in_chain;

  chain_size += ((int)outbuf2 - (int)outbuf) - smb_wct;
  if (was_inchain)
    {
      inbuf = last_inbuf;
      outbuf = last_outbuf;
    }
  else
    chain_size_in = 0;

  in_chain = True;
  last_inbuf = inbuf;
  last_outbuf = outbuf;


  inbuf2 -= chain_size_in;

  chain_size_in += ((int)inbuf2 - (int)inbuf) - smb_wct;


  /* allocate some space for the in and out buffers of the chained message */
  ibuf = (char *)malloc(size);
  obuf = (char *)malloc(bufsize);

  if (!ibuf || !obuf)
    {
      DEBUG(0,("Out of memory in chain reply\n"));
      return(ERROR(ERRSRV,ERRnoresource));
    }

  /* create the in buffer */
  memcpy(ibuf,inbuf,smb_wct);
  memcpy(ibuf+smb_wct,inbuf2,size-(int)(inbuf2-inbuf));
  CVAL(ibuf,smb_com) = type;

  /* create the out buffer */
  memset(obuf,0,smb_size);

  set_message(obuf,0,0,True);
  CVAL(obuf,smb_com) = CVAL(ibuf,smb_com);
  
  memcpy(obuf+4,ibuf+4,4);
  CVAL(obuf,smb_rcls) = SUCCESS;
  CVAL(obuf,smb_reh) = 0;
  CVAL(obuf,smb_flg) = 0x80; /* bit 7 set means a reply */
  SSVAL(obuf,smb_err,SUCCESS);
  SSVAL(obuf,smb_tid,SVAL(inbuf,smb_tid));
  SSVAL(obuf,smb_pid,SVAL(inbuf,smb_pid));
  SSVAL(obuf,smb_uid,SVAL(inbuf,smb_uid));
  SSVAL(obuf,smb_mid,SVAL(inbuf,smb_mid));

  /* process the request */
  outsize = switch_message(type,ibuf,obuf,
			   (size+smb_wct)-(int)(inbuf2-inbuf),
			   bufsize);
 
  /* copy the new reply header over the old one, but preserve 
     the smb_com field */
  memcpy(outbuf+smb_com+1,obuf+smb_com+1,smb_wct-(smb_com+1));

  /* and copy the data from the reply to the right spot */
  memcpy(outbuf2,obuf+smb_wct,outsize - smb_wct);

  /* free the allocated buffers */
  if (ibuf) free(ibuf);
  if (obuf) free(obuf);

  in_chain = was_inchain;

  /* return how much extra has been added to the packet */
  return(outsize - smb_wct);
}



/****************************************************************************
  construct a reply to the incoming packet
****************************************************************************/
int construct_reply(char *inbuf,char *outbuf,int size,int bufsize)
{
  int type = CVAL(inbuf,smb_com);
  int outsize = 0;
  int msg_type = CVAL(inbuf,0);

  chain_size = 0;
  
  if (msg_type != 0)
    return(reply_special(inbuf,outbuf,size,bufsize));
  
  memset(outbuf,0,smb_size);

  CVAL(outbuf,smb_com) = CVAL(inbuf,smb_com);
  set_message(outbuf,0,0,True);
  
  memcpy(outbuf+4,inbuf+4,4);
  CVAL(outbuf,smb_rcls) = SUCCESS;
  CVAL(outbuf,smb_reh) = 0;
  CVAL(outbuf,smb_flg) = 0x80; /* bit 7 set means a reply */
  SSVAL(outbuf,smb_err,SUCCESS);
  SSVAL(outbuf,smb_tid,SVAL(inbuf,smb_tid));
  SSVAL(outbuf,smb_pid,SVAL(inbuf,smb_pid));
  SSVAL(outbuf,smb_uid,SVAL(inbuf,smb_uid));
  SSVAL(outbuf,smb_mid,SVAL(inbuf,smb_mid));

  outsize = switch_message(type,inbuf,outbuf,size,bufsize);
 
  if(outsize != -1)
    smb_setlen(outbuf,outsize - 4);
  return(outsize);
}


/****************************************************************************
  process commands from the client
****************************************************************************/
void process(void )
{
  extern int Client;
  static int trans_num = 0;
  int nread;
  int deadtime = lp_deadtime()*60*1000*1000; /* the deadtime is in seconds */
  
  InBuffer = (char *)malloc(BUFFER_SIZE);
  OutBuffer = (char *)malloc(BUFFER_SIZE);
  if ((InBuffer == NULL) || (OutBuffer == NULL)) 
    return;
  
  while (True)
    {
      int32 len;      
      int msg_type;
      int msg_flags;
      int type;

      if (lp_readprediction())
	do_read_prediction();

      if (done_become_user != -1)
	{
	  if (!receive_smb(InBuffer,2*1000*1000))
	    {
	      unbecome_user();
	      if (!receive_smb(InBuffer,(num_files_open == 0?deadtime:0)))
		return;
	    }
	}
      else
	{
	  if (!receive_smb(InBuffer,(num_files_open == 0?deadtime:0)))	
	    return;
	}

      msg_type = CVAL(InBuffer,0);
      msg_flags = CVAL(InBuffer,1);
      type = CVAL(InBuffer,smb_com);

      len = smb_len(InBuffer);

      DEBUG(3,("got message type 0x%x of len 0x%x\n",msg_type,len));

      nread = len + 4;
      
      DEBUG(3,("%s Transaction %d\n",timestring(),trans_num));

      if (msg_type == 0)
	show_msg(InBuffer);

      nread = construct_reply(InBuffer,OutBuffer,nread,MIN(BUFFER_SIZE,lp_maxxmit()));
      
      if(nread > 0) {
        if (CVAL(OutBuffer,0) == 0)
	  show_msg(OutBuffer);
	
        if (nread != smb_len(OutBuffer) + 4) {
	  DEBUG(0,("ERROR: Invalid message response size! %d %d\n",
		nread,
		smb_len(OutBuffer)));
        }
	else
	  {
#if FORK_SEND	    
	    if (nread > 32000)
	      {
		signal(SIGCLD, SIGNAL_CAST sig_cld);		
		if (fork() == 0)
		  {
		    send_smb(OutBuffer);
		    exit(0);
		  }
	      }
	    else
	      send_smb(OutBuffer);
#else
	    send_smb(OutBuffer);
#endif
	  }
      }
      trans_num++;

      /* If we got a tree disconnect, see if this was the
	 last one in use, if so - exit */
      if(lp_deadtime() == 0 && type == SMBtdis) {
	int i;
	for(i=0;i<MAX_CONNECTIONS;i++)
	  if(Connections[i].open) 
	    break; /* Stay in loop */
	if(i != MAX_CONNECTIONS)
	  continue;
	/* There are no more connections open - shut down this socket
	   and terminate */
	DEBUG(3,("Last connection closed - shutting down\n"));
	shutdown(Client,2);
	return;
      }
    }
}


/****************************************************************************
  initialise connect, service and file structs
****************************************************************************/
void init_structs(void )
{
  int i;

  for (i=0;i<MAX_CONNECTIONS;i++)
    Connections[i].open = False;
  for (i=0;i<MAX_OPEN_FILES;i++)
    Files[i].open = False;

  num_files_open=0;

  init_dptrs();
}

/****************************************************************************
usage on the program
****************************************************************************/
void usage(char *pname)
{
  printf("Usage: %s [-D] [-p port] [-d debuglevel] [-l log basename] [-s services file]\n",pname);
  printf("Version %s\n",VERSION);
  printf("\t-D                    become a daemon\n");
  printf("\t-p port               listen on the specified port\n");
  printf("\t-d debuglevel         set the debuglevel\n");
  printf("\t-l log basename.      Basename for log/debug files\n");
  printf("\t-s services file.     Filename of services file\n");
  printf("\t-P                    passive only\n");
  printf("\t-a                    overwrite log file, don't append\n");
  printf("\n");
}


/****************************************************************************
  main program
****************************************************************************/
int main(int argc,char *argv[])
{
  /* shall I run as a daemon */
  BOOL is_daemon = False;
  int port = 139;
  int opt;
  extern FILE *dbf;
  extern char *optarg;

#ifdef SecureWare
  set_auth_parameters(argc,argv);
  setluid(0);
#endif

#ifdef OSF1_ENH_SEC
  set_auth_parameters();
#endif

  initial_uid = geteuid();
  initial_gid = getegid();

  if (initial_gid != 0 && initial_uid == 0)
    {
#ifdef HPUX
      setresgid(0,0,0);
#else
      setgid(0);
      setegid(0);
#endif
    }

  initial_uid = geteuid();
  initial_gid = getegid();

  while ((opt = getopt (argc, argv, "i:l:s:d:Dp:hPa")) != EOF)
    switch (opt)
      {
      case 'i':
	strcpy(scope,optarg);
	break;
      case 'P':
	{
	  extern BOOL passive;
	  passive = True;
	}
	break;	
      case 's':
	strcpy(servicesf,optarg);
	break;
      case 'l':
	strcpy(debugf,optarg);
	break;
      case 'a':
	append_log = !append_log;
	break;
      case 'D':
	is_daemon = True;
	break;
      case 'd':
	if (*optarg == 'A')
	  DEBUGLEVEL = 10000;
	else
	  DEBUGLEVEL = atoi(optarg);
	break;
      case 'p':
	port = atoi(optarg);
	break;
      case 'h':
	usage(argv[0]);
	exit(0);
	break;
      default:
	usage(argv[0]);
	exit(1);
      }

  
  NeedSwap = big_endian();
  
  if (DEBUGLEVEL > 100)
    {
      extern FILE *login,*logout;
      pstring fname="";
      sprintf(fname,"%s.in",debugf);
      login = fopen(fname,"w"); 
      if (login) chmod(fname,0600);
      sprintf(fname,"%s.out",debugf);
      logout = fopen(fname,"w");
      if (logout) chmod(fname,0600);
    }
  
  if (DEBUGLEVEL > 0)
    {
      pstring fname="";
      if (DEBUGLEVEL < 10)
	sprintf(fname,"%s.debug",debugf);      
      else
	sprintf(fname,"%s.debug.%d",debugf,getpid());      
      if (append_log)
	dbf = fopen(fname,"a");
      else
	dbf = fopen(fname,"w");
      if (dbf)
	setbuf(dbf,NULL);
      DEBUG(1,("%s smbd version %s started\n",timestring(),VERSION));
      DEBUG(1,("Copyright Andrew Tridgell 1992,1993,1994\n"));
    }

  GetWd(OriginalDir);
  get_machine_info();
  
  DEBUG(2,("uid=%d gid=%d euid=%d egid=%d\n",
	getuid(),getgid(),geteuid(),getegid()));

  init_structs();
  
  if (!lp_load(servicesf))
    return(-1);	
  
  if (DEBUGLEVEL > 1)
    DEBUG(3,("%s loaded services\n",timestring()));

  if (is_daemon)
    {
      DEBUG(3,("%s becoming a daemon\n",timestring()));
      become_daemon();
    }

  if (open_sockets(is_daemon,port))
    {
      /* reload the services file. It might have changed (if a daemon) */
      if (is_daemon && !lp_load(servicesf))
	return(-1);	

      if (lp_rootdir())
	{
	  chroot(lp_rootdir());
  
	  if (DEBUGLEVEL > 1)
	    DEBUG(2,("%s changed root to %s\n",timestring(),lp_rootdir()));
	}

      maxxmit = lp_maxxmit();

      create_mangled_stack(lp_mangledstack());
      {
	extern BOOL use_getwd_cache;
	use_getwd_cache = lp_getwdcache();
      }
      
      process();
      close_sockets();
    }
  if (dbf)
    fclose(dbf);
  return(0);
}

