# Copyright 2023 The Chromium Authors
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Presubmit script for the media/gpu/chromeos/shaders folder.

See http://dev.chromium.org/developers/how-tos/depottools/presubmit-scripts
for more details about the presubmit API built into depot_tools.
"""

import subprocess

PRESUBMIT_VERSION = '2.0.0'

def _FilterFile(affected_file):
  return affected_file.LocalPath().endswith(
      ('.h', '.vert', '.frag'))


def _CheckShadersCompiled(input_api, output_api):
  header_file_changed = False
  shader_file_changed = False
  header_file_content = None
  for file in input_api.AffectedSourceFiles(_FilterFile):
    if file.LocalPath().endswith('shaders.h'):
      header_file_changed = True
      header_file_content = input_api.ReadFile(file)
    elif file.LocalPath().endswith(('.vert', '.frag')):
      shader_file_changed = True

  if header_file_changed and not shader_file_changed:
    return [output_api.PresubmitError(
        'Modifications detected to shaders.h, but none of the shader source '
        'files have changed. This files autogenerated, please only use the '
        'compile_shaders.py script to modify this file.')]
  if shader_file_changed and not header_file_changed:
    return [output_api.PresubmitError(
        'Modifications detected to shader source files, but not shaders.h. '
        'Please recompile shaders.h using the compile_shaders.py script when '
        'making a change to shader source files.')]

  if shader_file_changed and header_file_changed:
    header_contents_expected = subprocess.run(
        ['python3', './compile_shaders.py', './'],
        capture_output=True, check=True).stdout.decode()
    if header_contents_expected != header_file_content:
      return [output_api.PresubmitError(
          'Contents of shaders.h do not match output of compile_shaders.py. '
          'Please recompile your shaders.')]

  return []


def CheckChangeOnUpload(input_api, output_api):
  return _CheckShadersCompiled(input_api, output_api)
