// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef FUCHSIA_WEB_WEBENGINE_BROWSER_WEB_ENGINE_MEMORY_INSPECTOR_H_
#define FUCHSIA_WEB_WEBENGINE_BROWSER_WEB_ENGINE_MEMORY_INSPECTOR_H_

#include <lib/inspect/cpp/vmo/types.h>
#include <memory>

#include "base/memory/weak_ptr.h"
#include "base/time/time.h"
#include "services/resource_coordinator/public/mojom/memory_instrumentation/memory_instrumentation.mojom.h"

namespace fpromise {
class context;
}

// Integrates with MemoryInstrumentation to publish memory details to Inspect
// on-demand.
class WebEngineMemoryInspector {
 public:
  // Creates a lazily-populated node called "memory" under |parent_node|.
  explicit WebEngineMemoryInspector(inspect::Node& parent_node);
  ~WebEngineMemoryInspector();

 private:
  // Returns the result of attempting to resolve the memory dump promise:
  // - If the dump is available then the contents are serialized and returned.
  // - Otherwise the |context| is suspended, a fresh dump requested, and a
  //   pending result returned. The |context| will be resumed only when the
  //   dump request completes.
  // This is used as a continuation function for promises generated by the
  // LazyNode's callback.
  fpromise::result<inspect::Inspector> ResolveMemoryDumpPromise(
      fpromise::context& context);

  // Handles completion of a memory dump request.
  void OnMemoryDumpComplete(
      base::TimeTicks requested_at,
      fpromise::suspended_task task,
      bool success,
      memory_instrumentation::mojom::GlobalMemoryDumpPtr raw_dump);

  // Node holding memory usage information.
  inspect::LazyNode node_;

  // Holds an Inspector populated with data from the memory dump received by
  // OnMemoryDumpComplete(), until the promise continuation is called.
  std::unique_ptr<inspect::Inspector> dump_results_;

  const base::WeakPtrFactory<WebEngineMemoryInspector> weak_this_{this};
};

#endif  // FUCHSIA_WEB_WEBENGINE_BROWSER_WEB_ENGINE_MEMORY_INSPECTOR_H_
