// Copyright 2019 The ChromiumOS Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef LIBIPP_IPP_ENUMS_H_
#define LIBIPP_IPP_ENUMS_H_

#include "ipp_export.h"

#include <cstdint>
#include <string>
#include <string_view>

namespace ipp {
// This is part of libipp library. See ipp.h for information how to use it.
// This file and the corresponding .cc file are generated from IPP
// specification. This file contains the following:
//  * GroupTag enum - contains names of all known IPP attributes groups.
//  * AttrName enum - contains names of all known IPP attributes and special
//  value _unknown representing attribute outside the schema.
//  * E_<name> enums - represent all known IPP keywords and enumerations, <name>
//  in the enum's name corresponds to a name of IPP keyword or enumeration (it
//  is the same as the name of the attribute).
//  * std::string ToString(AttrName, int) function - general function for
//  converting a pair (attribute_name,attribute_value) to a string
//  representation, returns empty string <=> if a corresponding
//  E_<attribute_name> is not defined or it doesn't contain the given
//  attribute_value.
//  * std::string ToString(enum_value) functions - these functions are similar
//  to the function above, the only difference is that enum type is specified by
//  the type of the parameter, instead of AttrName. There are also ToString(...)
//  functions for enum GroupTag and enum AttrName.
//  * bool FromString(const std::string&, AttrName, int*) function - general
//  function for converting a pair (string,attribute_name) to a corresponding
//  attribute value. Returns true <=> conversion was successful and an attribute
//  value was written to given int parameter. It fails when a corresponding
//  E_<attribute_name> is not defined or it doesn't contain any value with given
//  name, in this case given int parameter is not modified.
//  * bool FromString(const std::string&, enum_value*) functions - these
//  functions are similar to the function above, the only difference is that
//  enum type is specified by the type of the parameter, instead of AttrName.
//  There are also FromString(...) functions for enum GroupTag and enum
//  AttrName.
//
// This file contains subset of definitions from IPP registry at
// https://www.pwg.org/ipp/ipp-registrations.xml

enum class GroupTag : uint16_t {
  document_attributes = 0x0009,
  event_notification_attributes = 0x0007,
  job_attributes = 0x0002,
  operation_attributes = 0x0001,
  printer_attributes = 0x0004,
  resource_attributes = 0x0008,
  subscription_attributes = 0x0006,
  system_attributes = 0x000a,
  unsupported_attributes = 0x0005
};
enum class AttrName : uint16_t {
  _unknown = 0x0000,
  attributes_charset = 0x0001,
  attributes_natural_language = 0x0002,
  auth_info = 0x0003,
  auth_info_required = 0x0004,
  baling = 0x0005,
  baling_type = 0x0006,
  baling_type_supported = 0x0007,
  baling_when = 0x0008,
  baling_when_supported = 0x0009,
  binding = 0x000a,
  binding_reference_edge = 0x000b,
  binding_reference_edge_supported = 0x000c,
  binding_type = 0x000d,
  binding_type_supported = 0x000e,
  charset_configured = 0x000f,
  charset_supported = 0x0010,
  coating = 0x0011,
  coating_sides = 0x0012,
  coating_sides_supported = 0x0013,
  coating_type = 0x0014,
  coating_type_supported = 0x0015,
  color_supported = 0x0016,
  compression = 0x0017,
  compression_supported = 0x0018,
  contact_name = 0x0019,
  contact_uri = 0x001a,
  contact_vcard = 0x001b,
  copies = 0x001c,
  copies_actual = 0x001d,
  copies_default = 0x001e,
  copies_supported = 0x001f,
  cover_back = 0x0020,
  cover_back_actual = 0x0021,
  cover_back_default = 0x0022,
  cover_back_supported = 0x0023,
  cover_front = 0x0024,
  cover_front_actual = 0x0025,
  cover_front_default = 0x0026,
  cover_front_supported = 0x0027,
  cover_type = 0x0028,
  covering = 0x0029,
  covering_name = 0x002a,
  covering_name_supported = 0x002b,
  current_page_order = 0x002c,
  date_time_at_completed = 0x002d,
  date_time_at_creation = 0x002e,
  date_time_at_processing = 0x002f,
  detailed_status_message = 0x0030,
  device_service_count = 0x0031,
  device_uri = 0x0032,
  device_uuid = 0x0033,
  document_access_error = 0x0034,
  document_charset_default = 0x0035,
  document_charset_supplied = 0x0036,
  document_charset_supported = 0x0037,
  document_copies = 0x0038,
  document_digital_signature = 0x0039,
  document_digital_signature_default = 0x003a,
  document_digital_signature_supported = 0x003b,
  document_format = 0x003c,
  document_format_default = 0x003d,
  document_format_details_default = 0x003e,
  document_format_details_supplied = 0x003f,
  document_format_details_supported = 0x0040,
  document_format_device_id = 0x0041,
  document_format_ready = 0x0042,
  document_format_supplied = 0x0043,
  document_format_supported = 0x0044,
  document_format_varying_attributes = 0x0045,
  document_format_version = 0x0046,
  document_format_version_default = 0x0047,
  document_format_version_supplied = 0x0048,
  document_format_version_supported = 0x0049,
  document_message_supplied = 0x004a,
  document_metadata = 0x004b,
  document_name = 0x004c,
  document_name_supplied = 0x004d,
  document_natural_language = 0x004e,
  document_natural_language_default = 0x004f,
  document_natural_language_supplied = 0x0050,
  document_natural_language_supported = 0x0051,
  document_number = 0x0052,
  document_numbers = 0x0053,
  document_password_supported = 0x0054,
  document_source_application_name = 0x0055,
  document_source_application_version = 0x0056,
  document_source_os_name = 0x0057,
  document_source_os_version = 0x0058,
  document_uri = 0x0059,
  errors_count = 0x005a,
  feed_orientation = 0x005b,
  feed_orientation_supported = 0x005c,
  finishing_template = 0x005d,
  finishing_template_supported = 0x005e,
  finishings = 0x005f,
  finishings_col = 0x0060,
  finishings_col_actual = 0x0061,
  finishings_col_database = 0x0062,
  finishings_col_default = 0x0063,
  finishings_col_ready = 0x0064,
  finishings_default = 0x0065,
  finishings_ready = 0x0066,
  finishings_supported = 0x0067,
  first_printer_name = 0x0068,
  folding = 0x0069,
  folding_direction = 0x006a,
  folding_direction_supported = 0x006b,
  folding_offset = 0x006c,
  folding_offset_supported = 0x006d,
  folding_reference_edge = 0x006e,
  folding_reference_edge_supported = 0x006f,
  font_name_requested = 0x0070,
  font_name_requested_default = 0x0071,
  font_name_requested_supported = 0x0072,
  font_size_requested = 0x0073,
  font_size_requested_default = 0x0074,
  font_size_requested_supported = 0x0075,
  force_front_side = 0x0076,
  force_front_side_actual = 0x0077,
  generated_natural_language_supported = 0x0078,
  identify_actions = 0x0079,
  identify_actions_default = 0x007a,
  identify_actions_supported = 0x007b,
  imposition_template = 0x007c,
  impressions_completed_current_copy = 0x007d,
  input_orientation_requested = 0x007e,
  input_quality = 0x007f,
  input_sides = 0x0080,
  insert_after_page_number = 0x0081,
  insert_after_page_number_supported = 0x0082,
  insert_count = 0x0083,
  insert_count_supported = 0x0084,
  insert_sheet = 0x0085,
  insert_sheet_actual = 0x0086,
  insert_sheet_default = 0x0087,
  ipp_attribute_fidelity = 0x0088,
  ipp_features_supported = 0x0089,
  ipp_versions_supported = 0x008a,
  ippget_event_life = 0x008b,
  job_account_id = 0x008c,
  job_account_id_actual = 0x008d,
  job_account_id_default = 0x008e,
  job_account_id_supported = 0x008f,
  job_account_type = 0x0090,
  job_account_type_default = 0x0091,
  job_account_type_supported = 0x0092,
  job_accounting_output_bin = 0x0093,
  job_accounting_sheets = 0x0094,
  job_accounting_sheets_actual = 0x0095,
  job_accounting_sheets_default = 0x0096,
  job_accounting_sheets_type = 0x0097,
  job_accounting_user_id = 0x0098,
  job_accounting_user_id_actual = 0x0099,
  job_accounting_user_id_default = 0x009a,
  job_accounting_user_id_supported = 0x009b,
  job_attribute_fidelity = 0x009c,
  job_authorization_uri_supported = 0x009d,
  job_charge_info = 0x009e,
  job_collation_type = 0x009f,
  job_constraints_supported = 0x00a0,
  job_copies = 0x00a1,
  job_copies_actual = 0x00a2,
  job_copies_default = 0x00a3,
  job_copies_supported = 0x00a4,
  job_cover_back = 0x00a5,
  job_cover_back_actual = 0x00a6,
  job_cover_back_default = 0x00a7,
  job_cover_back_supported = 0x00a8,
  job_cover_front = 0x00a9,
  job_cover_front_actual = 0x00aa,
  job_cover_front_default = 0x00ab,
  job_cover_front_supported = 0x00ac,
  job_delay_output_until = 0x00ad,
  job_delay_output_until_default = 0x00ae,
  job_delay_output_until_supported = 0x00af,
  job_delay_output_until_time = 0x00b0,
  job_delay_output_until_time_supported = 0x00b1,
  job_detailed_status_messages = 0x00b2,
  job_document_access_errors = 0x00b3,
  job_error_action = 0x00b4,
  job_error_action_default = 0x00b5,
  job_error_action_supported = 0x00b6,
  job_error_sheet = 0x00b7,
  job_error_sheet_actual = 0x00b8,
  job_error_sheet_default = 0x00b9,
  job_error_sheet_type = 0x00ba,
  job_error_sheet_when = 0x00bb,
  job_finishings = 0x00bc,
  job_finishings_col = 0x00bd,
  job_finishings_col_actual = 0x00be,
  job_finishings_col_default = 0x00bf,
  job_finishings_col_ready = 0x00c0,
  job_finishings_default = 0x00c1,
  job_finishings_ready = 0x00c2,
  job_finishings_supported = 0x00c3,
  job_hold_until = 0x00c4,
  job_hold_until_default = 0x00c5,
  job_hold_until_supported = 0x00c6,
  job_hold_until_time = 0x00c7,
  job_hold_until_time_supported = 0x00c8,
  job_id = 0x00c9,
  job_ids_supported = 0x00ca,
  job_impressions = 0x00cb,
  job_impressions_completed = 0x00cc,
  job_impressions_supported = 0x00cd,
  job_k_octets = 0x00ce,
  job_k_octets_processed = 0x00cf,
  job_k_octets_supported = 0x00d0,
  job_mandatory_attributes = 0x00d1,
  job_media_sheets = 0x00d2,
  job_media_sheets_completed = 0x00d3,
  job_media_sheets_supported = 0x00d4,
  job_message_from_operator = 0x00d5,
  job_message_to_operator = 0x00d6,
  job_message_to_operator_actual = 0x00d7,
  job_message_to_operator_default = 0x00d8,
  job_message_to_operator_supported = 0x00d9,
  job_more_info = 0x00da,
  job_name = 0x00db,
  job_originating_user_name = 0x00dc,
  job_originating_user_uri = 0x00dd,
  job_pages = 0x00de,
  job_pages_completed = 0x00df,
  job_pages_completed_current_copy = 0x00e0,
  job_pages_per_set = 0x00e1,
  job_pages_per_set_supported = 0x00e2,
  job_password_encryption = 0x00e3,
  job_password_encryption_supported = 0x00e4,
  job_password_supported = 0x00e5,
  job_phone_number = 0x00e6,
  job_phone_number_default = 0x00e7,
  job_phone_number_supported = 0x00e8,
  job_printer_up_time = 0x00e9,
  job_printer_uri = 0x00ea,
  job_priority = 0x00eb,
  job_priority_actual = 0x00ec,
  job_priority_default = 0x00ed,
  job_priority_supported = 0x00ee,
  job_recipient_name = 0x00ef,
  job_recipient_name_default = 0x00f0,
  job_recipient_name_supported = 0x00f1,
  job_resolvers_supported = 0x00f2,
  job_resource_ids = 0x00f3,
  job_save_disposition = 0x00f4,
  job_save_printer_make_and_model = 0x00f5,
  job_sheet_message = 0x00f6,
  job_sheet_message_actual = 0x00f7,
  job_sheet_message_default = 0x00f8,
  job_sheet_message_supported = 0x00f9,
  job_sheets = 0x00fa,
  job_sheets_col = 0x00fb,
  job_sheets_col_actual = 0x00fc,
  job_sheets_col_default = 0x00fd,
  job_sheets_default = 0x00fe,
  job_sheets_supported = 0x00ff,
  job_spooling_supported = 0x0100,
  job_state = 0x0101,
  job_state_message = 0x0102,
  job_state_reasons = 0x0103,
  job_uri = 0x0104,
  job_uuid = 0x0105,
  jpeg_k_octets_supported = 0x0106,
  jpeg_x_dimension_supported = 0x0107,
  jpeg_y_dimension_supported = 0x0108,
  laminating = 0x0109,
  laminating_sides = 0x010a,
  laminating_sides_supported = 0x010b,
  laminating_type = 0x010c,
  laminating_type_supported = 0x010d,
  last_document = 0x010e,
  limit = 0x010f,
  material_color = 0x0110,
  max_save_info_supported = 0x0111,
  max_stitching_locations_supported = 0x0112,
  media = 0x0113,
  media_back_coating = 0x0114,
  media_back_coating_supported = 0x0115,
  media_bottom_margin = 0x0116,
  media_bottom_margin_supported = 0x0117,
  media_col = 0x0118,
  media_col_actual = 0x0119,
  media_col_database = 0x011a,
  media_col_default = 0x011b,
  media_col_ready = 0x011c,
  media_color = 0x011d,
  media_color_supported = 0x011e,
  media_default = 0x011f,
  media_front_coating = 0x0120,
  media_front_coating_supported = 0x0121,
  media_grain = 0x0122,
  media_grain_supported = 0x0123,
  media_hole_count = 0x0124,
  media_hole_count_supported = 0x0125,
  media_info = 0x0126,
  media_info_supported = 0x0127,
  media_input_tray_check = 0x0128,
  media_key = 0x0129,
  media_left_margin = 0x012a,
  media_left_margin_supported = 0x012b,
  media_order_count = 0x012c,
  media_order_count_supported = 0x012d,
  media_pre_printed = 0x012e,
  media_pre_printed_supported = 0x012f,
  media_ready = 0x0130,
  media_recycled = 0x0131,
  media_recycled_supported = 0x0132,
  media_right_margin = 0x0133,
  media_right_margin_supported = 0x0134,
  media_sheets_supported = 0x0135,
  media_size = 0x0136,
  media_size_name = 0x0137,
  media_size_supported = 0x0138,
  media_source = 0x0139,
  media_source_feed_direction = 0x013a,
  media_source_feed_orientation = 0x013b,
  media_source_properties = 0x013c,
  media_source_supported = 0x013d,
  media_supported = 0x013e,
  media_thickness = 0x013f,
  media_thickness_supported = 0x0140,
  media_tooth = 0x0141,
  media_tooth_supported = 0x0142,
  media_top_margin = 0x0143,
  media_top_margin_supported = 0x0144,
  media_type = 0x0145,
  media_type_supported = 0x0146,
  media_weight_metric = 0x0147,
  media_weight_metric_supported = 0x0148,
  member_names = 0x0149,
  member_uris = 0x014a,
  message = 0x014b,
  multiple_document_handling = 0x014c,
  multiple_document_handling_default = 0x014d,
  multiple_document_handling_supported = 0x014e,
  multiple_document_jobs_supported = 0x014f,
  multiple_operation_time_out = 0x0150,
  multiple_operation_time_out_action = 0x0151,
  my_jobs = 0x0152,
  natural_language_configured = 0x0153,
  notify_events = 0x0154,
  notify_events_default = 0x0155,
  notify_events_supported = 0x0156,
  notify_lease_duration_default = 0x0157,
  notify_lease_duration_supported = 0x0158,
  notify_pull_method = 0x0159,
  notify_pull_method_supported = 0x015a,
  notify_schemes_supported = 0x015b,
  number_of_documents = 0x015c,
  number_of_intervening_jobs = 0x015d,
  number_up = 0x015e,
  number_up_actual = 0x015f,
  number_up_default = 0x0160,
  number_up_supported = 0x0161,
  oauth_authorization_server_uri = 0x0162,
  operations_supported = 0x0163,
  orientation_requested = 0x0164,
  orientation_requested_default = 0x0165,
  orientation_requested_supported = 0x0166,
  original_requesting_user_name = 0x0167,
  output_bin = 0x0168,
  output_bin_default = 0x0169,
  output_bin_supported = 0x016a,
  output_device = 0x016b,
  output_device_actual = 0x016c,
  output_device_assigned = 0x016d,
  output_device_job_state_message = 0x016e,
  output_device_supported = 0x016f,
  output_device_uuid_assigned = 0x0170,
  output_device_uuid_supported = 0x0171,
  overrides = 0x0172,
  overrides_actual = 0x0173,
  page_delivery = 0x0174,
  page_delivery_default = 0x0175,
  page_delivery_supported = 0x0176,
  page_order_received = 0x0177,
  page_order_received_default = 0x0178,
  page_order_received_supported = 0x0179,
  page_ranges = 0x017a,
  page_ranges_actual = 0x017b,
  page_ranges_supported = 0x017c,
  pages = 0x017d,
  pages_per_minute = 0x017e,
  pages_per_minute_color = 0x017f,
  pages_per_subset = 0x0180,
  pages_per_subset_supported = 0x0181,
  parent_printers_supported = 0x0182,
  pdf_k_octets_supported = 0x0183,
  pdf_versions_supported = 0x0184,
  pdl_init_file = 0x0185,
  pdl_init_file_default = 0x0186,
  pdl_init_file_entry = 0x0187,
  pdl_init_file_entry_supported = 0x0188,
  pdl_init_file_location = 0x0189,
  pdl_init_file_location_supported = 0x018a,
  pdl_init_file_name = 0x018b,
  pdl_init_file_name_subdirectory_supported = 0x018c,
  pdl_init_file_name_supported = 0x018d,
  pdl_init_file_supported = 0x018e,
  pdl_override_supported = 0x018f,
  preferred_attributes_supported = 0x0190,
  presentation_direction_number_up = 0x0191,
  presentation_direction_number_up_default = 0x0192,
  presentation_direction_number_up_supported = 0x0193,
  print_color_mode = 0x0194,
  print_color_mode_default = 0x0195,
  print_color_mode_supported = 0x0196,
  print_content_optimize = 0x0197,
  print_content_optimize_default = 0x0198,
  print_content_optimize_supported = 0x0199,
  print_quality = 0x019a,
  print_quality_default = 0x019b,
  print_quality_supported = 0x019c,
  print_rendering_intent = 0x019d,
  print_rendering_intent_default = 0x019e,
  print_rendering_intent_supported = 0x019f,
  print_scaling = 0x01a0,
  printer_alert = 0x01a1,
  printer_alert_description = 0x01a2,
  printer_charge_info = 0x01a3,
  printer_charge_info_uri = 0x01a4,
  printer_config_change_date_time = 0x01a5,
  printer_config_change_time = 0x01a6,
  printer_config_changes = 0x01a7,
  printer_contact_col = 0x01a8,
  printer_current_time = 0x01a9,
  printer_detailed_status_messages = 0x01aa,
  printer_device_id = 0x01ab,
  printer_dns_sd_name = 0x01ac,
  printer_driver_installer = 0x01ad,
  printer_finisher = 0x01ae,
  printer_finisher_description = 0x01af,
  printer_finisher_supplies = 0x01b0,
  printer_finisher_supplies_description = 0x01b1,
  printer_geo_location = 0x01b2,
  printer_icc_profiles = 0x01b3,
  printer_icons = 0x01b4,
  printer_id = 0x01b5,
  printer_impressions_completed = 0x01b6,
  printer_info = 0x01b7,
  printer_input_tray = 0x01b8,
  printer_is_accepting_jobs = 0x01b9,
  printer_location = 0x01ba,
  printer_make_and_model = 0x01bb,
  printer_media_sheets_completed = 0x01bc,
  printer_message_date_time = 0x01bd,
  printer_message_from_operator = 0x01be,
  printer_message_time = 0x01bf,
  printer_more_info = 0x01c0,
  printer_more_info_manufacturer = 0x01c1,
  printer_name = 0x01c2,
  printer_organization = 0x01c3,
  printer_organizational_unit = 0x01c4,
  printer_output_tray = 0x01c5,
  printer_pages_completed = 0x01c6,
  printer_resolution = 0x01c7,
  printer_resolution_actual = 0x01c8,
  printer_resolution_default = 0x01c9,
  printer_resolution_supported = 0x01ca,
  printer_state = 0x01cb,
  printer_state_change_date_time = 0x01cc,
  printer_state_change_time = 0x01cd,
  printer_state_message = 0x01ce,
  printer_state_reasons = 0x01cf,
  printer_static_resource_directory_uri = 0x01d0,
  printer_static_resource_k_octets_free = 0x01d1,
  printer_static_resource_k_octets_supported = 0x01d2,
  printer_strings_languages_supported = 0x01d3,
  printer_strings_uri = 0x01d4,
  printer_supply = 0x01d5,
  printer_supply_description = 0x01d6,
  printer_supply_info_uri = 0x01d7,
  printer_type = 0x01d8,
  printer_type_mask = 0x01d9,
  printer_up_time = 0x01da,
  printer_uri = 0x01db,
  printer_uri_supported = 0x01dc,
  printer_uuid = 0x01dd,
  printer_xri_supported = 0x01de,
  profile_name = 0x01df,
  profile_url = 0x01e0,
  proof_print = 0x01e1,
  proof_print_copies = 0x01e2,
  proof_print_default = 0x01e3,
  proof_print_supported = 0x01e4,
  punching = 0x01e5,
  punching_hole_diameter_configured = 0x01e6,
  punching_locations = 0x01e7,
  punching_locations_supported = 0x01e8,
  punching_offset = 0x01e9,
  punching_offset_supported = 0x01ea,
  punching_reference_edge = 0x01eb,
  punching_reference_edge_supported = 0x01ec,
  pwg_raster_document_resolution_supported = 0x01ed,
  pwg_raster_document_sheet_back = 0x01ee,
  pwg_raster_document_type_supported = 0x01ef,
  queued_job_count = 0x01f0,
  reference_uri_schemes_supported = 0x01f1,
  requested_attributes = 0x01f2,
  requested_user_name = 0x01f3,
  requesting_user_name = 0x01f4,
  requesting_user_name_allowed = 0x01f5,
  requesting_user_name_denied = 0x01f6,
  requesting_user_uri_supported = 0x01f7,
  resolver_name = 0x01f8,
  save_disposition = 0x01f9,
  save_disposition_supported = 0x01fa,
  save_document_format = 0x01fb,
  save_document_format_default = 0x01fc,
  save_document_format_supported = 0x01fd,
  save_info = 0x01fe,
  save_location = 0x01ff,
  save_location_default = 0x0200,
  save_location_supported = 0x0201,
  save_name = 0x0202,
  save_name_subdirectory_supported = 0x0203,
  save_name_supported = 0x0204,
  separator_sheets = 0x0205,
  separator_sheets_actual = 0x0206,
  separator_sheets_default = 0x0207,
  separator_sheets_type = 0x0208,
  sheet_collate = 0x0209,
  sheet_collate_default = 0x020a,
  sheet_collate_supported = 0x020b,
  sheet_completed_copy_number = 0x020c,
  sheet_completed_document_number = 0x020d,
  sides = 0x020e,
  sides_default = 0x020f,
  sides_supported = 0x0210,
  status_code = 0x0211,
  status_message = 0x0212,
  stitching = 0x0213,
  stitching_angle = 0x0214,
  stitching_angle_supported = 0x0215,
  stitching_locations = 0x0216,
  stitching_locations_supported = 0x0217,
  stitching_method = 0x0218,
  stitching_method_supported = 0x0219,
  stitching_offset = 0x021a,
  stitching_offset_supported = 0x021b,
  stitching_reference_edge = 0x021c,
  stitching_reference_edge_supported = 0x021d,
  subordinate_printers_supported = 0x021e,
  time_at_completed = 0x021f,
  time_at_creation = 0x0220,
  time_at_processing = 0x0221,
  trimming = 0x0222,
  trimming_offset = 0x0223,
  trimming_offset_supported = 0x0224,
  trimming_reference_edge = 0x0225,
  trimming_reference_edge_supported = 0x0226,
  trimming_type = 0x0227,
  trimming_type_supported = 0x0228,
  trimming_when = 0x0229,
  trimming_when_supported = 0x022a,
  uri_authentication_supported = 0x022b,
  uri_security_supported = 0x022c,
  warnings_count = 0x022d,
  which_jobs = 0x022e,
  which_jobs_supported = 0x022f,
  x_dimension = 0x0230,
  x_image_position = 0x0231,
  x_image_position_default = 0x0232,
  x_image_position_supported = 0x0233,
  x_image_shift = 0x0234,
  x_image_shift_actual = 0x0235,
  x_image_shift_default = 0x0236,
  x_image_shift_supported = 0x0237,
  x_side1_image_shift = 0x0238,
  x_side1_image_shift_actual = 0x0239,
  x_side1_image_shift_default = 0x023a,
  x_side1_image_shift_supported = 0x023b,
  x_side2_image_shift = 0x023c,
  x_side2_image_shift_actual = 0x023d,
  x_side2_image_shift_default = 0x023e,
  x_side2_image_shift_supported = 0x023f,
  xri_authentication = 0x0240,
  xri_authentication_supported = 0x0241,
  xri_security = 0x0242,
  xri_security_supported = 0x0243,
  xri_uri = 0x0244,
  xri_uri_scheme_supported = 0x0245,
  y_dimension = 0x0246,
  y_image_position = 0x0247,
  y_image_position_default = 0x0248,
  y_image_position_supported = 0x0249,
  y_image_shift = 0x024a,
  y_image_shift_actual = 0x024b,
  y_image_shift_default = 0x024c,
  y_image_shift_supported = 0x024d,
  y_side1_image_shift = 0x024e,
  y_side1_image_shift_actual = 0x024f,
  y_side1_image_shift_default = 0x0250,
  y_side1_image_shift_supported = 0x0251,
  y_side2_image_shift = 0x0252,
  y_side2_image_shift_actual = 0x0253,
  y_side2_image_shift_default = 0x0254,
  y_side2_image_shift_supported = 0x0255
};
enum class E_auth_info_required : uint16_t {
  domain = 0x0000,
  none = 0x0001,
  password = 0x0002,
  username = 0x0003
};
enum class E_baling_type : uint16_t {
  band = 0x0000,
  shrink_wrap = 0x0001,
  wrap = 0x0002
};
enum class E_baling_when : uint16_t { after_job = 0x0000, after_sets = 0x0001 };
enum class E_binding_reference_edge : uint16_t {
  bottom = 0x0000,
  left = 0x0001,
  right = 0x0002,
  top = 0x0003
};
enum class E_binding_type : uint16_t {
  adhesive = 0x0000,
  comb = 0x0001,
  flat = 0x0002,
  padding = 0x0003,
  perfect = 0x0004,
  spiral = 0x0005,
  tape = 0x0006,
  velo = 0x0007
};
enum class E_coating_sides : uint16_t {
  back = 0x0000,
  both = 0x0001,
  front = 0x0002
};
enum class E_coating_type : uint16_t {
  archival = 0x0000,
  archival_glossy = 0x0001,
  archival_matte = 0x0002,
  archival_semi_gloss = 0x0003,
  glossy = 0x0004,
  high_gloss = 0x0005,
  matte = 0x0006,
  semi_gloss = 0x0007,
  silicone = 0x0008,
  translucent = 0x0009
};
enum class E_compression : uint16_t {
  compress = 0x0000,
  deflate = 0x0001,
  gzip = 0x0002,
  none = 0x0003
};
enum class E_cover_back_supported : uint16_t {
  cover_type = 0x0000,
  media = 0x0001,
  media_col = 0x0002
};
enum class E_cover_type : uint16_t {
  no_cover = 0x0000,
  print_back = 0x0001,
  print_both = 0x0002,
  print_front = 0x0003,
  print_none = 0x0004
};
enum class E_covering_name : uint16_t {
  plain = 0x0000,
  pre_cut = 0x0001,
  pre_printed = 0x0002
};
enum class E_current_page_order : uint16_t {
  _1_to_n_order = 0x0000,
  n_to_1_order = 0x0001
};
enum class E_document_digital_signature : uint16_t {
  dss = 0x0000,
  none = 0x0001,
  pgp = 0x0002,
  smime = 0x0003,
  xmldsig = 0x0004
};
enum class E_document_format_details_supported : uint16_t {
  document_format = 0x0000,
  document_format_device_id = 0x0001,
  document_format_version = 0x0002,
  document_natural_language = 0x0003,
  document_source_application_name = 0x0004,
  document_source_application_version = 0x0005,
  document_source_os_name = 0x0006,
  document_source_os_version = 0x0007
};
enum class E_document_format_varying_attributes : uint16_t { none = 0x0000 };
enum class E_feed_orientation : uint16_t {
  long_edge_first = 0x0000,
  short_edge_first = 0x0001
};
enum class E_finishing_template : uint16_t {
  bale = 0x0000,
  bind = 0x0001,
  bind_bottom = 0x0002,
  bind_left = 0x0003,
  bind_right = 0x0004,
  bind_top = 0x0005,
  booklet_maker = 0x0006,
  coat = 0x0007,
  cover = 0x0008,
  edge_stitch = 0x0009,
  edge_stitch_bottom = 0x000a,
  edge_stitch_left = 0x000b,
  edge_stitch_right = 0x000c,
  edge_stitch_top = 0x000d,
  fold = 0x000e,
  fold_accordion = 0x000f,
  fold_double_gate = 0x0010,
  fold_engineering_z = 0x0011,
  fold_gate = 0x0012,
  fold_half = 0x0013,
  fold_half_z = 0x0014,
  fold_left_gate = 0x0015,
  fold_letter = 0x0016,
  fold_parallel = 0x0017,
  fold_poster = 0x0018,
  fold_right_gate = 0x0019,
  fold_z = 0x001a,
  jdf_f2_1 = 0x001b,
  jdf_f4_1 = 0x001c,
  jdf_f4_2 = 0x001d,
  jdf_f6_1 = 0x001e,
  jdf_f6_2 = 0x001f,
  jdf_f6_3 = 0x0020,
  jdf_f6_4 = 0x0021,
  jdf_f6_5 = 0x0022,
  jdf_f6_6 = 0x0023,
  jdf_f6_7 = 0x0024,
  jdf_f6_8 = 0x0025,
  jdf_f8_1 = 0x0026,
  jdf_f8_2 = 0x0027,
  jdf_f8_3 = 0x0028,
  jdf_f8_4 = 0x0029,
  jdf_f8_5 = 0x002a,
  jdf_f8_6 = 0x002b,
  jdf_f8_7 = 0x002c,
  jdf_f10_1 = 0x002d,
  jdf_f10_2 = 0x002e,
  jdf_f10_3 = 0x002f,
  jdf_f12_1 = 0x0030,
  jdf_f12_2 = 0x0031,
  jdf_f12_3 = 0x0032,
  jdf_f12_4 = 0x0033,
  jdf_f12_5 = 0x0034,
  jdf_f12_6 = 0x0035,
  jdf_f12_7 = 0x0036,
  jdf_f12_8 = 0x0037,
  jdf_f12_9 = 0x0038,
  jdf_f12_10 = 0x0039,
  jdf_f12_11 = 0x003a,
  jdf_f12_12 = 0x003b,
  jdf_f12_13 = 0x003c,
  jdf_f12_14 = 0x003d,
  jdf_f14_1 = 0x003e,
  jdf_f16_1 = 0x003f,
  jdf_f16_2 = 0x0040,
  jdf_f16_3 = 0x0041,
  jdf_f16_4 = 0x0042,
  jdf_f16_5 = 0x0043,
  jdf_f16_6 = 0x0044,
  jdf_f16_7 = 0x0045,
  jdf_f16_8 = 0x0046,
  jdf_f16_9 = 0x0047,
  jdf_f16_10 = 0x0048,
  jdf_f16_11 = 0x0049,
  jdf_f16_12 = 0x004a,
  jdf_f16_13 = 0x004b,
  jdf_f16_14 = 0x004c,
  jdf_f18_1 = 0x004d,
  jdf_f18_2 = 0x004e,
  jdf_f18_3 = 0x004f,
  jdf_f18_4 = 0x0050,
  jdf_f18_5 = 0x0051,
  jdf_f18_6 = 0x0052,
  jdf_f18_7 = 0x0053,
  jdf_f18_8 = 0x0054,
  jdf_f18_9 = 0x0055,
  jdf_f20_1 = 0x0056,
  jdf_f20_2 = 0x0057,
  jdf_f24_1 = 0x0058,
  jdf_f24_2 = 0x0059,
  jdf_f24_3 = 0x005a,
  jdf_f24_4 = 0x005b,
  jdf_f24_5 = 0x005c,
  jdf_f24_6 = 0x005d,
  jdf_f24_7 = 0x005e,
  jdf_f24_8 = 0x005f,
  jdf_f24_9 = 0x0060,
  jdf_f24_10 = 0x0061,
  jdf_f24_11 = 0x0062,
  jdf_f28_1 = 0x0063,
  jdf_f32_1 = 0x0064,
  jdf_f32_2 = 0x0065,
  jdf_f32_3 = 0x0066,
  jdf_f32_4 = 0x0067,
  jdf_f32_5 = 0x0068,
  jdf_f32_6 = 0x0069,
  jdf_f32_7 = 0x006a,
  jdf_f32_8 = 0x006b,
  jdf_f32_9 = 0x006c,
  jdf_f36_1 = 0x006d,
  jdf_f36_2 = 0x006e,
  jdf_f40_1 = 0x006f,
  jdf_f48_1 = 0x0070,
  jdf_f48_2 = 0x0071,
  jdf_f64_1 = 0x0072,
  jdf_f64_2 = 0x0073,
  jog_offset = 0x0074,
  laminate = 0x0075,
  punch = 0x0076,
  punch_bottom_left = 0x0077,
  punch_bottom_right = 0x0078,
  punch_dual_bottom = 0x0079,
  punch_dual_left = 0x007a,
  punch_dual_right = 0x007b,
  punch_dual_top = 0x007c,
  punch_multiple_bottom = 0x007d,
  punch_multiple_left = 0x007e,
  punch_multiple_right = 0x007f,
  punch_multiple_top = 0x0080,
  punch_quad_bottom = 0x0081,
  punch_quad_left = 0x0082,
  punch_quad_right = 0x0083,
  punch_quad_top = 0x0084,
  punch_top_left = 0x0085,
  punch_top_right = 0x0086,
  punch_triple_bottom = 0x0087,
  punch_triple_left = 0x0088,
  punch_triple_right = 0x0089,
  punch_triple_top = 0x008a,
  saddle_stitch = 0x008b,
  staple = 0x008c,
  staple_bottom_left = 0x008d,
  staple_bottom_right = 0x008e,
  staple_dual_bottom = 0x008f,
  staple_dual_left = 0x0090,
  staple_dual_right = 0x0091,
  staple_dual_top = 0x0092,
  staple_top_left = 0x0093,
  staple_top_right = 0x0094,
  staple_triple_bottom = 0x0095,
  staple_triple_left = 0x0096,
  staple_triple_right = 0x0097,
  staple_triple_top = 0x0098,
  trim = 0x0099,
  trim_after_copies = 0x009a,
  trim_after_documents = 0x009b,
  trim_after_job = 0x009c,
  trim_after_pages = 0x009d
};
enum class E_finishings : uint16_t {
  none = 0x0003,
  staple = 0x0004,
  punch = 0x0005,
  cover = 0x0006,
  bind = 0x0007,
  saddle_stitch = 0x0008,
  edge_stitch = 0x0009,
  fold = 0x000a,
  trim = 0x000b,
  bale = 0x000c,
  booklet_maker = 0x000d,
  jog_offset = 0x000e,
  coat = 0x000f,
  laminate = 0x0010,
  staple_top_left = 0x0014,
  staple_bottom_left = 0x0015,
  staple_top_right = 0x0016,
  staple_bottom_right = 0x0017,
  edge_stitch_left = 0x0018,
  edge_stitch_top = 0x0019,
  edge_stitch_right = 0x001a,
  edge_stitch_bottom = 0x001b,
  staple_dual_left = 0x001c,
  staple_dual_top = 0x001d,
  staple_dual_right = 0x001e,
  staple_dual_bottom = 0x001f,
  staple_triple_left = 0x0020,
  staple_triple_top = 0x0021,
  staple_triple_right = 0x0022,
  staple_triple_bottom = 0x0023,
  bind_left = 0x0032,
  bind_top = 0x0033,
  bind_right = 0x0034,
  bind_bottom = 0x0035,
  trim_after_pages = 0x003c,
  trim_after_documents = 0x003d,
  trim_after_copies = 0x003e,
  trim_after_job = 0x003f,
  punch_top_left = 0x0046,
  punch_bottom_left = 0x0047,
  punch_top_right = 0x0048,
  punch_bottom_right = 0x0049,
  punch_dual_left = 0x004a,
  punch_dual_top = 0x004b,
  punch_dual_right = 0x004c,
  punch_dual_bottom = 0x004d,
  punch_triple_left = 0x004e,
  punch_triple_top = 0x004f,
  punch_triple_right = 0x0050,
  punch_triple_bottom = 0x0051,
  punch_quad_left = 0x0052,
  punch_quad_top = 0x0053,
  punch_quad_right = 0x0054,
  punch_quad_bottom = 0x0055,
  punch_multiple_left = 0x0056,
  punch_multiple_top = 0x0057,
  punch_multiple_right = 0x0058,
  punch_multiple_bottom = 0x0059,
  fold_accordion = 0x005a,
  fold_double_gate = 0x005b,
  fold_gate = 0x005c,
  fold_half = 0x005d,
  fold_half_z = 0x005e,
  fold_left_gate = 0x005f,
  fold_letter = 0x0060,
  fold_parallel = 0x0061,
  fold_poster = 0x0062,
  fold_right_gate = 0x0063,
  fold_z = 0x0064,
  fold_engineering_z = 0x0065
};
enum class E_folding_direction : uint16_t { inward = 0x0000, outward = 0x0001 };
enum class E_identify_actions : uint16_t {
  display = 0x0000,
  flash = 0x0001,
  sound = 0x0002,
  speak = 0x0003
};
enum class E_imposition_template : uint16_t {
  none = 0x0000,
  signature = 0x0001
};
enum class E_input_orientation_requested : uint16_t {
  portrait = 0x0003,
  landscape = 0x0004,
  reverse_landscape = 0x0005,
  reverse_portrait = 0x0006,
  none = 0x0007
};
enum class E_input_quality : uint16_t {
  draft = 0x0003,
  normal = 0x0004,
  high = 0x0005
};
enum class E_input_sides : uint16_t {
  one_sided = 0x0000,
  two_sided_long_edge = 0x0001,
  two_sided_short_edge = 0x0002
};
enum class E_ipp_features_supported : uint16_t {
  document_object = 0x0000,
  faxout = 0x0001,
  icc_color_matching = 0x0002,
  infrastructure_printer = 0x0003,
  ipp_3d = 0x0004,
  ipp_everywhere = 0x0005,
  job_save = 0x0006,
  none = 0x0007,
  page_overrides = 0x0008,
  proof_print = 0x0009,
  scan = 0x000a,
  subscription_object = 0x000b
};
enum class E_ipp_versions_supported : uint16_t {
  _1_0 = 0x0100,
  _1_1 = 0x0101,
  _2_0 = 0x0200,
  _2_1 = 0x0201,
  _2_2 = 0x0202
};
enum class E_job_account_type : uint16_t {
  general = 0x0000,
  group = 0x0001,
  none = 0x0002
};
enum class E_job_accounting_output_bin : uint16_t {
  auto_ = 0x0000,
  bottom = 0x0001,
  center = 0x0002,
  face_down = 0x0003,
  face_up = 0x0004,
  large_capacity = 0x0005,
  left = 0x0006,
  mailbox_1 = 0x0007,
  mailbox_2 = 0x0008,
  mailbox_3 = 0x0009,
  mailbox_4 = 0x000a,
  mailbox_5 = 0x000b,
  mailbox_6 = 0x000c,
  mailbox_7 = 0x000d,
  mailbox_8 = 0x000e,
  mailbox_9 = 0x000f,
  mailbox_10 = 0x0010,
  middle = 0x0011,
  my_mailbox = 0x0012,
  rear = 0x0013,
  right = 0x0014,
  side = 0x0015,
  stacker_1 = 0x0016,
  stacker_2 = 0x0017,
  stacker_3 = 0x0018,
  stacker_4 = 0x0019,
  stacker_5 = 0x001a,
  stacker_6 = 0x001b,
  stacker_7 = 0x001c,
  stacker_8 = 0x001d,
  stacker_9 = 0x001e,
  stacker_10 = 0x001f,
  top = 0x0020,
  tray_1 = 0x0021,
  tray_2 = 0x0022,
  tray_3 = 0x0023,
  tray_4 = 0x0024,
  tray_5 = 0x0025,
  tray_6 = 0x0026,
  tray_7 = 0x0027,
  tray_8 = 0x0028,
  tray_9 = 0x0029,
  tray_10 = 0x002a
};
enum class E_job_accounting_sheets_type : uint16_t {
  none = 0x0000,
  standard = 0x0001
};
enum class E_job_collation_type : uint16_t {
  uncollated_sheets = 0x0003,
  collated_documents = 0x0004,
  uncollated_documents = 0x0005
};
enum class E_job_delay_output_until : uint16_t {
  day_time = 0x0000,
  evening = 0x0001,
  indefinite = 0x0002,
  night = 0x0003,
  no_delay_output = 0x0004,
  second_shift = 0x0005,
  third_shift = 0x0006,
  weekend = 0x0007
};
enum class E_job_error_action : uint16_t {
  abort_job = 0x0000,
  cancel_job = 0x0001,
  continue_job = 0x0002,
  suspend_job = 0x0003
};
enum class E_job_error_sheet_when : uint16_t {
  always = 0x0000,
  on_error = 0x0001
};
enum class E_job_hold_until : uint16_t {
  day_time = 0x0000,
  evening = 0x0001,
  indefinite = 0x0002,
  night = 0x0003,
  no_hold = 0x0004,
  second_shift = 0x0005,
  third_shift = 0x0006,
  weekend = 0x0007
};
enum class E_job_mandatory_attributes : uint16_t {};
enum class E_job_password_encryption : uint16_t {
  md2 = 0x0000,
  md4 = 0x0001,
  md5 = 0x0002,
  none = 0x0003,
  sha = 0x0004,
  sha2_224 = 0x0005,
  sha2_256 = 0x0006,
  sha2_384 = 0x0007,
  sha2_512 = 0x0008,
  sha2_512_224 = 0x0009,
  sha2_512_256 = 0x000a,
  sha3_224 = 0x000b,
  sha3_256 = 0x000c,
  sha3_384 = 0x000d,
  sha3_512 = 0x000e,
  sha3_512_224 = 0x000f,
  sha3_512_256 = 0x0010,
  shake_128 = 0x0011,
  shake_256 = 0x0012
};
enum class E_job_sheets : uint16_t {
  first_print_stream_page = 0x0000,
  job_both_sheet = 0x0001,
  job_end_sheet = 0x0002,
  job_start_sheet = 0x0003,
  none = 0x0004,
  standard = 0x0005
};
enum class E_job_spooling_supported : uint16_t {
  automatic = 0x0000,
  spool = 0x0001,
  stream = 0x0002
};
enum class E_job_state : uint16_t {
  pending = 0x0003,
  pending_held = 0x0004,
  processing = 0x0005,
  processing_stopped = 0x0006,
  canceled = 0x0007,
  aborted = 0x0008,
  completed = 0x0009
};
enum class E_job_state_reasons : uint16_t {
  aborted_by_system = 0x0000,
  account_authorization_failed = 0x0001,
  account_closed = 0x0002,
  account_info_needed = 0x0003,
  account_limit_reached = 0x0004,
  compression_error = 0x0005,
  conflicting_attributes = 0x0006,
  connected_to_destination = 0x0007,
  connecting_to_destination = 0x0008,
  destination_uri_failed = 0x0009,
  digital_signature_did_not_verify = 0x000a,
  digital_signature_type_not_supported = 0x000b,
  document_access_error = 0x000c,
  document_format_error = 0x000d,
  document_password_error = 0x000e,
  document_permission_error = 0x000f,
  document_security_error = 0x0010,
  document_unprintable_error = 0x0011,
  errors_detected = 0x0012,
  job_canceled_at_device = 0x0013,
  job_canceled_by_operator = 0x0014,
  job_canceled_by_user = 0x0015,
  job_completed_successfully = 0x0016,
  job_completed_with_errors = 0x0017,
  job_completed_with_warnings = 0x0018,
  job_data_insufficient = 0x0019,
  job_delay_output_until_specified = 0x001a,
  job_digital_signature_wait = 0x001b,
  job_fetchable = 0x001c,
  job_held_for_review = 0x001d,
  job_hold_until_specified = 0x001e,
  job_incoming = 0x001f,
  job_interpreting = 0x0020,
  job_outgoing = 0x0021,
  job_password_wait = 0x0022,
  job_printed_successfully = 0x0023,
  job_printed_with_errors = 0x0024,
  job_printed_with_warnings = 0x0025,
  job_printing = 0x0026,
  job_queued = 0x0027,
  job_queued_for_marker = 0x0028,
  job_release_wait = 0x0029,
  job_restartable = 0x002a,
  job_resuming = 0x002b,
  job_saved_successfully = 0x002c,
  job_saved_with_errors = 0x002d,
  job_saved_with_warnings = 0x002e,
  job_saving = 0x002f,
  job_spooling = 0x0030,
  job_streaming = 0x0031,
  job_suspended = 0x0032,
  job_suspended_by_operator = 0x0033,
  job_suspended_by_system = 0x0034,
  job_suspended_by_user = 0x0035,
  job_suspending = 0x0036,
  job_transferring = 0x0037,
  job_transforming = 0x0038,
  none = 0x0039,
  printer_stopped = 0x003a,
  printer_stopped_partly = 0x003b,
  processing_to_stop_point = 0x003c,
  queued_in_device = 0x003d,
  resources_are_not_ready = 0x003e,
  resources_are_not_supported = 0x003f,
  service_off_line = 0x0040,
  submission_interrupted = 0x0041,
  unsupported_attributes_or_values = 0x0042,
  unsupported_compression = 0x0043,
  unsupported_document_format = 0x0044,
  waiting_for_user_action = 0x0045,
  warnings_detected = 0x0046
};
enum class E_laminating_type : uint16_t {
  archival = 0x0000,
  glossy = 0x0001,
  high_gloss = 0x0002,
  matte = 0x0003,
  semi_gloss = 0x0004,
  translucent = 0x0005
};
enum class E_material_color : uint16_t {
  black = 0x0000,
  blue = 0x0001,
  brown = 0x0002,
  buff = 0x0003,
  clear_black = 0x0004,
  clear_blue = 0x0005,
  clear_brown = 0x0006,
  clear_buff = 0x0007,
  clear_cyan = 0x0008,
  clear_gold = 0x0009,
  clear_goldenrod = 0x000a,
  clear_gray = 0x000b,
  clear_green = 0x000c,
  clear_ivory = 0x000d,
  clear_magenta = 0x000e,
  clear_multi_color = 0x000f,
  clear_mustard = 0x0010,
  clear_orange = 0x0011,
  clear_pink = 0x0012,
  clear_red = 0x0013,
  clear_silver = 0x0014,
  clear_turquoise = 0x0015,
  clear_violet = 0x0016,
  clear_white = 0x0017,
  clear_yellow = 0x0018,
  cyan = 0x0019,
  dark_blue = 0x001a,
  dark_brown = 0x001b,
  dark_buff = 0x001c,
  dark_cyan = 0x001d,
  dark_gold = 0x001e,
  dark_goldenrod = 0x001f,
  dark_gray = 0x0020,
  dark_green = 0x0021,
  dark_ivory = 0x0022,
  dark_magenta = 0x0023,
  dark_mustard = 0x0024,
  dark_orange = 0x0025,
  dark_pink = 0x0026,
  dark_red = 0x0027,
  dark_silver = 0x0028,
  dark_turquoise = 0x0029,
  dark_violet = 0x002a,
  dark_yellow = 0x002b,
  gold = 0x002c,
  goldenrod = 0x002d,
  gray = 0x002e,
  green = 0x002f,
  ivory = 0x0030,
  light_black = 0x0031,
  light_blue = 0x0032,
  light_brown = 0x0033,
  light_buff = 0x0034,
  light_cyan = 0x0035,
  light_gold = 0x0036,
  light_goldenrod = 0x0037,
  light_gray = 0x0038,
  light_green = 0x0039,
  light_ivory = 0x003a,
  light_magenta = 0x003b,
  light_mustard = 0x003c,
  light_orange = 0x003d,
  light_pink = 0x003e,
  light_red = 0x003f,
  light_silver = 0x0040,
  light_turquoise = 0x0041,
  light_violet = 0x0042,
  light_yellow = 0x0043,
  magenta = 0x0044,
  multi_color = 0x0045,
  mustard = 0x0046,
  no_color = 0x0047,
  orange = 0x0048,
  pink = 0x0049,
  red = 0x004a,
  silver = 0x004b,
  turquoise = 0x004c,
  violet = 0x004d,
  white = 0x004e,
  yellow = 0x004f
};
enum class E_media : uint16_t {
  a = 0x0000,
  a_translucent = 0x0001,
  a_transparent = 0x0002,
  a_white = 0x0003,
  arch_a = 0x0004,
  arch_a_translucent = 0x0005,
  arch_a_transparent = 0x0006,
  arch_a_white = 0x0007,
  arch_axsynchro_translucent = 0x0008,
  arch_axsynchro_transparent = 0x0009,
  arch_axsynchro_white = 0x000a,
  arch_b = 0x000b,
  arch_b_translucent = 0x000c,
  arch_b_transparent = 0x000d,
  arch_b_white = 0x000e,
  arch_bxsynchro_translucent = 0x000f,
  arch_bxsynchro_transparent = 0x0010,
  arch_bxsynchro_white = 0x0011,
  arch_c = 0x0012,
  arch_c_translucent = 0x0013,
  arch_c_transparent = 0x0014,
  arch_c_white = 0x0015,
  arch_cxsynchro_translucent = 0x0016,
  arch_cxsynchro_transparent = 0x0017,
  arch_cxsynchro_white = 0x0018,
  arch_d = 0x0019,
  arch_d_translucent = 0x001a,
  arch_d_transparent = 0x001b,
  arch_d_white = 0x001c,
  arch_dxsynchro_translucent = 0x001d,
  arch_dxsynchro_transparent = 0x001e,
  arch_dxsynchro_white = 0x001f,
  arch_e = 0x0020,
  arch_e_translucent = 0x0021,
  arch_e_transparent = 0x0022,
  arch_e_white = 0x0023,
  arch_exsynchro_translucent = 0x0024,
  arch_exsynchro_transparent = 0x0025,
  arch_exsynchro_white = 0x0026,
  asme_f_28x40in = 0x0027,
  auto_fixed_size_translucent = 0x0028,
  auto_fixed_size_transparent = 0x0029,
  auto_fixed_size_white = 0x002a,
  auto_synchro_translucent = 0x002b,
  auto_synchro_transparent = 0x002c,
  auto_synchro_white = 0x002d,
  auto_translucent = 0x002e,
  auto_transparent = 0x002f,
  auto_white = 0x0030,
  axsynchro_translucent = 0x0031,
  axsynchro_transparent = 0x0032,
  axsynchro_white = 0x0033,
  b = 0x0034,
  b_translucent = 0x0035,
  b_transparent = 0x0036,
  b_white = 0x0037,
  bond = 0x0038,
  bottom = 0x0039,
  bxsynchro_translucent = 0x003a,
  bxsynchro_transparent = 0x003b,
  bxsynchro_white = 0x003c,
  by_pass_tray = 0x003d,
  c = 0x003e,
  c_translucent = 0x003f,
  c_transparent = 0x0040,
  c_white = 0x0041,
  choice_iso_a4_210x297mm_na_letter_8_5x11in = 0x0042,
  custom1 = 0x0043,
  custom2 = 0x0044,
  custom3 = 0x0045,
  custom4 = 0x0046,
  custom5 = 0x0047,
  custom6 = 0x0048,
  custom7 = 0x0049,
  custom8 = 0x004a,
  custom9 = 0x004b,
  custom10 = 0x004c,
  cxsynchro_translucent = 0x004d,
  cxsynchro_transparent = 0x004e,
  cxsynchro_white = 0x004f,
  d = 0x0050,
  d_translucent = 0x0051,
  d_transparent = 0x0052,
  d_white = 0x0053,
  default_ = 0x0054,
  dxsynchro_translucent = 0x0055,
  dxsynchro_transparent = 0x0056,
  dxsynchro_white = 0x0057,
  e = 0x0058,
  e_translucent = 0x0059,
  e_transparent = 0x005a,
  e_white = 0x005b,
  envelope = 0x005c,
  executive = 0x005d,
  executive_white = 0x005e,
  exsynchro_translucent = 0x005f,
  exsynchro_transparent = 0x0060,
  exsynchro_white = 0x0061,
  f = 0x0062,
  folio = 0x0063,
  folio_white = 0x0064,
  heavyweight = 0x0065,
  invoice = 0x0066,
  invoice_white = 0x0067,
  iso_a0 = 0x0068,
  iso_a0_translucent = 0x0069,
  iso_a0_transparent = 0x006a,
  iso_a0_white = 0x006b,
  iso_a0xsynchro_translucent = 0x006c,
  iso_a0xsynchro_transparent = 0x006d,
  iso_a0xsynchro_white = 0x006e,
  iso_a1 = 0x006f,
  iso_a1_translucent = 0x0070,
  iso_a1_transparent = 0x0071,
  iso_a1_white = 0x0072,
  iso_a1x3_translucent = 0x0073,
  iso_a1x3_transparent = 0x0074,
  iso_a1x3_white = 0x0075,
  iso_a1x4_translucent = 0x0076,
  iso_a1x4_transparent = 0x0077,
  iso_a1x4_white = 0x0078,
  iso_a1xsynchro_translucent = 0x0079,
  iso_a1xsynchro_transparent = 0x007a,
  iso_a1xsynchro_white = 0x007b,
  iso_a2 = 0x007c,
  iso_a2_translucent = 0x007d,
  iso_a2_transparent = 0x007e,
  iso_a2_white = 0x007f,
  iso_a2x3_translucent = 0x0080,
  iso_a2x3_transparent = 0x0081,
  iso_a2x3_white = 0x0082,
  iso_a2x4_translucent = 0x0083,
  iso_a2x4_transparent = 0x0084,
  iso_a2x4_white = 0x0085,
  iso_a2x5_translucent = 0x0086,
  iso_a2x5_transparent = 0x0087,
  iso_a2x5_white = 0x0088,
  iso_a2xsynchro_translucent = 0x0089,
  iso_a2xsynchro_transparent = 0x008a,
  iso_a2xsynchro_white = 0x008b,
  iso_a3 = 0x008c,
  iso_a3_colored = 0x008d,
  iso_a3_translucent = 0x008e,
  iso_a3_transparent = 0x008f,
  iso_a3_white = 0x0090,
  iso_a3x3_translucent = 0x0091,
  iso_a3x3_transparent = 0x0092,
  iso_a3x3_white = 0x0093,
  iso_a3x4_translucent = 0x0094,
  iso_a3x4_transparent = 0x0095,
  iso_a3x4_white = 0x0096,
  iso_a3x5_translucent = 0x0097,
  iso_a3x5_transparent = 0x0098,
  iso_a3x5_white = 0x0099,
  iso_a3x6_translucent = 0x009a,
  iso_a3x6_transparent = 0x009b,
  iso_a3x6_white = 0x009c,
  iso_a3x7_translucent = 0x009d,
  iso_a3x7_transparent = 0x009e,
  iso_a3x7_white = 0x009f,
  iso_a3xsynchro_translucent = 0x00a0,
  iso_a3xsynchro_transparent = 0x00a1,
  iso_a3xsynchro_white = 0x00a2,
  iso_a4 = 0x00a3,
  iso_a4_colored = 0x00a4,
  iso_a4_translucent = 0x00a5,
  iso_a4_transparent = 0x00a6,
  iso_a4_white = 0x00a7,
  iso_a4x3_translucent = 0x00a8,
  iso_a4x3_transparent = 0x00a9,
  iso_a4x3_white = 0x00aa,
  iso_a4x4_translucent = 0x00ab,
  iso_a4x4_transparent = 0x00ac,
  iso_a4x4_white = 0x00ad,
  iso_a4x5_translucent = 0x00ae,
  iso_a4x5_transparent = 0x00af,
  iso_a4x5_white = 0x00b0,
  iso_a4x6_translucent = 0x00b1,
  iso_a4x6_transparent = 0x00b2,
  iso_a4x6_white = 0x00b3,
  iso_a4x7_translucent = 0x00b4,
  iso_a4x7_transparent = 0x00b5,
  iso_a4x7_white = 0x00b6,
  iso_a4x8_translucent = 0x00b7,
  iso_a4x8_transparent = 0x00b8,
  iso_a4x8_white = 0x00b9,
  iso_a4x9_translucent = 0x00ba,
  iso_a4x9_transparent = 0x00bb,
  iso_a4x9_white = 0x00bc,
  iso_a4xsynchro_translucent = 0x00bd,
  iso_a4xsynchro_transparent = 0x00be,
  iso_a4xsynchro_white = 0x00bf,
  iso_a5 = 0x00c0,
  iso_a5_colored = 0x00c1,
  iso_a5_translucent = 0x00c2,
  iso_a5_transparent = 0x00c3,
  iso_a5_white = 0x00c4,
  iso_a6 = 0x00c5,
  iso_a6_white = 0x00c6,
  iso_a7 = 0x00c7,
  iso_a7_white = 0x00c8,
  iso_a8 = 0x00c9,
  iso_a8_white = 0x00ca,
  iso_a9 = 0x00cb,
  iso_a9_white = 0x00cc,
  iso_a10 = 0x00cd,
  iso_a10_white = 0x00ce,
  iso_b0 = 0x00cf,
  iso_b0_white = 0x00d0,
  iso_b1 = 0x00d1,
  iso_b1_white = 0x00d2,
  iso_b2 = 0x00d3,
  iso_b2_white = 0x00d4,
  iso_b3 = 0x00d5,
  iso_b3_white = 0x00d6,
  iso_b4 = 0x00d7,
  iso_b4_colored = 0x00d8,
  iso_b4_envelope = 0x00d9,
  iso_b4_white = 0x00da,
  iso_b5 = 0x00db,
  iso_b5_colored = 0x00dc,
  iso_b5_envelope = 0x00dd,
  iso_b5_white = 0x00de,
  iso_b6 = 0x00df,
  iso_b6_white = 0x00e0,
  iso_b7 = 0x00e1,
  iso_b7_white = 0x00e2,
  iso_b8 = 0x00e3,
  iso_b8_white = 0x00e4,
  iso_b9 = 0x00e5,
  iso_b9_white = 0x00e6,
  iso_b10 = 0x00e7,
  iso_b10_white = 0x00e8,
  iso_c3 = 0x00e9,
  iso_c3_envelope = 0x00ea,
  iso_c4 = 0x00eb,
  iso_c4_envelope = 0x00ec,
  iso_c5 = 0x00ed,
  iso_c5_envelope = 0x00ee,
  iso_c6 = 0x00ef,
  iso_c6_envelope = 0x00f0,
  iso_designated_long = 0x00f1,
  iso_designated_long_envelope = 0x00f2,
  iso_2a0_1189x1682mm = 0x00f3,
  iso_a0_841x1189mm = 0x00f4,
  iso_a1_594x841mm = 0x00f5,
  iso_a1x3_841x1783mm = 0x00f6,
  iso_a1x4_841x2378mm = 0x00f7,
  iso_a2_420x594mm = 0x00f8,
  iso_a2x3_594x1261mm = 0x00f9,
  iso_a2x4_594x1682mm = 0x00fa,
  iso_a2x5_594x2102mm = 0x00fb,
  iso_a3_extra_322x445mm = 0x00fc,
  iso_a3_297x420mm = 0x00fd,
  iso_a0x3_1189x2523mm = 0x00fe,
  iso_a3x3_420x891mm = 0x00ff,
  iso_a3x4_420x1189mm = 0x0100,
  iso_a3x5_420x1486mm = 0x0101,
  iso_a3x6_420x1783mm = 0x0102,
  iso_a3x7_420x2080mm = 0x0103,
  iso_a4_extra_235_5x322_3mm = 0x0104,
  iso_a4_tab_225x297mm = 0x0105,
  iso_a4_210x297mm = 0x0106,
  iso_a4x3_297x630mm = 0x0107,
  iso_a4x4_297x841mm = 0x0108,
  iso_a4x5_297x1051mm = 0x0109,
  iso_a4x6_297x1261mm = 0x010a,
  iso_a4x7_297x1471mm = 0x010b,
  iso_a4x8_297x1682mm = 0x010c,
  iso_a4x9_297x1892mm = 0x010d,
  iso_a5_extra_174x235mm = 0x010e,
  iso_a5_148x210mm = 0x010f,
  iso_a6_105x148mm = 0x0110,
  iso_a7_74x105mm = 0x0111,
  iso_a8_52x74mm = 0x0112,
  iso_a9_37x52mm = 0x0113,
  iso_a10_26x37mm = 0x0114,
  iso_b0_1000x1414mm = 0x0115,
  iso_b1_707x1000mm = 0x0116,
  iso_b2_500x707mm = 0x0117,
  iso_b3_353x500mm = 0x0118,
  iso_b4_250x353mm = 0x0119,
  iso_b5_extra_201x276mm = 0x011a,
  iso_b5_176x250mm = 0x011b,
  iso_b6_125x176mm = 0x011c,
  iso_b6c4_125x324mm = 0x011d,
  iso_b7_88x125mm = 0x011e,
  iso_b8_62x88mm = 0x011f,
  iso_b9_44x62mm = 0x0120,
  iso_b10_31x44mm = 0x0121,
  iso_c0_917x1297mm = 0x0122,
  iso_c1_648x917mm = 0x0123,
  iso_c2_458x648mm = 0x0124,
  iso_c3_324x458mm = 0x0125,
  iso_c4_229x324mm = 0x0126,
  iso_c5_162x229mm = 0x0127,
  iso_c6_114x162mm = 0x0128,
  iso_c6c5_114x229mm = 0x0129,
  iso_c7_81x114mm = 0x012a,
  iso_c7c6_81x162mm = 0x012b,
  iso_c8_57x81mm = 0x012c,
  iso_c9_40x57mm = 0x012d,
  iso_c10_28x40mm = 0x012e,
  iso_dl_110x220mm = 0x012f,
  iso_id_1_53_98x85_6mm = 0x0130,
  iso_id_3_88x125mm = 0x0131,
  iso_ra0_860x1220mm = 0x0132,
  iso_ra1_610x860mm = 0x0133,
  iso_ra2_430x610mm = 0x0134,
  iso_ra3_305x430mm = 0x0135,
  iso_ra4_215x305mm = 0x0136,
  iso_sra0_900x1280mm = 0x0137,
  iso_sra1_640x900mm = 0x0138,
  iso_sra2_450x640mm = 0x0139,
  iso_sra3_320x450mm = 0x013a,
  iso_sra4_225x320mm = 0x013b,
  jis_b0 = 0x013c,
  jis_b0_translucent = 0x013d,
  jis_b0_transparent = 0x013e,
  jis_b0_white = 0x013f,
  jis_b1 = 0x0140,
  jis_b1_translucent = 0x0141,
  jis_b1_transparent = 0x0142,
  jis_b1_white = 0x0143,
  jis_b2 = 0x0144,
  jis_b2_translucent = 0x0145,
  jis_b2_transparent = 0x0146,
  jis_b2_white = 0x0147,
  jis_b3 = 0x0148,
  jis_b3_translucent = 0x0149,
  jis_b3_transparent = 0x014a,
  jis_b3_white = 0x014b,
  jis_b4 = 0x014c,
  jis_b4_colored = 0x014d,
  jis_b4_translucent = 0x014e,
  jis_b4_transparent = 0x014f,
  jis_b4_white = 0x0150,
  jis_b5 = 0x0151,
  jis_b5_colored = 0x0152,
  jis_b5_translucent = 0x0153,
  jis_b5_transparent = 0x0154,
  jis_b5_white = 0x0155,
  jis_b6 = 0x0156,
  jis_b6_white = 0x0157,
  jis_b7 = 0x0158,
  jis_b7_white = 0x0159,
  jis_b8 = 0x015a,
  jis_b8_white = 0x015b,
  jis_b9 = 0x015c,
  jis_b9_white = 0x015d,
  jis_b10 = 0x015e,
  jis_b10_white = 0x015f,
  jis_b0_1030x1456mm = 0x0160,
  jis_b1_728x1030mm = 0x0161,
  jis_b2_515x728mm = 0x0162,
  jis_b3_364x515mm = 0x0163,
  jis_b4_257x364mm = 0x0164,
  jis_b5_182x257mm = 0x0165,
  jis_b6_128x182mm = 0x0166,
  jis_b7_91x128mm = 0x0167,
  jis_b8_64x91mm = 0x0168,
  jis_b9_45x64mm = 0x0169,
  jis_b10_32x45mm = 0x016a,
  jis_exec_216x330mm = 0x016b,
  jpn_chou2_111_1x146mm = 0x016c,
  jpn_chou3_120x235mm = 0x016d,
  jpn_chou4_90x205mm = 0x016e,
  jpn_hagaki_100x148mm = 0x016f,
  jpn_kahu_240x322_1mm = 0x0170,
  jpn_kaku1_270x382mm = 0x0171,
  jpn_kaku2_240x332mm = 0x0172,
  jpn_kaku3_216x277mm = 0x0173,
  jpn_kaku4_197x267mm = 0x0174,
  jpn_kaku5_190x240mm = 0x0175,
  jpn_kaku7_142x205mm = 0x0176,
  jpn_kaku8_119x197mm = 0x0177,
  jpn_oufuku_148x200mm = 0x0178,
  jpn_you4_105x235mm = 0x0179,
  labels = 0x017a,
  large_capacity = 0x017b,
  ledger = 0x017c,
  ledger_white = 0x017d,
  letterhead = 0x017e,
  main = 0x017f,
  manual = 0x0180,
  middle = 0x0181,
  monarch = 0x0182,
  monarch_envelope = 0x0183,
  na_5x7 = 0x0184,
  na_6x9 = 0x0185,
  na_6x9_envelope = 0x0186,
  na_7x9 = 0x0187,
  na_7x9_envelope = 0x0188,
  na_8x10 = 0x0189,
  na_9x11 = 0x018a,
  na_9x11_envelope = 0x018b,
  na_9x12 = 0x018c,
  na_9x12_envelope = 0x018d,
  na_10x13 = 0x018e,
  na_10x13_envelope = 0x018f,
  na_10x14 = 0x0190,
  na_10x14_envelope = 0x0191,
  na_10x15 = 0x0192,
  na_10x15_envelope = 0x0193,
  na_legal = 0x0194,
  na_legal_colored = 0x0195,
  na_legal_white = 0x0196,
  na_letter = 0x0197,
  na_letter_colored = 0x0198,
  na_letter_transparent = 0x0199,
  na_letter_white = 0x019a,
  na_number_9 = 0x019b,
  na_number_9_envelope = 0x019c,
  na_number_10 = 0x019d,
  na_number_10_envelope = 0x019e,
  na_5x7_5x7in = 0x019f,
  na_6x9_6x9in = 0x01a0,
  na_7x9_7x9in = 0x01a1,
  na_9x11_9x11in = 0x01a2,
  na_10x11_10x11in = 0x01a3,
  na_10x13_10x13in = 0x01a4,
  na_10x14_10x14in = 0x01a5,
  na_10x15_10x15in = 0x01a6,
  na_11x12_11x12in = 0x01a7,
  na_11x15_11x15in = 0x01a8,
  na_12x19_12x19in = 0x01a9,
  na_a2_4_375x5_75in = 0x01aa,
  na_arch_a_9x12in = 0x01ab,
  na_arch_b_12x18in = 0x01ac,
  na_arch_c_18x24in = 0x01ad,
  na_arch_d_24x36in = 0x01ae,
  na_arch_e2_26x38in = 0x01af,
  na_arch_e3_27x39in = 0x01b0,
  na_arch_e_36x48in = 0x01b1,
  na_b_plus_12x19_17in = 0x01b2,
  na_c5_6_5x9_5in = 0x01b3,
  na_c_17x22in = 0x01b4,
  na_d_22x34in = 0x01b5,
  na_e_34x44in = 0x01b6,
  na_edp_11x14in = 0x01b7,
  na_eur_edp_12x14in = 0x01b8,
  na_executive_7_25x10_5in = 0x01b9,
  na_f_44x68in = 0x01ba,
  na_fanfold_eur_8_5x12in = 0x01bb,
  na_fanfold_us_11x14_875in = 0x01bc,
  na_foolscap_8_5x13in = 0x01bd,
  na_govt_legal_8x13in = 0x01be,
  na_govt_letter_8x10in = 0x01bf,
  na_index_3x5_3x5in = 0x01c0,
  na_index_4x6_ext_6x8in = 0x01c1,
  na_index_4x6_4x6in = 0x01c2,
  na_index_5x8_5x8in = 0x01c3,
  na_invoice_5_5x8_5in = 0x01c4,
  na_ledger_11x17in = 0x01c5,
  na_legal_extra_9_5x15in = 0x01c6,
  na_legal_8_5x14in = 0x01c7,
  na_letter_extra_9_5x12in = 0x01c8,
  na_letter_plus_8_5x12_69in = 0x01c9,
  na_letter_8_5x11in = 0x01ca,
  na_monarch_3_875x7_5in = 0x01cb,
  na_number_9_3_875x8_875in = 0x01cc,
  na_number_10_4_125x9_5in = 0x01cd,
  na_number_11_4_5x10_375in = 0x01ce,
  na_number_12_4_75x11in = 0x01cf,
  na_number_14_5x11_5in = 0x01d0,
  na_oficio_8_5x13_4in = 0x01d1,
  na_personal_3_625x6_5in = 0x01d2,
  na_quarto_8_5x10_83in = 0x01d3,
  na_super_a_8_94x14in = 0x01d4,
  na_super_b_13x19in = 0x01d5,
  na_wide_format_30x42in = 0x01d6,
  oe_12x16_12x16in = 0x01d7,
  oe_14x17_14x17in = 0x01d8,
  oe_18x22_18x22in = 0x01d9,
  oe_a2plus_17x24in = 0x01da,
  oe_business_card_2x3_5in = 0x01db,
  oe_photo_10r_10x12in = 0x01dc,
  oe_photo_20r_20x24in = 0x01dd,
  oe_photo_l_3_5x5in = 0x01de,
  oe_photo_s10r_10x15in = 0x01df,
  oe_square_photo_4x4in = 0x01e0,
  oe_square_photo_5x5in = 0x01e1,
  om_16k_184x260mm = 0x01e2,
  om_16k_195x270mm = 0x01e3,
  om_business_card_55x85mm = 0x01e4,
  om_business_card_55x91mm = 0x01e5,
  om_card_54x86mm = 0x01e6,
  om_dai_pa_kai_275x395mm = 0x01e7,
  om_dsc_photo_89x119mm = 0x01e8,
  om_folio_sp_215x315mm = 0x01e9,
  om_folio_210x330mm = 0x01ea,
  om_invite_220x220mm = 0x01eb,
  om_italian_110x230mm = 0x01ec,
  om_juuro_ku_kai_198x275mm = 0x01ed,
  om_large_photo_200x300 = 0x01ee,
  om_medium_photo_130x180mm = 0x01ef,
  om_pa_kai_267x389mm = 0x01f0,
  om_postfix_114x229mm = 0x01f1,
  om_small_photo_100x150mm = 0x01f2,
  om_square_photo_89x89mm = 0x01f3,
  om_wide_photo_100x200mm = 0x01f4,
  plain = 0x01f5,
  prc_1_102x165mm = 0x01f6,
  prc_2_102x176mm = 0x01f7,
  prc_3_125x176mm = 0x01f8,
  prc_4_110x208mm = 0x01f9,
  prc_5_110x220mm = 0x01fa,
  prc_6_120x320mm = 0x01fb,
  prc_7_160x230mm = 0x01fc,
  prc_8_120x309mm = 0x01fd,
  prc_10_324x458mm = 0x01fe,
  prc_16k_146x215mm = 0x01ff,
  prc_32k_97x151mm = 0x0200,
  pre_printed = 0x0201,
  pre_punched = 0x0202,
  quarto = 0x0203,
  quarto_white = 0x0204,
  recycled = 0x0205,
  roc_8k_10_75x15_5in = 0x0206,
  roc_16k_7_75x10_75in = 0x0207,
  side = 0x0208,
  super_b = 0x0209,
  tabloid = 0x020a,
  top = 0x020b,
  transparency = 0x020c,
  tray_1 = 0x020d,
  tray_2 = 0x020e,
  tray_3 = 0x020f,
  tray_4 = 0x0210,
  tray_5 = 0x0211,
  tray_6 = 0x0212,
  tray_7 = 0x0213,
  tray_8 = 0x0214,
  tray_9 = 0x0215,
  tray_10 = 0x0216
};
enum class E_media_back_coating : uint16_t {
  glossy = 0x0000,
  high_gloss = 0x0001,
  matte = 0x0002,
  none = 0x0003,
  satin = 0x0004,
  semi_gloss = 0x0005
};
enum class E_media_grain : uint16_t {
  x_direction = 0x0000,
  y_direction = 0x0001
};
enum class E_media_input_tray_check : uint16_t {
  bottom = 0x0000,
  by_pass_tray = 0x0001,
  envelope = 0x0002,
  large_capacity = 0x0003,
  main = 0x0004,
  manual = 0x0005,
  middle = 0x0006,
  side = 0x0007,
  top = 0x0008,
  tray_1 = 0x0009,
  tray_2 = 0x000a,
  tray_3 = 0x000b,
  tray_4 = 0x000c,
  tray_5 = 0x000d,
  tray_6 = 0x000e,
  tray_7 = 0x000f,
  tray_8 = 0x0010,
  tray_9 = 0x0011,
  tray_10 = 0x0012
};
enum class E_media_key : uint16_t {};
enum class E_media_pre_printed : uint16_t {
  blank = 0x0000,
  letter_head = 0x0001,
  pre_printed = 0x0002
};
enum class E_media_ready : uint16_t {
  a = 0x0000,
  a_translucent = 0x0001,
  a_transparent = 0x0002,
  a_white = 0x0003,
  arch_a = 0x0004,
  arch_a_translucent = 0x0005,
  arch_a_transparent = 0x0006,
  arch_a_white = 0x0007,
  arch_axsynchro_translucent = 0x0008,
  arch_axsynchro_transparent = 0x0009,
  arch_axsynchro_white = 0x000a,
  arch_b = 0x000b,
  arch_b_translucent = 0x000c,
  arch_b_transparent = 0x000d,
  arch_b_white = 0x000e,
  arch_bxsynchro_translucent = 0x000f,
  arch_bxsynchro_transparent = 0x0010,
  arch_bxsynchro_white = 0x0011,
  arch_c = 0x0012,
  arch_c_translucent = 0x0013,
  arch_c_transparent = 0x0014,
  arch_c_white = 0x0015,
  arch_cxsynchro_translucent = 0x0016,
  arch_cxsynchro_transparent = 0x0017,
  arch_cxsynchro_white = 0x0018,
  arch_d = 0x0019,
  arch_d_translucent = 0x001a,
  arch_d_transparent = 0x001b,
  arch_d_white = 0x001c,
  arch_dxsynchro_translucent = 0x001d,
  arch_dxsynchro_transparent = 0x001e,
  arch_dxsynchro_white = 0x001f,
  arch_e = 0x0020,
  arch_e_translucent = 0x0021,
  arch_e_transparent = 0x0022,
  arch_e_white = 0x0023,
  arch_exsynchro_translucent = 0x0024,
  arch_exsynchro_transparent = 0x0025,
  arch_exsynchro_white = 0x0026,
  asme_f_28x40in = 0x0027,
  auto_fixed_size_translucent = 0x0028,
  auto_fixed_size_transparent = 0x0029,
  auto_fixed_size_white = 0x002a,
  auto_synchro_translucent = 0x002b,
  auto_synchro_transparent = 0x002c,
  auto_synchro_white = 0x002d,
  auto_translucent = 0x002e,
  auto_transparent = 0x002f,
  auto_white = 0x0030,
  axsynchro_translucent = 0x0031,
  axsynchro_transparent = 0x0032,
  axsynchro_white = 0x0033,
  b = 0x0034,
  b_translucent = 0x0035,
  b_transparent = 0x0036,
  b_white = 0x0037,
  bxsynchro_translucent = 0x0038,
  bxsynchro_transparent = 0x0039,
  bxsynchro_white = 0x003a,
  c = 0x003b,
  c_translucent = 0x003c,
  c_transparent = 0x003d,
  c_white = 0x003e,
  choice_iso_a4_210x297mm_na_letter_8_5x11in = 0x003f,
  custom1 = 0x0040,
  custom2 = 0x0041,
  custom3 = 0x0042,
  custom4 = 0x0043,
  custom5 = 0x0044,
  custom6 = 0x0045,
  custom7 = 0x0046,
  custom8 = 0x0047,
  custom9 = 0x0048,
  custom10 = 0x0049,
  cxsynchro_translucent = 0x004a,
  cxsynchro_transparent = 0x004b,
  cxsynchro_white = 0x004c,
  d = 0x004d,
  d_translucent = 0x004e,
  d_transparent = 0x004f,
  d_white = 0x0050,
  default_ = 0x0051,
  dxsynchro_translucent = 0x0052,
  dxsynchro_transparent = 0x0053,
  dxsynchro_white = 0x0054,
  e = 0x0055,
  e_translucent = 0x0056,
  e_transparent = 0x0057,
  e_white = 0x0058,
  executive = 0x0059,
  executive_white = 0x005a,
  exsynchro_translucent = 0x005b,
  exsynchro_transparent = 0x005c,
  exsynchro_white = 0x005d,
  f = 0x005e,
  folio = 0x005f,
  folio_white = 0x0060,
  invoice = 0x0061,
  invoice_white = 0x0062,
  iso_a0 = 0x0063,
  iso_a0_translucent = 0x0064,
  iso_a0_transparent = 0x0065,
  iso_a0_white = 0x0066,
  iso_a0xsynchro_translucent = 0x0067,
  iso_a0xsynchro_transparent = 0x0068,
  iso_a0xsynchro_white = 0x0069,
  iso_a1 = 0x006a,
  iso_a1_translucent = 0x006b,
  iso_a1_transparent = 0x006c,
  iso_a1_white = 0x006d,
  iso_a1x3_translucent = 0x006e,
  iso_a1x3_transparent = 0x006f,
  iso_a1x3_white = 0x0070,
  iso_a1x4_translucent = 0x0071,
  iso_a1x4_transparent = 0x0072,
  iso_a1x4_white = 0x0073,
  iso_a1xsynchro_translucent = 0x0074,
  iso_a1xsynchro_transparent = 0x0075,
  iso_a1xsynchro_white = 0x0076,
  iso_a2 = 0x0077,
  iso_a2_translucent = 0x0078,
  iso_a2_transparent = 0x0079,
  iso_a2_white = 0x007a,
  iso_a2x3_translucent = 0x007b,
  iso_a2x3_transparent = 0x007c,
  iso_a2x3_white = 0x007d,
  iso_a2x4_translucent = 0x007e,
  iso_a2x4_transparent = 0x007f,
  iso_a2x4_white = 0x0080,
  iso_a2x5_translucent = 0x0081,
  iso_a2x5_transparent = 0x0082,
  iso_a2x5_white = 0x0083,
  iso_a2xsynchro_translucent = 0x0084,
  iso_a2xsynchro_transparent = 0x0085,
  iso_a2xsynchro_white = 0x0086,
  iso_a3 = 0x0087,
  iso_a3_colored = 0x0088,
  iso_a3_translucent = 0x0089,
  iso_a3_transparent = 0x008a,
  iso_a3_white = 0x008b,
  iso_a3x3_translucent = 0x008c,
  iso_a3x3_transparent = 0x008d,
  iso_a3x3_white = 0x008e,
  iso_a3x4_translucent = 0x008f,
  iso_a3x4_transparent = 0x0090,
  iso_a3x4_white = 0x0091,
  iso_a3x5_translucent = 0x0092,
  iso_a3x5_transparent = 0x0093,
  iso_a3x5_white = 0x0094,
  iso_a3x6_translucent = 0x0095,
  iso_a3x6_transparent = 0x0096,
  iso_a3x6_white = 0x0097,
  iso_a3x7_translucent = 0x0098,
  iso_a3x7_transparent = 0x0099,
  iso_a3x7_white = 0x009a,
  iso_a3xsynchro_translucent = 0x009b,
  iso_a3xsynchro_transparent = 0x009c,
  iso_a3xsynchro_white = 0x009d,
  iso_a4 = 0x009e,
  iso_a4_colored = 0x009f,
  iso_a4_translucent = 0x00a0,
  iso_a4_transparent = 0x00a1,
  iso_a4_white = 0x00a2,
  iso_a4x3_translucent = 0x00a3,
  iso_a4x3_transparent = 0x00a4,
  iso_a4x3_white = 0x00a5,
  iso_a4x4_translucent = 0x00a6,
  iso_a4x4_transparent = 0x00a7,
  iso_a4x4_white = 0x00a8,
  iso_a4x5_translucent = 0x00a9,
  iso_a4x5_transparent = 0x00aa,
  iso_a4x5_white = 0x00ab,
  iso_a4x6_translucent = 0x00ac,
  iso_a4x6_transparent = 0x00ad,
  iso_a4x6_white = 0x00ae,
  iso_a4x7_translucent = 0x00af,
  iso_a4x7_transparent = 0x00b0,
  iso_a4x7_white = 0x00b1,
  iso_a4x8_translucent = 0x00b2,
  iso_a4x8_transparent = 0x00b3,
  iso_a4x8_white = 0x00b4,
  iso_a4x9_translucent = 0x00b5,
  iso_a4x9_transparent = 0x00b6,
  iso_a4x9_white = 0x00b7,
  iso_a4xsynchro_translucent = 0x00b8,
  iso_a4xsynchro_transparent = 0x00b9,
  iso_a4xsynchro_white = 0x00ba,
  iso_a5 = 0x00bb,
  iso_a5_colored = 0x00bc,
  iso_a5_translucent = 0x00bd,
  iso_a5_transparent = 0x00be,
  iso_a5_white = 0x00bf,
  iso_a6 = 0x00c0,
  iso_a6_white = 0x00c1,
  iso_a7 = 0x00c2,
  iso_a7_white = 0x00c3,
  iso_a8 = 0x00c4,
  iso_a8_white = 0x00c5,
  iso_a9 = 0x00c6,
  iso_a9_white = 0x00c7,
  iso_a10 = 0x00c8,
  iso_a10_white = 0x00c9,
  iso_b0 = 0x00ca,
  iso_b0_white = 0x00cb,
  iso_b1 = 0x00cc,
  iso_b1_white = 0x00cd,
  iso_b2 = 0x00ce,
  iso_b2_white = 0x00cf,
  iso_b3 = 0x00d0,
  iso_b3_white = 0x00d1,
  iso_b4 = 0x00d2,
  iso_b4_colored = 0x00d3,
  iso_b4_white = 0x00d4,
  iso_b5 = 0x00d5,
  iso_b5_colored = 0x00d6,
  iso_b5_white = 0x00d7,
  iso_b6 = 0x00d8,
  iso_b6_white = 0x00d9,
  iso_b7 = 0x00da,
  iso_b7_white = 0x00db,
  iso_b8 = 0x00dc,
  iso_b8_white = 0x00dd,
  iso_b9 = 0x00de,
  iso_b9_white = 0x00df,
  iso_b10 = 0x00e0,
  iso_b10_white = 0x00e1,
  iso_c3 = 0x00e2,
  iso_c4 = 0x00e3,
  iso_c5 = 0x00e4,
  iso_c6 = 0x00e5,
  iso_designated_long = 0x00e6,
  iso_2a0_1189x1682mm = 0x00e7,
  iso_a0_841x1189mm = 0x00e8,
  iso_a1_594x841mm = 0x00e9,
  iso_a1x3_841x1783mm = 0x00ea,
  iso_a1x4_841x2378mm = 0x00eb,
  iso_a2_420x594mm = 0x00ec,
  iso_a2x3_594x1261mm = 0x00ed,
  iso_a2x4_594x1682mm = 0x00ee,
  iso_a2x5_594x2102mm = 0x00ef,
  iso_a3_extra_322x445mm = 0x00f0,
  iso_a3_297x420mm = 0x00f1,
  iso_a0x3_1189x2523mm = 0x00f2,
  iso_a3x3_420x891mm = 0x00f3,
  iso_a3x4_420x1189mm = 0x00f4,
  iso_a3x5_420x1486mm = 0x00f5,
  iso_a3x6_420x1783mm = 0x00f6,
  iso_a3x7_420x2080mm = 0x00f7,
  iso_a4_extra_235_5x322_3mm = 0x00f8,
  iso_a4_tab_225x297mm = 0x00f9,
  iso_a4_210x297mm = 0x00fa,
  iso_a4x3_297x630mm = 0x00fb,
  iso_a4x4_297x841mm = 0x00fc,
  iso_a4x5_297x1051mm = 0x00fd,
  iso_a4x6_297x1261mm = 0x00fe,
  iso_a4x7_297x1471mm = 0x00ff,
  iso_a4x8_297x1682mm = 0x0100,
  iso_a4x9_297x1892mm = 0x0101,
  iso_a5_extra_174x235mm = 0x0102,
  iso_a5_148x210mm = 0x0103,
  iso_a6_105x148mm = 0x0104,
  iso_a7_74x105mm = 0x0105,
  iso_a8_52x74mm = 0x0106,
  iso_a9_37x52mm = 0x0107,
  iso_a10_26x37mm = 0x0108,
  iso_b0_1000x1414mm = 0x0109,
  iso_b1_707x1000mm = 0x010a,
  iso_b2_500x707mm = 0x010b,
  iso_b3_353x500mm = 0x010c,
  iso_b4_250x353mm = 0x010d,
  iso_b5_extra_201x276mm = 0x010e,
  iso_b5_176x250mm = 0x010f,
  iso_b6_125x176mm = 0x0110,
  iso_b6c4_125x324mm = 0x0111,
  iso_b7_88x125mm = 0x0112,
  iso_b8_62x88mm = 0x0113,
  iso_b9_44x62mm = 0x0114,
  iso_b10_31x44mm = 0x0115,
  iso_c0_917x1297mm = 0x0116,
  iso_c1_648x917mm = 0x0117,
  iso_c2_458x648mm = 0x0118,
  iso_c3_324x458mm = 0x0119,
  iso_c4_229x324mm = 0x011a,
  iso_c5_162x229mm = 0x011b,
  iso_c6_114x162mm = 0x011c,
  iso_c6c5_114x229mm = 0x011d,
  iso_c7_81x114mm = 0x011e,
  iso_c7c6_81x162mm = 0x011f,
  iso_c8_57x81mm = 0x0120,
  iso_c9_40x57mm = 0x0121,
  iso_c10_28x40mm = 0x0122,
  iso_dl_110x220mm = 0x0123,
  iso_id_1_53_98x85_6mm = 0x0124,
  iso_id_3_88x125mm = 0x0125,
  iso_ra0_860x1220mm = 0x0126,
  iso_ra1_610x860mm = 0x0127,
  iso_ra2_430x610mm = 0x0128,
  iso_ra3_305x430mm = 0x0129,
  iso_ra4_215x305mm = 0x012a,
  iso_sra0_900x1280mm = 0x012b,
  iso_sra1_640x900mm = 0x012c,
  iso_sra2_450x640mm = 0x012d,
  iso_sra3_320x450mm = 0x012e,
  iso_sra4_225x320mm = 0x012f,
  jis_b0 = 0x0130,
  jis_b0_translucent = 0x0131,
  jis_b0_transparent = 0x0132,
  jis_b0_white = 0x0133,
  jis_b1 = 0x0134,
  jis_b1_translucent = 0x0135,
  jis_b1_transparent = 0x0136,
  jis_b1_white = 0x0137,
  jis_b2 = 0x0138,
  jis_b2_translucent = 0x0139,
  jis_b2_transparent = 0x013a,
  jis_b2_white = 0x013b,
  jis_b3 = 0x013c,
  jis_b3_translucent = 0x013d,
  jis_b3_transparent = 0x013e,
  jis_b3_white = 0x013f,
  jis_b4 = 0x0140,
  jis_b4_colored = 0x0141,
  jis_b4_translucent = 0x0142,
  jis_b4_transparent = 0x0143,
  jis_b4_white = 0x0144,
  jis_b5 = 0x0145,
  jis_b5_colored = 0x0146,
  jis_b5_translucent = 0x0147,
  jis_b5_transparent = 0x0148,
  jis_b5_white = 0x0149,
  jis_b6 = 0x014a,
  jis_b6_white = 0x014b,
  jis_b7 = 0x014c,
  jis_b7_white = 0x014d,
  jis_b8 = 0x014e,
  jis_b8_white = 0x014f,
  jis_b9 = 0x0150,
  jis_b9_white = 0x0151,
  jis_b10 = 0x0152,
  jis_b10_white = 0x0153,
  jis_b0_1030x1456mm = 0x0154,
  jis_b1_728x1030mm = 0x0155,
  jis_b2_515x728mm = 0x0156,
  jis_b3_364x515mm = 0x0157,
  jis_b4_257x364mm = 0x0158,
  jis_b5_182x257mm = 0x0159,
  jis_b6_128x182mm = 0x015a,
  jis_b7_91x128mm = 0x015b,
  jis_b8_64x91mm = 0x015c,
  jis_b9_45x64mm = 0x015d,
  jis_b10_32x45mm = 0x015e,
  jis_exec_216x330mm = 0x015f,
  jpn_chou2_111_1x146mm = 0x0160,
  jpn_chou3_120x235mm = 0x0161,
  jpn_chou4_90x205mm = 0x0162,
  jpn_hagaki_100x148mm = 0x0163,
  jpn_kahu_240x322_1mm = 0x0164,
  jpn_kaku1_270x382mm = 0x0165,
  jpn_kaku2_240x332mm = 0x0166,
  jpn_kaku3_216x277mm = 0x0167,
  jpn_kaku4_197x267mm = 0x0168,
  jpn_kaku5_190x240mm = 0x0169,
  jpn_kaku7_142x205mm = 0x016a,
  jpn_kaku8_119x197mm = 0x016b,
  jpn_oufuku_148x200mm = 0x016c,
  jpn_you4_105x235mm = 0x016d,
  ledger = 0x016e,
  ledger_white = 0x016f,
  monarch = 0x0170,
  na_5x7 = 0x0171,
  na_6x9 = 0x0172,
  na_7x9 = 0x0173,
  na_8x10 = 0x0174,
  na_9x11 = 0x0175,
  na_9x12 = 0x0176,
  na_10x13 = 0x0177,
  na_10x14 = 0x0178,
  na_10x15 = 0x0179,
  na_legal = 0x017a,
  na_legal_colored = 0x017b,
  na_legal_white = 0x017c,
  na_letter = 0x017d,
  na_letter_colored = 0x017e,
  na_letter_transparent = 0x017f,
  na_letter_white = 0x0180,
  na_number_9 = 0x0181,
  na_number_10 = 0x0182,
  na_5x7_5x7in = 0x0183,
  na_6x9_6x9in = 0x0184,
  na_7x9_7x9in = 0x0185,
  na_9x11_9x11in = 0x0186,
  na_10x11_10x11in = 0x0187,
  na_10x13_10x13in = 0x0188,
  na_10x14_10x14in = 0x0189,
  na_10x15_10x15in = 0x018a,
  na_11x12_11x12in = 0x018b,
  na_11x15_11x15in = 0x018c,
  na_12x19_12x19in = 0x018d,
  na_a2_4_375x5_75in = 0x018e,
  na_arch_a_9x12in = 0x018f,
  na_arch_b_12x18in = 0x0190,
  na_arch_c_18x24in = 0x0191,
  na_arch_d_24x36in = 0x0192,
  na_arch_e2_26x38in = 0x0193,
  na_arch_e3_27x39in = 0x0194,
  na_arch_e_36x48in = 0x0195,
  na_b_plus_12x19_17in = 0x0196,
  na_c5_6_5x9_5in = 0x0197,
  na_c_17x22in = 0x0198,
  na_d_22x34in = 0x0199,
  na_e_34x44in = 0x019a,
  na_edp_11x14in = 0x019b,
  na_eur_edp_12x14in = 0x019c,
  na_executive_7_25x10_5in = 0x019d,
  na_f_44x68in = 0x019e,
  na_fanfold_eur_8_5x12in = 0x019f,
  na_fanfold_us_11x14_875in = 0x01a0,
  na_foolscap_8_5x13in = 0x01a1,
  na_govt_legal_8x13in = 0x01a2,
  na_govt_letter_8x10in = 0x01a3,
  na_index_3x5_3x5in = 0x01a4,
  na_index_4x6_ext_6x8in = 0x01a5,
  na_index_4x6_4x6in = 0x01a6,
  na_index_5x8_5x8in = 0x01a7,
  na_invoice_5_5x8_5in = 0x01a8,
  na_ledger_11x17in = 0x01a9,
  na_legal_extra_9_5x15in = 0x01aa,
  na_legal_8_5x14in = 0x01ab,
  na_letter_extra_9_5x12in = 0x01ac,
  na_letter_plus_8_5x12_69in = 0x01ad,
  na_letter_8_5x11in = 0x01ae,
  na_monarch_3_875x7_5in = 0x01af,
  na_number_9_3_875x8_875in = 0x01b0,
  na_number_10_4_125x9_5in = 0x01b1,
  na_number_11_4_5x10_375in = 0x01b2,
  na_number_12_4_75x11in = 0x01b3,
  na_number_14_5x11_5in = 0x01b4,
  na_oficio_8_5x13_4in = 0x01b5,
  na_personal_3_625x6_5in = 0x01b6,
  na_quarto_8_5x10_83in = 0x01b7,
  na_super_a_8_94x14in = 0x01b8,
  na_super_b_13x19in = 0x01b9,
  na_wide_format_30x42in = 0x01ba,
  oe_12x16_12x16in = 0x01bb,
  oe_14x17_14x17in = 0x01bc,
  oe_18x22_18x22in = 0x01bd,
  oe_a2plus_17x24in = 0x01be,
  oe_business_card_2x3_5in = 0x01bf,
  oe_photo_10r_10x12in = 0x01c0,
  oe_photo_20r_20x24in = 0x01c1,
  oe_photo_l_3_5x5in = 0x01c2,
  oe_photo_s10r_10x15in = 0x01c3,
  oe_square_photo_4x4in = 0x01c4,
  oe_square_photo_5x5in = 0x01c5,
  om_16k_184x260mm = 0x01c6,
  om_16k_195x270mm = 0x01c7,
  om_business_card_55x85mm = 0x01c8,
  om_business_card_55x91mm = 0x01c9,
  om_card_54x86mm = 0x01ca,
  om_dai_pa_kai_275x395mm = 0x01cb,
  om_dsc_photo_89x119mm = 0x01cc,
  om_folio_sp_215x315mm = 0x01cd,
  om_folio_210x330mm = 0x01ce,
  om_invite_220x220mm = 0x01cf,
  om_italian_110x230mm = 0x01d0,
  om_juuro_ku_kai_198x275mm = 0x01d1,
  om_large_photo_200x300 = 0x01d2,
  om_medium_photo_130x180mm = 0x01d3,
  om_pa_kai_267x389mm = 0x01d4,
  om_postfix_114x229mm = 0x01d5,
  om_small_photo_100x150mm = 0x01d6,
  om_square_photo_89x89mm = 0x01d7,
  om_wide_photo_100x200mm = 0x01d8,
  prc_1_102x165mm = 0x01d9,
  prc_2_102x176mm = 0x01da,
  prc_3_125x176mm = 0x01db,
  prc_4_110x208mm = 0x01dc,
  prc_5_110x220mm = 0x01dd,
  prc_6_120x320mm = 0x01de,
  prc_7_160x230mm = 0x01df,
  prc_8_120x309mm = 0x01e0,
  prc_10_324x458mm = 0x01e1,
  prc_16k_146x215mm = 0x01e2,
  prc_32k_97x151mm = 0x01e3,
  quarto = 0x01e4,
  quarto_white = 0x01e5,
  roc_8k_10_75x15_5in = 0x01e6,
  roc_16k_7_75x10_75in = 0x01e7,
  super_b = 0x01e8,
  tabloid = 0x01e9
};
enum class E_media_source : uint16_t {
  alternate = 0x0000,
  alternate_roll = 0x0001,
  auto_ = 0x0002,
  bottom = 0x0003,
  by_pass_tray = 0x0004,
  center = 0x0005,
  disc = 0x0006,
  envelope = 0x0007,
  hagaki = 0x0008,
  large_capacity = 0x0009,
  left = 0x000a,
  main = 0x000b,
  main_roll = 0x000c,
  manual = 0x000d,
  middle = 0x000e,
  photo = 0x000f,
  rear = 0x0010,
  right = 0x0011,
  roll_1 = 0x0012,
  roll_2 = 0x0013,
  roll_3 = 0x0014,
  roll_4 = 0x0015,
  roll_5 = 0x0016,
  roll_6 = 0x0017,
  roll_7 = 0x0018,
  roll_8 = 0x0019,
  roll_9 = 0x001a,
  roll_10 = 0x001b,
  side = 0x001c,
  top = 0x001d,
  tray_1 = 0x001e,
  tray_2 = 0x001f,
  tray_3 = 0x0020,
  tray_4 = 0x0021,
  tray_5 = 0x0022,
  tray_6 = 0x0023,
  tray_7 = 0x0024,
  tray_8 = 0x0025,
  tray_9 = 0x0026,
  tray_10 = 0x0027,
  tray_11 = 0x0028,
  tray_12 = 0x0029,
  tray_13 = 0x002a,
  tray_14 = 0x002b,
  tray_15 = 0x002c,
  tray_16 = 0x002d,
  tray_17 = 0x002e,
  tray_18 = 0x002f,
  tray_19 = 0x0030,
  tray_20 = 0x0031
};
enum class E_media_tooth : uint16_t {
  antique = 0x0000,
  calendared = 0x0001,
  coarse = 0x0002,
  fine = 0x0003,
  linen = 0x0004,
  medium = 0x0005,
  smooth = 0x0006,
  stipple = 0x0007,
  uncalendared = 0x0008,
  vellum = 0x0009
};
enum class E_media_type : uint16_t {
  aluminum = 0x0000,
  auto_ = 0x0001,
  back_print_film = 0x0002,
  cardboard = 0x0003,
  cardstock = 0x0004,
  cd = 0x0005,
  continuous = 0x0006,
  continuous_long = 0x0007,
  continuous_short = 0x0008,
  corrugated_board = 0x0009,
  disc = 0x000a,
  disc_glossy = 0x000b,
  disc_high_gloss = 0x000c,
  disc_matte = 0x000d,
  disc_satin = 0x000e,
  disc_semi_gloss = 0x000f,
  double_wall = 0x0010,
  dry_film = 0x0011,
  dvd = 0x0012,
  embossing_foil = 0x0013,
  end_board = 0x0014,
  envelope = 0x0015,
  envelope_archival = 0x0016,
  envelope_bond = 0x0017,
  envelope_coated = 0x0018,
  envelope_cotton = 0x0019,
  envelope_fine = 0x001a,
  envelope_heavyweight = 0x001b,
  envelope_inkjet = 0x001c,
  envelope_lightweight = 0x001d,
  envelope_plain = 0x001e,
  envelope_preprinted = 0x001f,
  envelope_window = 0x0020,
  fabric = 0x0021,
  fabric_archival = 0x0022,
  fabric_glossy = 0x0023,
  fabric_high_gloss = 0x0024,
  fabric_matte = 0x0025,
  fabric_semi_gloss = 0x0026,
  fabric_waterproof = 0x0027,
  film = 0x0028,
  flexo_base = 0x0029,
  flexo_photo_polymer = 0x002a,
  flute = 0x002b,
  foil = 0x002c,
  full_cut_tabs = 0x002d,
  glass = 0x002e,
  glass_colored = 0x002f,
  glass_opaque = 0x0030,
  glass_surfaced = 0x0031,
  glass_textured = 0x0032,
  gravure_cylinder = 0x0033,
  image_setter_paper = 0x0034,
  imaging_cylinder = 0x0035,
  labels = 0x0036,
  labels_colored = 0x0037,
  labels_glossy = 0x0038,
  labels_high_gloss = 0x0039,
  labels_inkjet = 0x003a,
  labels_matte = 0x003b,
  labels_permanent = 0x003c,
  labels_satin = 0x003d,
  labels_security = 0x003e,
  labels_semi_gloss = 0x003f,
  laminating_foil = 0x0040,
  letterhead = 0x0041,
  metal = 0x0042,
  metal_glossy = 0x0043,
  metal_high_gloss = 0x0044,
  metal_matte = 0x0045,
  metal_satin = 0x0046,
  metal_semi_gloss = 0x0047,
  mounting_tape = 0x0048,
  multi_layer = 0x0049,
  multi_part_form = 0x004a,
  other = 0x004b,
  paper = 0x004c,
  photographic = 0x004d,
  photographic_archival = 0x004e,
  photographic_film = 0x004f,
  photographic_glossy = 0x0050,
  photographic_high_gloss = 0x0051,
  photographic_matte = 0x0052,
  photographic_satin = 0x0053,
  photographic_semi_gloss = 0x0054,
  plastic = 0x0055,
  plastic_archival = 0x0056,
  plastic_colored = 0x0057,
  plastic_glossy = 0x0058,
  plastic_high_gloss = 0x0059,
  plastic_matte = 0x005a,
  plastic_satin = 0x005b,
  plastic_semi_gloss = 0x005c,
  plate = 0x005d,
  polyester = 0x005e,
  pre_cut_tabs = 0x005f,
  roll = 0x0060,
  screen = 0x0061,
  screen_paged = 0x0062,
  self_adhesive = 0x0063,
  self_adhesive_film = 0x0064,
  shrink_foil = 0x0065,
  single_face = 0x0066,
  single_wall = 0x0067,
  sleeve = 0x0068,
  stationery = 0x0069,
  stationery_archival = 0x006a,
  stationery_coated = 0x006b,
  stationery_cotton = 0x006c,
  stationery_fine = 0x006d,
  stationery_heavyweight = 0x006e,
  stationery_heavyweight_coated = 0x006f,
  stationery_inkjet = 0x0070,
  stationery_letterhead = 0x0071,
  stationery_lightweight = 0x0072,
  stationery_preprinted = 0x0073,
  stationery_prepunched = 0x0074,
  tab_stock = 0x0075,
  tractor = 0x0076,
  transfer = 0x0077,
  transparency = 0x0078,
  triple_wall = 0x0079,
  wet_film = 0x007a
};
enum class E_multiple_document_handling : uint16_t {
  separate_documents_collated_copies = 0x0000,
  separate_documents_uncollated_copies = 0x0001,
  single_document = 0x0002,
  single_document_new_sheet = 0x0003
};
enum class E_multiple_operation_time_out_action : uint16_t {
  abort_job = 0x0000,
  hold_job = 0x0001,
  process_job = 0x0002
};
enum class E_notify_events : uint16_t {
  document_completed = 0x0000,
  document_config_changed = 0x0001,
  document_created = 0x0002,
  document_fetchable = 0x0003,
  document_state_changed = 0x0004,
  document_stopped = 0x0005,
  job_completed = 0x0006,
  job_config_changed = 0x0007,
  job_created = 0x0008,
  job_fetchable = 0x0009,
  job_progress = 0x000a,
  job_state_changed = 0x000b,
  job_stopped = 0x000c,
  none = 0x000d,
  printer_config_changed = 0x000e,
  printer_created = 0x000f,
  printer_deleted = 0x0010,
  printer_finishings_changed = 0x0011,
  printer_media_changed = 0x0012,
  printer_queue_order_changed = 0x0013,
  printer_restarted = 0x0014,
  printer_shutdown = 0x0015,
  printer_state_changed = 0x0016,
  printer_stopped = 0x0017,
  resource_canceled = 0x0018,
  resource_config_changed = 0x0019,
  resource_created = 0x001a,
  resource_installed = 0x001b,
  resource_state_changed = 0x001c,
  system_config_changed = 0x001d,
  system_restarted = 0x001e,
  system_shutdown = 0x001f,
  system_state_changed = 0x0020,
  system_stopped = 0x0021
};
enum class E_notify_pull_method : uint16_t { ippget = 0x0000 };
enum class E_operations_supported : int16_t {
  Print_Job = 0x0002,
  Print_URI = 0x0003,
  Validate_Job = 0x0004,
  Create_Job = 0x0005,
  Send_Document = 0x0006,
  Send_URI = 0x0007,
  Cancel_Job = 0x0008,
  Get_Job_Attributes = 0x0009,
  Get_Jobs = 0x000a,
  Get_Printer_Attributes = 0x000b,
  Hold_Job = 0x000c,
  Release_Job = 0x000d,
  Restart_Job = 0x000e,
  Pause_Printer = 0x0010,
  Resume_Printer = 0x0011,
  Purge_Jobs = 0x0012,
  Set_Printer_Attributes = 0x0013,
  Set_Job_Attributes = 0x0014,
  Get_Printer_Supported_Values = 0x0015,
  Create_Printer_Subscriptions = 0x0016,
  Create_Job_Subscriptions = 0x0017,
  Get_Subscription_Attributes = 0x0018,
  Get_Subscriptions = 0x0019,
  Renew_Subscription = 0x001a,
  Cancel_Subscription = 0x001b,
  Get_Notifications = 0x001c,
  Get_Resource_Attributes = 0x001e,
  Get_Resources = 0x0020,
  Enable_Printer = 0x0022,
  Disable_Printer = 0x0023,
  Pause_Printer_After_Current_Job = 0x0024,
  Hold_New_Jobs = 0x0025,
  Release_Held_New_Jobs = 0x0026,
  Deactivate_Printer = 0x0027,
  Activate_Printer = 0x0028,
  Restart_Printer = 0x0029,
  Shutdown_Printer = 0x002a,
  Startup_Printer = 0x002b,
  Reprocess_Job = 0x002c,
  Cancel_Current_Job = 0x002d,
  Suspend_Current_Job = 0x002e,
  Resume_Job = 0x002f,
  Promote_Job = 0x0030,
  Schedule_Job_After = 0x0031,
  Cancel_Document = 0x0033,
  Get_Document_Attributes = 0x0034,
  Get_Documents = 0x0035,
  Delete_Document = 0x0036,
  Set_Document_Attributes = 0x0037,
  Cancel_Jobs = 0x0038,
  Cancel_My_Jobs = 0x0039,
  Resubmit_Job = 0x003a,
  Close_Job = 0x003b,
  Identify_Printer = 0x003c,
  Validate_Document = 0x003d,
  Add_Document_Images = 0x003e,
  Acknowledge_Document = 0x003f,
  Acknowledge_Identify_Printer = 0x0040,
  Acknowledge_Job = 0x0041,
  Fetch_Document = 0x0042,
  Fetch_Job = 0x0043,
  Get_Output_Device_Attributes = 0x0044,
  Update_Active_Jobs = 0x0045,
  Deregister_Output_Device = 0x0046,
  Update_Document_Status = 0x0047,
  Update_Job_Status = 0x0048,
  Update_Output_Device_Attributes = 0x0049,
  Get_Next_Document_Data = 0x004a,
  Allocate_Printer_Resources = 0x004b,
  Create_Printer = 0x004c,
  Deallocate_Printer_Resources = 0x004d,
  Delete_Printer = 0x004e,
  Get_Printers = 0x004f,
  Shutdown_One_Printer = 0x0050,
  Startup_One_Printer = 0x0051,
  Cancel_Resource = 0x0052,
  Create_Resource = 0x0053,
  Install_Resource = 0x0054,
  Send_Resource_Data = 0x0055,
  Set_Resource_Attributes = 0x0056,
  Create_Resource_Subscriptions = 0x0057,
  Create_System_Subscriptions = 0x0058,
  Disable_All_Printers = 0x0059,
  Enable_All_Printers = 0x005a,
  Get_System_Attributes = 0x005b,
  Get_System_Supported_Values = 0x005c,
  Pause_All_Printers = 0x005d,
  Pause_All_Printers_After_Current_Job = 0x005e,
  Register_Output_Device = 0x005f,
  Restart_System = 0x0060,
  Resume_All_Printers = 0x0061,
  Set_System_Attributes = 0x0062,
  Shutdown_All_Printers = 0x0063,
  Startup_All_Printers = 0x0064,
  Get_Printer_Resources = 0x0065,
  Get_User_Printer_Attributes = 0x0066,
  Restart_One_Printer = 0x0067,
  CUPS_Get_Default = 0x4001,
  CUPS_Get_Printers = 0x4002,
  CUPS_Add_Modify_Printer = 0x4003,
  CUPS_Delete_Printer = 0x4004,
  CUPS_Get_Classes = 0x4005,
  CUPS_Add_Modify_Class = 0x4006,
  CUPS_Delete_Class = 0x4007,
  CUPS_Accept_Jobs = 0x4008,
  CUPS_Reject_Jobs = 0x4009,
  CUPS_Set_Default = 0x400a,
  CUPS_Get_Devices = 0x400b,
  CUPS_Get_PPDs = 0x400c,
  CUPS_Move_Job = 0x400d,
  CUPS_Authenticate_Job = 0x400e,
  CUPS_Get_PPD = 0x400f,
  CUPS_Get_Document = 0x4027,
  CUPS_Create_Local_Printer = 0x4028
};
enum class E_page_delivery : uint16_t {
  reverse_order_face_down = 0x0000,
  reverse_order_face_up = 0x0001,
  same_order_face_down = 0x0002,
  same_order_face_up = 0x0003,
  system_specified = 0x0004
};
enum class E_pdf_versions_supported : uint16_t {
  adobe_1_3 = 0x0000,
  adobe_1_4 = 0x0001,
  adobe_1_5 = 0x0002,
  adobe_1_6 = 0x0003,
  iso_15930_1_2001 = 0x0004,
  iso_15930_3_2002 = 0x0005,
  iso_15930_4_2003 = 0x0006,
  iso_15930_6_2003 = 0x0007,
  iso_15930_7_2010 = 0x0008,
  iso_15930_8_2010 = 0x0009,
  iso_16612_2_2010 = 0x000a,
  iso_19005_1_2005 = 0x000b,
  iso_19005_2_2011 = 0x000c,
  iso_19005_3_2012 = 0x000d,
  iso_32000_1_2008 = 0x000e,
  none = 0x000f,
  pwg_5102_3 = 0x0010
};
enum class E_pdl_init_file_supported : uint16_t {
  pdl_init_file_entry = 0x0000,
  pdl_init_file_location = 0x0001,
  pdl_init_file_name = 0x0002
};
enum class E_pdl_override_supported : uint16_t {
  attempted = 0x0000,
  guaranteed = 0x0001,
  not_attempted = 0x0002
};
enum class E_presentation_direction_number_up : uint16_t {
  tobottom_toleft = 0x0000,
  tobottom_toright = 0x0001,
  toleft_tobottom = 0x0002,
  toleft_totop = 0x0003,
  toright_tobottom = 0x0004,
  toright_totop = 0x0005,
  totop_toleft = 0x0006,
  totop_toright = 0x0007
};
enum class E_print_color_mode : uint16_t {
  auto_ = 0x0000,
  auto_monochrome = 0x0001,
  bi_level = 0x0002,
  color = 0x0003,
  highlight = 0x0004,
  monochrome = 0x0005,
  process_bi_level = 0x0006,
  process_monochrome = 0x0007
};
enum class E_print_content_optimize : uint16_t {
  auto_ = 0x0000,
  graphic = 0x0001,
  photo = 0x0002,
  text = 0x0003,
  text_and_graphic = 0x0004
};
enum class E_print_rendering_intent : uint16_t {
  absolute = 0x0000,
  auto_ = 0x0001,
  perceptual = 0x0002,
  relative = 0x0003,
  relative_bpc = 0x0004,
  saturation = 0x0005
};
enum class E_print_scaling : uint16_t {
  auto_ = 0x0000,
  auto_fit = 0x0001,
  fill = 0x0002,
  fit = 0x0003,
  none = 0x0004
};
enum class E_printer_state : uint16_t {
  idle = 0x0003,
  processing = 0x0004,
  stopped = 0x0005
};
enum class E_printer_state_reasons : uint16_t {
  alert_removal_of_binary_change_entry = 0x0000,
  bander_added = 0x0001,
  bander_almost_empty = 0x0002,
  bander_almost_full = 0x0003,
  bander_at_limit = 0x0004,
  bander_closed = 0x0005,
  bander_configuration_change = 0x0006,
  bander_cover_closed = 0x0007,
  bander_cover_open = 0x0008,
  bander_empty = 0x0009,
  bander_full = 0x000a,
  bander_interlock_closed = 0x000b,
  bander_interlock_open = 0x000c,
  bander_jam = 0x000d,
  bander_life_almost_over = 0x000e,
  bander_life_over = 0x000f,
  bander_memory_exhausted = 0x0010,
  bander_missing = 0x0011,
  bander_motor_failure = 0x0012,
  bander_near_limit = 0x0013,
  bander_offline = 0x0014,
  bander_opened = 0x0015,
  bander_over_temperature = 0x0016,
  bander_power_saver = 0x0017,
  bander_recoverable_failure = 0x0018,
  bander_recoverable_storage = 0x0019,
  bander_removed = 0x001a,
  bander_resource_added = 0x001b,
  bander_resource_removed = 0x001c,
  bander_thermistor_failure = 0x001d,
  bander_timing_failure = 0x001e,
  bander_turned_off = 0x001f,
  bander_turned_on = 0x0020,
  bander_under_temperature = 0x0021,
  bander_unrecoverable_failure = 0x0022,
  bander_unrecoverable_storage_error = 0x0023,
  bander_warming_up = 0x0024,
  binder_added = 0x0025,
  binder_almost_empty = 0x0026,
  binder_almost_full = 0x0027,
  binder_at_limit = 0x0028,
  binder_closed = 0x0029,
  binder_configuration_change = 0x002a,
  binder_cover_closed = 0x002b,
  binder_cover_open = 0x002c,
  binder_empty = 0x002d,
  binder_full = 0x002e,
  binder_interlock_closed = 0x002f,
  binder_interlock_open = 0x0030,
  binder_jam = 0x0031,
  binder_life_almost_over = 0x0032,
  binder_life_over = 0x0033,
  binder_memory_exhausted = 0x0034,
  binder_missing = 0x0035,
  binder_motor_failure = 0x0036,
  binder_near_limit = 0x0037,
  binder_offline = 0x0038,
  binder_opened = 0x0039,
  binder_over_temperature = 0x003a,
  binder_power_saver = 0x003b,
  binder_recoverable_failure = 0x003c,
  binder_recoverable_storage = 0x003d,
  binder_removed = 0x003e,
  binder_resource_added = 0x003f,
  binder_resource_removed = 0x0040,
  binder_thermistor_failure = 0x0041,
  binder_timing_failure = 0x0042,
  binder_turned_off = 0x0043,
  binder_turned_on = 0x0044,
  binder_under_temperature = 0x0045,
  binder_unrecoverable_failure = 0x0046,
  binder_unrecoverable_storage_error = 0x0047,
  binder_warming_up = 0x0048,
  camera_failure = 0x0049,
  chamber_cooling = 0x004a,
  chamber_failure = 0x004b,
  chamber_heating = 0x004c,
  chamber_temperature_high = 0x004d,
  chamber_temperature_low = 0x004e,
  cleaner_life_almost_over = 0x004f,
  cleaner_life_over = 0x0050,
  configuration_change = 0x0051,
  connecting_to_device = 0x0052,
  cover_open = 0x0053,
  deactivated = 0x0054,
  deleted = 0x0055,
  developer_empty = 0x0056,
  developer_low = 0x0057,
  die_cutter_added = 0x0058,
  die_cutter_almost_empty = 0x0059,
  die_cutter_almost_full = 0x005a,
  die_cutter_at_limit = 0x005b,
  die_cutter_closed = 0x005c,
  die_cutter_configuration_change = 0x005d,
  die_cutter_cover_closed = 0x005e,
  die_cutter_cover_open = 0x005f,
  die_cutter_empty = 0x0060,
  die_cutter_full = 0x0061,
  die_cutter_interlock_closed = 0x0062,
  die_cutter_interlock_open = 0x0063,
  die_cutter_jam = 0x0064,
  die_cutter_life_almost_over = 0x0065,
  die_cutter_life_over = 0x0066,
  die_cutter_memory_exhausted = 0x0067,
  die_cutter_missing = 0x0068,
  die_cutter_motor_failure = 0x0069,
  die_cutter_near_limit = 0x006a,
  die_cutter_offline = 0x006b,
  die_cutter_opened = 0x006c,
  die_cutter_over_temperature = 0x006d,
  die_cutter_power_saver = 0x006e,
  die_cutter_recoverable_failure = 0x006f,
  die_cutter_recoverable_storage = 0x0070,
  die_cutter_removed = 0x0071,
  die_cutter_resource_added = 0x0072,
  die_cutter_resource_removed = 0x0073,
  die_cutter_thermistor_failure = 0x0074,
  die_cutter_timing_failure = 0x0075,
  die_cutter_turned_off = 0x0076,
  die_cutter_turned_on = 0x0077,
  die_cutter_under_temperature = 0x0078,
  die_cutter_unrecoverable_failure = 0x0079,
  die_cutter_unrecoverable_storage_error = 0x007a,
  die_cutter_warming_up = 0x007b,
  door_open = 0x007c,
  extruder_cooling = 0x007d,
  extruder_failure = 0x007e,
  extruder_heating = 0x007f,
  extruder_jam = 0x0080,
  extruder_temperature_high = 0x0081,
  extruder_temperature_low = 0x0082,
  fan_failure = 0x0083,
  folder_added = 0x0084,
  folder_almost_empty = 0x0085,
  folder_almost_full = 0x0086,
  folder_at_limit = 0x0087,
  folder_closed = 0x0088,
  folder_configuration_change = 0x0089,
  folder_cover_closed = 0x008a,
  folder_cover_open = 0x008b,
  folder_empty = 0x008c,
  folder_full = 0x008d,
  folder_interlock_closed = 0x008e,
  folder_interlock_open = 0x008f,
  folder_jam = 0x0090,
  folder_life_almost_over = 0x0091,
  folder_life_over = 0x0092,
  folder_memory_exhausted = 0x0093,
  folder_missing = 0x0094,
  folder_motor_failure = 0x0095,
  folder_near_limit = 0x0096,
  folder_offline = 0x0097,
  folder_opened = 0x0098,
  folder_over_temperature = 0x0099,
  folder_power_saver = 0x009a,
  folder_recoverable_failure = 0x009b,
  folder_recoverable_storage = 0x009c,
  folder_removed = 0x009d,
  folder_resource_added = 0x009e,
  folder_resource_removed = 0x009f,
  folder_thermistor_failure = 0x00a0,
  folder_timing_failure = 0x00a1,
  folder_turned_off = 0x00a2,
  folder_turned_on = 0x00a3,
  folder_under_temperature = 0x00a4,
  folder_unrecoverable_failure = 0x00a5,
  folder_unrecoverable_storage_error = 0x00a6,
  folder_warming_up = 0x00a7,
  fuser_over_temp = 0x00a8,
  fuser_under_temp = 0x00a9,
  hold_new_jobs = 0x00aa,
  identify_printer_requested = 0x00ab,
  imprinter_added = 0x00ac,
  imprinter_almost_empty = 0x00ad,
  imprinter_almost_full = 0x00ae,
  imprinter_at_limit = 0x00af,
  imprinter_closed = 0x00b0,
  imprinter_configuration_change = 0x00b1,
  imprinter_cover_closed = 0x00b2,
  imprinter_cover_open = 0x00b3,
  imprinter_empty = 0x00b4,
  imprinter_full = 0x00b5,
  imprinter_interlock_closed = 0x00b6,
  imprinter_interlock_open = 0x00b7,
  imprinter_jam = 0x00b8,
  imprinter_life_almost_over = 0x00b9,
  imprinter_life_over = 0x00ba,
  imprinter_memory_exhausted = 0x00bb,
  imprinter_missing = 0x00bc,
  imprinter_motor_failure = 0x00bd,
  imprinter_near_limit = 0x00be,
  imprinter_offline = 0x00bf,
  imprinter_opened = 0x00c0,
  imprinter_over_temperature = 0x00c1,
  imprinter_power_saver = 0x00c2,
  imprinter_recoverable_failure = 0x00c3,
  imprinter_recoverable_storage = 0x00c4,
  imprinter_removed = 0x00c5,
  imprinter_resource_added = 0x00c6,
  imprinter_resource_removed = 0x00c7,
  imprinter_thermistor_failure = 0x00c8,
  imprinter_timing_failure = 0x00c9,
  imprinter_turned_off = 0x00ca,
  imprinter_turned_on = 0x00cb,
  imprinter_under_temperature = 0x00cc,
  imprinter_unrecoverable_failure = 0x00cd,
  imprinter_unrecoverable_storage_error = 0x00ce,
  imprinter_warming_up = 0x00cf,
  input_cannot_feed_size_selected = 0x00d0,
  input_manual_input_request = 0x00d1,
  input_media_color_change = 0x00d2,
  input_media_form_parts_change = 0x00d3,
  input_media_size_change = 0x00d4,
  input_media_type_change = 0x00d5,
  input_media_weight_change = 0x00d6,
  input_tray_elevation_failure = 0x00d7,
  input_tray_missing = 0x00d8,
  input_tray_position_failure = 0x00d9,
  inserter_added = 0x00da,
  inserter_almost_empty = 0x00db,
  inserter_almost_full = 0x00dc,
  inserter_at_limit = 0x00dd,
  inserter_closed = 0x00de,
  inserter_configuration_change = 0x00df,
  inserter_cover_closed = 0x00e0,
  inserter_cover_open = 0x00e1,
  inserter_empty = 0x00e2,
  inserter_full = 0x00e3,
  inserter_interlock_closed = 0x00e4,
  inserter_interlock_open = 0x00e5,
  inserter_jam = 0x00e6,
  inserter_life_almost_over = 0x00e7,
  inserter_life_over = 0x00e8,
  inserter_memory_exhausted = 0x00e9,
  inserter_missing = 0x00ea,
  inserter_motor_failure = 0x00eb,
  inserter_near_limit = 0x00ec,
  inserter_offline = 0x00ed,
  inserter_opened = 0x00ee,
  inserter_over_temperature = 0x00ef,
  inserter_power_saver = 0x00f0,
  inserter_recoverable_failure = 0x00f1,
  inserter_recoverable_storage = 0x00f2,
  inserter_removed = 0x00f3,
  inserter_resource_added = 0x00f4,
  inserter_resource_removed = 0x00f5,
  inserter_thermistor_failure = 0x00f6,
  inserter_timing_failure = 0x00f7,
  inserter_turned_off = 0x00f8,
  inserter_turned_on = 0x00f9,
  inserter_under_temperature = 0x00fa,
  inserter_unrecoverable_failure = 0x00fb,
  inserter_unrecoverable_storage_error = 0x00fc,
  inserter_warming_up = 0x00fd,
  interlock_closed = 0x00fe,
  interlock_open = 0x00ff,
  interpreter_cartridge_added = 0x0100,
  interpreter_cartridge_deleted = 0x0101,
  interpreter_complex_page_encountered = 0x0102,
  interpreter_memory_decrease = 0x0103,
  interpreter_memory_increase = 0x0104,
  interpreter_resource_added = 0x0105,
  interpreter_resource_deleted = 0x0106,
  interpreter_resource_unavailable = 0x0107,
  lamp_at_eol = 0x0108,
  lamp_failure = 0x0109,
  lamp_near_eol = 0x010a,
  laser_at_eol = 0x010b,
  laser_failure = 0x010c,
  laser_near_eol = 0x010d,
  make_envelope_added = 0x010e,
  make_envelope_almost_empty = 0x010f,
  make_envelope_almost_full = 0x0110,
  make_envelope_at_limit = 0x0111,
  make_envelope_closed = 0x0112,
  make_envelope_configuration_change = 0x0113,
  make_envelope_cover_closed = 0x0114,
  make_envelope_cover_open = 0x0115,
  make_envelope_empty = 0x0116,
  make_envelope_full = 0x0117,
  make_envelope_interlock_closed = 0x0118,
  make_envelope_interlock_open = 0x0119,
  make_envelope_jam = 0x011a,
  make_envelope_life_almost_over = 0x011b,
  make_envelope_life_over = 0x011c,
  make_envelope_memory_exhausted = 0x011d,
  make_envelope_missing = 0x011e,
  make_envelope_motor_failure = 0x011f,
  make_envelope_near_limit = 0x0120,
  make_envelope_offline = 0x0121,
  make_envelope_opened = 0x0122,
  make_envelope_over_temperature = 0x0123,
  make_envelope_power_saver = 0x0124,
  make_envelope_recoverable_failure = 0x0125,
  make_envelope_recoverable_storage = 0x0126,
  make_envelope_removed = 0x0127,
  make_envelope_resource_added = 0x0128,
  make_envelope_resource_removed = 0x0129,
  make_envelope_thermistor_failure = 0x012a,
  make_envelope_timing_failure = 0x012b,
  make_envelope_turned_off = 0x012c,
  make_envelope_turned_on = 0x012d,
  make_envelope_under_temperature = 0x012e,
  make_envelope_unrecoverable_failure = 0x012f,
  make_envelope_unrecoverable_storage_error = 0x0130,
  make_envelope_warming_up = 0x0131,
  marker_adjusting_print_quality = 0x0132,
  marker_developer_almost_empty = 0x0133,
  marker_developer_empty = 0x0134,
  marker_fuser_thermistor_failure = 0x0135,
  marker_fuser_timing_failure = 0x0136,
  marker_ink_almost_empty = 0x0137,
  marker_ink_empty = 0x0138,
  marker_print_ribbon_almost_empty = 0x0139,
  marker_print_ribbon_empty = 0x013a,
  marker_supply_empty = 0x013b,
  marker_supply_low = 0x013c,
  marker_toner_cartridge_missing = 0x013d,
  marker_waste_almost_full = 0x013e,
  marker_waste_full = 0x013f,
  marker_waste_ink_receptacle_almost_full = 0x0140,
  marker_waste_ink_receptacle_full = 0x0141,
  marker_waste_toner_receptacle_almost_full = 0x0142,
  marker_waste_toner_receptacle_full = 0x0143,
  material_empty = 0x0144,
  material_low = 0x0145,
  material_needed = 0x0146,
  media_empty = 0x0147,
  media_jam = 0x0148,
  media_low = 0x0149,
  media_needed = 0x014a,
  media_path_cannot_duplex_media_selected = 0x014b,
  media_path_media_tray_almost_full = 0x014c,
  media_path_media_tray_full = 0x014d,
  media_path_media_tray_missing = 0x014e,
  motor_failure = 0x014f,
  moving_to_paused = 0x0150,
  none = 0x0151,
  opc_life_over = 0x0152,
  opc_near_eol = 0x0153,
  other = 0x0154,
  output_area_almost_full = 0x0155,
  output_area_full = 0x0156,
  output_mailbox_select_failure = 0x0157,
  output_tray_missing = 0x0158,
  paused = 0x0159,
  perforater_added = 0x015a,
  perforater_almost_empty = 0x015b,
  perforater_almost_full = 0x015c,
  perforater_at_limit = 0x015d,
  perforater_closed = 0x015e,
  perforater_configuration_change = 0x015f,
  perforater_cover_closed = 0x0160,
  perforater_cover_open = 0x0161,
  perforater_empty = 0x0162,
  perforater_full = 0x0163,
  perforater_interlock_closed = 0x0164,
  perforater_interlock_open = 0x0165,
  perforater_jam = 0x0166,
  perforater_life_almost_over = 0x0167,
  perforater_life_over = 0x0168,
  perforater_memory_exhausted = 0x0169,
  perforater_missing = 0x016a,
  perforater_motor_failure = 0x016b,
  perforater_near_limit = 0x016c,
  perforater_offline = 0x016d,
  perforater_opened = 0x016e,
  perforater_over_temperature = 0x016f,
  perforater_power_saver = 0x0170,
  perforater_recoverable_failure = 0x0171,
  perforater_recoverable_storage = 0x0172,
  perforater_removed = 0x0173,
  perforater_resource_added = 0x0174,
  perforater_resource_removed = 0x0175,
  perforater_thermistor_failure = 0x0176,
  perforater_timing_failure = 0x0177,
  perforater_turned_off = 0x0178,
  perforater_turned_on = 0x0179,
  perforater_under_temperature = 0x017a,
  perforater_unrecoverable_failure = 0x017b,
  perforater_unrecoverable_storage_error = 0x017c,
  perforater_warming_up = 0x017d,
  platform_cooling = 0x017e,
  platform_failure = 0x017f,
  platform_heating = 0x0180,
  platform_temperature_high = 0x0181,
  platform_temperature_low = 0x0182,
  power_down = 0x0183,
  power_up = 0x0184,
  printer_manual_reset = 0x0185,
  printer_nms_reset = 0x0186,
  printer_ready_to_print = 0x0187,
  puncher_added = 0x0188,
  puncher_almost_empty = 0x0189,
  puncher_almost_full = 0x018a,
  puncher_at_limit = 0x018b,
  puncher_closed = 0x018c,
  puncher_configuration_change = 0x018d,
  puncher_cover_closed = 0x018e,
  puncher_cover_open = 0x018f,
  puncher_empty = 0x0190,
  puncher_full = 0x0191,
  puncher_interlock_closed = 0x0192,
  puncher_interlock_open = 0x0193,
  puncher_jam = 0x0194,
  puncher_life_almost_over = 0x0195,
  puncher_life_over = 0x0196,
  puncher_memory_exhausted = 0x0197,
  puncher_missing = 0x0198,
  puncher_motor_failure = 0x0199,
  puncher_near_limit = 0x019a,
  puncher_offline = 0x019b,
  puncher_opened = 0x019c,
  puncher_over_temperature = 0x019d,
  puncher_power_saver = 0x019e,
  puncher_recoverable_failure = 0x019f,
  puncher_recoverable_storage = 0x01a0,
  puncher_removed = 0x01a1,
  puncher_resource_added = 0x01a2,
  puncher_resource_removed = 0x01a3,
  puncher_thermistor_failure = 0x01a4,
  puncher_timing_failure = 0x01a5,
  puncher_turned_off = 0x01a6,
  puncher_turned_on = 0x01a7,
  puncher_under_temperature = 0x01a8,
  puncher_unrecoverable_failure = 0x01a9,
  puncher_unrecoverable_storage_error = 0x01aa,
  puncher_warming_up = 0x01ab,
  resuming = 0x01ac,
  separation_cutter_added = 0x01ad,
  separation_cutter_almost_empty = 0x01ae,
  separation_cutter_almost_full = 0x01af,
  separation_cutter_at_limit = 0x01b0,
  separation_cutter_closed = 0x01b1,
  separation_cutter_configuration_change = 0x01b2,
  separation_cutter_cover_closed = 0x01b3,
  separation_cutter_cover_open = 0x01b4,
  separation_cutter_empty = 0x01b5,
  separation_cutter_full = 0x01b6,
  separation_cutter_interlock_closed = 0x01b7,
  separation_cutter_interlock_open = 0x01b8,
  separation_cutter_jam = 0x01b9,
  separation_cutter_life_almost_over = 0x01ba,
  separation_cutter_life_over = 0x01bb,
  separation_cutter_memory_exhausted = 0x01bc,
  separation_cutter_missing = 0x01bd,
  separation_cutter_motor_failure = 0x01be,
  separation_cutter_near_limit = 0x01bf,
  separation_cutter_offline = 0x01c0,
  separation_cutter_opened = 0x01c1,
  separation_cutter_over_temperature = 0x01c2,
  separation_cutter_power_saver = 0x01c3,
  separation_cutter_recoverable_failure = 0x01c4,
  separation_cutter_recoverable_storage = 0x01c5,
  separation_cutter_removed = 0x01c6,
  separation_cutter_resource_added = 0x01c7,
  separation_cutter_resource_removed = 0x01c8,
  separation_cutter_thermistor_failure = 0x01c9,
  separation_cutter_timing_failure = 0x01ca,
  separation_cutter_turned_off = 0x01cb,
  separation_cutter_turned_on = 0x01cc,
  separation_cutter_under_temperature = 0x01cd,
  separation_cutter_unrecoverable_failure = 0x01ce,
  separation_cutter_unrecoverable_storage_error = 0x01cf,
  separation_cutter_warming_up = 0x01d0,
  sheet_rotator_added = 0x01d1,
  sheet_rotator_almost_empty = 0x01d2,
  sheet_rotator_almost_full = 0x01d3,
  sheet_rotator_at_limit = 0x01d4,
  sheet_rotator_closed = 0x01d5,
  sheet_rotator_configuration_change = 0x01d6,
  sheet_rotator_cover_closed = 0x01d7,
  sheet_rotator_cover_open = 0x01d8,
  sheet_rotator_empty = 0x01d9,
  sheet_rotator_full = 0x01da,
  sheet_rotator_interlock_closed = 0x01db,
  sheet_rotator_interlock_open = 0x01dc,
  sheet_rotator_jam = 0x01dd,
  sheet_rotator_life_almost_over = 0x01de,
  sheet_rotator_life_over = 0x01df,
  sheet_rotator_memory_exhausted = 0x01e0,
  sheet_rotator_missing = 0x01e1,
  sheet_rotator_motor_failure = 0x01e2,
  sheet_rotator_near_limit = 0x01e3,
  sheet_rotator_offline = 0x01e4,
  sheet_rotator_opened = 0x01e5,
  sheet_rotator_over_temperature = 0x01e6,
  sheet_rotator_power_saver = 0x01e7,
  sheet_rotator_recoverable_failure = 0x01e8,
  sheet_rotator_recoverable_storage = 0x01e9,
  sheet_rotator_removed = 0x01ea,
  sheet_rotator_resource_added = 0x01eb,
  sheet_rotator_resource_removed = 0x01ec,
  sheet_rotator_thermistor_failure = 0x01ed,
  sheet_rotator_timing_failure = 0x01ee,
  sheet_rotator_turned_off = 0x01ef,
  sheet_rotator_turned_on = 0x01f0,
  sheet_rotator_under_temperature = 0x01f1,
  sheet_rotator_unrecoverable_failure = 0x01f2,
  sheet_rotator_unrecoverable_storage_error = 0x01f3,
  sheet_rotator_warming_up = 0x01f4,
  shutdown = 0x01f5,
  slitter_added = 0x01f6,
  slitter_almost_empty = 0x01f7,
  slitter_almost_full = 0x01f8,
  slitter_at_limit = 0x01f9,
  slitter_closed = 0x01fa,
  slitter_configuration_change = 0x01fb,
  slitter_cover_closed = 0x01fc,
  slitter_cover_open = 0x01fd,
  slitter_empty = 0x01fe,
  slitter_full = 0x01ff,
  slitter_interlock_closed = 0x0200,
  slitter_interlock_open = 0x0201,
  slitter_jam = 0x0202,
  slitter_life_almost_over = 0x0203,
  slitter_life_over = 0x0204,
  slitter_memory_exhausted = 0x0205,
  slitter_missing = 0x0206,
  slitter_motor_failure = 0x0207,
  slitter_near_limit = 0x0208,
  slitter_offline = 0x0209,
  slitter_opened = 0x020a,
  slitter_over_temperature = 0x020b,
  slitter_power_saver = 0x020c,
  slitter_recoverable_failure = 0x020d,
  slitter_recoverable_storage = 0x020e,
  slitter_removed = 0x020f,
  slitter_resource_added = 0x0210,
  slitter_resource_removed = 0x0211,
  slitter_thermistor_failure = 0x0212,
  slitter_timing_failure = 0x0213,
  slitter_turned_off = 0x0214,
  slitter_turned_on = 0x0215,
  slitter_under_temperature = 0x0216,
  slitter_unrecoverable_failure = 0x0217,
  slitter_unrecoverable_storage_error = 0x0218,
  slitter_warming_up = 0x0219,
  spool_area_full = 0x021a,
  stacker_added = 0x021b,
  stacker_almost_empty = 0x021c,
  stacker_almost_full = 0x021d,
  stacker_at_limit = 0x021e,
  stacker_closed = 0x021f,
  stacker_configuration_change = 0x0220,
  stacker_cover_closed = 0x0221,
  stacker_cover_open = 0x0222,
  stacker_empty = 0x0223,
  stacker_full = 0x0224,
  stacker_interlock_closed = 0x0225,
  stacker_interlock_open = 0x0226,
  stacker_jam = 0x0227,
  stacker_life_almost_over = 0x0228,
  stacker_life_over = 0x0229,
  stacker_memory_exhausted = 0x022a,
  stacker_missing = 0x022b,
  stacker_motor_failure = 0x022c,
  stacker_near_limit = 0x022d,
  stacker_offline = 0x022e,
  stacker_opened = 0x022f,
  stacker_over_temperature = 0x0230,
  stacker_power_saver = 0x0231,
  stacker_recoverable_failure = 0x0232,
  stacker_recoverable_storage = 0x0233,
  stacker_removed = 0x0234,
  stacker_resource_added = 0x0235,
  stacker_resource_removed = 0x0236,
  stacker_thermistor_failure = 0x0237,
  stacker_timing_failure = 0x0238,
  stacker_turned_off = 0x0239,
  stacker_turned_on = 0x023a,
  stacker_under_temperature = 0x023b,
  stacker_unrecoverable_failure = 0x023c,
  stacker_unrecoverable_storage_error = 0x023d,
  stacker_warming_up = 0x023e,
  stapler_added = 0x023f,
  stapler_almost_empty = 0x0240,
  stapler_almost_full = 0x0241,
  stapler_at_limit = 0x0242,
  stapler_closed = 0x0243,
  stapler_configuration_change = 0x0244,
  stapler_cover_closed = 0x0245,
  stapler_cover_open = 0x0246,
  stapler_empty = 0x0247,
  stapler_full = 0x0248,
  stapler_interlock_closed = 0x0249,
  stapler_interlock_open = 0x024a,
  stapler_jam = 0x024b,
  stapler_life_almost_over = 0x024c,
  stapler_life_over = 0x024d,
  stapler_memory_exhausted = 0x024e,
  stapler_missing = 0x024f,
  stapler_motor_failure = 0x0250,
  stapler_near_limit = 0x0251,
  stapler_offline = 0x0252,
  stapler_opened = 0x0253,
  stapler_over_temperature = 0x0254,
  stapler_power_saver = 0x0255,
  stapler_recoverable_failure = 0x0256,
  stapler_recoverable_storage = 0x0257,
  stapler_removed = 0x0258,
  stapler_resource_added = 0x0259,
  stapler_resource_removed = 0x025a,
  stapler_thermistor_failure = 0x025b,
  stapler_timing_failure = 0x025c,
  stapler_turned_off = 0x025d,
  stapler_turned_on = 0x025e,
  stapler_under_temperature = 0x025f,
  stapler_unrecoverable_failure = 0x0260,
  stapler_unrecoverable_storage_error = 0x0261,
  stapler_warming_up = 0x0262,
  stitcher_added = 0x0263,
  stitcher_almost_empty = 0x0264,
  stitcher_almost_full = 0x0265,
  stitcher_at_limit = 0x0266,
  stitcher_closed = 0x0267,
  stitcher_configuration_change = 0x0268,
  stitcher_cover_closed = 0x0269,
  stitcher_cover_open = 0x026a,
  stitcher_empty = 0x026b,
  stitcher_full = 0x026c,
  stitcher_interlock_closed = 0x026d,
  stitcher_interlock_open = 0x026e,
  stitcher_jam = 0x026f,
  stitcher_life_almost_over = 0x0270,
  stitcher_life_over = 0x0271,
  stitcher_memory_exhausted = 0x0272,
  stitcher_missing = 0x0273,
  stitcher_motor_failure = 0x0274,
  stitcher_near_limit = 0x0275,
  stitcher_offline = 0x0276,
  stitcher_opened = 0x0277,
  stitcher_over_temperature = 0x0278,
  stitcher_power_saver = 0x0279,
  stitcher_recoverable_failure = 0x027a,
  stitcher_recoverable_storage = 0x027b,
  stitcher_removed = 0x027c,
  stitcher_resource_added = 0x027d,
  stitcher_resource_removed = 0x027e,
  stitcher_thermistor_failure = 0x027f,
  stitcher_timing_failure = 0x0280,
  stitcher_turned_off = 0x0281,
  stitcher_turned_on = 0x0282,
  stitcher_under_temperature = 0x0283,
  stitcher_unrecoverable_failure = 0x0284,
  stitcher_unrecoverable_storage_error = 0x0285,
  stitcher_warming_up = 0x0286,
  stopped_partly = 0x0287,
  stopping = 0x0288,
  subunit_added = 0x0289,
  subunit_almost_empty = 0x028a,
  subunit_almost_full = 0x028b,
  subunit_at_limit = 0x028c,
  subunit_closed = 0x028d,
  subunit_cooling_down = 0x028e,
  subunit_empty = 0x028f,
  subunit_full = 0x0290,
  subunit_life_almost_over = 0x0291,
  subunit_life_over = 0x0292,
  subunit_memory_exhausted = 0x0293,
  subunit_missing = 0x0294,
  subunit_motor_failure = 0x0295,
  subunit_near_limit = 0x0296,
  subunit_offline = 0x0297,
  subunit_opened = 0x0298,
  subunit_over_temperature = 0x0299,
  subunit_power_saver = 0x029a,
  subunit_recoverable_failure = 0x029b,
  subunit_recoverable_storage = 0x029c,
  subunit_removed = 0x029d,
  subunit_resource_added = 0x029e,
  subunit_resource_removed = 0x029f,
  subunit_thermistor_failure = 0x02a0,
  subunit_timing_Failure = 0x02a1,
  subunit_turned_off = 0x02a2,
  subunit_turned_on = 0x02a3,
  subunit_under_temperature = 0x02a4,
  subunit_unrecoverable_failure = 0x02a5,
  subunit_unrecoverable_storage = 0x02a6,
  subunit_warming_up = 0x02a7,
  timed_out = 0x02a8,
  toner_empty = 0x02a9,
  toner_low = 0x02aa,
  trimmer_added = 0x02ab,
  trimmer_almost_empty = 0x02ac,
  trimmer_almost_full = 0x02ad,
  trimmer_at_limit = 0x02ae,
  trimmer_closed = 0x02af,
  trimmer_configuration_change = 0x02b0,
  trimmer_cover_closed = 0x02b1,
  trimmer_cover_open = 0x02b2,
  trimmer_empty = 0x02b3,
  trimmer_full = 0x02b4,
  trimmer_interlock_closed = 0x02b5,
  trimmer_interlock_open = 0x02b6,
  trimmer_jam = 0x02b7,
  trimmer_life_almost_over = 0x02b8,
  trimmer_life_over = 0x02b9,
  trimmer_memory_exhausted = 0x02ba,
  trimmer_missing = 0x02bb,
  trimmer_motor_failure = 0x02bc,
  trimmer_near_limit = 0x02bd,
  trimmer_offline = 0x02be,
  trimmer_opened = 0x02bf,
  trimmer_over_temperature = 0x02c0,
  trimmer_power_saver = 0x02c1,
  trimmer_recoverable_failure = 0x02c2,
  trimmer_recoverable_storage = 0x02c3,
  trimmer_removed = 0x02c4,
  trimmer_resource_added = 0x02c5,
  trimmer_resource_removed = 0x02c6,
  trimmer_thermistor_failure = 0x02c7,
  trimmer_timing_failure = 0x02c8,
  trimmer_turned_off = 0x02c9,
  trimmer_turned_on = 0x02ca,
  trimmer_under_temperature = 0x02cb,
  trimmer_unrecoverable_failure = 0x02cc,
  trimmer_unrecoverable_storage_error = 0x02cd,
  trimmer_warming_up = 0x02ce,
  unknown = 0x02cf,
  wrapper_added = 0x02d0,
  wrapper_almost_empty = 0x02d1,
  wrapper_almost_full = 0x02d2,
  wrapper_at_limit = 0x02d3,
  wrapper_closed = 0x02d4,
  wrapper_configuration_change = 0x02d5,
  wrapper_cover_closed = 0x02d6,
  wrapper_cover_open = 0x02d7,
  wrapper_empty = 0x02d8,
  wrapper_full = 0x02d9,
  wrapper_interlock_closed = 0x02da,
  wrapper_interlock_open = 0x02db,
  wrapper_jam = 0x02dc,
  wrapper_life_almost_over = 0x02dd,
  wrapper_life_over = 0x02de,
  wrapper_memory_exhausted = 0x02df,
  wrapper_missing = 0x02e0,
  wrapper_motor_failure = 0x02e1,
  wrapper_near_limit = 0x02e2,
  wrapper_offline = 0x02e3,
  wrapper_opened = 0x02e4,
  wrapper_over_temperature = 0x02e5,
  wrapper_power_saver = 0x02e6,
  wrapper_recoverable_failure = 0x02e7,
  wrapper_recoverable_storage = 0x02e8,
  wrapper_removed = 0x02e9,
  wrapper_resource_added = 0x02ea,
  wrapper_resource_removed = 0x02eb,
  wrapper_thermistor_failure = 0x02ec,
  wrapper_timing_failure = 0x02ed,
  wrapper_turned_off = 0x02ee,
  wrapper_turned_on = 0x02ef,
  wrapper_under_temperature = 0x02f0,
  wrapper_unrecoverable_failure = 0x02f1,
  wrapper_unrecoverable_storage_error = 0x02f2,
  wrapper_warming_up = 0x02f3
};
enum class E_proof_print_supported : uint16_t {
  media = 0x0000,
  media_col = 0x0001,
  proof_print_copies = 0x0002
};
enum class E_pwg_raster_document_sheet_back : uint16_t {
  flipped = 0x0000,
  manual_tumble = 0x0001,
  normal = 0x0002,
  rotated = 0x0003
};
enum class E_pwg_raster_document_type_supported : uint16_t {
  adobe_rgb_8 = 0x0000,
  adobe_rgb_16 = 0x0001,
  black_1 = 0x0002,
  black_8 = 0x0003,
  black_16 = 0x0004,
  cmyk_8 = 0x0005,
  cmyk_16 = 0x0006,
  device1_8 = 0x0007,
  device1_16 = 0x0008,
  device2_8 = 0x0009,
  device2_16 = 0x000a,
  device3_8 = 0x000b,
  device3_16 = 0x000c,
  device4_8 = 0x000d,
  device4_16 = 0x000e,
  device5_8 = 0x000f,
  device5_16 = 0x0010,
  device6_8 = 0x0011,
  device6_16 = 0x0012,
  device7_8 = 0x0013,
  device7_16 = 0x0014,
  device8_8 = 0x0015,
  device8_16 = 0x0016,
  device9_8 = 0x0017,
  device9_16 = 0x0018,
  device10_8 = 0x0019,
  device10_16 = 0x001a,
  device11_8 = 0x001b,
  device11_16 = 0x001c,
  device12_8 = 0x001d,
  device12_16 = 0x001e,
  device13_8 = 0x001f,
  device13_16 = 0x0020,
  device14_8 = 0x0021,
  device14_16 = 0x0022,
  device15_8 = 0x0023,
  device15_16 = 0x0024,
  rgb_8 = 0x0025,
  rgb_16 = 0x0026,
  sgray_1 = 0x0027,
  sgray_8 = 0x0028,
  sgray_16 = 0x0029,
  srgb_8 = 0x002a,
  srgb_16 = 0x002b
};
enum class E_requested_attributes : uint16_t {
  all = 0x0000,
  document_description = 0x0001,
  document_template = 0x0002,
  job_actuals = 0x0003,
  job_description = 0x0004,
  job_template = 0x0005,
  printer_description = 0x0006,
  resource_description = 0x0007,
  resource_status = 0x0008,
  resource_template = 0x0009,
  subscription_description = 0x000a,
  subscription_template = 0x000b,
  system_description = 0x000c,
  system_status = 0x000d
};
enum class E_save_disposition : uint16_t {
  none = 0x0000,
  print_save = 0x0001,
  save_only = 0x0002
};
enum class E_separator_sheets_type : uint16_t {
  both_sheets = 0x0000,
  end_sheet = 0x0001,
  none = 0x0002,
  slip_sheets = 0x0003,
  start_sheet = 0x0004
};
enum class E_sheet_collate : uint16_t {
  collated = 0x0000,
  uncollated = 0x0001
};
enum class E_status_code : int16_t {
  successful_ok = 0x0000,
  successful_ok_ignored_or_substituted_attributes = 0x0001,
  successful_ok_conflicting_attributes = 0x0002,
  successful_ok_ignored_subscriptions = 0x0003,
  successful_ok_too_many_events = 0x0005,
  successful_ok_events_complete = 0x0007,
  client_error_bad_request = 0x0400,
  client_error_forbidden = 0x0401,
  client_error_not_authenticated = 0x0402,
  client_error_not_authorized = 0x0403,
  client_error_not_possible = 0x0404,
  client_error_timeout = 0x0405,
  client_error_not_found = 0x0406,
  client_error_gone = 0x0407,
  client_error_request_entity_too_large = 0x0408,
  client_error_request_value_too_long = 0x0409,
  client_error_document_format_not_supported = 0x040a,
  client_error_attributes_or_values_not_supported = 0x040b,
  client_error_uri_scheme_not_supported = 0x040c,
  client_error_charset_not_supported = 0x040d,
  client_error_conflicting_attributes = 0x040e,
  client_error_compression_not_supported = 0x040f,
  client_error_compression_error = 0x0410,
  client_error_document_format_error = 0x0411,
  client_error_document_access_error = 0x0412,
  client_error_attributes_not_settable = 0x0413,
  client_error_ignored_all_subscriptions = 0x0414,
  client_error_too_many_subscriptions = 0x0415,
  client_error_document_password_error = 0x0418,
  client_error_document_permission_error = 0x0419,
  client_error_document_security_error = 0x041a,
  client_error_document_unprintable_error = 0x041b,
  client_error_account_info_needed = 0x041c,
  client_error_account_closed = 0x041d,
  client_error_account_limit_reached = 0x041e,
  client_error_account_authorization_failed = 0x041f,
  client_error_not_fetchable = 0x0420,
  server_error_internal_error = 0x0500,
  server_error_operation_not_supported = 0x0501,
  server_error_service_unavailable = 0x0502,
  server_error_version_not_supported = 0x0503,
  server_error_device_error = 0x0504,
  server_error_temporary_error = 0x0505,
  server_error_not_accepting_jobs = 0x0506,
  server_error_busy = 0x0507,
  server_error_job_canceled = 0x0508,
  server_error_multiple_document_jobs_not_supported = 0x0509,
  server_error_printer_is_deactivated = 0x050a,
  server_error_too_many_jobs = 0x050b,
  server_error_too_many_documents = 0x050c
};
enum class E_stitching_method : uint16_t {
  auto_ = 0x0000,
  crimp = 0x0001,
  wire = 0x0002
};
enum class E_stitching_reference_edge : uint16_t {
  bottom = 0x0000,
  left = 0x0001,
  right = 0x0002,
  top = 0x0003
};
enum class E_trimming_type : uint16_t {
  draw_line = 0x0000,
  full = 0x0001,
  partial = 0x0002,
  perforate = 0x0003,
  score = 0x0004,
  tab = 0x0005
};
enum class E_trimming_when : uint16_t {
  after_documents = 0x0000,
  after_job = 0x0001,
  after_sets = 0x0002,
  after_sheets = 0x0003
};
enum class E_uri_authentication_supported : uint16_t {
  basic = 0x0000,
  certificate = 0x0001,
  digest = 0x0002,
  negotiate = 0x0003,
  none = 0x0004,
  oauth = 0x0005,
  requesting_user_name = 0x0006
};
enum class E_uri_security_supported : uint16_t {
  none = 0x0000,
  ssl3 = 0x0001,
  tls = 0x0002
};
enum class E_which_jobs : uint16_t {
  aborted = 0x0000,
  all = 0x0001,
  canceled = 0x0002,
  completed = 0x0003,
  fetchable = 0x0004,
  not_completed = 0x0005,
  pending = 0x0006,
  pending_held = 0x0007,
  processing = 0x0008,
  processing_stopped = 0x0009,
  proof_print = 0x000a,
  saved = 0x000b
};
enum class E_x_image_position : uint16_t {
  center = 0x0000,
  left = 0x0001,
  none = 0x0002,
  right = 0x0003
};
enum class E_xri_authentication : uint16_t {};
enum class E_xri_security : uint16_t {};
enum class E_y_image_position : uint16_t {
  bottom = 0x0000,
  center = 0x0001,
  none = 0x0002,
  top = 0x0003
};
typedef E_baling_type E_baling_type_supported;
typedef E_baling_when E_baling_when_supported;
typedef E_binding_reference_edge E_binding_reference_edge_supported;
typedef E_binding_type E_binding_type_supported;
typedef E_coating_sides E_coating_sides_supported;
typedef E_coating_type E_coating_type_supported;
typedef E_compression E_compression_supported;
typedef E_cover_back_supported E_cover_front_supported;
typedef E_covering_name E_covering_name_supported;
typedef E_document_digital_signature E_document_digital_signature_default;
typedef E_document_digital_signature E_document_digital_signature_supported;
typedef E_feed_orientation E_feed_orientation_supported;
typedef E_finishing_template E_finishing_template_supported;
typedef E_finishings E_finishings_default;
typedef E_finishings E_finishings_ready;
typedef E_finishings E_finishings_supported;
typedef E_folding_direction E_folding_direction_supported;
typedef E_binding_reference_edge E_folding_reference_edge;
typedef E_binding_reference_edge E_folding_reference_edge_supported;
typedef E_identify_actions E_identify_actions_default;
typedef E_identify_actions E_identify_actions_supported;
typedef E_job_account_type E_job_account_type_default;
typedef E_job_account_type E_job_account_type_supported;
typedef E_cover_back_supported E_job_cover_back_supported;
typedef E_cover_back_supported E_job_cover_front_supported;
typedef E_job_delay_output_until E_job_delay_output_until_default;
typedef E_job_delay_output_until E_job_delay_output_until_supported;
typedef E_job_error_action E_job_error_action_default;
typedef E_job_error_action E_job_error_action_supported;
typedef E_job_accounting_sheets_type E_job_error_sheet_type;
typedef E_finishings E_job_finishings;
typedef E_finishings E_job_finishings_default;
typedef E_finishings E_job_finishings_ready;
typedef E_finishings E_job_finishings_supported;
typedef E_job_hold_until E_job_hold_until_default;
typedef E_job_hold_until E_job_hold_until_supported;
typedef E_job_password_encryption E_job_password_encryption_supported;
typedef E_job_sheets E_job_sheets_default;
typedef E_job_sheets E_job_sheets_supported;
typedef E_coating_sides E_laminating_sides;
typedef E_coating_sides E_laminating_sides_supported;
typedef E_laminating_type E_laminating_type_supported;
typedef E_media_back_coating E_media_back_coating_supported;
typedef E_material_color E_media_color;
typedef E_material_color E_media_color_supported;
typedef E_media E_media_default;
typedef E_media_back_coating E_media_front_coating;
typedef E_media_back_coating E_media_front_coating_supported;
typedef E_media_grain E_media_grain_supported;
typedef E_media_pre_printed E_media_pre_printed_supported;
typedef E_job_accounting_sheets_type E_media_recycled;
typedef E_job_accounting_sheets_type E_media_recycled_supported;
typedef E_feed_orientation E_media_source_feed_direction;
typedef E_input_orientation_requested E_media_source_feed_orientation;
typedef E_media_source E_media_source_supported;
typedef E_media E_media_supported;
typedef E_media_tooth E_media_tooth_supported;
typedef E_media_type E_media_type_supported;
typedef E_multiple_document_handling E_multiple_document_handling_default;
typedef E_multiple_document_handling E_multiple_document_handling_supported;
typedef E_notify_events E_notify_events_default;
typedef E_notify_events E_notify_events_supported;
typedef E_notify_pull_method E_notify_pull_method_supported;
typedef E_input_orientation_requested E_orientation_requested;
typedef E_input_orientation_requested E_orientation_requested_default;
typedef E_input_orientation_requested E_orientation_requested_supported;
typedef E_job_accounting_output_bin E_output_bin;
typedef E_job_accounting_output_bin E_output_bin_default;
typedef E_job_accounting_output_bin E_output_bin_supported;
typedef E_page_delivery E_page_delivery_default;
typedef E_page_delivery E_page_delivery_supported;
typedef E_current_page_order E_page_order_received;
typedef E_current_page_order E_page_order_received_default;
typedef E_current_page_order E_page_order_received_supported;
typedef E_presentation_direction_number_up
    E_presentation_direction_number_up_default;
typedef E_presentation_direction_number_up
    E_presentation_direction_number_up_supported;
typedef E_print_color_mode E_print_color_mode_default;
typedef E_print_color_mode E_print_color_mode_supported;
typedef E_print_content_optimize E_print_content_optimize_default;
typedef E_print_content_optimize E_print_content_optimize_supported;
typedef E_input_quality E_print_quality;
typedef E_input_quality E_print_quality_default;
typedef E_input_quality E_print_quality_supported;
typedef E_print_rendering_intent E_print_rendering_intent_default;
typedef E_print_rendering_intent E_print_rendering_intent_supported;
typedef E_binding_reference_edge E_punching_reference_edge;
typedef E_binding_reference_edge E_punching_reference_edge_supported;
typedef E_save_disposition E_save_disposition_supported;
typedef E_sheet_collate E_sheet_collate_default;
typedef E_sheet_collate E_sheet_collate_supported;
typedef E_input_sides E_sides;
typedef E_input_sides E_sides_default;
typedef E_input_sides E_sides_supported;
typedef E_stitching_method E_stitching_method_supported;
typedef E_stitching_reference_edge E_stitching_reference_edge_supported;
typedef E_binding_reference_edge E_trimming_reference_edge;
typedef E_binding_reference_edge E_trimming_reference_edge_supported;
typedef E_trimming_type E_trimming_type_supported;
typedef E_trimming_when E_trimming_when_supported;
typedef E_which_jobs E_which_jobs_supported;
typedef E_x_image_position E_x_image_position_default;
typedef E_x_image_position E_x_image_position_supported;
typedef E_uri_authentication_supported E_xri_authentication_supported;
typedef E_uri_security_supported E_xri_security_supported;
typedef E_y_image_position E_y_image_position_default;
typedef E_y_image_position E_y_image_position_supported;

LIBIPP_EXPORT std::string ToString(AttrName, int value);

LIBIPP_EXPORT std::string ToString(GroupTag v);
LIBIPP_EXPORT std::string ToString(AttrName v);
LIBIPP_EXPORT std::string_view ToStrView(AttrName v);
LIBIPP_EXPORT std::string ToString(E_auth_info_required v);
LIBIPP_EXPORT std::string ToString(E_baling_type v);
LIBIPP_EXPORT std::string ToString(E_baling_when v);
LIBIPP_EXPORT std::string ToString(E_binding_reference_edge v);
LIBIPP_EXPORT std::string ToString(E_binding_type v);
LIBIPP_EXPORT std::string ToString(E_coating_sides v);
LIBIPP_EXPORT std::string ToString(E_coating_type v);
LIBIPP_EXPORT std::string ToString(E_compression v);
LIBIPP_EXPORT std::string ToString(E_cover_back_supported v);
LIBIPP_EXPORT std::string ToString(E_cover_type v);
LIBIPP_EXPORT std::string ToString(E_covering_name v);
LIBIPP_EXPORT std::string ToString(E_current_page_order v);
LIBIPP_EXPORT std::string ToString(E_document_digital_signature v);
LIBIPP_EXPORT std::string ToString(E_document_format_details_supported v);
LIBIPP_EXPORT std::string ToString(E_document_format_varying_attributes v);
LIBIPP_EXPORT std::string ToString(E_feed_orientation v);
LIBIPP_EXPORT std::string ToString(E_finishing_template v);
LIBIPP_EXPORT std::string ToString(E_finishings v);
LIBIPP_EXPORT std::string ToString(E_folding_direction v);
LIBIPP_EXPORT std::string ToString(E_identify_actions v);
LIBIPP_EXPORT std::string ToString(E_imposition_template v);
LIBIPP_EXPORT std::string ToString(E_input_orientation_requested v);
LIBIPP_EXPORT std::string ToString(E_input_quality v);
LIBIPP_EXPORT std::string ToString(E_input_sides v);
LIBIPP_EXPORT std::string ToString(E_ipp_features_supported v);
LIBIPP_EXPORT std::string ToString(E_ipp_versions_supported v);
LIBIPP_EXPORT std::string ToString(E_job_account_type v);
LIBIPP_EXPORT std::string ToString(E_job_accounting_output_bin v);
LIBIPP_EXPORT std::string ToString(E_job_accounting_sheets_type v);
LIBIPP_EXPORT std::string ToString(E_job_collation_type v);
LIBIPP_EXPORT std::string ToString(E_job_delay_output_until v);
LIBIPP_EXPORT std::string ToString(E_job_error_action v);
LIBIPP_EXPORT std::string ToString(E_job_error_sheet_when v);
LIBIPP_EXPORT std::string ToString(E_job_hold_until v);
LIBIPP_EXPORT std::string ToString(E_job_mandatory_attributes v);
LIBIPP_EXPORT std::string ToString(E_job_password_encryption v);
LIBIPP_EXPORT std::string ToString(E_job_sheets v);
LIBIPP_EXPORT std::string ToString(E_job_spooling_supported v);
LIBIPP_EXPORT std::string ToString(E_job_state v);
LIBIPP_EXPORT std::string ToString(E_job_state_reasons v);
LIBIPP_EXPORT std::string ToString(E_laminating_type v);
LIBIPP_EXPORT std::string ToString(E_material_color v);
LIBIPP_EXPORT std::string ToString(E_media v);
LIBIPP_EXPORT std::string ToString(E_media_back_coating v);
LIBIPP_EXPORT std::string ToString(E_media_grain v);
LIBIPP_EXPORT std::string ToString(E_media_input_tray_check v);
LIBIPP_EXPORT std::string ToString(E_media_key v);
LIBIPP_EXPORT std::string ToString(E_media_pre_printed v);
LIBIPP_EXPORT std::string ToString(E_media_ready v);
LIBIPP_EXPORT std::string ToString(E_media_source v);
LIBIPP_EXPORT std::string ToString(E_media_tooth v);
LIBIPP_EXPORT std::string ToString(E_media_type v);
LIBIPP_EXPORT std::string ToString(E_multiple_document_handling v);
LIBIPP_EXPORT std::string ToString(E_multiple_operation_time_out_action v);
LIBIPP_EXPORT std::string ToString(E_notify_events v);
LIBIPP_EXPORT std::string ToString(E_notify_pull_method v);
LIBIPP_EXPORT std::string ToString(E_operations_supported v);
LIBIPP_EXPORT std::string ToString(E_page_delivery v);
LIBIPP_EXPORT std::string ToString(E_pdf_versions_supported v);
LIBIPP_EXPORT std::string ToString(E_pdl_init_file_supported v);
LIBIPP_EXPORT std::string ToString(E_pdl_override_supported v);
LIBIPP_EXPORT std::string ToString(E_presentation_direction_number_up v);
LIBIPP_EXPORT std::string ToString(E_print_color_mode v);
LIBIPP_EXPORT std::string ToString(E_print_content_optimize v);
LIBIPP_EXPORT std::string ToString(E_print_rendering_intent v);
LIBIPP_EXPORT std::string ToString(E_print_scaling v);
LIBIPP_EXPORT std::string ToString(E_printer_state v);
LIBIPP_EXPORT std::string ToString(E_printer_state_reasons v);
LIBIPP_EXPORT std::string ToString(E_proof_print_supported v);
LIBIPP_EXPORT std::string ToString(E_pwg_raster_document_sheet_back v);
LIBIPP_EXPORT std::string ToString(E_pwg_raster_document_type_supported v);
LIBIPP_EXPORT std::string ToString(E_requested_attributes v);
LIBIPP_EXPORT std::string ToString(E_save_disposition v);
LIBIPP_EXPORT std::string ToString(E_separator_sheets_type v);
LIBIPP_EXPORT std::string ToString(E_sheet_collate v);
LIBIPP_EXPORT std::string ToString(E_status_code v);
LIBIPP_EXPORT std::string ToString(E_stitching_method v);
LIBIPP_EXPORT std::string ToString(E_stitching_reference_edge v);
LIBIPP_EXPORT std::string ToString(E_trimming_type v);
LIBIPP_EXPORT std::string ToString(E_trimming_when v);
LIBIPP_EXPORT std::string ToString(E_uri_authentication_supported v);
LIBIPP_EXPORT std::string ToString(E_uri_security_supported v);
LIBIPP_EXPORT std::string ToString(E_which_jobs v);
LIBIPP_EXPORT std::string ToString(E_x_image_position v);
LIBIPP_EXPORT std::string ToString(E_xri_authentication v);
LIBIPP_EXPORT std::string ToString(E_xri_security v);
LIBIPP_EXPORT std::string ToString(E_y_image_position v);

LIBIPP_EXPORT bool FromString(const std::string&, AttrName, int* value);

LIBIPP_EXPORT bool FromString(const std::string& s, GroupTag* v);
LIBIPP_EXPORT bool FromString(const std::string& s, AttrName* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_auth_info_required* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_baling_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_baling_when* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_binding_reference_edge* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_binding_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_coating_sides* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_coating_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_compression* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_cover_back_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_cover_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_covering_name* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_current_page_order* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_document_digital_signature* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_document_format_details_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_document_format_varying_attributes* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_feed_orientation* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_finishing_template* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_finishings* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_folding_direction* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_identify_actions* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_imposition_template* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_input_orientation_requested* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_input_quality* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_input_sides* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_ipp_features_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_ipp_versions_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_account_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_job_accounting_output_bin* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_job_accounting_sheets_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_collation_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_job_delay_output_until* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_error_action* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_error_sheet_when* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_hold_until* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_job_mandatory_attributes* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_job_password_encryption* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_sheets* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_job_spooling_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_state* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_job_state_reasons* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_laminating_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_material_color* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_back_coating* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_grain* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_media_input_tray_check* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_key* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_pre_printed* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_ready* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_source* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_tooth* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_media_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_multiple_document_handling* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_multiple_operation_time_out_action* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_notify_events* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_notify_pull_method* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_operations_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_page_delivery* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_pdf_versions_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_pdl_init_file_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_pdl_override_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_presentation_direction_number_up* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_print_color_mode* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_print_content_optimize* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_print_rendering_intent* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_print_scaling* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_printer_state* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_printer_state_reasons* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_proof_print_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_pwg_raster_document_sheet_back* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_pwg_raster_document_type_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_requested_attributes* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_save_disposition* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_separator_sheets_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_sheet_collate* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_status_code* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_stitching_method* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_stitching_reference_edge* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_trimming_type* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_trimming_when* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_uri_authentication_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s,
                              E_uri_security_supported* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_which_jobs* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_x_image_position* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_xri_authentication* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_xri_security* v);
LIBIPP_EXPORT bool FromString(const std::string& s, E_y_image_position* v);
}  // namespace ipp

#endif  //  LIBIPP_IPP_ENUMS_H_
