// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef MEDIA_FILTERS_HLS_CODEC_DETECTOR_H_
#define MEDIA_FILTERS_HLS_CODEC_DETECTOR_H_

#include "base/functional/callback.h"
#include "base/sequence_checker.h"
#include "media/base/media_log.h"
#include "media/base/media_tracks.h"
#include "media/base/stream_parser.h"
#include "media/filters/hls_data_source_provider.h"
#include "media/filters/hls_demuxer_status.h"
#include "media/filters/hls_rendition.h"

namespace media {

// Helper class for parsing a hls segment to determine a container and codec
// list, which can be used to initialize the chunk demuxer.
class MEDIA_EXPORT HlsCodecDetector {
 public:
  struct ContainerAndCodecs {
    std::string container;
    std::string codecs;
  };
  using CodecCallback = HlsDemuxerStatusCb<ContainerAndCodecs>;
  virtual ~HlsCodecDetector() = 0;

  virtual void DetermineContainerAndCodec(
      std::unique_ptr<HlsDataSourceStream> stream,
      CodecCallback cb) = 0;
  virtual void DetermineContainerOnly(
      std::unique_ptr<HlsDataSourceStream> stream,
      CodecCallback cb) = 0;
};

class MEDIA_EXPORT HlsCodecDetectorImpl : public HlsCodecDetector {
 public:
  ~HlsCodecDetectorImpl() override;

  // The HlsRenditionHost owns the only implementation of HlsCodecDetector
  // and uses it only on a single thread, where it is also deleted. Keeping
  // a raw pointer to that HlsRenditionHost is therefore safe.
  HlsCodecDetectorImpl(MediaLog* media_log, HlsRenditionHost* rendition_host_);

  void DetermineContainerAndCodec(std::unique_ptr<HlsDataSourceStream> stream,
                                  CodecCallback cb) override;
  void DetermineContainerOnly(std::unique_ptr<HlsDataSourceStream> stream,
                              CodecCallback cb) override;

 private:
  void PostSuccessToCallback(std::string container, std::string codecs);
  void OnStreamFetched(bool container_only,
                       HlsDataSourceProvider::ReadResult stream);
  HlsDemuxerStatus DetermineContainer(bool container_only,
                                      const uint8_t* data,
                                      size_t size);
  void ParserInit(const StreamParser::InitParameters& params);
  bool OnNewConfigMP2T(std::unique_ptr<MediaTracks> tracks);
  bool OnNewBuffers(const StreamParser::BufferQueueMap& buffers);
  void OnEncryptedMediaInit(EmeInitDataType type,
                            const std::vector<uint8_t>& data);
  void AddCodecToResponse(std::string codec);

  std::unique_ptr<MediaLog> log_;
  CodecCallback callback_;
  std::unique_ptr<media::StreamParser> parser_;
  std::string codec_response_;
  std::string container_;
  raw_ptr<HlsRenditionHost> rendition_host_ = nullptr;

  SEQUENCE_CHECKER(sequence_checker_);
  base::WeakPtrFactory<HlsCodecDetectorImpl> weak_factory_{this};
};

}  // namespace media

#endif  // MEDIA_FILTERS_HLS_CODEC_DETECTOR_H_
