// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/device/time_zone_monitor/time_zone_monitor_android.h"

#include <memory>
#include <string>

#include "base/android/jni_android.h"
#include "base/android/jni_string.h"
#include "base/task/sequenced_task_runner.h"
#include "third_party/icu/source/common/unicode/unistr.h"
#include "third_party/icu/source/i18n/unicode/timezone.h"

// Must come after all headers that specialize FromJniType() / ToJniType().
#include "services/device/time_zone_monitor/time_zone_monitor_jni_headers/TimeZoneMonitor_jni.h"

using base::android::JavaParamRef;

namespace device {

TimeZoneMonitorAndroid::TimeZoneMonitorAndroid() : TimeZoneMonitor() {
  impl_.Reset(
      Java_TimeZoneMonitor_getInstance(base::android::AttachCurrentThread(),
                                       reinterpret_cast<intptr_t>(this)));
}

TimeZoneMonitorAndroid::~TimeZoneMonitorAndroid() {
  Java_TimeZoneMonitor_stop(base::android::AttachCurrentThread(), impl_);
}

void TimeZoneMonitorAndroid::TimeZoneChangedFromJava(
    JNIEnv* env,
    const std::u16string& zone_id) {
  // See base/i18n/icu_util.cc:InitializeIcuTimeZone() for more information.
  std::unique_ptr<icu::TimeZone> new_zone(icu::TimeZone::createTimeZone(
      icu::UnicodeString(false, zone_id.data(), zone_id.length())));
  UpdateIcuAndNotifyClients(std::move(new_zone));
}

// static
std::unique_ptr<TimeZoneMonitor> TimeZoneMonitor::Create(
    scoped_refptr<base::SequencedTaskRunner> file_task_runner) {
  return std::unique_ptr<TimeZoneMonitor>(new TimeZoneMonitorAndroid());
}

}  // namespace device
