// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/cr_components/history_embeddings/history_embeddings_handler.h"

#include "base/i18n/time_formatting.h"
#include "base/strings/string_number_conversions.h"
#include "base/test/metrics/histogram_tester.h"
#include "base/test/mock_callback.h"
#include "base/test/scoped_feature_list.h"
#include "base/test/test_future.h"
#include "chrome/browser/history/history_service_factory.h"
#include "chrome/browser/history_embeddings/history_embeddings_service_factory.h"
#include "chrome/browser/optimization_guide/mock_optimization_guide_keyed_service.h"
#include "chrome/browser/optimization_guide/optimization_guide_keyed_service_factory.h"
#include "chrome/browser/page_content_annotations/page_content_annotations_service_factory.h"
#include "chrome/browser/ui/browser_commands.h"
#include "chrome/browser/ui/browser_window/public/browser_window_features.h"
#include "chrome/browser/ui/hats/hats_service_factory.h"
#include "chrome/browser/ui/hats/mock_hats_service.h"
#include "chrome/browser/ui/hats/survey_config.h"
#include "chrome/browser/user_education/user_education_service.h"
#include "chrome/browser/user_education/user_education_service_factory.h"
#include "chrome/common/chrome_features.h"
#include "chrome/test/base/browser_with_test_window_test.h"
#include "chrome/test/base/testing_browser_process.h"
#include "chrome/test/base/testing_profile.h"
#include "chrome/test/base/testing_profile_manager.h"
#include "chrome/test/user_education/mock_browser_user_education_interface.h"
#include "components/history_embeddings/answerer.h"
#include "components/history_embeddings/history_embeddings_features.h"
#include "components/history_embeddings/history_embeddings_service.h"
#include "components/page_content_annotations/core/test_page_content_annotations_service.h"
#include "components/passage_embeddings/passage_embeddings_test_util.h"
#include "components/user_education/test/mock_feature_promo_controller.h"
#include "content/public/browser/browser_context.h"
#include "content/public/test/test_web_ui.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "ui/base/l10n/time_format.h"
#include "ui/webui/resources/cr_components/history_embeddings/history_embeddings.mojom.h"

#if BUILDFLAG(IS_CHROMEOS)
#include "chromeos/constants/chromeos_features.h"
#endif  // BUILDFLAG(IS_CHROMEOS)

namespace {

class MockPage : public history_embeddings::mojom::Page {
 public:
  MockPage() = default;
  ~MockPage() override = default;

  mojo::PendingRemote<history_embeddings::mojom::Page> BindAndGetRemote() {
    DCHECK(!receiver_.is_bound());
    return receiver_.BindNewPipeAndPassRemote();
  }
  mojo::Receiver<history_embeddings::mojom::Page> receiver_{this};

  void FlushForTesting() { receiver_.FlushForTesting(); }

  MOCK_METHOD(void,
              SearchResultChanged,
              (history_embeddings::mojom::SearchResultPtr));
};

}  // namespace

std::unique_ptr<KeyedService> BuildTestHistoryEmbeddingsService(
    passage_embeddings::TestEnvironment* passage_embeddings_test_env,
    content::BrowserContext* browser_context) {
  return HistoryEmbeddingsServiceFactory::
      BuildServiceInstanceForBrowserContextForTesting(
          browser_context,
          passage_embeddings_test_env->embedder_metadata_provider(),
          passage_embeddings_test_env->embedder(),
          /*answerer=*/nullptr,
          /*intent_classifier=*/nullptr);
}

std::unique_ptr<KeyedService> BuildTestPageContentAnnotationsService(
    content::BrowserContext* browser_context) {
  auto* profile = Profile::FromBrowserContext(browser_context);
  auto* history_service = HistoryServiceFactory::GetForProfile(
      profile, ServiceAccessType::EXPLICIT_ACCESS);
  auto* optimization_guide_keyed_service =
      OptimizationGuideKeyedServiceFactory::GetForProfile(profile);
  return page_content_annotations::TestPageContentAnnotationsService::Create(
      optimization_guide_keyed_service, history_service);
}

std::unique_ptr<KeyedService> BuildTestOptimizationGuideKeyedService(
    content::BrowserContext* browser_context) {
  return std::make_unique<
      testing::NiceMock<MockOptimizationGuideKeyedService>>();
}

class HistoryEmbeddingsHandlerTest : public BrowserWithTestWindowTest {
 public:
  void SetUp() override {
    feature_list_.InitWithFeaturesAndParameters(
        /*enabled_features=*/{{history_embeddings::kHistoryEmbeddings,
                               {
                                   {"TrimAfterHostInResults", "true"},
                               }},
                              {history_embeddings::kHistoryEmbeddingsAnswers,
                               {}},
                              {feature_engagement::kIPHHistorySearchFeature,
                               {}},
#if BUILDFLAG(IS_CHROMEOS)
                              {chromeos::features::
                                   kFeatureManagementHistoryEmbedding,
                               {{}}}
#endif  // BUILDFLAG(IS_CHROMEOS)
        },
        /*disabled_features=*/{});

    user_ed_override_ =
        BrowserWindowFeatures::GetUserDataFactoryForTesting()
            .AddOverrideForTesting(
                base::BindRepeating([](BrowserWindowInterface& window) {
                  return std::make_unique<MockBrowserUserEducationInterface>(
                      &window);
                }));

    BrowserWithTestWindowTest::SetUp();

    TestingProfile* profile_ = profile_manager()->CreateTestingProfile(
        "History Embeddings Test User",
        {
            TestingProfile::TestingFactory{
                HistoryServiceFactory::GetInstance(),
                HistoryServiceFactory::GetDefaultFactory()},
            TestingProfile::TestingFactory{
                HistoryEmbeddingsServiceFactory::GetInstance(),
                base::BindRepeating(&BuildTestHistoryEmbeddingsService,
                                    &passage_embeddings_test_env_)},
            TestingProfile::TestingFactory{
                PageContentAnnotationsServiceFactory::GetInstance(),
                base::BindRepeating(&BuildTestPageContentAnnotationsService)},
            TestingProfile::TestingFactory{
                OptimizationGuideKeyedServiceFactory::GetInstance(),
                base::BindRepeating(&BuildTestOptimizationGuideKeyedService)},
        });

    web_contents_ = content::WebContents::Create(
        content::WebContents::CreateParams(profile_));
    web_ui_.set_web_contents(web_contents_.get());
    browser()->tab_strip_model()->AppendWebContents(std::move(web_contents_),
                                                    true);
    mock_hats_service_ = static_cast<MockHatsService*>(
        HatsServiceFactory::GetInstance()->SetTestingFactoryAndUse(
            profile_, base::BindRepeating(&BuildMockHatsService)));

    handler_ = std::make_unique<HistoryEmbeddingsHandler>(
        mojo::PendingReceiver<history_embeddings::mojom::PageHandler>(),
        profile_->GetWeakPtr(), web_ui(), false);
    handler_->SetPage(page_.BindAndGetRemote());
  }

  void TearDown() override {
    browser()->tab_strip_model()->CloseAllTabs();
    mock_hats_service_ = nullptr;
    web_contents_.reset();
    handler_.reset();
    BrowserWithTestWindowTest::TearDown();
  }

  content::TestWebUI* web_ui() { return &web_ui_; }

  base::HistogramTester& histogram_tester() { return histogram_tester_; }

  MockBrowserUserEducationInterface* user_education() {
    return static_cast<MockBrowserUserEducationInterface*>(
        BrowserUserEducationInterface::From(browser()));
  }

 protected:
  base::test::ScopedFeatureList feature_list_;
  std::unique_ptr<content::WebContents> web_contents_;
  content::TestWebUI web_ui_;
  passage_embeddings::TestEnvironment passage_embeddings_test_env_;
  std::unique_ptr<HistoryEmbeddingsHandler> handler_;
  testing::NiceMock<MockPage> page_;
  raw_ptr<MockHatsService> mock_hats_service_;
  base::HistogramTester histogram_tester_;
  ui::UserDataFactory::ScopedOverride user_ed_override_;
};

TEST_F(HistoryEmbeddingsHandlerTest, Searches) {
  auto query = history_embeddings::mojom::SearchQuery::New();
  query->query = "search query for empty result";
  base::test::TestFuture<history_embeddings::mojom::SearchResultPtr> future;
  EXPECT_CALL(page_, SearchResultChanged)
      .WillOnce(base::test::InvokeFuture(future));
  handler_->Search(std::move(query));
  auto result = future.Take();
  ASSERT_EQ(result->items.size(), 0u);
}

TEST_F(HistoryEmbeddingsHandlerTest, FormatsMojoResults) {
  history_embeddings::ScoredUrlRow scored_url_row(
      history_embeddings::ScoredUrl(0, 0, {}, 0.5f, 0.2f));
  scored_url_row.row = history::URLRow{GURL{"https://google.com/search"}};
  scored_url_row.row.set_title(u"my title");
  scored_url_row.row.set_last_visit(base::Time::Now() - base::Hours(1));
  history_embeddings::ScoredUrlRow other_scored_url_row = scored_url_row;
  other_scored_url_row.row = history::URLRow(GURL("http://other.com"));
  other_scored_url_row.is_url_known_to_sync = true;

  history_embeddings::SearchResult embeddings_result;
  embeddings_result.scored_url_rows = {
      scored_url_row,
      other_scored_url_row,
  };
  embeddings_result.query = "search query";
  embeddings_result.answerer_result.status =
      history_embeddings::ComputeAnswerStatus::kSuccess;
  embeddings_result.answerer_result.answer.set_text("the answer");
  embeddings_result.answerer_result.url = "http://other.com";
  embeddings_result.answerer_result.text_directives = {"text fragment"};

  base::test::TestFuture<history_embeddings::mojom::SearchResultPtr> future;
  EXPECT_CALL(page_, SearchResultChanged)
      .WillOnce(base::test::InvokeFuture(future));
  handler_->PublishResultToPageForTesting(std::move(embeddings_result));

  auto mojo_result = future.Take();
  EXPECT_EQ(mojo_result->query, "search query");
  EXPECT_EQ(mojo_result->answer_status,
            history_embeddings::mojom::AnswerStatus::kSuccess);
  EXPECT_EQ(mojo_result->answer, "the answer");
  ASSERT_EQ(mojo_result->items.size(), 2u);
  EXPECT_EQ(mojo_result->items[0]->title, "my title");
  EXPECT_EQ(mojo_result->items[0]->url.spec(), "https://google.com/search");
  EXPECT_EQ(mojo_result->items[0]->relative_time,
            base::UTF16ToUTF8(ui::TimeFormat::Simple(
                ui::TimeFormat::FORMAT_ELAPSED, ui::TimeFormat::LENGTH_SHORT,
                base::Time::Now() - scored_url_row.row.last_visit())));
  EXPECT_EQ(mojo_result->items[0]->short_date_time,
            base::UTF16ToUTF8(
                base::TimeFormatShortDate(scored_url_row.row.last_visit())));
  EXPECT_EQ(mojo_result->items[0]->last_url_visit_timestamp,
            scored_url_row.row.last_visit().InMillisecondsFSinceUnixEpoch());
  EXPECT_EQ(mojo_result->items[0]->url_for_display, "google.com");
  EXPECT_EQ(mojo_result->items[0]->answer_data.is_null(), true);
  EXPECT_EQ(mojo_result->items[0]->is_url_known_to_sync, false);
  EXPECT_EQ(mojo_result->items[1]->url.spec(), "http://other.com/");
  EXPECT_EQ(mojo_result->items[1]->url_for_display, "other.com");
  EXPECT_EQ(mojo_result->items[1]->answer_data.is_null(), false);
  EXPECT_EQ(mojo_result->items[1]->answer_data->answer_text_directives.size(),
            1u);
  EXPECT_EQ(mojo_result->items[1]->answer_data->answer_text_directives[0],
            "text fragment");
  EXPECT_EQ(mojo_result->items[1]->is_url_known_to_sync, true);
}

TEST_F(HistoryEmbeddingsHandlerTest, RecordsMetrics) {
  handler_->RecordSearchResultsMetrics(false, false, false, false, false, 2);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kEmbeddingsSearch, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kEmbeddingsNonEmptyResultsShown, 0);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kEmbeddingsResultClicked, 0);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kAnswerShown, 0);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kAnswerCitationClicked, 0);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kOtherHistoryResultClicked, 0);

  handler_->RecordSearchResultsMetrics(true, true, true, true, true, 2);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kEmbeddingsSearch, 2);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kEmbeddingsNonEmptyResultsShown, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kEmbeddingsResultClicked, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kAnswerShown, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kAnswerCitationClicked, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions",
      HistoryEmbeddingsUserActions::kOtherHistoryResultClicked, 1);

  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions.HistoryPage",
      HistoryEmbeddingsUserActions::kEmbeddingsSearch, 2);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions.HistoryPage",
      HistoryEmbeddingsUserActions::kEmbeddingsNonEmptyResultsShown, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions.HistoryPage",
      HistoryEmbeddingsUserActions::kEmbeddingsResultClicked, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions.HistoryPage",
      HistoryEmbeddingsUserActions::kAnswerShown, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions.HistoryPage",
      HistoryEmbeddingsUserActions::kAnswerCitationClicked, 1);
  histogram_tester().ExpectBucketCount(
      "History.Embeddings.UserActions.HistoryPage",
      HistoryEmbeddingsUserActions::kOtherHistoryResultClicked, 1);
}

TEST_F(HistoryEmbeddingsHandlerTest, ShowsPromo) {
  EXPECT_CALL(
      *user_education(),
      MaybeShowFeaturePromo(user_education::test::MatchFeaturePromoParams(
          feature_engagement::kIPHHistorySearchFeature)))
      .Times(1);
  handler_->MaybeShowFeaturePromo();
}

TEST_F(HistoryEmbeddingsHandlerTest, LaunchesDelayedHaTSSurvey) {
  base::test::ScopedFeatureList features;
  features.InitWithFeatures(
      {features::kHappinessTrackingSurveysForHistoryEmbeddings}, {});

  bool non_empty_results = true;
  bool user_clicked_results = true;
  bool answer_shown = true;
  bool answer_citation_clicked = false;
  bool other_history_result_clicked = false;
  const SurveyBitsData survey_bits_data = {
      {"non empty results", non_empty_results},
      {"best matches result clicked", user_clicked_results},
      {"result clicked", other_history_result_clicked},
      {"answer shown", answer_shown},
      {"answer citation clicked", answer_citation_clicked},
  };
  int query_word_count = 2;
  const SurveyStringData product_specific_string_data = {
      {"query word count", base::NumberToString(query_word_count)}};
  EXPECT_CALL(
      *mock_hats_service_,
      LaunchDelayedSurvey(kHatsSurveyTriggerHistoryEmbeddings, testing::_,
                          survey_bits_data, product_specific_string_data))
      .Times(1);
  handler_->RecordSearchResultsMetrics(
      non_empty_results, user_clicked_results, answer_shown,
      answer_citation_clicked, other_history_result_clicked, query_word_count);
}
