// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/common/credential_manager_types.h"

#include <memory>

#include "base/strings/string_number_conversions.h"

namespace password_manager {

std::string CredentialTypeToString(CredentialType value) {
  switch (value) {
    case CredentialType::CREDENTIAL_TYPE_EMPTY:
      return "CredentialType::CREDENTIAL_TYPE_EMPTY";
    case CredentialType::CREDENTIAL_TYPE_PASSWORD:
      return "CredentialType::CREDENTIAL_TYPE_PASSWORD";
    case CredentialType::CREDENTIAL_TYPE_FEDERATED:
      return "CredentialType::CREDENTIAL_TYPE_FEDERATED";
  }
  return "Unknown CredentialType value: " +
         base::NumberToString(static_cast<int>(value));
}

std::ostream& operator<<(std::ostream& os, CredentialType value) {
  return os << CredentialTypeToString(value);
}

CredentialInfo::CredentialInfo() = default;

CredentialInfo::CredentialInfo(CredentialType type,
                               base::Optional<base::string16> id,
                               base::Optional<base::string16> name,
                               GURL icon,
                               base::Optional<base::string16> password,
                               url::Origin federation)
    : type(type),
      id(std::move(id)),
      name(std::move(name)),
      icon(std::move(icon)),
      password(std::move(password)),
      federation(std::move(federation)) {
  switch (type) {
    case CredentialType::CREDENTIAL_TYPE_EMPTY:
      password = base::string16();
      federation = url::Origin();
      break;
    case CredentialType::CREDENTIAL_TYPE_PASSWORD:
      federation = url::Origin();
      break;
    case CredentialType::CREDENTIAL_TYPE_FEDERATED:
      password = base::string16();
      break;
  }
}

CredentialInfo::CredentialInfo(const CredentialInfo& other) = default;

CredentialInfo::~CredentialInfo() = default;

bool CredentialInfo::operator==(const CredentialInfo& rhs) const {
  return (type == rhs.type && id == rhs.id && name == rhs.name &&
          icon == rhs.icon && password == rhs.password &&
          federation.Serialize() == rhs.federation.Serialize());
}

}  // namespace password_manager
