// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_RENDERER_PLUGINS_CHROME_PLUGIN_PLACEHOLDER_H_
#define CHROME_RENDERER_PLUGINS_CHROME_PLUGIN_PLACEHOLDER_H_

#include <stdint.h>
#include <string>

#include "base/macros.h"
#include "chrome/common/buildflags.h"
#include "chrome/common/plugin.mojom.h"
#include "components/no_state_prefetch/renderer/prerender_observer.h"
#include "components/plugins/renderer/loadable_plugin_placeholder.h"
#include "content/public/renderer/render_thread_observer.h"
#include "mojo/public/cpp/bindings/pending_remote.h"
#include "mojo/public/cpp/bindings/receiver.h"
#include "third_party/blink/public/mojom/context_menu/context_menu.mojom.h"

class ChromePluginPlaceholder final
    : public plugins::LoadablePluginPlaceholder,
      public content::RenderThreadObserver,
      public blink::mojom::ContextMenuClient,
      public chrome::mojom::PluginRenderer,
      public prerender::PrerenderObserver,
      public gin::Wrappable<ChromePluginPlaceholder> {
 public:
  static gin::WrapperInfo kWrapperInfo;

  static ChromePluginPlaceholder* CreateBlockedPlugin(
      content::RenderFrame* render_frame,
      const blink::WebPluginParams& params,
      const content::WebPluginInfo& info,
      const std::string& identifier,
      const base::string16& name,
      int resource_id,
      const base::string16& message);

  // Creates a new WebViewPlugin with a MissingPlugin as a delegate.
  static ChromePluginPlaceholder* CreateLoadableMissingPlugin(
      content::RenderFrame* render_frame,
      const blink::WebPluginParams& params);

  // Runs |callback| over each plugin placeholder for the given RenderFrame.
  static void ForEach(
      content::RenderFrame* render_frame,
      const base::RepeatingCallback<void(ChromePluginPlaceholder*)>& callback);

  void SetStatus(chrome::mojom::PluginStatus status);

  mojo::PendingRemote<chrome::mojom::PluginRenderer> BindPluginRenderer();

 private:
  ChromePluginPlaceholder(content::RenderFrame* render_frame,
                          const blink::WebPluginParams& params,
                          const std::string& html_data,
                          const base::string16& title);
  ~ChromePluginPlaceholder() override;

  // content::LoadablePluginPlaceholder overrides.
  blink::WebPlugin* CreatePlugin() override;
  void OnBlockedContent(content::RenderFrame::PeripheralContentStatus status,
                        bool is_same_origin) override;

  // gin::Wrappable (via PluginPlaceholder) method
  gin::ObjectTemplateBuilder GetObjectTemplateBuilder(
      v8::Isolate* isolate) final;

  // WebViewPlugin::Delegate (via PluginPlaceholder) methods:
  v8::Local<v8::Value> GetV8Handle(v8::Isolate* isolate) override;
  void ShowContextMenu(const blink::WebMouseEvent&) override;

  // content::RenderThreadObserver methods:
  void PluginListChanged() override;

  // Show the Plugins permission bubble.
  void ShowPermissionBubbleCallback();

  // chrome::mojom::PluginRenderer methods.
  void FinishedDownloading() override;
  void UpdateDownloading() override;
  void UpdateSuccess() override;
  void UpdateFailure() override;

  // blink::mojom::ContextMenuClient methods.
  void CustomContextMenuAction(uint32_t action) override;
  void ContextMenuClosed(const GURL& link_followed) override;

  // prerender::PrerenderObserver methods:
  void SetIsPrerendering(bool is_prerendering) override;

  chrome::mojom::PluginStatus status_;

  base::string16 title_;

  base::string16 plugin_name_;

  mojo::Receiver<chrome::mojom::PluginRenderer> plugin_renderer_receiver_{this};

  mojo::AssociatedReceiver<blink::mojom::ContextMenuClient>
      context_menu_client_receiver_{this};

  DISALLOW_COPY_AND_ASSIGN(ChromePluginPlaceholder);
};

#endif  // CHROME_RENDERER_PLUGINS_CHROME_PLUGIN_PLACEHOLDER_H_
