/** \file wlib.h
 * Common definitions and declarations for the wlib library
 */

#ifndef HAVE_WLIB_H
#define HAVE_WLIB_H
#ifdef WINDOWS
#include <stdio.h>
#define FILE_SEP_CHAR "\\"
#include "getline.h"
#else
#define FILE_SEP_CHAR "/"
#endif

#include <stdbool.h>

#ifdef USE_SIMPLE_GETTEXT
char *bindtextdomain( char *domainname, char *dirname );
char *bind_textdomain_codeset(char *domainname, char *codeset );
char *textdomain( char *domainname );
char *gettext( const char *msgid );

char *g_win32_getlocale (void);
#endif

// conversion routines to and from UTF-8
bool wSystemToUTF8(const char *inString, char *outString,
                   unsigned outStringLength);
bool wUTF8ToSystem(const char *inString, char *outString,
                   unsigned outStringLength);
bool wIsUTF8(const char * string);

/*
 * Interface types
 */

// a big integer
typedef long wInteger_t;
// Position/Size of objects drawn on a WDraw canvas (fractional pixels)
typedef double wDrawPix_t;
// Position/Size of controls/windows (integral pixels)
typedef long wWinPix_t;
// Boolean
typedef int wBool_t;
// index for lists etc
typedef int wIndex_t;

/*
 * Opaque Pointers
 */
typedef struct wWin_t       * wWin_p;
typedef struct wControl_t   * wControl_p;
typedef struct wButton_t    * wButton_p;
typedef struct wString_t    * wString_p;
typedef struct wInteger_t   * wInteger_p;
typedef struct wFloat_t     * wFloat_p;
typedef struct wList_t      * wList_p;
typedef struct wChoice_t    * wChoice_p;
typedef struct wDraw_t      * wDraw_p;
typedef struct wMenu_t      * wMenu_p;
typedef struct wText_t      * wText_p;
typedef struct wMessage_t   * wMessage_p;
typedef struct wLine_t      * wLine_p;
typedef struct wMenuList_t  * wMenuList_p;
typedef struct wMenuPush_t  * wMenuPush_p;
typedef struct wMenuRadio_t * wMenuRadio_p;
typedef struct wMenuToggle_t* wMenuToggle_p;
typedef struct wBox_t       * wBox_p;
typedef struct wIcon_t      * wIcon_p;
typedef struct wDrawBitMap_t * wDrawBitMap_p;
typedef struct wFont_t      * wFont_p;
typedef struct wBitmap_t	* wBitmap_p;
typedef struct wStatus_t    * wStatus_p;
typedef int wDrawWidth;
typedef int wDrawColor;

typedef struct {
	const char * name;
	const char * value;
} wBalloonHelp_t;

extern long debugWindow;
extern long wDebugFont;

/*------------------------------------------------------------------------------
 *
 * Bitmap Controls bitmap.c
 */

wControl_p wBitmapCreate(wWin_p parent, wWinPix_t x, wWinPix_t y, long options,
                         const struct wIcon_t * iconP);
wIcon_p wIconCreateBitMap(wWinPix_t w, wWinPix_t h, const char *bits,
                          wDrawColor color);
#ifndef WINDOWS
// png's are a string pointer:
typedef const unsigned char * wIconBitMap_t;
#else
// xpm's are an array of string pointers:
typedef const unsigned char ** wIconBitMap_t;
#endif
wIcon_p wIconCreatePixMap(	const wIconBitMap_t  );
void wIconSetColor(		wIcon_p, wDrawColor );

/*------------------------------------------------------------------------------
 *
 * Frames around widgets boxes.c
 *
 */

typedef enum {
	wBoxThinB,
	wBoxThinW,
	wBoxAbove,
	wBoxBelow,
	wBoxThickB,
	wBoxThickW,
	wBoxRidge,
	wBoxTrough
}
wBoxType_e;

void wBoxSetSize(wBox_p b, wWinPix_t w, wWinPix_t h);
void wlibDrawBox(wWin_p win, wBoxType_e style, wWinPix_t x, wWinPix_t y,
                 wWinPix_t w, wWinPix_t h);
wBox_p wBoxCreate(wWin_p parent, wWinPix_t bx, wWinPix_t by,
                  const char *labelStr, wBoxType_e boxTyp, wWinPix_t bw, wWinPix_t bh);

/*------------------------------------------------------------------------------
 *
 * Buttons, toggles and radiobuttons button.c
 *
 */

/* Creation CallBacks */
typedef void (*wButtonCallBack_p)( void * );

/* Creation Options */
#define BB_DEFAULT	(1L<<5)
#define BB_CANCEL	(1L<<6)
#define BB_HELP (1L<<7)

/* Creation CallBacks */
typedef void (*wChoiceCallBack_p)( long, void * );

/* Creation Options */
#define BC_ICON 	(1L<<0)
#define BC_HORZ 	(1L<<22)
#define BC_NONE 	(1L<<19)
#define BC_NOBORDER 	(1L<<15)

void wButtonSetLabel(wButton_p bb, const char *labelStr);
void wButtonSetBusy(wButton_p bb, int value);
wButton_p wButtonCreate(wWin_p parent, wWinPix_t x, wWinPix_t y,
                        const char *helpStr, const char *labelStr, long option, wWinPix_t width,
                        wButtonCallBack_p action, void *data);
void wRadioSetValue(wChoice_p bc, long value);
long wRadioGetValue(wChoice_p bc);
void wToggleSetValue(wChoice_p bc, long value);
long wToggleGetValue(wChoice_p b);
wChoice_p wRadioCreate(wWin_p parent, wWinPix_t x, wWinPix_t y,
                       const char *helpStr, const char *labelStr, long option,
                       const char * const *labels, long *valueP, wChoiceCallBack_p action, void *data);
wChoice_p wToggleCreate(wWin_p parent, wWinPix_t x, wWinPix_t y,
                        const char *helpStr, const char *labelStr, long option,
                        const char * const *labels, long *valueP, wChoiceCallBack_p action, void *data);


/*------------------------------------------------------------------------------
 *
 * System Interface
 */

void wInitAppName(char *appName);

const char * wGetAppLibDir(			void );
const char * wGetAppWorkDir(			void );
const char * wGetUserHomeDir( void );

void wSetAudio(bool setting);
void wBeep( void );
wBool_t wNotice(		const char *, const char *, const char * );
int wNotice3(			const char *, const char *, const char *, const char * );
void wHelp(			const char * );


#define NT_INFORMATION 1
#define NT_WARNING	   2
#define NT_ERROR	   4

wBool_t wNoticeEx( int type, const char * msg, const char * yes,
                   const char * no );

unsigned wOpenFileExternal(char *filename);


void wSetBalloonHelp ( wBalloonHelp_t * );
void wEnableBalloonHelp ( int );
void wBalloonHelpUpdate ( void );

void wFlush(			void );

typedef void (*wAlarmCallBack_p)( void );
void wAlarm(			long, wAlarmCallBack_p );
void wPause(			long );
unsigned long wGetTimer(	void );

void wExit(			int );

typedef enum {	wCursorNormal,
                wCursorNone,
                wCursorAppStart,
                wCursorHand,
                wCursorNo,
                wCursorSizeAll,
                wCursorSizeNESW,
                wCursorSizeNS,
                wCursorSizeNWSE,
                wCursorSizeWE,
                wCursorWait,
                wCursorIBeam,
                wCursorCross,
                wCursorQuestion
             } wCursor_t;
void wSetCursor( wDraw_p, wCursor_t );
#define defaultCursor wCursorCross

const char * wMemStats( void );

#define WKEY_SHIFT	(1<<1)
#define WKEY_CTRL	(1<<2)
#define WKEY_ALT	(1<<3)
int wGetKeyState(		void );

void wGetDisplaySize(		wWinPix_t*, wWinPix_t* );

wIcon_p wIconCreateBitMap(	wWinPix_t, wWinPix_t, const char * bits,
                                wDrawColor );
void wIconDraw( wDraw_p d, wIcon_p bm, wWinPix_t x, wWinPix_t y );

void wConvertToCharSet(		char *, int );
void wConvertFromCharSet(	char *, int );
#ifdef WINDOWS
FILE * wFileOpen(		const char *, const char * );
#endif

/*------------------------------------------------------------------------------
 *
 * Main and Popup Windows
 */

/* Creation CallBacks */
typedef enum {
	wClose_e,
	wResize_e,
	wState_e,
	wQuit_e,
	wRedraw_e
}
winProcEvent;
typedef void (*wWinCallBack_p)( wWin_p, winProcEvent, void *, void * );

/* Creation Options */
#define F_AUTOSIZE	(1L<<1)
#define F_HEADER 	(1L<<2)
#define F_RESIZE 	(1L<<3)
#define F_BLOCK  	(1L<<4)
#define F_MENUBAR 	(1L<<5)
#define F_NOTAB		(1L<<8)
#define F_RECALLPOS	(1L<<9)
#define F_RECALLSIZE	(1L<<10)
#define F_TOP		(1L<<11)
#define F_CENTER	(1L<<12)
#define F_HIDE		(1L<<13)
#define F_MAXIMIZE  (1L<<14)
#define F_RESTRICT  (1L<<15)
#define F_NOTTRANSIENT (1L<<16)

wWin_p wWinMainCreate(	        const char *, wWinPix_t, wWinPix_t, const char *,
                                const char *, const char *,
                                long, wWinCallBack_p, void * );
wWin_p wWinPopupCreate(		wWin_p, wWinPix_t, wWinPix_t, const char *,
                                const char *, const char *,
                                long, wWinCallBack_p, void * );

wWin_p wMain(			int, char *[] );
void wWinSetBigIcon(		wWin_p, wIcon_p );
void wWinSetSmallIcon(		wWin_p, wIcon_p );
#define DONTGRABFOCUS 0x100
void wWinShow(			wWin_p, unsigned show );
wBool_t wWinIsVisible(		wWin_p );
wBool_t wWinIsMaximized( wWin_p win);
void wWinGetSize (		wWin_p, wWinPix_t *, wWinPix_t * );
void wWinSetSize(		wWin_p, wWinPix_t, wWinPix_t );
void wWinSetTitle(		wWin_p, const char * );
void wWinSetBusy(		wWin_p, wBool_t );
const char * wWinGetTitle(		wWin_p );
void wWinClear(			wWin_p, wWinPix_t, wWinPix_t, wWinPix_t, wWinPix_t );
void wMessage(			wWin_p, const char *, wBool_t );
void wWinTop(			wWin_p );
void wWinDoCancel(		wWin_p );
void wWinBlockEnable(		wBool_t );
void wSetGeometry(wWin_p, wWinPix_t min_width, wWinPix_t max_width,
                  wWinPix_t min_height, wWinPix_t max_height, wWinPix_t base_width,
                  wWinPix_t base_height, double aspect_ratio);

int wCreateSplash( char *appName, char *appVer );
int wSetSplashInfo( char *msg );
void wDestroySplash( void );

/*------------------------------------------------------------------------------
 *
 * Controls in general
 */

/* Creation Options */
#define BO_ICON		(1L<<0)
#define BO_DISABLED	(1L<<1)
#define BO_READONLY	(1L<<2)
#define BO_NOTAB	(1L<<8)
#define BO_BORDER	(1L<<9)
//#define BO_ENTER    (1L<<10)
#define BO_ENTER    0
#define BO_REPEAT   (1L<<11)
#define BO_IGNFOCUS	(1L<<12)

wWinPix_t wLabelWidth(		const char * );
const char * wControlGetHelp(		wControl_p );
void wControlSetHelp(		wControl_p, const char * );
void wControlShow(		wControl_p, wBool_t );
wWinPix_t wControlGetWidth(	wControl_p );
wWinPix_t wControlGetHeight(	wControl_p );
wWinPix_t wControlGetPosX(		wControl_p );
wWinPix_t wControlGetPosY(		wControl_p );
void wControlSetPos(		wControl_p, wWinPix_t, wWinPix_t );
void wControlSetFocus(		wControl_p );
void wControlActive(		wControl_p, wBool_t );
void wControlSetBalloon(	wControl_p, wWinPix_t, wWinPix_t, const char * );
void wControlSetLabel(		wControl_p, const char * );
void wControlSetBalloonText(	wControl_p, const char * );
void wControlSetContext(	wControl_p, void * );
void wControlHilite(		wControl_p, wBool_t );

void wControlLinkedSet( wControl_p b1, wControl_p b2 );
void wControlLinkedActive( wControl_p b, int active );

/*------------------------------------------------------------------------------
 *
 * String entry
 */

#define BS_TRIM			(1<<12)
/* Creation CallBacks */
typedef void (*wStringCallBack_p)( const char *, void *);
wString_p wStringCreate(	wWin_p, wWinPix_t, wWinPix_t, const char *,
                                const char *, long,
                                wWinPix_t, char *, wIndex_t, wStringCallBack_p,
                                void * );
void wStringSetValue(		wString_p, const char * );
void wStringSetWidth(		wString_p, wWinPix_t );
const char * wStringGetValue(		wString_p );


/*------------------------------------------------------------------------------
 *
 * Numeric Entry
 */

/* Creation CallBacks */
typedef void (*wIntegerCallBack_p)( long, void *, int);
typedef void (*wFloatCallBack_p)( double, void *, int);
wInteger_p wIntegerCreate(	wWin_p, wWinPix_t, wWinPix_t, const char *,
                                const char *, long,
                                wWinPix_t, wInteger_t, wInteger_t, wInteger_t *,
                                wIntegerCallBack_p, void * );
wFloat_p wFloatCreate(		wWin_p, wWinPix_t, wWinPix_t, const char *,
                                const char *, long,
                                wWinPix_t, double, double, double *,
                                wFloatCallBack_p, void * );
void wIntegerSetValue(		wInteger_p, wInteger_t );
void wFloatSetValue(		wFloat_p, double );
wInteger_t wIntegerGetValue(	wInteger_p );
double wFloatGetValue(		wFloat_p );


/*------------------------------------------------------------------------------
 *
 * Lists
 */

/* Creation CallBacks */
typedef void (*wListCallBack_p)( wIndex_t, const char *, wIndex_t, void *,
                                 void * );

/* Creation Options */
#define BL_DUP		(1L<<16)
#define BL_SORT		(1L<<17)
#define BL_MANY 	(1L<<18)
#define BL_NONE 	(1L<<19)
#define BL_SETSTAY 	(1L<<20)
#define BL_DBLCLICK 	(1L<<21)
#define BL_FIXFONT 	(1L<<22)
#define BL_EDITABLE	(1L<<23)
#define BL_ICON		(1L<<0)


/* lists, droplists and combo boxes */

wList_p wListCreate(		wWin_p, wWinPix_t, wWinPix_t, const char *, const char *,
                                long,
                                long, wWinPix_t, int, wWinPix_t *, wBool_t *, const char **, long *,
                                wListCallBack_p, void * );
wList_p wDropListCreate(	wWin_p, wWinPix_t, wWinPix_t, const char *,
                                const char *, long,
                                long, wWinPix_t, long *, wListCallBack_p, void * );

wList_p wComboListCreate(wWin_p parent, wWinPix_t x, wWinPix_t y,
                         const char *helpStr, const char *labelStr, long option, long number,
                         wWinPix_t width, long *valueP, wListCallBack_p action, void *data);
void wListClear(wList_p b);
void wListSetIndex(wList_p b, int element);
wIndex_t wListFindValue(wList_p b, const char *val);
wIndex_t wListGetCount(wList_p b);
wIndex_t wListGetIndex(		wList_p );
void *wListGetItemContext(wList_p b, wIndex_t inx);
wBool_t wListGetItemSelected(wList_p b, wIndex_t inx);
wIndex_t wListGetSelectedCount(wList_p b);
void wListSelectAll(wList_p bl);
wBool_t wListSetValues(wList_p b, wIndex_t row, const char *labelStr,
                       wIcon_p bm, void *itemData);
void wListDelete(wList_p b, wIndex_t inx);
int wListGetColumnWidths(wList_p bl, int colCnt, wWinPix_t *colWidths);
wIndex_t wListAddValue(wList_p b, const char *labelStr, wIcon_p bm,
                       void *itemData);
void wListSetSize(wList_p bl, wWinPix_t w, wWinPix_t h);
wIndex_t wListGetValues(	wList_p, char *, int, void * *, void * * );

/** \todo Check for the existance of following functions */
void  wListSetValue(		wList_p, const char * );
void wListSetActive(		wList_p, wIndex_t, wBool_t );
void wListSetEditable(		wList_p, wBool_t );


/*------------------------------------------------------------------------------
 *
 * Messages
 */

#define BM_LARGE (1L<<24)
#define BM_SMALL (1L<<25)
#define COMBOBOX (1L)

#define wMessageSetFont( x ) ( x & (BM_LARGE | BM_SMALL ))

#define wMessageCreate( w, p1, p2, l, p3, m ) wMessageCreateEx( w, p1, p2, l, p3, m, 0 )
wMessage_p wMessageCreateEx(	wWin_p, wWinPix_t, wWinPix_t, const char *,
                                wWinPix_t, const char *, long );

void wMessageSetValue(		wMessage_p, const char * );
void wMessageSetWidth(		wMessage_p, wWinPix_t );
wWinPix_t wMessageGetWidth( const char *testString );
wWinPix_t wMessageGetHeight( long );


/*------------------------------------------------------------------------------
 *
 * Lines
 */

typedef struct {
	int width;
	int x0, y0;
	int x1, y1;
} wLines_t, * wLines_p;

wLine_p wLineCreate(		wWin_p, const char *, int, wLines_t *);


/*------------------------------------------------------------------------------
 *
 * Text
 */

/* Creation Options */
#define BT_HSCROLL 	(1L<<24)
#define BT_CHARUNITS	(1L<<23)
#define BT_FIXEDFONT	(1L<<22)
#define BT_DOBOLD	(1L<<21)
#define BT_TOP		(1L<<20)	/* Show the top of the text */

wText_p wTextCreate(		wWin_p, wWinPix_t, wWinPix_t, const char *, const char *,
                                long,
                                wWinPix_t, wWinPix_t );
void wTextClear(		wText_p );
void wTextAppend(		wText_p, const char * );
void wTextSetReadonly(		wText_p, wBool_t );
int wTextGetSize(		wText_p );
void wTextGetText(		wText_p, char *, int );
wBool_t wTextGetModified(	wText_p );
void wTextReadFile(		wText_p, const char * );
wBool_t wTextSave(		wText_p, const char * );
wBool_t wTextPrint(		wText_p );
void wTextSetSize(		wText_p, wWinPix_t, wWinPix_t );
void wTextComputeSize(		wText_p, wWinPix_t, wWinPix_t, wWinPix_t *,
                                wWinPix_t * );
void wTextSetPosition(		wText_p bt, int pos );


/*------------------------------------------------------------------------------
 *
 * Draw
 */


typedef int wDrawOpts;
#define wDrawOptTemp	(1<<0)
#define wDrawOptNoClip	(1<<1)
#define wDrawOptTransparent  (1<<2)
#define wDrawOutlineFont (1<<3)
#ifdef CURSOR_SURFACE
#define wDrawOptCursor  (1<<4)
#define wDrawOptCursorClr (1<<5)
#define wDrawOptCursorClr (1<<6)
#define wDrawOptCursorRmv (1<<7)
#define wDrawOptCursorQuit (1<<8)
#define wDrawOptOpaque   (1<<9)
#endif


typedef enum {
	wDrawLineSolid,
	wDrawLineDash,
	wDrawLineDot,
	wDrawLineDashDot,
	wDrawLineDashDotDot,
	wDrawLineCenter,
	wDrawLinePhantom
}
wDrawLineType_e;

typedef enum {
	wPolyLineStraight,
	wPolyLineSmooth,
	wPolyLineRound
}
wPolyLine_e;

typedef int wAction_t;
#define wActionMove		(1)
#define wActionLDown		(2)
#define wActionLDrag		(3)
#define wActionLUp		(4)
#define wActionRDown		(5)
#define wActionRDrag		(6)
#define wActionRUp		(7)
#define wActionText		(8)
#define wActionExtKey		(9)
#define wActionWheelUp (10)
#define wActionWheelDown (11)
#define wActionLDownDouble (12)
#define wActionModKey (13)
#define wActionScrollUp (14)
#define wActionScrollDown (15)
#define wActionScrollLeft (16)
#define wActionScrollRight (17)
#define wActionMDown (18)
#define wActionMDrag (19)
#define wActionMUp (20)
#define wActionLast		wActionMUp


#define wRGB(R,G,B)\
	(long)(((((long)(R)<<16))&0xFF0000L)|((((long)(G))<<8)&0x00FF00L)|(((long)(B))&0x0000FFL))


/* Creation CallBacks */
typedef void (*wDrawRedrawCallBack_p)( wDraw_p, void *, wWinPix_t, wWinPix_t );
typedef void (*wDrawActionCallBack_p)(	wDraw_p, void*, wAction_t, wDrawPix_t,
                                        wDrawPix_t );

/* Creation Options */
#define BD_TICKS	(1L<<25)
#define BD_DIRECT	(1L<<26)
#define BD_NOCAPTURE (1L<<27)
#define BD_NOFOCUS  (1L<<28)
#define BD_MODKEYS  (1L<<29)

/* Create: */
wDraw_p wDrawCreate(		wWin_p, wWinPix_t, wWinPix_t, const char *, long,
                                wWinPix_t, wWinPix_t, void *,
                                wDrawRedrawCallBack_p, wDrawActionCallBack_p );

/* Draw: */
void wDrawLine(			wDraw_p, wDrawPix_t, wDrawPix_t, wDrawPix_t, wDrawPix_t,
                                wDrawWidth, wDrawLineType_e, wDrawColor,
                                wDrawOpts );
#define double2wAngle_t( A )	(A)
typedef double wAngle_t;
void wDrawArc(			wDraw_p, wDrawPix_t, wDrawPix_t, wDrawPix_t, wAngle_t,
                                wAngle_t,
                                int, wDrawWidth, wDrawLineType_e, wDrawColor,
                                wDrawOpts );
void wDrawPoint(		wDraw_p, wDrawPix_t, wDrawPix_t, wDrawColor, wDrawOpts );
#define double2wFontSize_t( FS )	(FS)
typedef double wFontSize_t;
void wDrawString(		wDraw_p, wDrawPix_t, wDrawPix_t, wAngle_t, const char *,
                                wFont_p,
                                wFontSize_t, wDrawColor, wDrawOpts );
void wDrawFilledRectangle(	wDraw_p, wDrawPix_t, wDrawPix_t, wDrawPix_t,
                                wDrawPix_t,
                                wDrawColor, wDrawOpts );
void wDrawPolygon(	wDraw_p, wDrawPix_t [][2], wPolyLine_e [], wIndex_t,
                        wDrawColor, wDrawWidth, wDrawLineType_e,
                        wDrawOpts, int, int );
void wDrawFilledCircle(		wDraw_p, wDrawPix_t, wDrawPix_t, wDrawPix_t,
                                wDrawColor, wDrawOpts );

void wDrawGetTextSize(		wDrawPix_t *, wDrawPix_t *, wDrawPix_t *, wDrawPix_t *,
                                wDraw_p, const char *, wFont_p,
                                wFontSize_t );
void wDrawClear(		wDraw_p );
void wDrawClearTemp(		wDraw_p );
wBool_t wDrawSetTempMode(	wDraw_p, wBool_t );

void wDrawDelayUpdate(		wDraw_p, wBool_t );
void wDrawClip(			wDraw_p, wDrawPix_t, wDrawPix_t, wDrawPix_t, wDrawPix_t );
wDrawColor wDrawColorGray(	int );
wDrawColor wDrawFindColor(	long );
long wDrawGetRGB(		wDrawColor );

/* Geometry */
double wDrawGetDPI(		wDraw_p );
double wDrawGetMaxRadius(	wDraw_p );
void wDrawSetSize(		wDraw_p, wWinPix_t, wWinPix_t, void * );
void wDrawGetSize(		wDraw_p, wWinPix_t *, wWinPix_t * );

/* Bitmaps */
wDrawBitMap_p wDrawBitMapCreate( wDraw_p, int, int, int, int,
                                 const unsigned char * );
void wDrawBitMap(		wDraw_p, wDrawBitMap_p, wDrawPix_t, wDrawPix_t,
                                wDrawColor, wDrawOpts );

wDraw_p wBitMapCreate(		wWinPix_t, wWinPix_t, int );
wBool_t wBitMapDelete(		wDraw_p );
wBool_t wBitMapWriteFile(	wDraw_p, const char * );

/* Misc */
void * wDrawGetContext(		wDraw_p );
void wDrawSaveImage(		wDraw_p );
void wDrawRestoreImage(		wDraw_p );
int wDrawSetBackground(    wDraw_p, char * path, char ** error);
void wDrawCloneBackground(wDraw_p from, wDraw_p to);
void wDrawShowBackground(   wDraw_p, wWinPix_t pos_x, wWinPix_t pos_y,
                            wWinPix_t width, wAngle_t angle, int screen);

/*------------------------------------------------------------------------------
 *
 * Fonts
 */
void wInitializeFonts();
void wSelectFont(		const char * );
wFontSize_t wSelectedFontSize(	void );
void wSetSelectedFontSize(wFontSize_t size);
#define F_TIMES	(1)
#define F_HELV	(2)
wFont_p wStandardFont(		int, wBool_t, wBool_t );


/*------------------------------------------------------------------------------
 *
 * Printing
 */

typedef void (*wPrintSetupCallBack_p)( wBool_t );

wBool_t wPrintInit(		void );
void wPrintSetup(		wPrintSetupCallBack_p );
void wPrintGetMargins(		double *, double *, double *, double * );
void wPrintGetPageSize(		double *, double * );
wBool_t wPrintDocStart(		const char *, int, int * );
wDraw_p wPrintPageStart(	void );
wBool_t wPrintPageEnd(		wDraw_p );
void wPrintDocEnd(		void );
wBool_t wPrintQuit(		void );
void wPrintClip(		wDrawPix_t, wDrawPix_t, wDrawPix_t, wDrawPix_t );
const char * wPrintGetName(	void );


/*------------------------------------------------------------------------------
 *
 * Menus
 */

#define WACCL_BASE	(1000)
#define WALT		(1<<10)
#define WCTL		(1<<11)
#define WMETA		(1<<12)
#define WSHIFT		(1<<13)

typedef enum {
	wAccelKey_None,
	wAccelKey_Del,
	wAccelKey_Ins,
	wAccelKey_Home,
	wAccelKey_End,
	wAccelKey_Pgup,
	wAccelKey_Pgdn,
	wAccelKey_Up,
	wAccelKey_Down,
	wAccelKey_Right,
	wAccelKey_Left,
	wAccelKey_Back,
	wAccelKey_F1,
	wAccelKey_F2,
	wAccelKey_F3,
	wAccelKey_F4,
	wAccelKey_F5,
	wAccelKey_F6,
	wAccelKey_F7,
	wAccelKey_F8,
	wAccelKey_F9,
	wAccelKey_F10,
	wAccelKey_F11,
	wAccelKey_F12,
	wAccelKey_Numpad_Add,
	wAccelKey_Numpad_Subtract,
	wAccelKey_LineFeed
}
wAccelKey_e;

typedef enum {
	wModKey_None,
	wModKey_Alt,
	wModKey_Shift,
	wModKey_Ctrl
}
wModKey_e;

void wDoAccelHelp( wAccelKey_e key, void * );

/* Creation CallBacks */
typedef void (*wMenuCallBack_p)( void * );
typedef void (*wMenuListCallBack_p)( int, const char *, void * );
typedef void (*wMenuCallBack_p)( void * );
typedef void (*wAccelKeyCallBack_p)( wAccelKey_e, void * );
typedef void (*wMenuTraceCallBack_p)( wMenu_p, const char *, void * );

/* Creation Options */
#define BM_ICON		(1L<<0)

wMenu_p wMenuCreate(		wWin_p, wWinPix_t, wWinPix_t, const char *, const char *,
                                long );
wMenu_p wMenuBarAdd(		wWin_p, const char *, const char * );

wMenuPush_p wMenuPushCreate(	wMenu_p, const char *, const char *, long,
                                wMenuCallBack_p, void * );
wMenuRadio_p wMenuRadioCreate(	wMenu_p, const char *, const char *, long,
                                wMenuCallBack_p, void * );

wMenu_p wMenuMenuCreate(	wMenu_p, const char *, const char * );
wMenu_p wMenuPopupCreate(	wWin_p, const char * );
void wMenuSeparatorCreate(	wMenu_p );
wMenuList_p wMenuListCreate(	wMenu_p, const char *, int, wMenuListCallBack_p );
void wMenuRadioSetActive( wMenuRadio_p );
void wMenuPushEnable(		wMenuPush_p, wBool_t );
void wMenuListAdd(		wMenuList_p, int, const char *, const void * );
void wMenuListDelete(		wMenuList_p, const char * );
const char * wMenuListGet(	wMenuList_p, int, void ** );
void wMenuListClear(		wMenuList_p );

wMenuToggle_p wMenuToggleCreate(	wMenu_p, const char *, const char *, long,
                                        wBool_t, wMenuCallBack_p, void * );
wBool_t wMenuToggleSet(		wMenuToggle_p, wBool_t );
wBool_t wMenuToggleGet(		wMenuToggle_p );
void wMenuToggleEnable(		wMenuToggle_p, wBool_t );

void wMenuPopupShow(		wMenu_p );

void wMenuAddHelp(		wMenu_p );

void wMenuSetTraceCallBack(	wMenu_p, wMenuTraceCallBack_p, void * );
wBool_t wMenuAction(		wMenu_p, const char * );

void wAttachAccelKey( wAccelKey_e, int, wAccelKeyCallBack_p, void * );

/*------------------------------------------------------------------------------
 *
 * File Selection
 */

#define FS_MULTIPLEFILES	1
#define FS_PICTURES         2

struct wFilSel_t;
typedef enum {
	FS_SAVE,
	FS_LOAD,
	FS_UPDATE
}
wFilSelMode_e;
typedef int (*wFilSelCallBack_p)( int files, char ** fileName, void * );
struct wFilSel_t * wFilSelCreate(wWin_p, wFilSelMode_e, int, const char *,
                                 const char *,
                                 wFilSelCallBack_p, void * );
int wFilSelect(			struct wFilSel_t *, const char * );


/*------------------------------------------------------------------------------
 *
 * Color Selection
 */
/* Creation CallBacks */
typedef void (*wColorSelectButtonCallBack_p)( void *, wDrawColor );

wBool_t wColorSelect( const char *, wDrawColor * );
wButton_p wColorSelectButtonCreate( wWin_p, wWinPix_t, wWinPix_t, const char *,
                                    const char *,
                                    long, wWinPix_t, wDrawColor *, wColorSelectButtonCallBack_p, void * );
void wColorSelectButtonSetColor( wButton_p, wDrawColor );
wDrawColor wColorSelectButtonGetColor( wButton_p );

/*------------------------------------------------------------------------------
 *
 * Preferences
 */

void wPrefSetString(const char *, const char *, const char * );
char * wPrefGetString(const char *section, const char *name);
char * wPrefGetStringBasic( const char *section, const char *name );
char * wPrefGetStringExt(const char *section, const char *name);

void wPrefsLoad(char * name);

void wPrefSetInteger(const char *, const char *, long );
wBool_t wPrefGetInteger(const char *section, const char *name, long *result,
                        long defaultValue);
wBool_t wPrefGetIntegerBasic(const char *section, const char *name,
                             long *result, long defaultValue);
wBool_t wPrefGetIntegerExt(const char *section, const char *name, long *result,
                           long defaultValue);

void wPrefSetFloat(		const char *, const char *, double );
wBool_t wPrefGetFloat(const char *section, const char *name, double *result,
                      double defaultValue);
wBool_t wPrefGetFloatBasic(const char *section, const char *name,
                           double *result, double defaultValue);
wBool_t wPrefGetFloatExt(const char *section, const char *name, double *result,
                         double defaultValue);

//const char * wPrefGetSectionItem( const char * sectionName, wIndex_t * index,
//                                  const char ** name );
void wPrefFlush( char * name);
void wPrefReset( void );
void wPrefTokenize(char* line, char** section, char** name, char** value);
void wPrefFormatLine(const char* section, const char* name,
                    const char* value, char* result);

//void CleanupCustom( void );

/*------------------------------------------------------------------------------
 *
 * Statusbar
 */

wStatus_p wStatusCreate(
        wWin_p	parent,
        wWinPix_t	x,
        wWinPix_t	y,
        const char 	* labelStr,
        wWinPix_t	width,
        const char	*message );

wWinPix_t wStatusGetWidth(const char *testString);
wWinPix_t wStatusGetHeight(long flags);

void wStatusSetValue(wStatus_p b, const char * arg);
void wStatusSetWidth(wStatus_p b, wWinPix_t width);

/*------------------------------------------------------------------------------
 *
 * System-Information
 */

char* wGetTempPath(void);
char* wGetOSVersion(void);
char* wGetProfileFilename(void);
char* wGetUserID(void);
const char* wGetUserHomeRootDir(void);
const char *wGetPlatformVersion(void);

/*-------------------------------------------------------------------------------
 * User Preferences
 */

#define PREFSECTION "Preference"
#define LARGEICON   "LargeIcons"
#define DPISET      "ScreenDPI"
#define PRINTSCALE    "PrintScale"
#define PRINTTEXTSCALE  "PrintTextScale"
#endif
