///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __PLUGIN_RENDERER_H
#define __PLUGIN_RENDERER_H

#include <core/Core.h>
#include <core/reference/RefTarget.h>
#include <core/scene/animation/TimeInterval.h>
#include <core/viewport/Viewport.h>

namespace Core {
	
class DataSet;				// defined in DataSet.h
class RenderSettings;		// defined in RenderSettings.h
class FrameBuffer;			// defined in FrameBuffer.h
	
/******************************************************************************
* Abstract base class for scene renderers.
******************************************************************************/
class CORE_DLLEXPORT PluginRenderer : public RefTarget
{
protected:
	/// Constructor.
	PluginRenderer(bool isLoading = false);
	
public:

	/// Returns the general render settings for this renderer. 
	RenderSettings* renderSettings() const { return _renderSettings; }

	///	Prepares the renderer for rendering of the given scene.
	/// Throws an exception on error. Returns false when the operation has been aborted by the user.
	virtual bool startRender(DataSet* dataset) = 0;
	
	/// Renders a single animation frame into the given frame buffer.
	/// Throws an exception on error. Returns false when the operation has been aborted by the user.
	virtual bool renderFrame(TimeTicks time, CameraViewDescription view, FrameBuffer* frameBuffer) = 0;
	 
	///	Finishes the rendering pass. This is called after all animation frames have been rendered
	/// or when the rendering operation has been aborted.	
	virtual void endRender() = 0;
	
	/// Returns the list of installed plugin renderer classes.
	static QList<PluginClassDescriptor*> availableRendererClasses();
	
protected:

	/// Saves the class' contents to the given stream. 
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream. 
	virtual void loadFromStream(ObjectLoadStream& stream);

private:

	RenderSettings* _renderSettings;
	
	friend class RenderSettings;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(PluginRenderer)
};

};

#endif // __PLUGIN_RENDERER_H
