///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/ModifiedObject.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>

#include "AtomsObjectAnalyzerBase.h"

namespace AtomViz {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(AtomsObjectAnalyzerBase, AtomsObjectModifierBase)
DEFINE_PROPERTY_FIELD(AtomsObjectAnalyzerBase, "UpdateOnTimeChange", _autoUpdateOnTimeChange)
DEFINE_FLAGS_REFERENCE_FIELD(AtomsObjectAnalyzerBase, NearestNeighborList, "NearestNeighborList", PROPERTY_FIELD_ALWAYS_DEEP_COPY, _nearestNeighborList)
SET_PROPERTY_FIELD_LABEL(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange, "Auto-update on time change")
SET_PROPERTY_FIELD_LABEL(AtomsObjectAnalyzerBase, _nearestNeighborList, "Nearest neighbor list")

/******************************************************************************
* Constructor.
******************************************************************************/
AtomsObjectAnalyzerBase::AtomsObjectAnalyzerBase(bool isLoading)
	: AtomsObjectModifierBase(isLoading), _autoUpdateOnTimeChange(false)
{
	INIT_PROPERTY_FIELD(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange);
	INIT_PROPERTY_FIELD(AtomsObjectAnalyzerBase, _nearestNeighborList);
	if(!isLoading) {
		setNearestNeighborList(new NearestNeighborList());
	}
}

/******************************************************************************
* Asks the modifier for its validity interval at the given time.
******************************************************************************/
TimeInterval AtomsObjectAnalyzerBase::modifierValidity(TimeTicks time)
{
	return TimeForever;
}

/******************************************************************************
* Builds the nearest neighbor list for the input AtomsObject.
******************************************************************************/
bool AtomsObjectAnalyzerBase::buildNeighborList(bool noProgressIndicator)
{
	if(!_nearestNeighborList)
		throw Exception("The modifier does not have a nearest neighbor list sub-object.");

	// Rebuild list.
	if(!_nearestNeighborList->build(input()))
		_nearestNeighborList->clear();

	return _nearestNeighborList->isValid();
}

/******************************************************************************
* This will recalculate the analysis for the first input object.
* Throws an exception on error.
******************************************************************************/
EvaluationStatus AtomsObjectAnalyzerBase::performAnalysis(TimeTicks time, bool suppressDialogs)
{
	if(modifierApplications().empty())
		return EvaluationStatus(EvaluationStatus::EVALUATION_WARNING, tr("Cannot perform analysis when modifier has not been applied to any input object."));

	return performAnalysis(time, modifierApplications().front(), suppressDialogs);
}

/******************************************************************************
* This will recalculate the analysis for one input object of the modifier.
* Throws an exception on error.
******************************************************************************/
EvaluationStatus AtomsObjectAnalyzerBase::performAnalysis(TimeTicks time, ModifierApplication* modApp, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(modApp);
	// To get the input atoms, evaluate modifier stack up to this modifier.
	ModifiedObject* modObj = modApp->modifiedObject();
	CHECK_OBJECT_POINTER(modObj);
	this->modApp = modApp;

	UndoSuspender noUndo;
	ViewportSuspender noVPUpdates;

	AtomsObject* oldInputAtoms = this->inputAtoms;

	PipelineFlowState flowState;
	try {
		if(!inputAtoms) {
			flowState = modObj->evalObject(time, modApp, false);
			inputAtoms = dynamic_object_cast<AtomsObject>(flowState.result());
			if(!inputAtoms)
				throw Exception(tr("This modifier cannot be evaluated because the input object does not contain any atoms."));
		}
		setAnalysisValidityInterval(TimeForever);

		// Let the sub-class do the actual analysis.
		CHECK_POINTER(inputAtoms);
		_analysisStatus = doAnalysis(time, suppressDialogs);
		CHECK_POINTER(inputAtoms);

		// Save the validity interval of the analysis.
		if(!oldInputAtoms)
			intersectAnalysisValidityInterval(flowState.stateValidity());
		intersectAnalysisValidityInterval(input()->objectValidity(time));
	}
	catch(const Exception& ex) {
		// Transfer exception message to evaluation status.
		QString msg = ex.message();
		for(int i=1; i<ex.messages().size(); i++) {
			msg += "\n";
			msg += ex.messages()[i];
		}
		_analysisStatus = EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, msg);
		inputAtoms = oldInputAtoms;
		setAnalysisValidityInterval(TimeNever);
		modApp->setStatus(_analysisStatus);

		throw;
	}

	// Cleanup
	inputAtoms = oldInputAtoms;

	// Update user interface.
	notifyDependents(REFTARGET_CHANGED);

	modApp->setStatus(_analysisStatus);
	return _analysisStatus;
}

/******************************************************************************
* This modifies the input object.
******************************************************************************/
EvaluationStatus AtomsObjectAnalyzerBase::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if the analysis result is still valid.
	if(analysisValidityInterval().contains(time) == false) {
		// Analysis results are out-dated.
		// We have to recompute them if auto-update is enabled.
		if(autoUpdateOnTimeChangeEnabled()) {
			performAnalysis(time, modifierApplication(), true);
			intersectAnalysisValidityInterval(validityInterval);
		}
	}

	if(_analysisStatus.type() == EvaluationStatus::EVALUATION_ERROR)
		return _analysisStatus;

	if(!analysisValidityInterval().contains(time))
		return EvaluationStatus(EvaluationStatus::EVALUATION_WARNING, tr("No analysis results available."));

	validityInterval.intersect(analysisValidityInterval());

	return applyResult(time, validityInterval);
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void AtomsObjectAnalyzerBase::saveToStream(ObjectSaveStream& stream)
{
	AtomsObjectModifierBase::saveToStream(stream);

	stream.beginChunk(0x66234A);
	stream << _analysisValidityInterval;
	stream << _analysisStatus;
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void AtomsObjectAnalyzerBase::loadFromStream(ObjectLoadStream& stream)
{
	AtomsObjectModifierBase::loadFromStream(stream);

	stream.expectChunk(0x66234A);
	stream >> _analysisValidityInterval;
	stream >> _analysisStatus;
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr AtomsObjectAnalyzerBase::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	AtomsObjectAnalyzerBase::SmartPtr clone = static_object_cast<AtomsObjectAnalyzerBase>(AtomsObjectModifierBase::clone(deepCopy, cloneHelper));

	clone->_analysisValidityInterval = this->_analysisValidityInterval;
	clone->_analysisStatus = this->_analysisStatus;

	return clone;
}

}; // End of namespace AtomViz
