/* $Id: trigger.c,v 1.12 2005/12/28 23:54:29 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Implements the Python interface to a Postgres trigger fire
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <commands/trigger.h>
#include <nodes/params.h>
#include <nodes/execnodes.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <compile.h>
#include <eval.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/function.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/trigger.h>

static PyObj
pull_arguments(int16 tgnargs, char **tgargs)
{
	PyObj args = PyTuple_New(tgnargs);

	if (tgnargs > 0)
	{
		int i;
		for (i = 0; i < tgnargs; ++i)
		{
			PyObj ob = PyString_FromString(tgargs[i]);
			if (ob == NULL) goto fail;
			PyTuple_SET_ITEM(args, i, ob);
		}
	}

	return(args);
fail:
	Py_DECREF(args);
	return(NULL);
}

static PyMemberDef PyPgTriggerPull_Members[] = {
	{"oid", T_OBJECT, offsetof(struct PyPgTriggerPull, tp_oid), RO,
	PyDoc_STR("the oid of the trigger")},
	{"old", T_OBJECT, offsetof(struct PyPgTriggerPull, tp_old), RO,
	PyDoc_STR("the old tuple in the pull")},
	{"new", T_OBJECT, offsetof(struct PyPgTriggerPull, tp_new), RO,
	PyDoc_STR("the new tuple in the pull")},
	{"event", T_INT, offsetof(struct PyPgTriggerPull, tp_event), RO,
	PyDoc_STR("event code of the trigger")},
	{NULL}
};

static int
pull_clear(PyObj self)
{
	PyObj ob;

	ob = PyPgTriggerPull_FetchOid(self);
	if (ob != NULL)
	{
		PyPgTriggerPull_FixOid(self, NULL);
		DECREF(ob);
	}

	ob = PyPgTriggerPull_FetchOld(self);
	if (ob != NULL)
	{
		PyPgTriggerPull_FixOld(self, NULL);
		DECREF(ob);
	}

	ob = PyPgTriggerPull_FetchNew(self);
	if (ob != NULL)
	{
		PyPgTriggerPull_FixNew(self, NULL);
		DECREF(ob);
	}

	return(0);
}

static PyObj
pull_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob;

	rob = PyPgCall_Type.tp_call(self, args, kw);
	if (rob == NULL || rob == Py_None || rob == Py_True || rob == Py_False)
		return(rob);

	if (PyPgHeapTuple_Check(rob))
	{
		PyObj ot, nt;
		ot = PyPgTriggerPull_FetchOld(self);
		nt = PyPgTriggerPull_FetchNew(self);
		Assert(ot != Py_None || nt != Py_None);

		if ((nt != Py_None && rob == nt) || (ot))
		{
			DECREF(rob);
			rob = Py_True;
			INCREF(rob);
		}
	}
	else
	{
		PyObj nr;
		nr = Py_Call(PyPgCall_FetchOutput(self), rob);
		DECREF(rob);
		rob = nr;
	}

	return(rob);
}

PyDoc_STRVAR(PyPgTriggerPull_Doc,
"Python interface to a PostgreSQL trigger pull");

PyTypeObject PyPgTriggerPull_Type = {
	PyObject_HEAD_INIT(NULL)
	0,											/* ob_size */
	"Postgres.TriggerPull",				/* tp_name */
	sizeof(struct PyPgTriggerPull),	/* tp_basicsize */
	0,											/* tp_itemsize */
	NULL,										/* tp_dealloc */
	NULL,										/* tp_print */
	NULL,										/* tp_getattr */
	NULL,										/* tp_setattr */
	NULL,										/* tp_compare */
	NULL,										/* tp_repr */
	NULL,										/* tp_as_number */
	NULL,										/* tp_as_sequence */
	NULL,										/* tp_as_mapping */
	NULL,										/* tp_hash */
	pull_call,								/* tp_call */
	NULL,										/* tp_str */
	NULL,										/* tp_getattro */
	NULL,										/* tp_setattro */
	NULL,										/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,					/* tp_flags */
	PyPgTriggerPull_Doc,					/* tp_doc */
	NULL,										/* tp_traverse */
	pull_clear,								/* tp_clear */
	NULL,										/* tp_richcompare */
	0,											/* tp_weaklistoffset */
	NULL,										/* tp_iter */
	NULL,										/* tp_iternext */
	NULL,										/* tp_methods */
	PyPgTriggerPull_Members,			/* tp_members */
	NULL,										/* tp_getset */
	&PyPgProceduralCall_Type,			/* tp_base */
	NULL,										/* tp_dict */
	NULL,										/* tp_descr_get */
	NULL,										/* tp_descr_set */
	0,											/* tp_dictoffset */
	NULL,										/* tp_init */
	NULL,										/* tp_alloc */
	NULL,										/* tp_new */
};

PyObj
PyPgTriggerPull_Initialize(PyObj self, FunctionCallInfo fcinfo)
{
	PyObj func, args, out, old, new;
	TriggerData *tgd;
	Trigger *tg;

	Assert(fcinfo != NULL && CALLED_AS_TRIGGER(fcinfo));
	if (self == NULL) return(NULL);

	tgd = (TriggerData *) fcinfo->context;
	tg = tgd->tg_trigger;

	func = PyPgFunction_FromOid(fcinfo->flinfo->fn_oid);
	if (func == NULL) goto fail;
	PyPgCall_FixFunction(self, func);

	args = pull_arguments(tg->tgnargs, tg->tgargs);
	if (args == NULL) goto fail;
	PyPgCall_FixArguments(self, args);

	out = PyPgTupleDesc_New(tgd->tg_relation->rd_att);
	if (out == NULL) goto fail;
	PyPgCall_FixOutput(self, out);

	if (tgd->tg_trigtuple != NULL)
	{
		old = PyPgHeapTuple_New(out, tgd->tg_trigtuple);
		if (old == NULL) goto fail;
		PyPgHeapTuple_FixTableOid(old, tgd->tg_relation->rd_id);
	}
	else
	{
		old = Py_None;
		INCREF(old);
	}
	PyPgTriggerPull_FixOld(self, old);

	if (tgd->tg_newtuple != NULL)
	{
		new = PyPgHeapTuple_New(out, tgd->tg_newtuple);
		if (new == NULL) goto fail;
		PyPgHeapTuple_FixTableOid(new, tgd->tg_relation->rd_id);
	}
	else
	{
		new = Py_None;
		INCREF(new);
	}
	PyPgTriggerPull_FixNew(self, new);

	PyPgCall_FixReturned(self, Py_None);

	return(self);
fail:
	self->ob_type->tp_free(self);
	return(NULL);
}
