/* $Id: function.c,v 1.7 2005/06/29 23:40:13 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/func.c,v 1.6 2004/12/15 13:05:14 flaw
 * if/src/func.c,v 1.1 2004/09/28 15:40:49 flaw
 *//*
 * Postgres function type implementation
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <access/heapam.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <mb/pg_wchar.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <marshal.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/utils.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/conv.h>
#include <pypg/error.h>
#include <pypg/function.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/function.h>
#include <pypg/cis.h>

static PyObj
kwd_from_array(ArrayType *at)
{
	Datum *argnames = NULL;
	int elms = 0, i;
	PyObj rob = NULL;

	deconstruct_array(at, ARR_ELEMTYPE(at), -1, false, 'i', &argnames, &elms);
	Assert(elms < FUNC_MAX_ARGS);

	rob = PyDict_New();
	if (rob == NULL) return(NULL);

	for (i = 0; i < elms; ++i)
	{
		char *val;

		val = DatumGetPointer(DirectFunctionCall1(textout, argnames[i]));
		if (val == NULL) goto fail;
		if (strlen(val) > 0)
		{
			int err;
			err = PyDict_SetItem(rob, PyInt_FromLong(i), PyString_FromString(val));
			pfree(val);
			if (err < 0) goto fail;
		}
	}
	pfree(argnames);

	return(rob);
fail:
	pfree(argnames);
	DECREF(rob);
	return(NULL);
}

static PyMethodDef PyPgFunction_Methods[] = {
	/*{"name", FunctionRef, METH_NOARGS|METH_O|METH_VARARGS, "docstring"},*/
	{NULL}
};

static PyMemberDef PyPgFunction_Members[] = {
	{"code", T_OBJECT, offsetof(struct PyPgFunction, fn_code), RO,
	"Postgres.Function's code object"},
	{"returns", T_OBJECT, offsetof(struct PyPgFunction, fn_prorettype), RO,
	"Postgres.Function's Postgres.Type return type"},
	{"argtypes", T_OBJECT, offsetof(struct PyPgFunction, fn_proargtypes), RO,
	"A tuple of Postgres.Type's specifying the function's argument types"},
	{"argnames", T_OBJECT, offsetof(struct PyPgFunction, fn_proargnames), RO,
	"A dictionary of argument numbers set to their given name"},
	{NULL}
};

static PyObj
func_repr(PyObj self)
{
	Form_pg_proc ps = PyPgFunction_FetchProcStruct(self);
	PyObj rep;

	rep = PyString_FromFormat("<Postgres.Function %s()>",
													NameStr(ps->proname));
	return(rep);
}

static int
func_traverse(PyObj self, visitproc visit, void *arg)
{
	int err = 0;

	err = self->ob_type->tp_base->tp_traverse(self, visit, arg);
	if (err) return(err);
	err = visit(PyPgFunction_FetchReturnType(self), arg);
	if (err) return(err);
	err = visit(PyPgFunction_FetchArgTypes(self), arg);
	if (err) return(err);

	return(0);
}

static int
func_clear(PyObj self)
{
	PyObj ob;

	ob = PyPgFunction_FetchReturnType(self);
	if (ob != NULL)
	{
		PyPgFunction_FixReturnType(self, NULL);
		DECREF(ob);
	}

	ob = PyPgFunction_FetchArgNames(self);
	if (ob != NULL)
	{
		PyPgFunction_FixArgNames(self, NULL);
		DECREF(ob);
	}

	ob = PyPgFunction_FetchArgTypes(self);
	if (ob != NULL)
	{
		PyPgFunction_FixArgTypes(self, NULL);
		DECREF(ob);
	}

	ob = PyPgFunction_FetchCode(self);
	if (ob != NULL)
	{
		PyPgFunction_FixCode(self, NULL);
		DECREF(ob);
	}

	self->ob_type->tp_base->tp_clear(self);
	return(0);
}

static PyObj
func_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj call, xtd = NULL, rob = NULL;
	PyObj argtypes, pgargs;
	int nargs;

	if (PyCFunctionErr_NoKeywordsAllowed(kw) < 0)
		return(NULL);

	nargs = PyObject_Length(args);
	argtypes = PyPgFunction_FetchArgTypes(self);
	if (argtypes == Py_None)
	{
		PyErr_SetString(PyExc_NotImplementedError,
				"variable arguments not supported");
		return(NULL);
	}
	else
	{
		int argtlen;
		argtlen = PyObject_Length(argtypes);
		if (nargs != argtlen)
		{
			PyErr_Format(PyExc_TypeError,
					"Postgres.Function requires %d arguments, given %d",
					nargs, argtlen);
			return(NULL);
		}
	}

	pgargs = PyList_New(nargs);
	if (pgargs == NULL)
		return(NULL);

	if (nargs > 0)
	{
		int i;
		for (i = 0; i < nargs; ++i)
		{
			PyObj narg;
			narg = PyPgObject_FromPyPgTypeAndPyObject(
						PyTuple_GET_ITEM(argtypes, i),
						PyTuple_GET_ITEM(args, i)
					);
			if (narg == NULL)
			{
				DECREF(pgargs);
				return(NULL);
			}
			PyList_SetItem(pgargs, i, narg);
		}
	}

	if (PyPgFunction_IsPython(self))
		/* XXX: call = PyPgPythonCall_New(self, pgargs, xtd); */
		call = PyPgCall_New(self, pgargs, xtd);
	else
		call = PyPgFunctionCall_New(self, pgargs, xtd);
	DECREF(pgargs);
	if (call != NULL)
	{
		/* return the call if it explicitly requested, as well as: */
		if (PyPgFunction_IsSRF(self) || PyPgCall_FetchExpected(call) != Py_None)
			rob = call;
		else
		{
			rob = PyObject_CallObject(call, NULL);
			DECREF(call);
		}
	}

	return(rob);
}

static PyObj
func_new_from_oid(PyTypeObject *subtype, Oid fn_oid)
{
	HeapTuple ht = NULL;
	int i;
	PyObj lo;
	HeapTupleHeader hth;
	Form_pg_proc ps;
	
	PGFunction fn_addr;

	PyObj pg_proc_type, prorettype, fat, pan, rob = NULL;

	pg_proc_type = pg_proc_PyPgType();
	if (pg_proc_type == NULL) return(NULL);

	lo = PyLong_FromUnsignedLong(fn_oid);
	if (lo == NULL) return(NULL);

	if (PyMapping_HasKey(PyPgFunction_Cache, lo))
	{
		rob = PyObject_GetItem(PyPgFunction_Cache, lo);
		Py_DECREF(lo);
		return(rob);
	}

	PG_TRY();
	{
		ht = SearchSysCache(PROCOID, fn_oid, 0, 0, 0);
		if (ht == NULL)
		{
			ereport(ERROR,(
				errmsg("failed to fetch procedure")
			));
		}
		hth = HeapTupleHeader_FromHeapTuple(ht);
		if (hth == NULL)
		{
			ereport(ERROR,(
				errmsg("failed to duplicate procedure into Python context")
			));
		}
	}
	PYPG_CATCH_END(if (ht) ReleaseSysCache(ht); goto fail);
	ReleaseSysCache(ht);

	ps = (Form_pg_proc) HeapTupleHeaderStructure(hth);

	rob = subtype->tp_alloc(subtype, 0);
	if (rob == NULL) goto fail;
	PyPgObject_FixType(rob, pg_proc_type);
	PyPgObject_FixDatum(rob, PointerGetDatum(hth));

	if (ps->prolang == PyPgCI.PL.Oid)
	{
		fn_addr = PyPgCI.PL.Handler;
		if (PyPgCI.PL.Fixer(rob) < 0) goto fail;
	}
	else
	{
		FmgrInfo flinfo;

		PG_TRY();
		{
			fmgr_info_cxt(fn_oid, &flinfo, PythonMemoryContext);
		}
		PYPG_CATCH_END(goto fail);

		fn_addr = flinfo.fn_addr;
		Py_INCREF(Py_None);
		PyPgFunction_FixCode(rob, Py_None);
	}
	PyPgFunction_FixPGFunction(rob, fn_addr);

	prorettype = PyPgType_FromOid(ps->prorettype);
	if (prorettype == NULL) goto fail;
	PyPgFunction_FixReturnType(rob, prorettype);

	fat = PyTuple_New(ps->pronargs);
	if (fat == NULL) goto fail;
	PyPgFunction_FixArgTypes(rob, fat);

	for (i = 0; i < ps->pronargs; ++i)
	{
		Oid to;
#if PGV_MM == 80
		to = ps->proargtypes[i];
#else
		to = ps->proargtypes.values[i];
#endif
		PyObj argtype = PyPgType_FromOid(to);
		if (argtype == NULL) goto fail;
		PyTuple_SET_ITEM(fat, i, argtype);
	}

	if (ps->pronargs > 0)
	{
		bool isnull = true;
		ArrayType *at;

		ht->t_len = HeapTupleHeaderGetDatumLength(hth);
		ht->t_datamcxt = PythonMemoryContext;
		ht->t_data = hth;

		at = (ArrayType *) fastgetattr(ht, Anum_pg_proc_proargnames,
									PyPgType_FetchTupleDesc(pg_proc_type), &isnull);

		pan = isnull ? PyDict_New() : kwd_from_array(at);
	}
	else
		pan = PyDict_New();
	if (pan == NULL) goto fail;
	PyPgFunction_FixArgNames(rob, pan);

	PyObject_SetItem(PyPgFunction_Cache, lo, rob);
	Py_DECREF(lo);
	return(rob);

fail:
	Py_DECREF(lo);
	Py_XDECREF(rob);

	return(NULL);
}

static PyObj
func_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	Oid fn_oid = InvalidOid;
	PyObj flo;
	PyObj rob;

	flo = PySequence_GetItem(args, 0);
	if (flo == NULL)
	{
		PyErr_SetString(PyExc_TypeError,
			"Postgres.Function requires at least one argument");
		return(NULL);
	}

	fn_oid = Oid_FromPyObject(flo);
	if (fn_oid == InvalidOid)
	{
		if (!PyErr_Occurred())
		{
			PyErr_Format(PyExc_TypeError,
				"unable to fetch Oid from '%s' object",
				flo->ob_type->tp_name);
		}

		return(NULL);
	}

	rob = func_new_from_oid(subtype, fn_oid);
	return(rob);
}

static const char doc[] =
"Postgres function interface type";

PyTypeObject PyPgFunction_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Function",				/* tp_name */
	sizeof(struct PyPgFunction),	/* tp_basicsize */
	0,										/* tp_itemsize */
	NULL,									/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	func_repr,							/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	func_call,							/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) doc,						/* tp_doc */
	func_traverse,						/* tp_traverse */
	func_clear,							/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgFunction_Methods,			/* tp_methods */
	PyPgFunction_Members,			/* tp_members */
	NULL,									/* tp_getset */
	&PyPgObject_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	func_new,							/* tp_new */
};

PyObj
PyPgFunction_FromOid(Oid po)
{
	return(func_new_from_oid(&PyPgFunction_Type, po));
}

PyObj
pg_proc_PyPgType(void)
{
	static PyObj rob = NULL;
	if (rob == NULL)
		rob = PyPgType_FromOid(PG_PROC_RELTYPE_OID);

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
