/************************************************************************
 *                                                                      *
 * ODBC Simple Wrapper V2.0                                             *
 * Copyright (C) 2009 Claudio Rocchini                                  *
 * Istituto Geografico Militare Italiano                                *
 * web:   www.igmi.org                                                  *
 * email: ad2prod@geomil.esercito.difesa.it                             *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the  Free Software Foundation, either  version 3 of the  License, or *
 * any later version.                                                   *
 *                                                                      *
 *   This program is distributed in the hope that it will be useful,    *
 *   but WITHOUT ANY WARRANTY; without  even the implied warranty of    *
 *   MERCHANTABILITY or  FITNESS  FOR A PARTICULAR PURPOSE.  See the    *
 *   GNU General Public License for more details.                       *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program.  If not, see <http://www.gnu.org/licenses/> *
 *                                                                      *
 ************************************************************************/


#include <stdio.h>
#include <assert.h>

#include "rodbc2.h"

namespace sql2
{

static void get_message_status( SQLSMALLINT handle_type, SQLHANDLE handle, char * status )
{
	const SQLSMALLINT msize = 1024;
	SQLSMALLINT mlength = 0;
	SQLCHAR message[msize];
	SQLCHAR state[6];
	SQLINTEGER native_error = 0;

	state[5] = 0;
	state[0] = 0;
	::SQLGetDiagRec(handle_type,handle,1,state,&native_error,message,msize,&mlength);
	message[mlength] = 0;
	sprintf(status,"%s (%ld) %s",state,native_error,message);
}


void Environment::get_message( char * message ) const
{
	get_message_status(SQL_HANDLE_ENV,handle,message);
}


bool Environment::get_drivers( std::vector<Driver> & drivers ) const
{
	const int BSIZE = 4096;
	unsigned char desc[BSIZE];
	SQLSMALLINT ldesc;
	unsigned char attr[BSIZE];
	SQLSMALLINT lattr;
	bool      first = true;
	SQLRETURN r;

	drivers.clear();
	for(;;)
	{
		r = ::SQLDrivers( handle,first ? SQL_FETCH_FIRST : SQL_FETCH_NEXT,
				desc,BSIZE,&ldesc,
				attr,BSIZE,&lattr);
		if(r==SQL_ERROR  ) return false;
		if(r!=SQL_SUCCESS) break;
		
		drivers.push_back( Driver() );
		drivers.back().name = std::string( (char *)desc );
		char * p = (char *)attr;
		while(*p)
		{
			drivers.back().params.push_back( std::string(p) );
			p = p + strlen(p)+1;
		}
		first = false;
	}
	return true;
}


bool Environment::get_data_source( std::vector< std::string > & sources ) const
{
	SQLRETURN r;
	bool      first = true;

	assert(handle);
	sources.clear();
	for(;;)
	{
		const int BSIZE = 2032;
		char server_name[BSIZE]; SQLSMALLINT server_name_lenght;
		char server_desc[BSIZE]; SQLSMALLINT server_desc_lenght;

		r = ::SQLDataSources(
			handle,
			SQLSMALLINT( first ? SQL_FETCH_FIRST : SQL_FETCH_NEXT ),
			(SQLCHAR *)server_name,BSIZE,&server_name_lenght,
			(SQLCHAR *)server_desc,BSIZE,&server_desc_lenght
		);
		first = false;
		if(r==SQL_ERROR  ) return false;
		if(r!=SQL_SUCCESS) break;
		sources.push_back( std::string(server_name) + "-" + std::string(server_desc) );
	}
	return true;
}


inline bool Connection::connect(const char * conn, int mode )
{
	const SQLSMALLINT bsize = 256;
	char buff[bsize];
	SQLSMALLINT blength;

	SQLRETURN r;

	r = ::SQLDriverConnect(handle,0,(unsigned char *)conn,SQL_NTS,(unsigned char *)buff,bsize,&blength,SQL_DRIVER_NOPROMPT);

	if(r==SQL_ERROR || r==SQL_INVALID_HANDLE) return false;

	connected = true;
	::SQLSetConnectAttr(handle,SQL_ATTR_ACCESS_MODE,SQLPOINTER(mode),0);
	return true;
}


void Connection::get_message( char * message ) const
{
	get_message_status(SQL_HANDLE_DBC,handle,message);
}


	// What driver is select?
	// The first string matching whit type!


bool Connection::connect_file(const char * type, const char * filename, int mode, const char * passw )
{
	std::vector< Driver > drivers;
	if( !envp->get_drivers( drivers ) )
		return false;
	
	std::string driver;
	std::vector< Driver >::iterator i;
	for(i=drivers.begin();i!=drivers.end();++i)
		if( strstr( i->name.c_str(),type) )
		{
			driver = i->name;
			break;
		}
	if( driver.empty() ) return false;

	std::string conn = "DRIVER=" + driver + ";DBQ=" + filename + ";";
	if(passw)
		conn = conn + "PWD=" + passw + ";"; 

	return connect(conn.c_str(),mode);
}

	// Statement implementation

void Statement::get_message( char * message ) const
{
	get_message_status(SQL_HANDLE_STMT,handle,message);
}


bool Statement::tables( std::vector< TableDesc > & tables )
{
	const int NC = 5;
	const int BSIZE = 4096;
	char buf[NC][BSIZE]; SQLINTEGER lbuf[NC];
	int i;
	SQLRETURN r;

	for(i=0;i<NC;++i)
	{
		::SQLBindCol(handle,i+1,SQL_C_CHAR,SQLPOINTER(buf[i]),BSIZE,lbuf+i);
	}

	if( ::SQLTables(handle,0,0,0,0,0,0,0,0)==SQL_ERROR)
		return false;

	for(i=0;i<NC;++i)
		buf[i][0] = 0;

	tables.clear();
	for(;;)
	{
		r = ::SQLFetch(handle);
		if(r==SQL_ERROR) return false;
		if(r==SQL_NO_DATA) break;

		tables.push_back( TableDesc() );
		tables.back().catalog = buf[0];
		tables.back().schema  = buf[1];
		tables.back().name = buf[2];
		tables.back().type = buf[3];
		tables.back().desc = buf[4];
		for(i=0;i<NC;++i)
			buf[i][0] = 0;
	}
	::SQLCloseCursor(handle);
	::SQLFreeStmt(handle,SQL_UNBIND);
	return true;
}


bool Statement::columns( const char * table, std::vector< ColumnDesc > & cols )
{
	const int NC = 13;
	const int BSIZE = 4096;
	char buf[NC][BSIZE]; SQLINTEGER lbuf[NC];
	long ibuf[NC];
	int i;
	SQLRETURN r;

	for(i=0;i<NC;++i)
	{
		if(i==4)
		::SQLBindCol(handle,i+1,SQL_C_SLONG,&ibuf[i],sizeof(long),0);
		else
		::SQLBindCol(handle,i+1,SQL_C_CHAR,SQLPOINTER(buf[i]),BSIZE,lbuf+i);
	}

	if( ::SQLColumns(handle,0,0, 0,0, (unsigned char *)table,strlen(table), 0,0)==SQL_ERROR)
		return false;

	for(i=0;i<NC;++i)
		buf[i][0] = 0;

	cols.clear();
	for(;;)
	{
		r = ::SQLFetch(handle);
		if(r==SQL_ERROR) return false;
		if(r==SQL_NO_DATA) break;

		cols.push_back( ColumnDesc() );
		cols.back().catalog = buf[0];
		cols.back().schema  = buf[1];
		cols.back().table   = buf[2];
		cols.back().name    = buf[3];
		cols.back().type    = ibuf[4];
		cols.back().type_name     = buf[5];
		cols.back().size          = buf[6];
		cols.back().buffer_size   = buf[7];
		cols.back().digit         = buf[8];
		cols.back().radix         = buf[9];
		cols.back().nullable      = buf[10];
		cols.back().remark        = buf[11];
		cols.back().default_value = buf[12];
		for(i=0;i<NC;++i)
			buf[i][0] = 0;
	}
	::SQLCloseCursor(handle);
	::SQLFreeStmt(handle,SQL_UNBIND);

	return true;
}


}	// End namespace sql2

