/*
 * Copyright (c) 2021, Arne Elster <arne@elster.li>
 *
 * SPDX-License-Identifier: BSD-2-Clause
 */

#pragma once

#include <AK/Array.h>
#include <AK/HashMap.h>
#include <AK/Math.h>

namespace Audio::MP3::Tables {

// ISO/IEC 11172-3 (2.4.2.3)
Array<int, 4> LayerNumberLookup { -1, 3, 2, 1 };

// ISO/IEC 11172-3 (2.4.2.3)
Array<Array<int, 16>, 3> BitratesPerLayerLookup { {
    { 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, -1 }, // Layer I
    { 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, -1 },    // Layer II
    { 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, -1 }      // Layer III
} };

// ISO/IEC 11172-3 (2.4.2.3)
Array<int, 4> SampleratesLookup { { 44100, 48000, 32000, -1 } };

// ISO/IEC 11172-3 (2.4.2.7)
Array<int, 16> ScalefacCompressSlen1 { { 0, 0, 0, 0, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4 } };
Array<int, 16> ScalefacCompressSlen2 { { 0, 1, 2, 3, 0, 1, 2, 3, 1, 2, 3, 1, 2, 3, 2, 3 } };

// ISO/IEC 11172-3 (Table B.6)
Array<int, 22> Pretab { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 3, 3, 3, 2, 0 } };

// ISO/IEC 11172-3 (Table B.9)
Array<float, 8> AliasReductionCoefficients {
    -0.6,
    -0.535,
    -0.33,
    -0.185,
    -0.095,
    -0.041,
    -0.0142,
    -0.0032
};

// This is using the cs[i] formula taken from ISO/IEC 11172-3 (below Table B.9)
Array<float, 8> AliasReductionCs { {
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[0], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[1], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[2], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[3], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[4], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[5], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[6], 2.0)),
    1.0f / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[7], 2.0)),
} };

// This is using the ca[i] formula taken from ISO/IEC 11172-3 (below Table B.9)
Array<float, 8> AliasReductionCa { {
    AliasReductionCoefficients[0] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[0], 2.0)),
    AliasReductionCoefficients[1] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[1], 2.0)),
    AliasReductionCoefficients[2] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[2], 2.0)),
    AliasReductionCoefficients[3] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[3], 2.0)),
    AliasReductionCoefficients[4] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[4], 2.0)),
    AliasReductionCoefficients[5] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[5], 2.0)),
    AliasReductionCoefficients[6] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[6], 2.0)),
    AliasReductionCoefficients[7] / AK::sqrt<float>(1 + AK::pow<float>(AliasReductionCoefficients[7], 2.0)),
} };

struct ScaleFactorBand {
    size_t width;
    size_t start;
    size_t end;
};

template<auto sizes, size_t offset = 0>
constexpr auto MakeShortScaleFactorBandArray()
{
    constexpr size_t N = sizes.size();
    Array<ScaleFactorBand, 3 * N> result {};
    size_t start = offset;

    for (size_t i = 0; i < N; i++) {
        result[3 * i + 0] = { sizes[i], start, start + sizes[i] - 1 };
        start += sizes[i];
        result[3 * i + 1] = { sizes[i], start, start + sizes[i] - 1 };
        start += sizes[i];
        result[3 * i + 2] = { sizes[i], start, start + sizes[i] - 1 };
        start += sizes[i];
    }

    return result;
}

template<auto sizes>
constexpr auto MakeLongScaleFactorBandArray()
{
    constexpr size_t N = sizes.size();
    Array<ScaleFactorBand, N> result {};
    size_t start = 0;

    for (size_t i = 0; i < N; i++) {
        result[i] = { sizes[i], start, start + sizes[i] - 1 };
        start += sizes[i];
    }

    return result;
}

template<auto sizes_long, auto sizes_short>
constexpr auto MakeMixedScaleFactorBandArray()
{
    constexpr size_t N = sizes_long.size() + sizes_short.size() * 3 + 1;
    Array<ScaleFactorBand, N> result {};

    constexpr auto long_bands = MakeLongScaleFactorBandArray<sizes_long>();
    constexpr auto short_bands = MakeShortScaleFactorBandArray<sizes_short, long_bands.last().end + 1>();

    for (size_t i = 0; i < long_bands.size(); i++) {
        result[i] = long_bands[i];
    }

    for (size_t i = 0; i < short_bands.size(); i++) {
        result[i + long_bands.size()] = short_bands[i];
    }

    for (size_t i = long_bands.size() + short_bands.size(); i < N; i++) {
        result[i] = { 0, 576, 576 };
    }

    return result;
}

// ISO/IEC 11172-3 (Table B.8)
constexpr auto ScaleFactorBandShort32000 = MakeShortScaleFactorBandArray<Array<size_t, 13> { 4, 4, 4, 4, 6, 8, 12, 16, 20, 26, 34, 42, 12 }>();
constexpr auto ScaleFactorBandShort44100 = MakeShortScaleFactorBandArray<Array<size_t, 13> { 4, 4, 4, 4, 6, 8, 10, 12, 14, 18, 22, 30, 56 }>();
constexpr auto ScaleFactorBandShort48000 = MakeShortScaleFactorBandArray<Array<size_t, 13> { 4, 4, 4, 4, 6, 6, 10, 12, 14, 16, 20, 26, 66 }>();

constexpr auto ScaleFactorBandMixed32000 = MakeMixedScaleFactorBandArray<Array<size_t, 8> { 4, 4, 4, 4, 4, 4, 6, 6 }, Array<size_t, 10> { 4, 6, 8, 12, 16, 20, 26, 34, 42, 12 }>();
constexpr auto ScaleFactorBandMixed44100 = MakeMixedScaleFactorBandArray<Array<size_t, 8> { 4, 4, 4, 4, 4, 4, 6, 6 }, Array<size_t, 10> { 4, 6, 8, 10, 12, 14, 18, 22, 30, 56 }>();
constexpr auto ScaleFactorBandMixed48000 = MakeMixedScaleFactorBandArray<Array<size_t, 8> { 4, 4, 4, 4, 4, 4, 6, 6 }, Array<size_t, 10> { 4, 6, 6, 10, 12, 14, 16, 20, 26, 66 }>();

constexpr auto ScaleFactorBandLong32000 = MakeLongScaleFactorBandArray<Array<size_t, 23> { 4, 4, 4, 4, 4, 4, 6, 6, 8, 10, 12, 16, 20, 24, 30, 38, 46, 56, 68, 84, 102, 26, 0 }>();
constexpr auto ScaleFactorBandLong44100 = MakeLongScaleFactorBandArray<Array<size_t, 23> { 4, 4, 4, 4, 4, 4, 6, 6, 8, 8, 10, 12, 16, 20, 24, 28, 34, 42, 50, 54, 76, 158, 0 }>();
constexpr auto ScaleFactorBandLong48000 = MakeLongScaleFactorBandArray<Array<size_t, 23> { 4, 4, 4, 4, 4, 4, 6, 6, 6, 8, 10, 12, 16, 18, 22, 28, 34, 40, 46, 54, 54, 192, 0 }>();

// ISO/IEC 11172-3 (2.4.3.4.10.3 a)
Array<float, 36> WindowBlockTypeNormal { {
    AK::sin<float>(AK::Pi<float> / 36 * (0 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (1 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (2 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (3 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (4 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (5 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (7 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (8 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (9 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (10 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (11 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (12 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (13 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (14 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (15 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (16 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (17 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (19 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (20 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (21 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (22 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (23 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (24 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (25 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (26 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (27 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (28 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (29 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (30 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (31 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (32 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (33 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (34 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (35 + 0.5f)),
} };

// ISO/IEC 11172-3 (2.4.3.4.10.3 b)
AK::Array<float, 36> WindowBlockTypeStart { {
    AK::sin<float>(AK::Pi<float> / 36 * (0 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (1 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (2 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (3 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (4 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (5 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (7 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (8 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (9 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (10 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (11 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (12 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (13 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (14 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (15 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (16 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (17 + 0.5f)),
    1,
    1,
    1,
    1,
    1,
    1,
    AK::sin<float>(AK::Pi<float> / 12 * (24 - 18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (25 - 18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (26 - 18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (27 - 18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (28 - 18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (29 - 18 + 0.5f)),
    0,
    0,
    0,
    0,
    0,
    0,
} };

// ISO/IEC 11172-3 (2.4.3.4.10.3 d)
AK::Array<float, 36> WindowBlockTypeShort { {
    AK::sin<float>(AK::Pi<float> / 12 * (0 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (1 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (2 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (3 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (4 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (5 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (7 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (8 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (9 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (10 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (11 + 0.5f)),

    AK::sin<float>(AK::Pi<float> / 12 * (0 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (1 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (2 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (3 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (4 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (5 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (7 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (8 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (9 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (10 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (11 + 0.5f)),

    AK::sin<float>(AK::Pi<float> / 12 * (0 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (1 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (2 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (3 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (4 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (5 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (7 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (8 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (9 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (10 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (11 + 0.5f)),
} };

// ISO/IEC 11172-3 (2.4.3.4.10.3 c)
AK::Array<float, 36> WindowBlockTypeEnd { {
    0,
    0,
    0,
    0,
    0,
    0,
    AK::sin<float>(AK::Pi<float> / 12 * (6 - 6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (7 - 6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (8 - 6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (9 - 6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (10 - 6 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 12 * (11 - 6 + 0.5f)),
    1,
    1,
    1,
    1,
    1,
    1,
    AK::sin<float>(AK::Pi<float> / 36 * (18 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (19 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (20 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (21 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (22 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (23 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (24 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (25 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (26 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (27 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (28 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (29 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (30 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (31 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (32 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (33 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (34 + 0.5f)),
    AK::sin<float>(AK::Pi<float> / 36 * (35 + 0.5f)),
} };

// ISO/IEC 11172-3 (Table B.3)
AK::Array<float, 512> WindowSynthesis {
    0.000000000, -0.000015259, -0.000015259, -0.000015259, -0.000015259, -0.000015259, -0.000015259, -0.000030518,
    -0.000030518, -0.000030518, -0.000030518, -0.000045776, -0.000045776, -0.000061035, -0.000061035, -0.000076294,
    -0.000076294, -0.000091553, -0.000106812, -0.000106812, -0.000122070, -0.000137329, -0.000152588, -0.000167847,
    -0.000198364, -0.000213623, -0.000244141, -0.000259399, -0.000289917, -0.000320435, -0.000366211, -0.000396729,
    -0.000442505, -0.000473022, -0.000534058, -0.000579834, -0.000625610, -0.000686646, -0.000747681, -0.000808716,
    -0.000885010, -0.000961304, -0.001037598, -0.001113892, -0.001205444, -0.001296997, -0.001388550, -0.001480103,
    -0.001586914, -0.001693726, -0.001785278, -0.001907349, -0.002014160, -0.002120972, -0.002243042, -0.002349854,
    -0.002456665, -0.002578735, -0.002685547, -0.002792358, -0.002899170, -0.002990723, -0.003082275, -0.003173828,
    0.003250122, 0.003326416, 0.003387451, 0.003433228, 0.003463745, 0.003479004, 0.003479004, 0.003463745,
    0.003417969, 0.003372192, 0.003280640, 0.003173828, 0.003051758, 0.002883911, 0.002700806, 0.002487183,
    0.002227783, 0.001937866, 0.001617432, 0.001266479, 0.000869751, 0.000442505, -0.000030518, -0.000549316,
    -0.001098633, -0.001693726, -0.002334595, -0.003005981, -0.003723145, -0.004486084, -0.005294800, -0.006118774,
    -0.007003784, -0.007919312, -0.008865356, -0.009841919, -0.010848999, -0.011886597, -0.012939453, -0.014022827,
    -0.015121460, -0.016235352, -0.017349243, -0.018463135, -0.019577026, -0.020690918, -0.021789551, -0.022857666,
    -0.023910522, -0.024932861, -0.025909424, -0.026840210, -0.027725220, -0.028533936, -0.029281616, -0.029937744,
    -0.030532837, -0.031005859, -0.031387329, -0.031661987, -0.031814575, -0.031845093, -0.031738281, -0.031478882,
    0.031082153, 0.030517578, 0.029785156, 0.028884888, 0.027801514, 0.026535034, 0.025085449, 0.023422241,
    0.021575928, 0.019531250, 0.017257690, 0.014801025, 0.012115479, 0.009231567, 0.006134033, 0.002822876,
    -0.000686646, -0.004394531, -0.008316040, -0.012420654, -0.016708374, -0.021179199, -0.025817871, -0.030609131,
    -0.035552979, -0.040634155, -0.045837402, -0.051132202, -0.056533813, -0.061996460, -0.067520142, -0.073059082,
    -0.078628540, -0.084182739, -0.089706421, -0.095169067, -0.100540161, -0.105819702, -0.110946655, -0.115921021,
    -0.120697021, -0.125259399, -0.129562378, -0.133590698, -0.137298584, -0.140670776, -0.143676758, -0.146255493,
    -0.148422241, -0.150115967, -0.151306152, -0.151962280, -0.152069092, -0.151596069, -0.150497437, -0.148773193,
    -0.146362305, -0.143264771, -0.139450073, -0.134887695, -0.129577637, -0.123474121, -0.116577148, -0.108856201,
    0.100311279, 0.090927124, 0.080688477, 0.069595337, 0.057617187, 0.044784546, 0.031082153, 0.016510010,
    0.001068115, -0.015228271, -0.032379150, -0.050354004, -0.069168091, -0.088775635, -0.109161377, -0.130310059,
    -0.152206421, -0.174789429, -0.198059082, -0.221984863, -0.246505737, -0.271591187, -0.297210693, -0.323318481,
    -0.349868774, -0.376800537, -0.404083252, -0.431655884, -0.459472656, -0.487472534, -0.515609741, -0.543823242,
    -0.572036743, -0.600219727, -0.628295898, -0.656219482, -0.683914185, -0.711318970, -0.738372803, -0.765029907,
    -0.791213989, -0.816864014, -0.841949463, -0.866363525, -0.890090942, -0.913055420, -0.935195923, -0.956481934,
    -0.976852417, -0.996246338, -1.014617920, -1.031936646, -1.048156738, -1.063217163, -1.077117920, -1.089782715,
    -1.101211548, -1.111373901, -1.120223999, -1.127746582, -1.133926392, -1.138763428, -1.142211914, -1.144287109,
    1.144989014, 1.144287109, 1.142211914, 1.138763428, 1.133926392, 1.127746582, 1.120223999, 1.111373901,
    1.101211548, 1.089782715, 1.077117920, 1.063217163, 1.048156738, 1.031936646, 1.014617920, 0.996246338,
    0.976852417, 0.956481934, 0.935195923, 0.913055420, 0.890090942, 0.866363525, 0.841949463, 0.816864014,
    0.791213989, 0.765029907, 0.738372803, 0.711318970, 0.683914185, 0.656219482, 0.628295898, 0.600219727,
    0.572036743, 0.543823242, 0.515609741, 0.487472534, 0.459472656, 0.431655884, 0.404083252, 0.376800537,
    0.349868774, 0.323318481, 0.297210693, 0.271591187, 0.246505737, 0.221984863, 0.198059082, 0.174789429,
    0.152206421, 0.130310059, 0.109161377, 0.088775635, 0.069168091, 0.050354004, 0.032379150, 0.015228271,
    -0.001068115, -0.016510010, -0.031082153, -0.044784546, -0.057617187, -0.069595337, -0.080688477, -0.090927124,
    0.100311279, 0.108856201, 0.116577148, 0.123474121, 0.129577637, 0.134887695, 0.139450073, 0.143264771,
    0.146362305, 0.148773193, 0.150497437, 0.151596069, 0.152069092, 0.151962280, 0.151306152, 0.150115967,
    0.148422241, 0.146255493, 0.143676758, 0.140670776, 0.137298584, 0.133590698, 0.129562378, 0.125259399,
    0.120697021, 0.115921021, 0.110946655, 0.105819702, 0.100540161, 0.095169067, 0.089706421, 0.084182739,
    0.078628540, 0.073059082, 0.067520142, 0.061996460, 0.056533813, 0.051132202, 0.045837402, 0.040634155,
    0.035552979, 0.030609131, 0.025817871, 0.021179199, 0.016708374, 0.012420654, 0.008316040, 0.004394531,
    0.000686646, -0.002822876, -0.006134033, -0.009231567, -0.012115479, -0.014801025, -0.017257690, -0.019531250,
    -0.021575928, -0.023422241, -0.025085449, -0.026535034, -0.027801514, -0.028884888, -0.029785156, -0.030517578,
    0.031082153, 0.031478882, 0.031738281, 0.031845093, 0.031814575, 0.031661987, 0.031387329, 0.031005859,
    0.030532837, 0.029937744, 0.029281616, 0.028533936, 0.027725220, 0.026840210, 0.025909424, 0.024932861,
    0.023910522, 0.022857666, 0.021789551, 0.020690918, 0.019577026, 0.018463135, 0.017349243, 0.016235352,
    0.015121460, 0.014022827, 0.012939453, 0.011886597, 0.010848999, 0.009841919, 0.008865356, 0.007919312,
    0.007003784, 0.006118774, 0.005294800, 0.004486084, 0.003723145, 0.003005981, 0.002334595, 0.001693726,
    0.001098633, 0.000549316, 0.000030518, -0.000442505, -0.000869751, -0.001266479, -0.001617432, -0.001937866,
    -0.002227783, -0.002487183, -0.002700806, -0.002883911, -0.003051758, -0.003173828, -0.003280640, -0.003372192,
    -0.003417969, -0.003463745, -0.003479004, -0.003479004, -0.003463745, -0.003433228, -0.003387451, -0.003326416,
    0.003250122, 0.003173828, 0.003082275, 0.002990723, 0.002899170, 0.002792358, 0.002685547, 0.002578735,
    0.002456665, 0.002349854, 0.002243042, 0.002120972, 0.002014160, 0.001907349, 0.001785278, 0.001693726,
    0.001586914, 0.001480103, 0.001388550, 0.001296997, 0.001205444, 0.001113892, 0.001037598, 0.000961304,
    0.000885010, 0.000808716, 0.000747681, 0.000686646, 0.000625610, 0.000579834, 0.000534058, 0.000473022,
    0.000442505, 0.000396729, 0.000366211, 0.000320435, 0.000289917, 0.000259399, 0.000244141, 0.000213623,
    0.000198364, 0.000167847, 0.000152588, 0.000137329, 0.000122070, 0.000106812, 0.000106812, 0.000091553,
    0.000076294, 0.000076294, 0.000061035, 0.000061035, 0.000045776, 0.000045776, 0.000030518, 0.000030518,
    0.000030518, 0.000030518, 0.000015259, 0.000015259, 0.000015259, 0.000015259, 0.000015259, 0.000015259
};

// ISO/IEC 11172-3 (2.4.3.2.2)
// cos((16 + i) * (2 * k + 1) * pi / 64.0), k=0..31, i=0..63
Array<Array<float, 32>, 64> SynthesisSubbandFilterCoefficients = {
    Array<float, 32> { { 0.7071067811865476, -0.7071067811865475, -0.7071067811865477, 0.7071067811865474, 0.7071067811865477, -0.7071067811865467, -0.7071067811865471, 0.7071067811865466, 0.7071067811865472, -0.7071067811865465, -0.7071067811865474, 0.7071067811865464, 0.7071067811865475, -0.7071067811865464, -0.7071067811865476, 0.7071067811865462, 0.7071067811865476, -0.7071067811865461, -0.7071067811865477, 0.707106781186546, 0.7071067811865503, -0.707106781186546, -0.7071067811865479, 0.7071067811865483, 0.7071067811865505, -0.7071067811865458, -0.707106781186548, 0.7071067811865482, 0.7071067811865507, -0.7071067811865456, -0.7071067811865482, 0.707106781186548 } },
    Array<float, 32> { { 0.6715589548470183, -0.8032075314806448, -0.5141027441932218, 0.9039892931234431, 0.33688985339222005, -0.9700312531945441, -0.14673047445536166, 0.9987954562051724, -0.04906767432741729, -0.9891765099647811, 0.24298017990326243, 0.9415440651830208, -0.42755509343028003, -0.8577286100002726, 0.5956993044924337, 0.7409511253549602, -0.7409511253549589, -0.5956993044924354, 0.8577286100002715, 0.4275550934302851, -0.9415440651830214, -0.24298017990326443, 0.9891765099647806, 0.04906767432742292, -0.9987954562051724, 0.1467304744553596, 0.9700312531945451, -0.3368898533922206, -0.903989293123444, 0.5141027441932186, 0.8032075314806442, -0.6715589548470177 } },
    Array<float, 32> { { 0.6343932841636455, -0.8819212643483549, -0.29028467725446244, 0.9951847266721969, -0.09801714032955997, -0.9569403357322087, 0.47139673682599736, 0.7730104533627377, -0.773010453362737, -0.4713967368259983, 0.9569403357322089, 0.09801714032956282, -0.995184726672197, 0.2902846772544622, 0.8819212643483563, -0.6343932841636443, -0.6343932841636459, 0.8819212643483553, 0.29028467725446433, -0.9951847266721968, 0.09801714032956063, 0.9569403357322086, -0.47139673682599326, -0.7730104533627394, 0.7730104533627351, 0.4713967368259993, -0.9569403357322086, -0.09801714032956038, 0.9951847266721968, -0.2902846772544578, -0.8819212643483568, 0.6343932841636434 } },
    Array<float, 32> { { 0.5956993044924335, -0.9415440651830207, -0.04906767432741803, 0.9700312531945441, -0.5141027441932214, -0.6715589548470181, 0.903989293123443, 0.1467304744553618, -0.9891765099647811, 0.42755509343028014, 0.7409511253549601, -0.8577286100002717, -0.24298017990326395, 0.9987954562051724, -0.33688985339221794, -0.8032075314806458, 0.8032075314806444, 0.33688985339222016, -0.9987954562051723, 0.24298017990326515, 0.8577286100002729, -0.7409511253549561, -0.4275550934302822, 0.9891765099647806, -0.146730474455363, -0.903989293123444, 0.6715589548470151, 0.5141027441932219, -0.9700312531945433, 0.04906767432741926, 0.9415440651830214, -0.5956993044924298 } },
    Array<float, 32> { { 0.5555702330196023, -0.9807852804032304, 0.1950903220161283, 0.8314696123025455, -0.8314696123025451, -0.19509032201612803, 0.9807852804032307, -0.5555702330196015, -0.5555702330196026, 0.9807852804032304, -0.19509032201612858, -0.8314696123025449, 0.8314696123025438, 0.19509032201613036, -0.9807852804032308, 0.5555702330196011, 0.5555702330196061, -0.9807852804032297, 0.19509032201612447, 0.8314696123025471, -0.8314696123025435, -0.19509032201613097, 0.9807852804032309, -0.5555702330196005, -0.5555702330196036, 0.9807852804032302, -0.19509032201612736, -0.8314696123025456, 0.8314696123025451, 0.19509032201612808, -0.9807852804032303, 0.555570233019603 } },
    Array<float, 32> { { 0.5141027441932217, -0.9987954562051724, 0.42755509343028214, 0.5956993044924332, -0.989176509964781, 0.3368898533922202, 0.6715589548470182, -0.9700312531945441, 0.24298017990326243, 0.7409511253549601, -0.9415440651830203, 0.14673047445536033, 0.8032075314806457, -0.9039892931234428, 0.04906767432741668, 0.8577286100002728, -0.8577286100002696, -0.04906767432741925, 0.9039892931234453, -0.8032075314806442, -0.1467304744553664, 0.9415440651830211, -0.740951125354956, -0.24298017990326493, 0.9700312531945451, -0.6715589548470177, -0.3368898533922243, 0.9891765099647811, -0.5956993044924298, -0.42755509343028286, 0.9987954562051726, -0.5141027441932149 } },
    Array<float, 32> { { 0.4713967368259978, -0.9951847266721969, 0.6343932841636456, 0.29028467725446255, -0.9569403357322087, 0.773010453362737, 0.09801714032956081, -0.8819212643483562, 0.8819212643483555, -0.09801714032956124, -0.7730104533627368, 0.9569403357322088, -0.2902846772544621, -0.6343932841636459, 0.9951847266721968, -0.4713967368259935, -0.47139673682599587, 0.9951847266721967, -0.6343932841636466, -0.2902846772544613, 0.9569403357322086, -0.7730104533627373, -0.09801714032956038, 0.881921264348355, -0.8819212643483548, 0.0980171403295599, 0.7730104533627375, -0.9569403357322085, 0.29028467725446083, 0.634393284163647, -0.9951847266721959, 0.47139673682598915 } },
    Array<float, 32> { { 0.4275550934302822, -0.970031253194544, 0.803207531480645, -0.04906767432741754, -0.7409511253549599, 0.989176509964781, -0.5141027441932212, -0.33688985339221955, 0.9415440651830208, -0.8577286100002717, 0.14673047445536033, 0.6715589548470199, -0.9987954562051723, 0.5956993044924335, 0.24298017990326776, -0.9039892931234438, 0.9039892931234441, -0.2429801799032616, -0.5956993044924329, 0.9987954562051726, -0.6715589548470179, -0.14673047445536663, 0.8577286100002731, -0.941544065183021, 0.33688985339221694, 0.5141027441932221, -0.9891765099647817, 0.740951125354958, 0.04906767432741681, -0.8032075314806467, 0.9700312531945422, -0.42755509343028464 } },
    Array<float, 32> { { 0.38268343236508984, -0.9238795325112868, 0.9238795325112865, -0.3826834323650899, -0.38268343236509056, 0.9238795325112867, -0.9238795325112864, 0.38268343236508956, 0.3826834323650909, -0.9238795325112876, 0.9238795325112868, -0.3826834323650892, -0.3826834323650912, 0.9238795325112877, -0.9238795325112854, 0.38268343236508556, 0.3826834323650883, -0.9238795325112865, 0.9238795325112866, -0.3826834323650885, -0.38268343236509195, 0.9238795325112881, -0.9238795325112851, 0.3826834323650849, 0.382683432365089, -0.9238795325112868, 0.9238795325112863, -0.3826834323650813, -0.3826834323650926, 0.9238795325112856, -0.9238795325112849, 0.3826834323650908 } },
    Array<float, 32> { { 0.33688985339222005, -0.8577286100002721, 0.9891765099647809, -0.6715589548470177, 0.04906767432741742, 0.5956993044924335, -0.9700312531945443, 0.9039892931234429, -0.42755509343028003, -0.24298017990326395, 0.8032075314806457, -0.9987954562051723, 0.7409511253549588, -0.1467304744553635, -0.5141027441932244, 0.941544065183021, -0.9415440651830213, 0.5141027441932188, 0.146730474455363, -0.7409511253549584, 0.9987954562051726, -0.8032075314806439, 0.24298017990326443, 0.42755509343028597, -0.9039892931234442, 0.9700312531945441, -0.5956993044924352, -0.04906767432742757, 0.6715589548470239, -0.9891765099647817, 0.8577286100002706, -0.33688985339221944 } },
    Array<float, 32> { { 0.29028467725446233, -0.7730104533627371, 0.9951847266721969, -0.8819212643483548, 0.47139673682599736, 0.09801714032956081, -0.6343932841636456, 0.9569403357322094, -0.9569403357322089, 0.6343932841636444, -0.09801714032956099, -0.47139673682599875, 0.8819212643483564, -0.9951847266721968, 0.7730104533627352, -0.2902846772544582, -0.2902846772544613, 0.7730104533627373, -0.9951847266721972, 0.8819212643483533, -0.4713967368259928, -0.09801714032956063, 0.6343932841636468, -0.9569403357322098, 0.9569403357322074, -0.6343932841636404, 0.09801714032955235, 0.47139673682599387, -0.8819212643483538, 0.9951847266721969, -0.7730104533627365, 0.2902846772544601 } },
    Array<float, 32> { { 0.24298017990326398, -0.6715589548470187, 0.9415440651830209, -0.989176509964781, 0.8032075314806448, -0.4275550934302818, -0.04906767432741852, 0.5141027441932239, -0.8577286100002726, 0.9987954562051724, -0.9039892931234428, 0.5956993044924335, -0.1467304744553635, -0.3368898533922236, 0.7409511253549605, -0.9700312531945442, 0.9700312531945442, -0.740951125354956, 0.33688985339221716, 0.14673047445536325, -0.5956993044924334, 0.9039892931234457, -0.9987954562051722, 0.8577286100002709, -0.5141027441932149, 0.049067674327418764, 0.4275550934302864, -0.8032075314806426, 0.9891765099647812, -0.9415440651830184, 0.6715589548470194, -0.24298017990325993 } },
    Array<float, 32> { { 0.19509032201612833, -0.5555702330196022, 0.8314696123025455, -0.9807852804032307, 0.9807852804032304, -0.831469612302545, 0.5555702330196015, -0.19509032201612858, -0.19509032201613025, 0.5555702330196028, -0.831469612302545, 0.9807852804032309, -0.9807852804032297, 0.8314696123025456, -0.5555702330196007, 0.19509032201612425, 0.1950903220161276, -0.5555702330196036, 0.8314696123025475, -0.9807852804032303, 0.9807852804032301, -0.8314696123025431, 0.555570233019603, -0.1950903220161269, -0.19509032201612497, 0.5555702330196073, -0.8314696123025459, 0.9807852804032298, -0.9807852804032293, 0.8314696123025446, -0.5555702330196052, 0.19509032201612256 } },
    Array<float, 32> { { 0.14673047445536175, -0.4275550934302825, 0.6715589548470188, -0.8577286100002723, 0.9700312531945443, -0.9987954562051724, 0.9415440651830204, -0.8032075314806446, 0.5956993044924337, -0.33688985339221794, 0.04906767432741668, 0.24298017990326776, -0.5141027441932244, 0.7409511253549605, -0.9039892931234439, 0.989176509964781, -0.989176509964781, 0.9039892931234439, -0.7409511253549558, 0.5141027441932183, -0.24298017990326087, -0.04906767432742023, 0.33688985339222133, -0.5956993044924337, 0.8032075314806446, -0.9415440651830204, 0.9987954562051723, -0.9700312531945448, 0.8577286100002668, -0.6715589548470114, 0.4275550934302745, -0.14673047445535428 } },
    Array<float, 32> { { 0.09801714032956077, -0.29028467725446244, 0.471396736825998, -0.6343932841636454, 0.7730104533627377, -0.8819212643483562, 0.9569403357322094, -0.995184726672197, 0.9951847266721968, -0.9569403357322088, 0.8819212643483553, -0.7730104533627375, 0.6343932841636439, -0.47139673682599326, 0.2902846772544615, -0.09801714032955673, -0.09801714032956038, 0.29028467725446505, -0.4713967368259965, 0.6343932841636468, -0.7730104533627399, 0.8819212643483553, -0.9569403357322078, 0.9951847266721968, -0.9951847266721966, 0.9569403357322073, -0.881921264348351, 0.7730104533627387, -0.6343932841636453, 0.47139673682599476, -0.29028467725445634, 0.09801714032955137 } },
    Array<float, 32> { { 0.049067674327418126, -0.1467304744553623, 0.24298017990326423, -0.336889853392221, 0.4275550934302828, -0.5141027441932238, 0.595699304492435, -0.6715589548470199, 0.7409511253549602, -0.8032075314806458, 0.8577286100002728, -0.9039892931234438, 0.941544065183021, -0.9700312531945442, 0.989176509964781, -0.9987954562051724, 0.9987954562051724, -0.989176509964781, 0.9700312531945441, -0.941544065183021, 0.9039892931234437, -0.8577286100002726, 0.8032075314806414, -0.7409511253549601, 0.6715589548470144, -0.5956993044924349, 0.5141027441932174, -0.4275550934302842, 0.3368898533922158, -0.2429801799032666, 0.14673047445535767, -0.049067674327421214 } },
    Array<float, 32> { { 6.123233995736766e-17, -1.8369701987210297e-16, 3.061616997868383e-16, -4.286263797015736e-16, 5.51091059616309e-16, -2.4499125789312946e-15, -9.803364199544708e-16, -2.6948419387607653e-15, -7.354070601250002e-16, -2.939771298590236e-15, -4.904777002955296e-16, -3.1847006584197066e-15, -2.45548340466059e-16, -3.4296300182491773e-15, -6.189806365883577e-19, -3.674559378078648e-15, 2.443103791928823e-16, -3.919488737908119e-15, 4.892397390223529e-16, -4.164418097737589e-15, 7.839596456452825e-15, -4.40934745756706e-15, 9.790984586812941e-16, 2.4511505402044715e-15, 8.329455176111767e-15, -4.899206177226001e-15, 1.4689571783402355e-15, 1.96129182054553e-15, 8.819313895770708e-15, -5.389064896884942e-15, 1.9588158979991767e-15, 1.471433100886589e-15 } },
    Array<float, 32> { { -0.04906767432741801, 0.14673047445536194, -0.2429801799032628, 0.3368898533922202, -0.4275550934302818, 0.5141027441932227, -0.5956993044924338, 0.6715589548470184, -0.7409511253549589, 0.8032075314806444, -0.8577286100002696, 0.9039892931234441, -0.9415440651830213, 0.9700312531945442, -0.989176509964781, 0.9987954562051724, -0.9987954562051724, 0.9891765099647811, -0.9700312531945443, 0.9415440651830214, -0.9039892931234473, 0.8577286100002698, -0.8032075314806426, 0.7409511253549568, -0.6715589548470162, 0.5956993044924314, -0.51410274419322, 0.42755509343028064, -0.336889853392219, 0.24298017990326326, -0.14673047445536155, 0.04906767432741827 } },
    Array<float, 32> { { -0.09801714032956065, 0.29028467725446205, -0.4713967368259975, 0.6343932841636447, -0.773010453362737, 0.8819212643483555, -0.9569403357322089, 0.9951847266721968, -0.995184726672197, 0.9569403357322095, -0.8819212643483565, 0.7730104533627371, -0.634393284163649, 0.4713967368259993, -0.2902846772544615, 0.09801714032956405, 0.0980171403295599, -0.29028467725445756, 0.47139673682599564, -0.6343932841636404, 0.773010453362739, -0.8819212643483545, 0.9569403357322073, -0.9951847266721959, 0.9951847266721969, -0.9569403357322102, 0.8819212643483592, -0.7730104533627362, 0.6343932841636479, -0.47139673682600425, 0.2902846772544601, -0.09801714032956259 } },
    Array<float, 32> { { -0.14673047445536164, 0.42755509343028214, -0.6715589548470177, 0.8577286100002719, -0.9700312531945441, 0.9987954562051724, -0.9415440651830209, 0.8032075314806457, -0.5956993044924354, 0.33688985339222016, -0.04906767432741925, -0.2429801799032616, 0.5141027441932188, -0.740951125354956, 0.9039892931234439, -0.989176509964781, 0.9891765099647811, -0.9039892931234442, 0.7409511253549612, -0.5141027441932254, 0.2429801799032623, 0.04906767432741142, -0.33688985339221944, 0.5956993044924263, -0.8032075314806432, 0.9415440651830218, -0.9987954562051722, 0.9700312531945438, -0.8577286100002759, 0.6715589548470194, -0.42755509343029086, 0.14673047445536544 } },
    Array<float, 32> { { -0.1950903220161282, 0.5555702330196018, -0.8314696123025451, 0.9807852804032304, -0.9807852804032307, 0.8314696123025448, -0.5555702330196027, 0.19509032201613036, 0.19509032201612822, -0.555570233019601, 0.8314696123025456, -0.9807852804032295, 0.9807852804032309, -0.8314696123025455, 0.5555702330196066, -0.19509032201613144, -0.19509032201612714, 0.555570233019603, -0.8314696123025429, 0.9807852804032301, -0.9807852804032304, 0.831469612302544, -0.5555702330196105, 0.19509032201613602, 0.19509032201612256, -0.5555702330195991, 0.8314696123025443, -0.9807852804032305, 0.98078528040323, -0.8314696123025506, 0.5555702330196085, -0.1950903220161336 } },
    Array<float, 32> { { -0.24298017990326387, 0.6715589548470183, -0.9415440651830205, 0.9891765099647811, -0.8032075314806455, 0.4275550934302814, 0.049067674327416926, -0.5141027441932223, 0.8577286100002715, -0.9987954562051723, 0.9039892931234453, -0.5956993044924329, 0.146730474455363, 0.33688985339221716, -0.7409511253549558, 0.9700312531945441, -0.9700312531945443, 0.7409511253549612, -0.33688985339221805, -0.14673047445536205, 0.5956993044924321, -0.9039892931234419, 0.9987954562051726, -0.8577286100002757, 0.5141027441932292, -0.04906767432742855, -0.42755509343028375, 0.8032075314806449, -0.9891765099647807, 0.941544065183022, -0.6715589548470223, 0.24298017990327087 } },
    Array<float, 32> { { -0.29028467725446216, 0.7730104533627367, -0.9951847266721969, 0.8819212643483553, -0.4713967368259983, -0.09801714032956124, 0.6343932841636444, -0.9569403357322088, 0.9569403357322095, -0.6343932841636489, 0.09801714032956356, 0.47139673682599625, -0.8819212643483549, 0.9951847266721968, -0.7730104533627398, 0.2902846772544653, 0.29028467725446083, -0.7730104533627368, 0.9951847266721963, -0.8819212643483538, 0.47139673682600036, 0.09801714032955186, -0.6343932841636453, 0.9569403357322072, -0.9569403357322082, 0.6343932841636479, -0.09801714032956942, -0.47139673682599736, 0.8819212643483522, -0.9951847266721966, 0.773010453362739, -0.2902846772544709 } },
    Array<float, 32> { { -0.33688985339221994, 0.857728610000272, -0.9891765099647811, 0.6715589548470182, -0.04906767432741852, -0.5956993044924338, 0.9700312531945435, -0.9039892931234437, 0.4275550934302851, 0.24298017990326515, -0.8032075314806442, 0.9987954562051726, -0.7409511253549584, 0.14673047445536325, 0.5141027441932183, -0.941544065183021, 0.9415440651830214, -0.5141027441932254, -0.14673047445536205, 0.7409511253549529, -0.9987954562051722, 0.8032075314806449, -0.24298017990327322, -0.4275550934302776, 0.9039892931234431, -0.9700312531945464, 0.5956993044924377, 0.0490676743274173, -0.6715589548470108, 0.9891765099647801, -0.8577286100002726, 0.33688985339223004 } },
    Array<float, 32> { { -0.3826834323650897, 0.9238795325112865, -0.9238795325112867, 0.38268343236509067, 0.38268343236508956, -0.923879532511287, 0.9238795325112876, -0.3826834323650912, -0.382683432365089, 0.9238795325112867, -0.9238795325112865, 0.3826834323650885, 0.3826834323650851, -0.9238795325112851, 0.9238795325112881, -0.3826834323650924, -0.3826834323650813, 0.9238795325112835, -0.9238795325112897, 0.3826834323650962, 0.382683432365084, -0.9238795325112846, 0.9238795325112886, -0.3826834323650935, -0.38268343236508673, 0.9238795325112857, -0.9238795325112874, 0.3826834323650908, 0.38268343236508945, -0.9238795325112868, 0.9238795325112863, -0.38268343236508806 } },
    Array<float, 32> { { -0.42755509343028186, 0.970031253194544, -0.8032075314806454, 0.0490676743274184, 0.7409511253549591, -0.9891765099647809, 0.5141027441932241, 0.33688985339221783, -0.9415440651830214, 0.8577286100002729, -0.1467304744553664, -0.6715589548470179, 0.9987954562051726, -0.5956993044924334, -0.24298017990326087, 0.9039892931234437, -0.9039892931234473, 0.2429801799032623, 0.5956993044924321, -0.9987954562051722, 0.6715589548470242, 0.14673047445536494, -0.8577286100002721, 0.9415440651830218, -0.33688985339222594, -0.5141027441932137, 0.9891765099647812, -0.7409511253549601, -0.04906767432741338, 0.8032075314806403, -0.9700312531945466, 0.427555093430282 } },
    Array<float, 32> { { -0.4713967368259977, 0.9951847266721969, -0.6343932841636454, -0.29028467725446255, 0.9569403357322089, -0.7730104533627368, -0.09801714032956099, 0.8819212643483553, -0.8819212643483565, 0.09801714032956356, 0.7730104533627351, -0.9569403357322097, 0.29028467725446505, 0.6343932841636434, -0.9951847266721972, 0.4713967368259999, 0.47139673682598915, -0.9951847266721966, 0.6343932841636528, 0.2902846772544601, -0.9569403357322062, 0.7730104533627383, 0.09801714032955137, -0.881921264348354, 0.8819212643483594, -0.09801714032956259, -0.7730104533627312, 0.9569403357322094, -0.2902846772544709, -0.6343932841636442, 0.9951847266721977, -0.47139673682601163 } },
    Array<float, 32> { { -0.5141027441932217, 0.9987954562051724, -0.4275550934302827, -0.5956993044924326, 0.9891765099647809, -0.33688985339221983, -0.6715589548470184, 0.9700312531945441, -0.24298017990326443, -0.7409511253549561, 0.9415440651830211, -0.14673047445536663, -0.8032075314806439, 0.9039892931234457, -0.04906767432742023, -0.8577286100002726, 0.8577286100002698, 0.04906767432741142, -0.9039892931234419, 0.8032075314806449, 0.14673047445536494, -0.9415440651830181, 0.7409511253549621, 0.2429801799032628, -0.9700312531945445, 0.6715589548470249, 0.33688985339221483, -0.9891765099647807, 0.5956993044924325, 0.42755509343027315, -0.9987954562051727, 0.5141027441932245 } },
    Array<float, 32> { { -0.555570233019602, 0.9807852804032304, -0.19509032201612803, -0.831469612302545, 0.8314696123025448, 0.19509032201612844, -0.9807852804032303, 0.5555702330196061, 0.5555702330196038, -0.9807852804032302, 0.1950903220161276, 0.8314696123025452, -0.8314696123025456, -0.19509032201612714, 0.9807852804032301, -0.5555702330196102, -0.5555702330196056, 0.9807852804032298, -0.19509032201612544, -0.8314696123025465, 0.8314696123025443, 0.1950903220161293, -0.9807852804032305, 0.5555702330196024, 0.5555702330196015, -0.9807852804032308, 0.19509032201613025, 0.8314696123025438, -0.831469612302547, -0.19509032201612447, 0.9807852804032268, -0.5555702330196183 } },
    Array<float, 32> { { -0.5956993044924334, 0.9415440651830209, 0.04906767432741742, -0.9700312531945441, 0.5141027441932239, 0.6715589548470184, -0.9039892931234437, -0.1467304744553635, 0.9891765099647806, -0.4275550934302822, -0.740951125354956, 0.8577286100002731, 0.24298017990326443, -0.9987954562051722, 0.33688985339222133, 0.8032075314806414, -0.8032075314806426, -0.33688985339221944, 0.9987954562051726, -0.24298017990327322, -0.8577286100002721, 0.7409511253549621, 0.42755509343027404, -0.9891765099647809, 0.14673047445536544, 0.9039892931234398, -0.6715589548470173, -0.5141027441932191, 0.9700312531945459, -0.04906767432741582, -0.9415440651830153, 0.5956993044924388 } },
    Array<float, 32> { { -0.6343932841636454, 0.881921264348355, 0.29028467725446266, -0.9951847266721969, 0.09801714032956282, 0.9569403357322088, -0.47139673682599875, -0.7730104533627375, 0.7730104533627371, 0.47139673682599625, -0.9569403357322097, -0.09801714032955648, 0.9951847266721964, -0.290284677254462, -0.8819212643483513, 0.6343932841636472, 0.6343932841636483, -0.8819212643483573, -0.2902846772544634, 0.9951847266721976, -0.09801714032956209, -0.956940335732206, 0.47139673682600125, 0.7730104533627381, -0.7730104533627412, -0.4713967368259969, 0.9569403357322115, 0.09801714032955722, -0.9951847266721972, 0.2902846772544681, 0.8819212643483483, -0.6343932841636412 } },
    Array<float, 32> { { -0.6715589548470184, 0.8032075314806453, 0.5141027441932213, -0.9039892931234434, -0.33688985339221816, 0.970031253194544, 0.1467304744553601, -0.9987954562051724, 0.04906767432742292, 0.9891765099647806, -0.24298017990326493, -0.941544065183021, 0.42755509343028597, 0.8577286100002709, -0.5956993044924337, -0.7409511253549601, 0.7409511253549568, 0.5956993044924263, -0.8577286100002757, -0.4275550934302776, 0.9415440651830218, 0.2429801799032628, -0.9891765099647809, -0.04906767432742072, 0.998795456205172, -0.1467304744553693, -0.9700312531945426, 0.3368898533922236, 0.9039892931234365, -0.5141027441932339, -0.8032075314806376, 0.671558954847026 } },
    Array<float, 32> { { -0.7071067811865475, 0.7071067811865477, 0.7071067811865466, -0.7071067811865474, -0.7071067811865464, 0.7071067811865476, 0.707106781186546, -0.7071067811865479, -0.7071067811865458, 0.7071067811865507, 0.707106781186548, -0.7071067811865483, -0.7071067811865452, 0.7071067811865511, 0.7071067811865425, -0.7071067811865539, -0.7071067811865498, 0.7071067811865467, 0.707106781186547, -0.7071067811865495, -0.7071067811865442, 0.7071067811865522, 0.7071067811865415, -0.707106781186555, -0.7071067811865487, 0.7071067811865477, 0.707106781186546, -0.7071067811865606, -0.7071067811865432, 0.7071067811865432, 0.7071067811865405, -0.707106781186546 } },
    Array<float, 32> { { -0.7409511253549589, 0.5956993044924332, 0.8577286100002719, -0.4275550934302813, -0.9415440651830203, 0.2429801799032641, 0.9891765099647806, -0.04906767432741925, -0.9987954562051724, -0.146730474455363, 0.9700312531945451, 0.33688985339221694, -0.9039892931234442, -0.5141027441932149, 0.8032075314806446, 0.6715589548470144, -0.6715589548470162, -0.8032075314806432, 0.5141027441932292, 0.9039892931234431, -0.33688985339222594, -0.9700312531945445, 0.14673047445536544, 0.998795456205172, 0.04906767432741681, -0.989176509964782, -0.24298017990326515, 0.9415440651830271, 0.4275550934302855, -0.8577286100002731, -0.595699304492427, 0.7409511253549683 } },
    Array<float, 32> { { -0.773010453362737, 0.471396736825998, 0.9569403357322085, -0.0980171403295627, -0.995184726672197, -0.2902846772544621, 0.8819212643483564, 0.6343932841636439, -0.634393284163649, -0.8819212643483549, 0.29028467725446505, 0.9951847266721964, 0.09801714032955966, -0.9569403357322078, -0.47139673682599215, 0.7730104533627381, 0.7730104533627387, -0.47139673682600386, -0.9569403357322082, 0.09801714032955867, 0.9951847266721977, 0.29028467725445917, -0.8819212643483545, -0.6343932841636388, 0.6343932841636487, 0.8819212643483552, -0.2902846772544578, -0.995184726672195, -0.098017140329546, 0.9569403357322118, 0.4713967368259926, -0.7730104533627378 } },
    Array<float, 32> { { -0.8032075314806448, 0.33688985339222005, 0.9987954562051724, 0.24298017990326243, -0.8577286100002726, -0.7409511253549589, 0.4275550934302851, 0.9891765099647806, 0.1467304744553596, -0.903989293123444, -0.6715589548470177, 0.5141027441932221, 0.9700312531945441, 0.049067674327418764, -0.9415440651830204, -0.5956993044924349, 0.5956993044924314, 0.9415440651830218, -0.04906767432742855, -0.9700312531945464, -0.5141027441932137, 0.6715589548470249, 0.9039892931234398, -0.1467304744553693, -0.989176509964782, -0.42755509343027626, 0.7409511253549631, 0.857728610000262, -0.24298017990326848, -0.9987954562051733, -0.3368898533922167, 0.8032075314806552 } },
    Array<float, 32> { { -0.8314696123025453, 0.19509032201612878, 0.9807852804032307, 0.5555702330196015, -0.5555702330196027, -0.9807852804032303, -0.19509032201612808, 0.8314696123025471, 0.8314696123025455, -0.19509032201613122, -0.9807852804032303, -0.5555702330196002, 0.5555702330196071, 0.9807852804032301, 0.19509032201612303, -0.83146961230255, -0.8314696123025465, 0.19509032201612928, 0.9807852804032313, 0.5555702330195958, -0.5555702330196114, -0.9807852804032304, -0.19509032201612497, 0.8314696123025489, 0.8314696123025397, -0.1950903220161413, -0.9807852804032337, -0.5555702330196093, 0.5555702330195978, 0.9807852804032309, 0.1950903220161269, -0.8314696123025479 } },
    Array<float, 32> { { -0.857728610000272, 0.049067674327418154, 0.9039892931234434, 0.8032075314806447, -0.14673047445536216, -0.9415440651830209, -0.7409511253549563, 0.242980179903268, 0.9700312531945451, 0.6715589548470151, -0.3368898533922243, -0.9891765099647817, -0.5956993044924352, 0.4275550934302864, 0.9987954562051723, 0.5141027441932174, -0.51410274419322, -0.9987954562051722, -0.42755509343028375, 0.5956993044924377, 0.9891765099647812, 0.33688985339221483, -0.6715589548470173, -0.9700312531945426, -0.24298017990326515, 0.7409511253549631, 0.9415440651830211, 0.1467304744553698, -0.8032075314806528, -0.9039892931234407, -0.049067674327418764, 0.8577286100002681 } },
    Array<float, 32> { { -0.8819212643483549, -0.09801714032955997, 0.7730104533627377, 0.9569403357322089, 0.2902846772544622, -0.6343932841636459, -0.9951847266721968, -0.47139673682599326, 0.4713967368259993, 0.9951847266721968, 0.6343932841636434, -0.290284677254462, -0.9569403357322078, -0.7730104533627321, 0.09801714032956502, 0.8819212643483556, 0.8819212643483558, 0.09801714032955137, -0.7730104533627409, -0.9569403357322079, -0.29028467725446244, 0.634393284163654, 0.9951847266721962, 0.4713967368259935, -0.47139673682601163, -0.9951847266721967, -0.634393284163638, 0.29028467725445495, 0.9569403357322098, 0.7730104533627278, -0.0980171403295577, -0.8819212643483589 } },
    Array<float, 32> { { -0.9039892931234433, -0.2429801799032628, 0.5956993044924335, 0.9987954562051724, 0.6715589548470184, -0.14673047445536241, -0.857728610000271, -0.9415440651830213, -0.3368898533922206, 0.5141027441932219, 0.9891765099647811, 0.740951125354958, -0.04906767432742757, -0.8032075314806426, -0.9700312531945448, -0.4275550934302842, 0.42755509343028064, 0.9700312531945438, 0.8032075314806449, 0.0490676743274173, -0.7409511253549601, -0.9891765099647807, -0.5141027441932191, 0.3368898533922236, 0.9415440651830271, 0.857728610000262, 0.1467304744553698, -0.6715589548470129, -0.9987954562051727, -0.595699304492438, 0.24298017990325896, 0.9039892931234415 } },
    Array<float, 32> { { -0.9238795325112867, -0.3826834323650899, 0.38268343236509067, 0.9238795325112875, 0.9238795325112868, 0.3826834323650891, -0.38268343236509145, -0.9238795325112865, -0.9238795325112852, -0.3826834323650883, 0.382683432365089, 0.9238795325112882, 0.9238795325112835, 0.3826834323650908, -0.38268343236509306, -0.9238795325112898, -0.9238795325112873, -0.38268343236508673, 0.3826834323650971, 0.9238795325112862, 0.9238795325112856, 0.3826834323650826, -0.38268343236508806, -0.9238795325112878, -0.9238795325112893, -0.38268343236507857, 0.38268343236509217, 0.9238795325112841, 0.9238795325112822, 0.3826834323650876, -0.38268343236508306, -0.9238795325112912 } },
    Array<float, 32> { { -0.9415440651830207, -0.5141027441932214, 0.1467304744553618, 0.7409511253549601, 0.9987954562051724, 0.8032075314806444, 0.24298017990326515, -0.4275550934302822, -0.903989293123444, -0.9700312531945433, -0.5956993044924298, 0.04906767432741681, 0.6715589548470239, 0.9891765099647812, 0.8577286100002668, 0.3368898533922158, -0.336889853392219, -0.8577286100002759, -0.9891765099647807, -0.6715589548470108, -0.04906767432741338, 0.5956993044924325, 0.9700312531945459, 0.9039892931234365, 0.4275550934302855, -0.24298017990326848, -0.8032075314806528, -0.9987954562051727, -0.7409511253549578, -0.14673047445535137, 0.5141027441932381, 0.9415440651830205 } },
    Array<float, 32> { { -0.9569403357322088, -0.6343932841636448, -0.09801714032955972, 0.4713967368259984, 0.8819212643483563, 0.9951847266721968, 0.7730104533627352, 0.2902846772544615, -0.2902846772544615, -0.7730104533627398, -0.9951847266721972, -0.8819212643483513, -0.47139673682599215, 0.09801714032956502, 0.6343932841636476, 0.9569403357322092, 0.9569403357322092, 0.6343932841636476, 0.09801714032955088, -0.4713967368260047, -0.8819212643483579, -0.9951847266721965, -0.7730104533627352, -0.2902846772544615, 0.2902846772544615, 0.7730104533627352, 0.9951847266721965, 0.8819212643483579, 0.47139673682597966, -0.09801714032957916, -0.6343932841636586, -0.9569403357322133 } },
    Array<float, 32> { { -0.970031253194544, -0.7409511253549593, -0.3368898533922201, 0.14673047445536203, 0.5956993044924352, 0.9039892931234438, 0.9987954562051724, 0.8577286100002712, 0.5141027441932186, 0.04906767432741926, -0.42755509343028286, -0.8032075314806467, -0.9891765099647817, -0.9415440651830184, -0.6715589548470114, -0.2429801799032666, 0.24298017990326326, 0.6715589548470194, 0.941544065183022, 0.9891765099647801, 0.8032075314806403, 0.42755509343027315, -0.04906767432741582, -0.5141027441932339, -0.8577286100002731, -0.9987954562051733, -0.9039892931234407, -0.595699304492438, -0.14673047445535137, 0.33688985339221855, 0.740951125354969, 0.9700312531945446 } },
    Array<float, 32> { { -0.9807852804032304, -0.8314696123025451, -0.5555702330196015, -0.19509032201612858, 0.19509032201613036, 0.5555702330196061, 0.8314696123025471, 0.9807852804032309, 0.9807852804032302, 0.8314696123025451, 0.555570233019603, 0.19509032201613025, -0.19509032201613216, -0.5555702330196105, -0.8314696123025462, -0.980785280403232, -0.9807852804032305, -0.8314696123025421, -0.5555702330196044, -0.19509032201612497, 0.19509032201613746, 0.5555702330196032, 0.8314696123025413, 0.9807852804032302, 0.9807852804032294, 0.8314696123025391, 0.5555702330195881, 0.1950903220161336, -0.1950903220161288, -0.5555702330196076, -0.8314696123025522, -0.9807852804032341 } },
    Array<float, 32> { { -0.989176509964781, -0.9039892931234431, -0.7409511253549592, -0.5141027441932225, -0.24298017990326207, 0.04906767432742268, 0.3368898533922204, 0.5956993044924359, 0.8032075314806442, 0.9415440651830214, 0.9987954562051726, 0.9700312531945422, 0.8577286100002706, 0.6715589548470194, 0.4275550934302745, 0.14673047445535767, -0.14673047445536155, -0.42755509343029086, -0.6715589548470223, -0.8577286100002726, -0.9700312531945466, -0.9987954562051727, -0.9415440651830153, -0.8032075314806376, -0.595699304492427, -0.3368898533922167, -0.049067674327418764, 0.24298017990325896, 0.5141027441932381, 0.740951125354969, 0.9039892931234478, 0.9891765099647819 } },
    Array<float, 32> { { -0.9951847266721968, -0.9569403357322085, -0.8819212643483547, -0.7730104533627357, -0.6343932841636443, -0.4713967368259935, -0.2902846772544582, -0.09801714032955673, 0.09801714032956405, 0.2902846772544653, 0.4713967368259999, 0.6343932841636472, 0.7730104533627381, 0.8819212643483556, 0.9569403357322092, 0.9951847266721969, 0.9951847266721969, 0.9569403357322089, 0.8819212643483554, 0.7730104533627378, 0.6343932841636468, 0.47139673682599953, 0.29028467725445123, 0.09801714032956356, -0.09801714032957136, -0.2902846772544587, -0.4713967368260064, -0.6343932841636418, -0.7730104533627428, -0.8819212643483524, -0.9569403357322113, -0.9951847266721963 } },
    Array<float, 32> { { -0.9987954562051724, -0.989176509964781, -0.9700312531945441, -0.9415440651830203, -0.9039892931234428, -0.8577286100002696, -0.8032075314806442, -0.740951125354956, -0.6715589548470177, -0.5956993044924298, -0.5141027441932149, -0.42755509343028464, -0.33688985339221944, -0.24298017990325993, -0.14673047445535428, -0.049067674327421214, 0.04906767432741827, 0.14673047445536544, 0.24298017990327087, 0.33688985339223004, 0.427555093430282, 0.5141027441932245, 0.5956993044924388, 0.671558954847026, 0.7409511253549683, 0.8032075314806552, 0.8577286100002681, 0.9039892931234415, 0.9415440651830205, 0.9700312531945446, 0.9891765099647819, 0.9987954562051728 } },
    Array<float, 32> { { -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0 } },
    Array<float, 32> { { -0.9987954562051724, -0.9891765099647811, -0.9700312531945443, -0.9415440651830209, -0.9039892931234437, -0.857728610000271, -0.803207531480646, -0.7409511253549584, -0.6715589548470208, -0.5956993044924335, -0.5141027441932254, -0.4275550934302833, -0.33688985339221855, -0.24298017990327322, -0.14673047445536833, -0.0490676743274217, 0.0490676743274173, 0.14673047445536397, 0.24298017990325518, 0.3368898533922144, 0.42755509343027936, 0.5141027441932094, 0.5956993044924357, 0.6715589548470122, 0.7409511253549459, 0.8032075314806435, 0.857728610000265, 0.9039892931234449, 0.9415440651830181, 0.9700312531945394, 0.9891765099647807, 0.9987954562051717 } },
    Array<float, 32> { { -0.9951847266721969, -0.9569403357322087, -0.8819212643483562, -0.7730104533627368, -0.6343932841636459, -0.47139673682599587, -0.2902846772544613, -0.09801714032956038, 0.0980171403295599, 0.29028467725446083, 0.47139673682598915, 0.6343932841636483, 0.7730104533627387, 0.8819212643483558, 0.9569403357322092, 0.9951847266721969, 0.9951847266721969, 0.9569403357322094, 0.8819212643483564, 0.7730104533627393, 0.63439328416366, 0.47139673682599, 0.29028467725445495, 0.0980171403295538, -0.09801714032956649, -0.29028467725446716, -0.47139673682600125, -0.6343932841636479, -0.7730104533627383, -0.8819212643483556, -0.9569403357322089, -0.9951847266721968 } },
    Array<float, 32> { { -0.989176509964781, -0.9039892931234433, -0.74095112535496, -0.514102744193224, -0.2429801799032642, 0.049067674327419986, 0.3368898533922174, 0.5956993044924329, 0.8032075314806417, 0.9415440651830198, 0.9987954562051724, 0.9700312531945453, 0.8577286100002701, 0.6715589548470191, 0.4275550934302873, 0.1467304744553722, -0.14673047445536058, -0.4275550934302767, -0.6715589548470104, -0.857728610000264, -0.9700312531945425, -0.9987954562051722, -0.9415440651830261, -0.8032075314806487, -0.5956993044924309, -0.33688985339223515, -0.049067674327424635, 0.2429801799032666, 0.5141027441932078, 0.7409511253549544, 0.9039892931234444, 0.9891765099647786 } },
    Array<float, 32> { { -0.9807852804032304, -0.8314696123025456, -0.5555702330196026, -0.19509032201613025, 0.19509032201612822, 0.5555702330196038, 0.8314696123025455, 0.9807852804032302, 0.980785280403231, 0.8314696123025477, 0.5555702330196073, 0.1950903220161288, -0.1950903220161192, -0.5555702330195991, -0.8314696123025462, -0.9807852804032291, -0.9807852804032308, -0.8314696123025509, -0.5555702330196061, -0.1950903220161413, 0.19509032201613458, 0.5555702330196003, 0.8314696123025391, 0.9807852804032267, 0.9807852804032304, 0.83146961230255, 0.5555702330196166, 0.19509032201612592, -0.1950903220161221, -0.5555702330195897, -0.8314696123025479, -0.9807852804032297 } },
    Array<float, 32> { { -0.970031253194544, -0.7409511253549599, -0.33688985339221955, 0.14673047445536033, 0.5956993044924335, 0.9039892931234441, 0.9987954562051726, 0.8577286100002731, 0.5141027441932221, 0.04906767432741681, -0.42755509343028464, -0.8032075314806391, -0.9891765099647798, -0.9415440651830229, -0.671558954847022, -0.24298017990326706, 0.2429801799032623, 0.6715589548470184, 0.9415440651830214, 0.9891765099647826, 0.8032075314806505, 0.4275550934302891, -0.049067674327411916, -0.5141027441932179, -0.8577286100002706, -0.9987954562051723, -0.9039892931234431, -0.5956993044924317, -0.14673047445535817, 0.336889853392225, 0.7409511253549447, 0.9700312531945392 } },
    Array<float, 32> { { -0.9569403357322089, -0.6343932841636454, -0.0980171403295627, 0.4713967368259969, 0.8819212643483553, 0.9951847266721967, 0.7730104533627373, 0.29028467725446505, -0.29028467725445756, -0.7730104533627368, -0.9951847266721966, -0.8819212643483573, -0.47139673682600386, 0.09801714032955137, 0.6343932841636476, 0.9569403357322089, 0.9569403357322094, 0.6343932841636487, 0.09801714032956697, -0.47139673682599, -0.8819212643483566, -0.9951847266721981, -0.7730104533627378, -0.2902846772544793, 0.29028467725445684, 0.7730104533627409, 0.9951847266721959, 0.8819212643483543, 0.47139673682601074, -0.0980171403295577, -0.6343932841636305, -0.9569403357322067 } },
    Array<float, 32> { { -0.9415440651830208, -0.514102744193222, 0.14673047445536058, 0.7409511253549589, 0.9987954562051723, 0.8032075314806439, 0.24298017990326823, -0.4275550934302789, -0.9039892931234422, -0.9700312531945444, -0.5956993044924396, 0.04906767432741828, 0.671558954847014, 0.9891765099647812, 0.8577286100002741, 0.3368898533922296, -0.33688985339221805, -0.8577286100002678, -0.989176509964781, -0.6715589548470231, -0.049067674327430505, 0.5956993044924184, 0.9700312531945449, 0.9039892931234444, 0.42755509343028997, -0.24298017990324947, -0.8032075314806324, -0.9987954562051723, -0.7409511253549624, -0.1467304744553727, 0.514102744193207, 0.9415440651830225 } },
    Array<float, 32> { { -0.9238795325112868, -0.38268343236509056, 0.38268343236508956, 0.9238795325112868, 0.9238795325112877, 0.3826834323650883, -0.3826834323650885, -0.9238795325112851, -0.9238795325112868, -0.3826834323650926, 0.3826834323650908, 0.9238795325112833, 0.9238795325112886, 0.38268343236509034, -0.3826834323650799, -0.9238795325112843, -0.9238795325112876, -0.38268343236508806, 0.3826834323650822, 0.9238795325112852, 0.9238795325112867, 0.3826834323650858, -0.38268343236507135, -0.9238795325112807, -0.9238795325112912, -0.38268343236509667, 0.38268343236508673, 0.9238795325112871, 0.9238795325112849, 0.38268343236510755, -0.38268343236507585, -0.9238795325112826 } },
    Array<float, 32> { { -0.9039892931234434, -0.24298017990326348, 0.5956993044924339, 0.9987954562051723, 0.6715589548470174, -0.14673047445536325, -0.8577286100002693, -0.9415440651830225, -0.33688985339222455, 0.5141027441932179, 0.9891765099647803, 0.7409511253549618, -0.04906767432741436, -0.8032075314806429, -0.9700312531945448, -0.42755509343028464, 0.4275550934302798, 0.9700312531945434, 0.8032075314806546, 0.04906767432741974, -0.7409511253549486, -0.9891765099647811, -0.5141027441932347, 0.33688985339221944, 0.9415440651830159, 0.8577286100002721, 0.1467304744553756, -0.6715589548470188, -0.9987954562051731, -0.5956993044924325, 0.24298017990325138, 0.903989293123444 } },
    Array<float, 32> { { -0.881921264348355, -0.09801714032956069, 0.7730104533627358, 0.9569403357322095, 0.29028467725446433, -0.6343932841636466, -0.9951847266721972, -0.4713967368259965, 0.47139673682599564, 0.9951847266721963, 0.6343932841636528, -0.2902846772544634, -0.9569403357322082, -0.7730104533627409, 0.09801714032955088, 0.8819212643483554, 0.8819212643483564, 0.09801714032956697, -0.7730104533627397, -0.9569403357322087, -0.2902846772544653, 0.6343932841636404, 0.9951847266721979, 0.4713967368260099, -0.4713967368259822, -0.9951847266721948, -0.6343932841636426, 0.29028467725446244, 0.9569403357322078, 0.7730104533627414, -0.0980171403295499, -0.8819212643483483 } },
    Array<float, 32> { { -0.8577286100002721, 0.04906767432741742, 0.9039892931234429, 0.8032075314806457, -0.1467304744553635, -0.9415440651830213, -0.7409511253549584, 0.24298017990326443, 0.9700312531945441, 0.6715589548470239, -0.33688985339221944, -0.9891765099647798, -0.5956993044924345, 0.42755509343027404, 0.9987954562051723, 0.5141027441932301, -0.5141027441932191, -0.998795456205173, -0.4275550934302855, 0.5956993044924357, 0.9891765099647838, 0.33688985339223143, -0.6715589548470144, -0.9700312531945436, -0.2429801799032837, 0.7409511253549499, 0.9415440651830231, 0.14673047445536203, -0.8032075314806318, -0.9039892931234499, -0.04906767432742659, 0.8577286100002711 } },
    Array<float, 32> { { -0.8314696123025455, 0.1950903220161272, 0.9807852804032304, 0.5555702330196028, -0.555570233019601, -0.9807852804032302, -0.19509032201613122, 0.8314696123025451, 0.8314696123025477, -0.19509032201611967, -0.9807852804032293, -0.5555702330196048, 0.555570233019602, 0.98078528040323, 0.195090322016137, -0.8314696123025419, -0.831469612302547, 0.19509032201612786, 0.9807852804032281, 0.5555702330195978, -0.5555702330195971, -0.9807852804032339, -0.1950903220161288, 0.8314696123025386, 0.8314696123025425, -0.1950903220161221, -0.980785280403227, -0.5555702330196027, 0.5555702330195922, 0.9807852804032294, 0.19509032201613458, -0.8314696123025354 } },
    Array<float, 32> { { -0.8032075314806449, 0.3368898533922202, 0.9987954562051724, 0.2429801799032641, -0.8577286100002696, -0.7409511253549583, 0.4275550934302822, 0.9891765099647811, 0.14673047445537077, -0.903989293123445, -0.6715589548470162, 0.5141027441932233, 0.9700312531945438, 0.04906767432741827, -0.9415440651830204, -0.5956993044924352, 0.5956993044924306, 0.9415440651830271, -0.0490676743274266, -0.9700312531945459, -0.5141027441932162, 0.6715589548470224, 0.9039892931234415, -0.14673047445536494, -0.9891765099647812, -0.4275550934302811, 0.7409511253549591, 0.8577286100002726, -0.24298017990326182, -0.9987954562051722, -0.33688985339222405, 0.8032075314806417 } },
    Array<float, 32> { { -0.7730104533627371, 0.47139673682599736, 0.9569403357322094, -0.09801714032956099, -0.9951847266721968, -0.2902846772544613, 0.8819212643483533, 0.6343932841636468, -0.6343932841636404, -0.8819212643483538, 0.2902846772544601, 0.9951847266721976, 0.09801714032955867, -0.9569403357322079, -0.4713967368260047, 0.7730104533627378, 0.7730104533627393, -0.47139673682599, -0.9569403357322087, 0.0980171403295421, 0.9951847266721959, 0.29028467725446244, -0.881921264348346, -0.6343932841636533, 0.6343932841636449, 0.8819212643483644, -0.2902846772544521, -0.995184726672197, -0.09801714032958111, 0.9569403357322056, 0.47139673682599953, -0.7730104533627234 } },
    Array<float, 32> { { -0.7409511253549591, 0.5956993044924327, 0.8577286100002725, -0.4275550934302798, -0.9415440651830222, 0.24298017990326493, 0.9891765099647811, -0.049067674327415586, -0.9987954562051722, -0.14673047445536058, 0.9700312531945421, 0.3368898533922222, -0.9039892931234447, -0.5141027441932267, 0.8032075314806446, 0.6715589548470253, -0.6715589548470154, -0.803207531480644, 0.5141027441932153, 0.9039892931234503, -0.33688985339222316, -0.9700312531945453, 0.1467304744553475, 0.9987954562051722, 0.0490676743274217, -0.9891765099647791, -0.24298017990328463, 0.9415440651830201, 0.42755509343029174, -0.857728610000262, -0.5956993044924334, 0.7409511253549532 } },
};

}
