/***************************************************************************
   faderwidget.h -  smoothly fades one widget to reveal a second
                    based on the Qt "Fade-Effects" demo; updated to use a 
                    QWidget source.

                             -------------------
    begin                : Wednesday 05 Jan 2011
    copyright            : (C) 2011 by Adam Goossens
    email                : adam.goossens@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <QTimeLine>
#include <QPainter>
#include <QSize>
#include <stdint.h>

#include "../kmessdebug.h"
#include "faderwidget.h"


/**
 * Construct a new FaderWidget.
 * 
 * \arg begin is the source widget. A pixmap will be grabbed from this widget and will be
 * used as the fading source.
 * 
 * \arg parent must be a valid QWidget and should be the widget you are fading out to.
 * 
 * @param begin Source widget which will "fade out"
 * @param parent Widget which is being revealed.
 */
FaderWidget::FaderWidget( QWidget *begin, QWidget *parent )
: QWidget( parent )
{
  timeline_ = new QTimeLine( 1000, this ); // 1sec fade by default
  timeline_->setFrameRange( 1000, 0 );

  connect( timeline_, SIGNAL( frameChanged( int ) ),
           this,        SLOT( update() ) );

  KMESS_ASSERT( parent );
  KMESS_ASSERT( begin );

  resize( parent->size() );

  setAttribute( Qt::WA_DeleteOnClose );

  pixmap_ = QPixmap::grabWidget( begin );
}



/**
 * Change the duration of the fade.
 * 
 * The default is 1000ms (1sec)
 * @param milliseconds Fade duration
 */
void FaderWidget::setDuration( uint16_t milliseconds )
{
  if ( timeline_->state() == QTimeLine::Running )
  {
    kWarning() << "Cannot change duration whilst fade is running";
    return;
  }
  
  timeline_->setDuration( milliseconds );
}



/**
 * Start the fade effect.
 * 
 * Calling this whilst the fade is already running has no effect.
 */
void FaderWidget::start()
{
  if ( timeline_->state() == QTimeLine::Running )
  {
    kWarning() << "Fade is already running.";
    return;
  }
  
  timeline_->start();
  show();
}



/**
 * Stop the fade effect.
 * 
 * Calling this whilst already stopped has no effect.
 */
void FaderWidget::stop()
{
  timeline_->stop();
}



/**
 * Paint the grabbed pixmap with a given opacity over our parent widget.
 */
void FaderWidget::paintEvent( QPaintEvent * )
{
  QPainter painter( this );
  qreal frame = timeline_->currentFrame();
  painter.setOpacity( frame / 1000 ); // 1000 frames possible
  painter.drawPixmap( 0, 0, pixmap_ );

  if ( frame <= 0 )
  {
    close();
  }
}

#include "faderwidget.moc"
