(** Generic Input/Ouput *)

(*IF-OCAML*)
type pos = int64
(** The type of a position in a file *)
(*ENDIF-OCAML*)
(*i*)(*F#
type pos = int
F#*)(*i*)

val pos_succ : (pos -> pos)
val pos_pred : (pos -> pos)
val pos_max : (pos -> pos -> pos)
val possub : (pos -> pos -> pos)
val posadd : (pos -> pos -> pos)
(** Arithmetic on positions *)

val postoi : (pos -> int)
val posofi : (int -> pos)
(** Convert to and from integers *)

val postoi64 : (pos -> int64)
val posofi64 : (int64 -> pos)
(** Convert to and from 64 bit integers *)

(** Type for inputs. *)
type input =
  {pos_in : unit -> pos;
   seek_in : pos -> unit;
   input_char : unit -> char option;
   input_byte : unit -> int;
   in_channel_length : unit -> pos;
   set_offset : pos -> unit}

(** Type for outputs *)
type output =
  {pos_out : unit -> pos;
   seek_out : pos -> unit;
   output_char : char -> unit;
   output_byte : int -> unit;
   out_channel_length : unit -> pos}

val no_more : int
(** A distinguished byte value indicating "no more input" *)

val output_of_stream : Utility.stream -> output

val output_of_channel : out_channel -> output

val input_of_channel : in_channel -> input

val input_of_stream : Utility.stream -> input

val input_of_bytestream : Utility.bytestream -> input

val input_of_string : string -> input
(** Convert between various input and output and data types. *)

(** Move forward one character *)
val nudge : input -> unit

val rewind : input -> unit
val rewind2 : input -> unit
val rewind3 : input -> unit
(** Move backward one, two or three positions. *)

(** Look at the next character without advancing the pointer. *)
val peek_char : input -> char option

(** Look at the next byte without advancing the pointer. *)
val peek_byte : input -> int

(** Output a string. *)
val output_string : output -> string -> unit

(** Read the previous character, moving the pointer back one. *)
val read_char_back : input -> char option

(** Read a line. cf Pervasives.read_line. *)
val read_line : input -> string

(** Extract a bytestream from an input or output. *)
val bytestream_of_input_channel :
  in_channel -> Utility.bytestream

(** Write a bytestream to an output channel *)
val bytestream_to_output_channel :
  out_channel -> Utility.bytestream -> unit

(** {2 Read Bit streams} *)

(** The type of MSB-first bitstreams *)
type bitstream

(** Make a bitstream from an input. *)
val bitstream_of_input : input -> bitstream

(** Get a bit *)
val getbit : bitstream -> bool

(** Ditto but as an integer, 0 or 1. *)
val getbitint : bitstream -> int

(** Align on a byte boundary *)
val align : bitstream -> unit

(** Get a 32-bit value *)
val getval_32 : bitstream -> int -> int32

(** {2 Write Bit streams} *)

(** The type of MSB-first bitstreams for writing. *)
type bitstream_write

(** Return a new write bistream. *)
val make_write_bitstream : unit -> bitstream_write

(** Debug printer. *)
val print_bitstream : bitstream_write -> unit

(** Build a bytestream from a write bitstream, padding with zeroes. *)
val bytestream_of_write_bitstream : bitstream_write -> Utility.bytestream

(** Put a single bit, 0 or 1. *)
val putbit : bitstream_write -> int -> unit

(** Same, but input is boolean *)
val putbool : bitstream_write -> bool -> unit

(** Put a multi-bit value (given as an int32) containing the given number of
useful bits into a bitstream *)
val putval : bitstream_write -> int -> int32 -> unit

(** Byte-align. *)
val align_write : bitstream_write -> unit

(** Append two write bitstreams *)
val write_bitstream_append :
  bitstream_write -> bitstream_write -> bitstream_write

(** Same, but align at boundary *)
val write_bitstream_append_aligned :
  bitstream_write -> bitstream_write -> bitstream_write

(** Join several write bitstreams *)
val join_write_bitstreams :
  bitstream_write list -> bitstream_write

(**/**)

(* for debug only *)
val input_in_bitstream : bitstream -> input

