/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "ArgumentCoders.h"
#include "Connection.h"
#include "MessageNames.h"
#include <wtf/Forward.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/text/WTFString.h>

namespace WebCore {
struct ClientOrigin;
struct MessageWithMessagePorts;
}

namespace Messages {
namespace NetworkBroadcastChannelRegistry {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::NetworkBroadcastChannelRegistry;
}

class RegisterChannel {
public:
    using Arguments = std::tuple<WebCore::ClientOrigin, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkBroadcastChannelRegistry_RegisterChannel; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    RegisterChannel(const WebCore::ClientOrigin& origin, const String& name)
        : m_arguments(origin, name)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ClientOrigin&, const String&> m_arguments;
};

class UnregisterChannel {
public:
    using Arguments = std::tuple<WebCore::ClientOrigin, String>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkBroadcastChannelRegistry_UnregisterChannel; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    UnregisterChannel(const WebCore::ClientOrigin& origin, const String& name)
        : m_arguments(origin, name)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ClientOrigin&, const String&> m_arguments;
};

class PostMessage {
public:
    using Arguments = std::tuple<WebCore::ClientOrigin, String, WebCore::MessageWithMessagePorts>;

    static IPC::MessageName name() { return IPC::MessageName::NetworkBroadcastChannelRegistry_PostMessage; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NetworkBroadcastChannelRegistry_PostMessageReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    PostMessage(const WebCore::ClientOrigin& origin, const String& name, const WebCore::MessageWithMessagePorts& message)
        : m_arguments(origin, name, message)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::ClientOrigin&, const String&, const WebCore::MessageWithMessagePorts&> m_arguments;
};

} // namespace NetworkBroadcastChannelRegistry
} // namespace Messages
