﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/budgets/BudgetsRequest.h>
#include <aws/budgets/Budgets_EXPORTS.h>
#include <aws/budgets/model/Notification.h>
#include <aws/budgets/model/Subscriber.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Budgets {
namespace Model {

/**
 * <p> Request of CreateNotification </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/budgets-2016-10-20/CreateNotificationRequest">AWS
 * API Reference</a></p>
 */
class CreateNotificationRequest : public BudgetsRequest {
 public:
  AWS_BUDGETS_API CreateNotificationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateNotification"; }

  AWS_BUDGETS_API Aws::String SerializePayload() const override;

  AWS_BUDGETS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>accountId</code> that is associated with the budget that you want
   * to create a notification for.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  CreateNotificationRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the budget that you want Amazon Web Services to notify you about.
   * Budget names must be unique within an account.</p>
   */
  inline const Aws::String& GetBudgetName() const { return m_budgetName; }
  inline bool BudgetNameHasBeenSet() const { return m_budgetNameHasBeenSet; }
  template <typename BudgetNameT = Aws::String>
  void SetBudgetName(BudgetNameT&& value) {
    m_budgetNameHasBeenSet = true;
    m_budgetName = std::forward<BudgetNameT>(value);
  }
  template <typename BudgetNameT = Aws::String>
  CreateNotificationRequest& WithBudgetName(BudgetNameT&& value) {
    SetBudgetName(std::forward<BudgetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The notification that you want to create.</p>
   */
  inline const Notification& GetNotification() const { return m_notification; }
  inline bool NotificationHasBeenSet() const { return m_notificationHasBeenSet; }
  template <typename NotificationT = Notification>
  void SetNotification(NotificationT&& value) {
    m_notificationHasBeenSet = true;
    m_notification = std::forward<NotificationT>(value);
  }
  template <typename NotificationT = Notification>
  CreateNotificationRequest& WithNotification(NotificationT&& value) {
    SetNotification(std::forward<NotificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of subscribers that you want to associate with the notification. Each
   * notification can have one SNS subscriber and up to 10 email subscribers.</p>
   */
  inline const Aws::Vector<Subscriber>& GetSubscribers() const { return m_subscribers; }
  inline bool SubscribersHasBeenSet() const { return m_subscribersHasBeenSet; }
  template <typename SubscribersT = Aws::Vector<Subscriber>>
  void SetSubscribers(SubscribersT&& value) {
    m_subscribersHasBeenSet = true;
    m_subscribers = std::forward<SubscribersT>(value);
  }
  template <typename SubscribersT = Aws::Vector<Subscriber>>
  CreateNotificationRequest& WithSubscribers(SubscribersT&& value) {
    SetSubscribers(std::forward<SubscribersT>(value));
    return *this;
  }
  template <typename SubscribersT = Subscriber>
  CreateNotificationRequest& AddSubscribers(SubscribersT&& value) {
    m_subscribersHasBeenSet = true;
    m_subscribers.emplace_back(std::forward<SubscribersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_budgetName;

  Notification m_notification;

  Aws::Vector<Subscriber> m_subscribers;
  bool m_accountIdHasBeenSet = false;
  bool m_budgetNameHasBeenSet = false;
  bool m_notificationHasBeenSet = false;
  bool m_subscribersHasBeenSet = false;
};

}  // namespace Model
}  // namespace Budgets
}  // namespace Aws
