/*
 * Copyright (c) 2020, Andreas Kling <kling@serenityos.org>
 *
 * SPDX-License-Identifier: BSD-2-Clause
 */

#pragma once

#include <AK/Forward.h>
#include <AK/NonnullOwnPtr.h>
#include <AK/WeakPtr.h>
#include <Kernel/API/KeyCode.h>
#include <LibGUI/Forward.h>
#include <LibGfx/Forward.h>
#include <LibWeb/Forward.h>
#include <LibWeb/Page/EditEventHandler.h>

namespace Web {

class EventHandler {
public:
    explicit EventHandler(Badge<HTML::BrowsingContext>, HTML::BrowsingContext&);
    ~EventHandler();

    bool handle_mouseup(Gfx::IntPoint const&, unsigned button, unsigned buttons, unsigned modifiers);
    bool handle_mousedown(Gfx::IntPoint const&, unsigned button, unsigned buttons, unsigned modifiers);
    bool handle_mousemove(Gfx::IntPoint const&, unsigned buttons, unsigned modifiers);
    bool handle_mousewheel(Gfx::IntPoint const&, unsigned button, unsigned buttons, unsigned modifiers, int wheel_delta_x, int wheel_delta_y);
    bool handle_doubleclick(Gfx::IntPoint const&, unsigned button, unsigned buttons, unsigned modifiers);

    bool handle_keydown(KeyCode, unsigned modifiers, u32 code_point);
    bool handle_keyup(KeyCode, unsigned modifiers, u32 code_point);

    void set_mouse_event_tracking_layout_node(Layout::Node*);

    void set_edit_event_handler(NonnullOwnPtr<EditEventHandler> value) { m_edit_event_handler = move(value); }

private:
    bool focus_next_element();
    bool focus_previous_element();

    bool fire_keyboard_event(FlyString const& event_name, HTML::BrowsingContext& browsing_context, KeyCode key, unsigned modifiers, u32 code_point);

    Layout::InitialContainingBlock* layout_root();
    Layout::InitialContainingBlock const* layout_root() const;

    Painting::PaintableBox* paint_root();
    Painting::PaintableBox const* paint_root() const;

    HTML::BrowsingContext& m_browsing_context;

    bool m_in_mouse_selection { false };

    WeakPtr<Layout::Node> m_mouse_event_tracking_layout_node;

    NonnullOwnPtr<EditEventHandler> m_edit_event_handler;

    WeakPtr<DOM::EventTarget> m_mousedown_target;
};

}
