#!/usr/bin/env python
# License: GPLv3 Copyright: 2024, Kovid Goyal <kovid at kovidgoyal.net>

from enum import IntEnum
from typing import NamedTuple


class Type(IntEnum):
    boolean: int = 1
    index: int = 2
    hidden: int = 3


class FeatureDefinition(NamedTuple):
    name: str
    type: Type

# From: https://learn.microsoft.com/en-ca/typography/opentype/spec/featurelist
known_features: dict[str, FeatureDefinition] = {  # {{{
    'aalt':	FeatureDefinition('Access All Alternates', Type.index),
    'abvf':	FeatureDefinition('Above-base Forms', Type.hidden),
    'abvm':	FeatureDefinition('Above-base Mark Positioning', Type.hidden),
    'abvs':	FeatureDefinition('Above-base Substitutions', Type.hidden),
    'afrc':	FeatureDefinition('Alternative Fractions', Type.boolean),
    'akhn':	FeatureDefinition('Akhand', Type.hidden),
    'blwf':	FeatureDefinition('Below-base Forms', Type.hidden),
    'blwm':	FeatureDefinition('Below-base Mark Positioning', Type.hidden),
    'blws':	FeatureDefinition('Below-base Substitutions', Type.hidden),
    'calt':	FeatureDefinition('Contextual Alternates', Type.boolean),
    'case':	FeatureDefinition('Case-Sensitive Forms', Type.hidden),
    'ccmp':	FeatureDefinition('Glyph Composition / Decomposition', Type.hidden),
    'cfar':	FeatureDefinition('Conjunct Form After Ro', Type.hidden),
    'chws':	FeatureDefinition('Contextual Half-width Spacing', Type.boolean),
    'cjct':	FeatureDefinition('Conjunct Forms', Type.hidden),
    'clig':	FeatureDefinition('Contextual Ligatures', Type.boolean),
    'cpct':	FeatureDefinition('Centered CJK Punctuation', Type.boolean),
    'cpsp':	FeatureDefinition('Capital Spacing', Type.boolean),
    'cswh':	FeatureDefinition('Contextual Swash', Type.boolean),
    'curs':	FeatureDefinition('Cursive Positioning', Type.hidden),
    'c2pc':	FeatureDefinition('Petite Capitals From Capitals', Type.boolean),
    'c2sc':	FeatureDefinition('Small Capitals From Capitals', Type.boolean),
    'dist':	FeatureDefinition('Distances', Type.hidden),
    'dlig':	FeatureDefinition('Discretionary Ligatures', Type.boolean),
    'dnom':	FeatureDefinition('Denominators', Type.hidden),
    'dtls':	FeatureDefinition('Dotless Forms', Type.hidden),
    'expt':	FeatureDefinition('Expert Forms', Type.boolean),
    'falt':	FeatureDefinition('Final Glyph on Line Alternates', Type.boolean),
    'fin2':	FeatureDefinition('Terminal Forms #2', Type.hidden),
    'fin3':	FeatureDefinition('Terminal Forms #3', Type.hidden),
    'fina':	FeatureDefinition('Terminal Forms', Type.hidden),
    'flac':	FeatureDefinition('Flattened accent forms', Type.hidden),
    'frac':	FeatureDefinition('Fractions', Type.boolean),
    'fwid':	FeatureDefinition('Full Widths', Type.boolean),
    'half':	FeatureDefinition('Half Forms', Type.hidden),
    'haln':	FeatureDefinition('Halant Forms', Type.hidden),
    'halt':	FeatureDefinition('Alternate Half Widths', Type.boolean),
    'hist':	FeatureDefinition('Historical Forms', Type.boolean),
    'hkna':	FeatureDefinition('Horizontal Kana Alternates', Type.boolean),
    'hlig':	FeatureDefinition('Historical Ligatures', Type.boolean),
    'hngl':	FeatureDefinition('Hangul', Type.boolean),
    'hojo':	FeatureDefinition('Hojo Kanji Forms (JIS X 0212-1990 Kanji Forms)', Type.boolean),
    'hwid':	FeatureDefinition('Half Widths', Type.boolean),
    'init':	FeatureDefinition('Initial Forms', Type.hidden),
    'isol':	FeatureDefinition('Isolated Forms', Type.hidden),
    'ital':	FeatureDefinition('Italics', Type.boolean),
    'jalt':	FeatureDefinition('Justification Alternates', Type.boolean),
    'jp78':	FeatureDefinition('JIS78 Forms', Type.boolean),
    'jp83':	FeatureDefinition('JIS83 Forms', Type.boolean),
    'jp90':	FeatureDefinition('JIS90 Forms', Type.boolean),
    'jp04':	FeatureDefinition('JIS2004 Forms', Type.boolean),
    'kern':	FeatureDefinition('Kerning', Type.boolean),
    'lfbd':	FeatureDefinition('Left Bounds', Type.boolean),
    'liga':	FeatureDefinition('Standard Ligatures', Type.boolean),
    'ljmo':	FeatureDefinition('Leading Jamo Forms', Type.hidden),
    'lnum':	FeatureDefinition('Lining Figures', Type.boolean),
    'locl':	FeatureDefinition('Localized Forms', Type.hidden),
    'ltra':	FeatureDefinition('Left-to-right alternates', Type.hidden),
    'ltrm':	FeatureDefinition('Left-to-right mirrored forms', Type.hidden),
    'mark':	FeatureDefinition('Mark Positioning', Type.hidden),
    'med2':	FeatureDefinition('Medial Forms #2', Type.hidden),
    'medi':	FeatureDefinition('Medial Forms', Type.hidden),
    'mgrk':	FeatureDefinition('Mathematical Greek', Type.boolean),
    'mkmk':	FeatureDefinition('Mark to Mark Positioning', Type.hidden),
    'mset':	FeatureDefinition('Mark Positioning via Substitution', Type.hidden),
    'nalt':	FeatureDefinition('Alternate Annotation Forms', Type.index),
    'nlck':	FeatureDefinition('NLC Kanji Forms', Type.boolean),
    'nukt':	FeatureDefinition('Nukta Forms', Type.hidden),
    'numr':	FeatureDefinition('Numerators', Type.hidden),
    'onum':	FeatureDefinition('Oldstyle Figures', Type.boolean),
    'opbd':	FeatureDefinition('Optical Bounds', Type.boolean),
    'ordn':	FeatureDefinition('Ordinals', Type.boolean),
    'ornm':	FeatureDefinition('Ornaments', Type.index),
    'palt':	FeatureDefinition('Proportional Alternate Widths', Type.boolean),
    'pcap':	FeatureDefinition('Petite Capitals', Type.boolean),
    'pkna':	FeatureDefinition('Proportional Kana', Type.boolean),
    'pnum':	FeatureDefinition('Proportional Figures', Type.boolean),
    'pref':	FeatureDefinition('Pre-Base Forms', Type.hidden),
    'pres':	FeatureDefinition('Pre-base Substitutions', Type.hidden),
    'pstf':	FeatureDefinition('Post-base Forms', Type.hidden),
    'psts':	FeatureDefinition('Post-base Substitutions', Type.hidden),
    'pwid':	FeatureDefinition('Proportional Widths', Type.boolean),
    'qwid':	FeatureDefinition('Quarter Widths', Type.boolean),
    'rand':	FeatureDefinition('Randomize', Type.boolean),
    'rclt':	FeatureDefinition('Required Contextual Alternates', Type.hidden),
    'rkrf':	FeatureDefinition('Rakar Forms', Type.hidden),
    'rlig':	FeatureDefinition('Required Ligatures', Type.hidden),
    'rphf':	FeatureDefinition('Reph Forms', Type.hidden),
    'rtbd':	FeatureDefinition('Right Bounds', Type.boolean),
    'rtla':	FeatureDefinition('Right-to-left alternates', Type.hidden),
    'rtlm':	FeatureDefinition('Right-to-left mirrored forms', Type.hidden),
    'ruby':	FeatureDefinition('Ruby Notation Forms', Type.boolean),
    'rvrn':	FeatureDefinition('Required Variation Alternates', Type.hidden),
    'salt':	FeatureDefinition('Stylistic Alternates', Type.index),
    'sinf':	FeatureDefinition('Scientific Inferiors', Type.boolean),
    'size':	FeatureDefinition('Optical size', Type.hidden),
    'smcp':	FeatureDefinition('Small Capitals', Type.boolean),
    'smpl':	FeatureDefinition('Simplified Forms', Type.boolean),
    'ssty':	FeatureDefinition('Math script style alternates', Type.hidden),
    'stch':	FeatureDefinition('Stretching Glyph Decomposition', Type.hidden),
    'subs':	FeatureDefinition('Subscript', Type.boolean),
    'sups':	FeatureDefinition('Superscript', Type.boolean),
    'swsh':	FeatureDefinition('Swash', Type.index),
    'titl':	FeatureDefinition('Titling', Type.boolean),
    'tjmo':	FeatureDefinition('Trailing Jamo Forms', Type.hidden),
    'tnam':	FeatureDefinition('Traditional Name Forms', Type.boolean),
    'tnum':	FeatureDefinition('Tabular Figures', Type.boolean),
    'trad':	FeatureDefinition('Traditional Forms', Type.index),
    'twid':	FeatureDefinition('Third Widths', Type.boolean),
    'unic':	FeatureDefinition('Unicase', Type.boolean),
    'valt':	FeatureDefinition('Alternate Vertical Metrics', Type.boolean),
    'vatu':	FeatureDefinition('Vattu Variants', Type.hidden),
    'vchw':	FeatureDefinition('Vertical Contextual Half-width Spacing', Type.hidden),
    'vert':	FeatureDefinition('Vertical Writing', Type.boolean),
    'vhal':	FeatureDefinition('Alternate Vertical Half Metrics', Type.boolean),
    'vjmo':	FeatureDefinition('Vowel Jamo Forms', Type.hidden),
    'vkna':	FeatureDefinition('Vertical Kana Alternates', Type.boolean),
    'vkrn':	FeatureDefinition('Vertical Kerning', Type.boolean),
    'vpal':	FeatureDefinition('Proportional Alternate Vertical Metrics', Type.boolean),
    'vrt2':	FeatureDefinition('Vertical Alternates and Rotation', Type.boolean),
    'vrtr':	FeatureDefinition('Vertical Alternates for Rotation', Type.boolean),
    'zero':	FeatureDefinition('Slashed Zero', Type.boolean),
}
for i in range(1, 100):
    known_features[f'cv{i:02d}'] = FeatureDefinition(f'Character Variant {i}', Type.index)
for i in range(1, 20):
    known_features[f'ss{i:02d}'] = FeatureDefinition(f'Stylistic Set {i}', Type.boolean)
# }}}


