%define OT_PointToFieldFunction_doc
"Function mapping a point into a field.

Parameters
----------
inputDim : int, :math:`\\geq 1`
    Dimension :math:`\\inputDim` of the input vector
outputMesh : :class:`~openturns.Mesh`
    The output mesh
outputDim : int, :math:`\\geq 1`
    Dimension :math:`d'` of the output field

Notes
-----
A PointToFieldFunction acts on points to produce fields:

.. math::

    f: \\left| \\begin{array}{rcl}
               \\Rset^{d} &  \\rightarrow & \\cM_{N'} \\times (\\Rset^{d'})^{N'}  \\\\
                \\vect{v} & \\mapsto  & F' 
              \\end{array} \\right.

with  :math:`\\cM_{N'}` a mesh of :math:`\\cD' \\subset \\Rset^{n'}`.

A field is represented by a collection :math:`(\\vect{t}'_i, \\vect{v}'_i)_{1 \\leq i \\leq N'}` of elements of :math:`\\cM_{N'} \\times (\\Rset^{d'})^{N'}` where :math:`\\vect{t}'_i` is a vertex of :math:`\\cM_{N'}` and :math:`\\vect{v}'_i` the associated value in :math:`\\Rset^{d'}`.

The two first constructors build an object which evaluation operator is not defined (it throws a *NotYetImplementedException*).
The instantiation of such an object is used to extract an actual :class:`~openturns.PointToFieldFunction` from a  :class:`~openturns.Study`.

See also
--------
PythonPointToFieldFunction, OpenTURNSPythonPointToFieldFunction, FieldToPointFunction 

Examples
--------
Use the class :class:`~openturns.OpenTURNSPythonPointToFieldFunction` to create a function that acts a vector :math:`\\vect{v}` of dimension :math:`d=2` and returns a field defined by:

    - the mesh that discretizes :math:`[0, 1]` into 10 regular intervals of length 0.1 (:math:`n=1`)
    - the value associated to the vertex number :math:`i` is :math:`\\vect{v}'_i = i*\\vect{v}` (:math:`d'=2`)

Using the class :class:`~openturns.OpenTURNSPythonPointToFieldFunction` allows one to define a persistent state between the evaluations of the function.

For example, we create the function which maps the point :math:`\\vect{x}\\in \\Rset^2` into the field where the output values are :math:`(\\vect{O}, \\vect{x}, 2\\vect{x}, \\dots, 10\\vect{x})` on the regular grid :math:`(0, 0.1, \\dots, 1.0)`.

>>> import openturns as ot
>>> class FUNC(ot.OpenTURNSPythonPointToFieldFunction):
...     def __init__(self):
...         mesh = ot.RegularGrid(0.0, 0.1, 11)
...         super(FUNC, self).__init__(2, mesh, 2)
...         self.setInputDescription(['R', 'S'])
...         self.setOutputDescription(['T', 'U'])
...     def _exec(self, X):
...         size = self.getOutputMesh().getVerticesNumber()
...         Y = [ot.Point(X)*i for i in range(size)]
...         return Y
>>> F = FUNC()

Create the associated PointToFieldFunction: 

>>> myFunc = ot.PointToFieldFunction(F)

It is also possible to create a PointToFieldFunction from a python function as follows: 

>>> mesh = ot.RegularGrid(0.0, 0.1, 11)
>>> def myPyFunc(X):
...     size = 11
...     Y = [ot.Point(X)*i for i in range(size)]
...     return Y
>>> inputDim = 2
>>> outputDim = 2
>>> myFunc = ot.PythonPointToFieldFunction(inputDim, mesh, outputDim, myPyFunc)

Evaluation the function on a vector:

>>> Yfield = myFunc([1.1, 2.2])
"
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation
OT_PointToFieldFunction_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getCallsNumber_doc
"Get the number of calls of the function.

Returns
-------
callsNumber : int
    Counts the number of times the function has been called since its
    creation."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getCallsNumber
OT_PointToFieldFunction_getCallsNumber_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getInputDescription_doc
"Get the description of the input vector.

Returns
-------
inputDescription : :class:`~openturns.Description`
    Description of the input vector."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getInputDescription
OT_PointToFieldFunction_getInputDescription_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getInputDimension_doc
"Get the dimension of the input vector.

Returns
-------
d : int
    Dimension :math:`d` of the input vector."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getInputDimension
OT_PointToFieldFunction_getInputDimension_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getMarginal_doc
"Get the marginal(s) at given indice(s).

Parameters
----------
i : int or list of ints, :math:`0 \\leq i < d`
    Indice(s) of the marginal(s) to be extracted.

Returns
-------
function : :class:`~openturns.PointToFieldFunction`
    The initial function restricted to the concerned marginal(s) at the indice(s)
    :math:`i`."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getMarginal
OT_PointToFieldFunction_getMarginal_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getOutputMesh_doc
"Get the output mesh.

Returns
-------
outputMesh : :class:`~openturns.Mesh`
    The mesh :math:`\\cM_{N'}` of the output field."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getOutputMesh
OT_PointToFieldFunction_getOutputMesh_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getOutputDescription_doc
"Get the description of the output field values.

Returns
-------
outputDescription : :class:`~openturns.Description`
    Description of the output field values."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getOutputDescription
OT_PointToFieldFunction_getOutputDescription_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_getOutputDimension_doc
"Get the dimension of the output field values.

Returns
-------
d' : int
    Dimension :math:`d'` of the output field values."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::getOutputDimension
OT_PointToFieldFunction_getOutputDimension_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_setInputDescription_doc
"Set the description of the input vector.

Parameters
----------
inputDescription : sequence of str
    Description of the input vector."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::setInputDescription
OT_PointToFieldFunction_setInputDescription_doc

// ---------------------------------------------------------------------

%define OT_PointToFieldFunction_setOutputDescription_doc
"Set the description of the output field values.

Parameters
----------
outputDescription : sequence of str
    Description of the output field values."
%enddef
%feature("docstring") OT::PointToFieldFunctionImplementation::setOutputDescription
OT_PointToFieldFunction_setOutputDescription_doc
