%feature("docstring") OT::LinearModelAnalysis
"Analyse a linear model.

Parameters
----------
linearModelResult : :class:`~openturns.LinearModelResult`
    A linear model result.

See Also
--------
LinearModelResult

Notes
-----
This class relies on a linear model result structure and analyses the results.

By default, on graphs, labels of the 3 most significant points are displayed.
This number can be changed by modifying the :class:`~openturns.ResourceMap` key
(``LinearModelAnalysis-IdentifiersNumber``).

The class has a pretty-print method which is triggered by
the `print()` function.
This prints the following results, where we focus on the properties
of a satisfactory regression model.

- Each row of the table of coefficients tests if one single coefficient is zero.
  For a single coefficient, if the p-value of the T-test is close to zero,
  we can reject the hypothesis that this coefficient is zero. See :meth:`getCoefficientsTScores`
  to get the scores
  and :meth:`getCoefficientsPValues` to get  the related p-values.
- The :math:`R^2` score measures how the predicted output values are close to the
  observed values.
  If the :math:`R^2` is close to 1 (e.g. larger than 0.95), then the predictions are
  accurate on average. See :meth:`~openturns.LinearModelResult.getRSquared`.
  Furthermore, the adjusted :math:`R^2` value, denoted by :math:`R_{ad}^2`, takes into account the data set
  size and the number of hyperparameters. See :meth:`~openturns.LinearModelResult.getAdjustedRSquared`.
- The Fisher-test tests if all the coefficients are simultaneously zero.
  If the p-value is close to zero, then we can reject this hypothesis:
  there is at least one nonzero coefficient. See :meth:`getFisherScore` to get the scores
  and :meth:`getFisherPValue` to get related the p-values.
- The normality tests check if the residuals are Gaussian.
  The normality assumption can be accepted (or, more precisely, cannot be
  rejected) if the p-value is larger than a threshold (e.g. 0.05). See 
  :meth:`getNormalityTestCramerVonMises`, :meth:`getNormalityTestResultAndersonDarling`,
  :meth:`getNormalityTestResultChiSquared()` and :meth:`getNormalityTestResultKolmogorovSmirnov`.

The basics on regression theory are presented in :ref:`regression_analysis`.
The goodness of fit tests for normality are presented in :ref:`graphical_fitting_test`, 
:ref:`chi2_fitting_test`, :ref:`kolmogorov_smirnov_test`, :ref:`cramer_vonmises_test`  and
:ref:`anderson_darling_test`.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal()
>>> Xsample = distribution.getSample(30)
>>> func = ot.SymbolicFunction(['x'], ['2 * x + 1'])
>>> Ysample = func(Xsample) + ot.Normal().getSample(30)
>>> algo = ot.LinearModelAlgorithm(Ysample, Xsample)
>>> result = algo.getResult()
>>> analysis = ot.LinearModelAnalysis(result)
>>> # print(analysis)  # Pretty-print"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getLinearModelResult
"Accessor to the linear model result.

Returns
-------
linearModelResult : :class:`~openturns.LinearModelResult`
    The  linear model result which has been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsTScores
"Accessor to the coefficients of linear expansion over their standard error.

Returns
-------
tScores : :class:`~openturns.Point`
   The Student score of each coefficient estimate :math:`\\hat{\\vect{a}}`.

Notes
-----
The T-test checks if the coefficient :math`\\hat{a}_k` is statistically different from zero and is used under the Gaussian assumption of the error :math:`\\varepsilon`. See [rawlings2001]_ (section 4.5.2 *Special cases of the general form* page 122) for more details.

For each coefficient estimate :math:`\\hat{\\vect{a}}`, the Student score :math:`t_k` is computed as: 

.. math::
   :label: tScores

   t_k = \\dfrac{\\hat{a}_k}{\\sigma(a_k)}

where :math:`\\sigma(a_k)` is the standard deviation of the distribution of the estimator :math:`\\hat{a}_k`
defined in :eq:`std_dev_estim`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsPValues
"Accessor to the coefficients of the p values.

Returns
-------
pValues : :class:`~openturns.Point`
    Student P-values of the coefficient estimates.

Notes
-----
The T-test checks if the coefficient :math`\\hat{a}_k` is statistically different from zero  and is used under the Gaussian assumption of the error :math:`\\varepsilon`.

The p-values of each coefficient estimate is computed from the t-scores defined in :eq:`tScores` with
respect to the :class:`~openturns.Student` distribution
with :math:`dof` degrees of freedom defined in :eq:`dofWithBase` or :eq:`dofNoBase`.

These p-values are used under the Gaussian assumption of the error :math:`\\varepsilon`. If the p-value is close to zero, we can
reject the hypothesis that this coefficient is zero.

If the residuals are not Gaussian, this test is not appropriate and should not be used.
"
// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsConfidenceInterval
"Accessor to the confidence interval of level :math:`\\alpha` for the coefficients
of the linear expansion.

Parameters
----------
alpĥa : float, :math:`0 \\leq \\alpha \\leq 1`
    The confidence level :math:`\\alpha`.

Returns
-------
confidenceInterval : :class:`~openturns.Interval`
    The confidence interval.

Notes
-----
Under the Gaussian assumption of the error, the confidence interval of the coefficient :math:`a_k` of level :math:`\\alpha` is defined by:

.. math::
   \\left[\\hat{a}_k \\pm \\sqrt{\\left((\\mat{\\Psi}^t\\mat{\\Psi})^{-1} \\right)_{k+1, k+1}}\\hat{\\sigma} t_{(1+\\alpha)/2}\\right]

where:

- :math:`t_{(1+\\alpha)/2}` is the quantile of order :math:`(1+\\alpha)/2` of the Student(:math:`dof`) distribution,
- with :math:`dof` the degrees of freedom defined in :eq:`dofNoBase` or :eq:`dofWithBase`,
- :math:`\\mat{\\Psi}` the design matrix defined in :eq:`designMatDef` or :eq:`designMatDef2`.

The interval returned is multivariate and contains the intervals of all the coefficients.

If the residuals are not Gaussian, this test is not appropriate and should not be used.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getFisherScore
"Accessor to the Fisher statistics.

Returns
-------
fisherScore : float
    The Fisher score of the model.

Notes
-----
The Fisher-test tests if all the coefficients are simultaneously equal to zero  and is used under the Gaussian assumption of the error :math:`\\varepsilon`.

The Fisher score is computed as follows. Let be :math:`dofM` the degrees of freedom of the model, equal to the number of coefficients to estimate (:math:`p+1` of :math:`p'`). If the basis contains an intercept, then we subtract 1 from :math:`dofM`.

Let :math:`dof` be the degrees of freedom defined in :eq:`dofNoBase` or :eq:`dofWithBase`.

Let *SSR* be the Sum of Squared Residuals (sometimes called *SSE* as Sum of Squared Errors) defined by: 

.. math::
   SSR = \\sum_{i=1}^\\sampleSize \\varepsilon_i^2

Let *SST* be the Sum of Squared Total defined by: 

.. math::
   SST = \\left| \\begin{array}{ll}
       \\sum_{i=1}^\\sampleSize (Y_i - \\bar{Y}_i)^2 & \\mbox{if the basis contains an intercept} \\\\ 
       \\sum_{i=1}^\\sampleSize Y_i^2               & \\mbox{if not.}
                \\end{array}
         \\right.

where :math:`\\bar{Y}_i = \\dfrac{1}{\\sampleSize} Y_i`.

We denote by *SSM* the Sum of Squared Model defined by:

.. math::
   SSM = SST - SSR

Then, the Fisher score :math:`f` is defined by:

.. math::
   :label: FisherScore

   f = \\dfrac{SSM/dofM}{SSE/dof}
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getFisherPValue
"Accessor to the Fisher p-values.

Returns
-------
fisherPValue : float
    Fisher P-value of the model estimate.

Notes
-----
The F-test tests if all the coefficients are simultaneously equal to zero  and is used under the Gaussian assumption of the error :math:`\\varepsilon`.

The Fisher p-value of each coefficient estimate is computed from the Fisher score defined in :eq:`FisherScore` with
respect to the :class:`~openturns.FisherSnedecor` distribution parameterized by :math:`(dofM, dof)` where:

- :math:`dofM` is the degrees of freedom of the model (equal to the number of coefficients to estimate (:math:`p+1` of :math:`p'`). If the basis contains an intercept, then we subtract 1 from :math:`dofM`.
- :math:`dof` is the degrees of freedom defined in :eq:`dofNoBase` or :eq:`dofWithBase`.

This p-value is used under the Gaussian assumption of the error :math:`\\varepsilon`. It tests
if all the coefficients are statistically useful to the model. If the p-value is close to zero, then we can reject this hypothesis: *there is at least one nonzero coefficient*.

If the residuals are not Gaussian, this test is not appropriate and should not be used." 

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getResidualsStandardError
"Accessor to the standard error of the residuals.

Returns
-------
stdError : float
    The residuals standard deviation estimate.

Notes
-----
The standard error is also called the *root mean squared error* or the *standard error of regression*. It is
the residual standard deviation :math:`\\hat{\\sigma}` defined in :eq:`estimSigma2Noise` which is the unbiaised
residuals variance.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultChiSquared
"Performs Chi-Square test.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.

Notes
-----
The Chi-Square test is a goodness of fit test which objective is to check the
normality assumption (null hypothesis) of residuals (and thus the model).

Usually, Chi-Square test applies for discrete distributions. Here we rely on
the :meth:`~openturns.FittingTest.ChiSquared` to check the normality.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultKolmogorovSmirnov
"Performs Kolmogorov test.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.

Notes
-----
We check if the residuals are Gaussian  thanks to :meth:`~openturns.FittingTest.Kolmogorov`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultAndersonDarling
"Performs Anderson-Darling test.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check if the residuals are Gaussian  thanks to :meth:`~openturns.NormalityTest.AndersonDarlingNormal`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestCramerVonMises
"Performs Cramer-Von Mises test.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check if the residuals are Gaussian thanks to :meth:`~openturns.NormalityTest.CramerVonMisesNormal`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawModelVsFitted
"Accessor to plot of model versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
The graph plots the sample :math:`(Y_i, \\hat{Y}_i)` where :math:`Y_i` is the real value of experience :math:`i`
and :math:`\\hat{Y}_i` is the value fitted by the linear model, defined in :eq:`fittedValue` or :eq:`fittedValue2`.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::LinearModelAnalysis::drawResidualsVsFitted
"Accessor to plot of residuals versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
The graph plots the sample :math:`(\\varepsilon_i, \\hat{Y}_i)` where :math:`\\varepsilon_i` is the residual of experience :math:`i` defined in :eq:`residualDef`
and :math:`\\hat{Y}_i` is the value fitted by the linear model, defined in :eq:`fittedValue` or :eq:`fittedValue2`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawScaleLocation
"Accessor to a Scale-Location plot of sqrt(abs(standardized residuals)) versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
The graph plots the sample :math:`(\\hat{Y}_i, \\sqrt{|\\varepsilon_i^{st}|})` where :math:`\\varepsilon_i^{st}` is the standardized residual of experience :math:`i` defined in :eq:`stdRes`
and :math:`\\hat{Y}_i` is the value fitted by the linear model, defined in :eq:`fittedValue` or :eq:`fittedValue2`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawQQplot
"Accessor to plot a Normal quantiles-quantiles plot of standardized residuals.

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
The graph plots the empirical quantiles of the standardized residuals defined in :eq:`stdRes` versus
the quantiles of the Normal distribution with zero mean and unit variance.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawCookDistance
"Accessor to plot of Cook's distances versus row labels.

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
The graph plots the Cook distance of each experience :math:`i` is defined in :eq:`cookDef`.
The Cook's distance measures the impact of every individual data point on the linear regression. See [rawlings2001]_ (section 11.2.1, *Cook's D* page 362) for more details."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawResidualsVsLeverages
"Accessor to plot of residuals versus leverages that adds bands corresponding to Cook's distances of 0.5 and 1.

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
This graph plots the residuals :math:`\\varepsilon_i` defined in :eq:`residualDef` and the leverage :math:`\\ell_i`  of experience :math:`i` defined in :eq:`leverageDef`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawCookVsLeverages
"Accessor to plot of Cook's distances versus leverage/(1-leverage). 

Returns
-------
graph : :class:`~openturns.Graph`

Notes
-----
This graph plots the Cook distance defined in :eq:`cookDef` and the the ration :math:`\\ell_i/(1-\\ell_i)` where :math:`\\ell_i` is the leverage of experience :math:`i` defined in :eq:`leverageDef`."

