pragma Ada_2012;

pragma Style_Checks (Off);
pragma Warnings (Off, "-gnatwu");

with Interfaces.C; use Interfaces.C;
with libfswatch_types_h;
with cmonitor_h;
with Interfaces.C.Strings;
with Interfaces.C.Extensions;
with cevent_h;
with System;
with cfilter_h;

package libfswatch_h is

  -- * Copyright (c) 2014-2022 Enrico M. Crisostomo
  -- *
  -- * This program is free software; you can redistribute it and/or modify it under
  -- * the terms of the GNU General Public License as published by the Free Software
  -- * Foundation; either version 3, or (at your option) any later version.
  -- *
  -- * This program is distributed in the hope that it will be useful, but WITHOUT
  -- * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  -- * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  -- * details.
  -- *
  -- * You should have received a copy of the GNU General Public License along with
  -- * this program.  If not, see <http://www.gnu.org/licenses/>.
  --  

  --*
  -- * @file
  -- * @brief Header of the `libfswatch` library.
  -- *
  -- * This header file defines the API of the `libfswatch` library.
  -- *
  -- * @copyright Copyright (c) 2014-2022 Enrico M. Crisostomo
  -- * @license GNU General Public License v. 3.0
  -- * @author Enrico M. Crisostomo
  -- * @version 1.8.0
  --  

  --*
  --   * The `libfswatch` C API let users create monitor sessions and receive file
  --   * system events matching the specified criteria.  Most API functions return
  --   * a status code of type FSW_STATUS which can take any value specified in
  --   * the error.h header.  A successful API call returns FSW_OK and the last
  --   * error can be obtained calling the fsw_last_error() function.
  --   *
  --   * If the compiler and the C++ library used to build `libfswatch` support the
  --   * thread_local storage specified then this API is thread safe and a
  --   * different state is maintained on a per-thread basis.
  --   *
  --   * Session-modifying API calls (such as fsw_add_path) will take effect the
  --   * next time a monitor is started with fsw_start_monitor.
  --   *
  --   * Currently not all monitors supports being stopped, in which case
  --   * fsw_start_monitor is a non-returning API call.
  --   *
  --   * A basic session needs at least:
  --   *
  --   *   * A path to watch.
  --   *   * A callback to process the events sent by the monitor.
  --   *
  --   * as shown in the next example (error checking code was omitted).
  --   *
  --   *     // Use the default monitor.
  --   *     const FSW_HANDLE handle = fsw_init_session(system_default_monitor_type);
  --   *
  --   *     fsw_add_path(handle, "my/path");
  --   *     fsw_set_callback(handle, my_callback);
  --   *
  --   *     fsw_start_monitor(handle);
  --   *
  --   * A suitable callback function is a function pointer of type
  --   * FSW_CEVENT_CALLBACK, that is it is a function conforming with the
  --   * following signature:
  --   *
  --   *     void c_process_events(fsw_cevent const * const events,
  --   *                           const unsigned int event_num,
  --   *                           void * data);
  --   *
  --   * When a monitor receives change events satisfying all the session criteria,
  --   * the callback is invoked and passed a copy of the events.
  --    

  --*
  --   * This function initializes the `libfswatch` library and must be invoked
  --   * before any other calls to the C or C++ API.  If the function succeeds, it
  --   * returns FSW_OK, otherwise the initialization routine failed and the library
  --   * should not be usable.
  --    

   function fsw_init_library return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:93
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_init_library";

  --*
  --   * This function creates a new monitor session using the specified monitor
  --   * and returns an handle to it.  This function is the `libfswatch` API entry
  --   * point.
  --   *
  --   * @see cmonitor.h for a list of all the available monitors.
  --    

   function fsw_init_session (c_type : cmonitor_h.fsw_monitor_type) return libfswatch_types_h.FSW_HANDLE  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:102
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_init_session";

  --*
  --   * Adds a path to watch to the specified session.  At least one path must be
  --   * added to the current session in order for it to be valid.
  --    

   function fsw_add_path (handle : libfswatch_types_h.FSW_HANDLE; path : Interfaces.C.Strings.chars_ptr) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:108
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_add_path";

  --*
  --   * Adds the specified monitor property.
  --    

   function fsw_add_property
     (handle : libfswatch_types_h.FSW_HANDLE;
      name : Interfaces.C.Strings.chars_ptr;
      value : Interfaces.C.Strings.chars_ptr) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:113
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_add_property";

  --*
  --   * Sets the allow overflow flag of the monitor.  When this flag is set, a
  --   * monitor is allowed to overflow and report it as a change event.
  --    

   function fsw_set_allow_overflow (handle : libfswatch_types_h.FSW_HANDLE; allow_overflow : Extensions.bool) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:119
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_allow_overflow";

  --*
  --   * Sets the callback the monitor invokes when some events are received.  The
  --   * callback must be set in the current session in order for it to be valid.
  --   *
  --   * See cevent.h for the definition of FSW_CEVENT_CALLBACK.
  --    

   function fsw_set_callback
     (handle : libfswatch_types_h.FSW_HANDLE;
      callback : cevent_h.FSW_CEVENT_CALLBACK;
      data : System.Address) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:127
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_callback";

  --*
  --   * Sets the latency of the monitor.  By default, the latency is set to 1 s.
  --    

   function fsw_set_latency (handle : libfswatch_types_h.FSW_HANDLE; latency : double) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:134
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_latency";

  --*
  --   * Determines whether the monitor recursively scans each watched path or not.
  --   * Recursive scanning is an optional feature which could not be implemented
  --   * by all the monitors.  By default, recursive scanning is disabled.
  --    

   function fsw_set_recursive (handle : libfswatch_types_h.FSW_HANDLE; recursive : Extensions.bool) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:141
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_recursive";

  --*
  --   * Determines whether the monitor only watches a directory when performing a
  --   * recursive scan.  By default, a monitor accepts all kinds of files.
  --    

   function fsw_set_directory_only (handle : libfswatch_types_h.FSW_HANDLE; directory_only : Extensions.bool) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:147
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_directory_only";

  --*
  --   * Determines whether a symbolic link is followed or not.  By default, a
  --   * symbolic link are not followed.
  --    

   function fsw_set_follow_symlinks (handle : libfswatch_types_h.FSW_HANDLE; follow_symlinks : Extensions.bool) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:153
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_follow_symlinks";

  --*
  --   * Adds an event type filter to the current session.
  --   *
  --   * See cfilter.h for the definition of fsw_event_type_filter.
  --    

   function fsw_add_event_type_filter (handle : libfswatch_types_h.FSW_HANDLE; event_type : cfilter_h.fsw_event_type_filter) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:161
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_add_event_type_filter";

  --*
  --   * Adds a filter to the current session.  A filter is a regular expression
  --   * that, depending on whether the filter type is exclusion or not, must or
  --   * must not be matched for an event path for the event to be accepted.
  --   *
  --   * See cfilter.h for the definition of fsw_cmonitor_filter.
  --    

   function fsw_add_filter (handle : libfswatch_types_h.FSW_HANDLE; filter : cfilter_h.fsw_cmonitor_filter) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:171
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_add_filter";

  --*
  --   * Starts the monitor if it is properly configured.  Depending on the type of
  --   * monitor this call might return when a monitor is stopped or not.
  --    

   function fsw_start_monitor (handle : libfswatch_types_h.FSW_HANDLE) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:178
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_start_monitor";

  --*
  --   * Stops a running monitor.
  --    

   function fsw_stop_monitor (handle : libfswatch_types_h.FSW_HANDLE) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:183
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_stop_monitor";

  --*
  --   * Checks if a monitor exists and is running.
  --    

   function fsw_is_running (handle : libfswatch_types_h.FSW_HANDLE) return Extensions.bool  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:188
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_is_running";

  --*
  --   * Destroys an existing session and invalidates its handle.
  --    

   function fsw_destroy_session (handle : libfswatch_types_h.FSW_HANDLE) return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:193
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_destroy_session";

  --*
  --   * Gets the last error code.
  --    

   function fsw_last_error return libfswatch_types_h.FSW_STATUS  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:198
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_last_error";

  --*
  --   * Check whether the verbose mode is active.
  --    

   function fsw_is_verbose return Extensions.bool  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:203
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_is_verbose";

  --*
  --   * Set the verbose mode.
  --    

   procedure fsw_set_verbose (verbose : Extensions.bool)  -- D:/M/msys64/ucrt64/include/libfswatch/c/libfswatch.h:208
   with Import => True, 
        Convention => C, 
        External_Name => "fsw_set_verbose";

end libfswatch_h;

pragma Style_Checks (On);
pragma Warnings (On, "-gnatwu");
