/**
 * @license
 * Copyright 2025 Qwen
 * SPDX-License-Identifier: Apache-2.0
 */
import { AuthType } from '@qwen-code/qwen-code-core';
import { CommandKind } from './types.js';
import { AVAILABLE_MODELS_QWEN, getOpenAIAvailableModelFromEnv, } from '../models/availableModels.js';
function getAvailableModelsForAuthType(authType) {
    switch (authType) {
        case AuthType.QWEN_OAUTH:
            return AVAILABLE_MODELS_QWEN;
        case AuthType.USE_OPENAI: {
            const openAIModel = getOpenAIAvailableModelFromEnv();
            return openAIModel ? [openAIModel] : [];
        }
        default:
            // For other auth types, return empty array for now
            // This can be expanded later according to the design doc
            return [];
    }
}
export const modelCommand = {
    name: 'model',
    description: 'Switch the model for this session',
    kind: CommandKind.BUILT_IN,
    action: async (context) => {
        const { services } = context;
        const { config } = services;
        if (!config) {
            return {
                type: 'message',
                messageType: 'error',
                content: 'Configuration not available.',
            };
        }
        const contentGeneratorConfig = config.getContentGeneratorConfig();
        if (!contentGeneratorConfig) {
            return {
                type: 'message',
                messageType: 'error',
                content: 'Content generator configuration not available.',
            };
        }
        const authType = contentGeneratorConfig.authType;
        if (!authType) {
            return {
                type: 'message',
                messageType: 'error',
                content: 'Authentication type not available.',
            };
        }
        const availableModels = getAvailableModelsForAuthType(authType);
        if (availableModels.length === 0) {
            return {
                type: 'message',
                messageType: 'error',
                content: `No models available for the current authentication type (${authType}).`,
            };
        }
        // Trigger model selection dialog
        return {
            type: 'dialog',
            dialog: 'model',
        };
    },
};
//# sourceMappingURL=modelCommand.js.map