# document.py
#
# Copyright 2025 Jamie Gravendeel
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

import logging
from pathlib import Path
from shlex import quote

from gi.repository import Gio, GLib, GObject

from morphosis import supported_formats


class MorphosisDocument(GObject.Object):
    """Document file and all of its data."""

    __gtype_name__ = "MorphosisDocument"

    file = GObject.Property(type=Gio.File)
    name = GObject.Property(type=str, default=_("Document"))
    path = GObject.Property(type=str)
    suffix = GObject.Property(type=str)
    icon = GObject.Property(type=str)

    convert_finished = GObject.Signal(arg_types=[Gio.File])
    invalid_content_type = GObject.Signal()

    def set_from_file(self, file: Gio.File) -> bool:
        """Get information about a document and verify it."""
        content_type = file.query_info(
            Gio.FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE,
            Gio.FileQueryInfoFlags.NONE,
        ).get_content_type()

        name = file.query_info(
            Gio.FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME,
            Gio.FileQueryInfoFlags.NONE,
        ).get_display_name()

        suffix = Path(name).suffix[1:]

        if (
            content_type not in supported_formats.MIME_TYPES
            and suffix not in supported_formats.EXTENSIONS
        ):
            logging.warning("File format %s is not supported", content_type)
            self.emit("invalid-content-type")
            return False

        self.file = file
        self.name = name
        self.path = file.get_path()
        self.suffix = suffix
        self.icon = "text-html" if suffix == "html" else "x-office-document"
        return True

    def convert(
        self,
        output_file: Gio.File,
        format: str,
        font_style: str,
        has_font_options: bool,
    ) -> None:
        """Convert a document."""
        if not (output_path := output_file.get_path()):
            logging.error("Cannot get path for output file")
            return self.emit("convert-finished")

        pandoc_args = " ".join(
            (
                f"--variable=mainfont:{'Serif' if font_style == 'Serif' else 'Sans-Serif'}"
                if has_font_options
                else "",
                f"--to={supported_formats.FORMATS[format].format}",
                "--variable=lang:und",
                "--embed-resources",
                "--standalone",
                "--wrap=preserve",
                "--pdf-engine weasyprint",
            )
        )

        try:
            _success, _stdout, stderr, status = GLib.spawn_command_line_sync(
                f"pandoc {quote(self.path)} {pandoc_args} -o {quote(output_path)}"
            )
            GLib.spawn_check_wait_status(status)
        except GLib.Error:
            logging.error("Cannot convert document: %s", stderr)
            return self.emit("convert-finished")

        return self.emit("convert-finished", output_file)
