/* recipient.c - mainatin recipient sets
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2002, 2003, 2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "util.h"
#include "context.h"
#include "rungpg.h"
#include "ops.h"

gpgme_error_t
gpgme_recipients_new (gpgme_recipients_t * r_rset)
{
    gpgme_recipients_t rset;
    
    if (!r_rset)
	return mk_error (Invalid_Value);
    *r_rset = NULL;
    rset = calloc (1, sizeof *rset);
    if (!rset)
        return mk_error (Out_Of_Core);
    *r_rset = rset;
    return 0;
}


void
gpgme_recipients_release( gpgme_recipients_t rset )
{
    struct user_id_s * u, * u2;

    if (!rset)
	return;
    for (u = rset->list; u; u = u2) {
        u2 = u->next;
	safe_free (u);
    }
    rset->count = 0;
    safe_free (rset);
}


gpgme_error_t
gpgme_recipients_add_name (gpgme_recipients_t rset, const char *name)
{
    int val = GPGME_VALIDITY_UNKNOWN;
    return gpgme_recipients_add_name_with_validity( rset, name, val );
}


gpgme_error_t
gpgme_recipients_add_name_with_validity( gpgme_recipients_t rset,
                                         const char * name,
                                         gpgme_validity_t val )
{
    struct user_id_s * r;
    
    if (!name || !rset)
        return mk_error (Invalid_Value);
    r = malloc (sizeof *r + strlen (name) + 1);
    if (!r)
        return mk_error (Out_Of_Core);
    r->validity = val;
    r->name_part = "";
    r->email_part = "";
    r->comment_part = "";
    strcpy (r->name, name);
    r->next = rset->list;
    rset->list = r;
    rset->count++;
    return 0;
}



unsigned int 
gpgme_recipients_count (const gpgme_recipients_t rset)
{
    struct user_id_s * r;
    unsigned int count = 0;
    
    if (rset) {
        for (r = rset->list ; r; r = r->next)
            count++;
    }
    return count;
}


const char*
gpgme_recipients_get_name (const gpgme_recipients_t rset, int pos)
{
    struct user_id_s *r;

    if (!rset || pos > rset->count)
	return NULL;
    for (r = rset->list; r && pos > 0; r = r->next)
	pos--;
    return r->name;
}


gpgme_error_t
gpgme_recipients_enum_open( const gpgme_recipients_t rset, void **ctx )
{
    if( !rset || !ctx )
        return mk_error( Invalid_Value );
    *ctx = rset->list;
    return 0;
}


const char *
gpgme_recipients_enum_read( const gpgme_recipients_t rset, void **ctx )
{
    struct user_id_s *r;
    
    if( !rset || !ctx )
        return NULL; /* oops */
    
    r = *ctx;
    if( r ) {
        const char *s = r->name;
        r = r->next;
        *ctx = r;
        return s;
    }
    
    return NULL;
}


gpgme_error_t
gpgme_recipients_enum_close( const gpgme_recipients_t rset, void **ctx )
{
    if( !rset || !ctx )
        return mk_error (Invalid_Value);
    *ctx = NULL;
    return 0;
}


void
_gpgme_append_gpg_args_from_recipients (const gpgme_recipients_t rset, 
				        _gpg_object_t gpg)
{
    struct user_id_s *r;

    assert (rset);
    for (r = rset->list ; r; r = r->next) {
        _gpgme_gpg_add_arg (gpg, "-r");
        _gpgme_gpg_add_arg (gpg, r->name);
    }    
}


int
_gpgme_recipients_all_valid( const gpgme_recipients_t rset )
{
    struct user_id_s *r;
    
    assert( rset );
    for ( r = rset->list ; r; r = r->next ) {
        if( r->validity != GPGME_VALIDITY_FULL
            && r->validity != GPGME_VALIDITY_ULTIMATE )
            return 0; /*no*/
    }	
    return 1; /*yes*/
}
