package gnu.crypto.jce;

// --------------------------------------------------------------------------
// $Id: GnuCrypto.java,v 1.12 2003/01/11 06:11:37 raif Exp $
//
// Copyright (C) 2001, 2002, 2003, Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at
// your option) any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the
//
//    Free Software Foundation, Inc.,
//    59 Temple Place, Suite 330,
//    Boston, MA  02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
//
// --------------------------------------------------------------------------

import gnu.crypto.Registry;

import java.security.Provider;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * <p>The GNU Crypto implementation of the Java Cryptographic Extension (JCE)
 * Provider.</p>
 *
 * @version $Revision: 1.12 $
 * @see java.security.Provider
 */
public final class GnuCrypto extends Provider {

   // Constants and variables
   // -------------------------------------------------------------------------

   // Constructor(s)
   // -------------------------------------------------------------------------

   /**
    * <p>The <a href="http://www.gnu.org/software/gnu-crypto/">GNU Crypto</a>
    * Provider.</p>
    */
   public GnuCrypto() {
      super(Registry.GNU_CRYPTO, 1.1, "GNU Crypto JCE Provider");

      // MessageDigest
      put("MessageDigest.MD2", "gnu.crypto.jce.hash.MD2Spi");
      put("MessageDigest.MD2 ImplementedIn", "Software");
      put("MessageDigest.MD4", "gnu.crypto.jce.hash.MD4Spi");
      put("MessageDigest.MD4 ImplementedIn", "Software");
      put("MessageDigest.MD5", "gnu.crypto.jce.hash.MD5Spi");
      put("MessageDigest.MD5 ImplementedIn", "Software");
      put("MessageDigest.RIPEMD128", "gnu.crypto.jce.hash.RipeMD128Spi");
      put("MessageDigest.RIPEMD128 ImplementedIn", "Software");
      put("MessageDigest.RIPEMD160", "gnu.crypto.jce.hash.RipeMD160Spi");
      put("MessageDigest.RIPEMD160 ImplementedIn", "Software");
      put("MessageDigest.SHA-160", "gnu.crypto.jce.hash.Sha160Spi");
      put("MessageDigest.SHA-160 ImplementedIn", "Software");
      put("MessageDigest.WHIRLPOOL", "gnu.crypto.jce.hash.WhirlpoolSpi");
      put("MessageDigest.WHIRLPOOL ImplementedIn", "Software");

      // SecureRandom
      put("SecureRandom.ARCFOUR", "gnu.crypto.jce.prng.ARCFourRandomSpi");
      put("SecureRandom.MD2PRNG", "gnu.crypto.jce.prng.MD2RandomSpi");
      put("SecureRandom.MD2PRNG ImplementedIn", "Software");
      put("SecureRandom.MD4PRNG", "gnu.crypto.jce.prng.MD4RandomSpi");
      put("SecureRandom.MD4PRNG ImplementedIn", "Software");
      put("SecureRandom.MD5PRNG", "gnu.crypto.jce.prng.MD5RandomSpi");
      put("SecureRandom.MD5PRNG ImplementedIn", "Software");
      put("SecureRandom.RIPEMD128PRNG", "gnu.crypto.jce.prng.RipeMD128RandomSpi");
      put("SecureRandom.RIPEMD128PRNG ImplementedIn", "Software");
      put("SecureRandom.RIPEMD160PRNG", "gnu.crypto.jce.prng.RipeMD160RandomSpi");
      put("SecureRandom.RIPEMD160PRNG ImplementedIn", "Software");
      put("SecureRandom.SHA-160PRNG", "gnu.crypto.jce.prng.Sha160RandomSpi");
      put("SecureRandom.SHA-160PRNG ImplementedIn", "Software");
      put("SecureRandom.WHIRLPOOLPRNG", "gnu.crypto.jce.prng.WhirlpoolRandomSpi");
      put("SecureRandom.WHIRLPOOLPRNG ImplementedIn", "Software");
      put("SecureRandom.ICM", "gnu.crypto.jce.prng.ICMRandomSpi");
      put("SecureRandom.ICM ImplementedIn", "Software");
      put("SecureRandom.UMAC-KDF", "gnu.crypto.jce.prng.UMacRandomSpi");
      put("SecureRandom.UMAC-KDF ImplementedIn", "Software");

      // KeyPairGenerator
      put("KeyPairGenerator.DSS", "gnu.crypto.jce.sig.DSSKeyPairGeneratorSpi");
      put("KeyPairGenerator.DSS KeySize", "1024");
      put("KeyPairGenerator.DSS ImplementedIn", "Software");
      put("KeyPairGenerator.RSA", "gnu.crypto.jce.sig.RSAKeyPairGeneratorSpi");
      put("KeyPairGenerator.RSA KeySize", "1024");
      put("KeyPairGenerator.RSA ImplementedIn", "Software");

      // Signature
      put("Signature.DSS/RAW", "gnu.crypto.jce.sig.DSSRawSignatureSpi");
      put("Signature.DSS/RAW KeySize", "1024");
      put("Signature.DSS/RAW ImplementedIn", "Software");
      put("Signature.RSA-PSS/RAW", "gnu.crypto.jce.sig.RSAPSSRawSignatureSpi");
      put("Signature.RSA-PSS/RAW KeySize", "1024");
      put("Signature.RSA-PSS/RAW ImplementedIn", "Software");

      // Cipher
      put("Cipher.ANUBIS",   "gnu.crypto.jce.cipher.AnubisSpi");
      put("Cipher.ANUBIS ImplementedIn", "Software");
      put("Cipher.ARCFOUR",  "gnu.crypto.jce.cipher.ARCFourSpi");
      put("Cipher.ARCFOUR ImplementedIn", "Software");
      put("Cipher.BLOWFISH", "gnu.crypto.jce.cipher.BlowfishSpi");
      put("Cipher.BLOWFISH ImplementedIn", "Software");
      put("Cipher.DES",      "gnu.crypto.jce.cipher.DESSpi");
      put("Cipher.DES ImplementedIn", "Software");
      put("Cipher.KHAZAD",   "gnu.crypto.jce.cipher.KhazadSpi");
      put("Cipher.KHAZAD ImplementedIn", "Software");
      put("Cipher.NULL",     "gnu.crypto.jce.cipher.NullCipherSpi");
      put("Cipher.NULL ImplementedIn", "Software");
      put("Cipher.AES",      "gnu.crypto.jce.cipher.RijndaelSpi");
      put("Cipher.AES ImplementedIn", "Software");
      put("Cipher.RIJNDAEL", "gnu.crypto.jce.cipher.RijndaelSpi");
      put("Cipher.RIJNDAEL ImplementedIn", "Software");
      put("Cipher.SERPENT",  "gnu.crypto.jce.cipher.SerpentSpi");
      put("Cipher.SERPENT ImplementedIn", "Software");
      put("Cipher.SQUARE",   "gnu.crypto.jce.cipher.SquareSpi");
      put("Cipher.SQUARE ImplementedIn", "Software");
      put("Cipher.TRIPLEDES", "gnu.crypto.jce.cipher.TripleDESSpi");
      put("Cipher.TRIPLEDES ImplementedIn", "Software");
      put("Cipher.TWOFISH",  "gnu.crypto.jce.cipher.TwofishSpi");
      put("Cipher.TWOFISH ImplementedIn", "Software");
      put("AlgorithmParameters.BlockCipherParameters",
         "gnu.crypto.jce.params.BlockCipherParameters");

      // MAC
      put("Mac.HMAC-MD2",        "gnu.crypto.jce.mac.HMacMD2Spi");
      put("Mac.HMAC-MD4",        "gnu.crypto.jce.mac.HMacMD4Spi");
      put("Mac.HMAC-MD5",        "gnu.crypto.jce.mac.HMacMD5Spi");
      put("Mac.HMAC-SHA160",     "gnu.crypto.jce.mac.HMacSHA160Spi");
      put("Mac.HMAC-RIPEMD160",  "gnu.crypto.jce.mac.HMacRipeMD160Spi");
      put("Mac.HMAC-RIPEMD128",  "gnu.crypto.jce.mac.HMacRipeMD128Spi");
      put("Mac.HMAC-WHIRLPOOL",  "gnu.crypto.jce.mac.HMacWhirlpoolSpi");
      put("Mac.TMMH16",          "gnu.crypto.jce.mac.TMMH16Spi");
      put("Mac.UHASH32",         "gnu.crypto.jce.mac.UHash32Spi");
      put("Mac.UMAC32",          "gnu.crypto.jce.mac.UMac32Spi");

      // Aliases
      put("Alg.Alias.AlgorithmParameters.AES",      "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.BLOWFISH", "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.ANUBIS",   "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.KHAZAD",   "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.NULL",     "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.RIJNDAEL", "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.SERPENT",  "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.SQUARE",   "BlockCipherParameters");
      put("Alg.Alias.AlgorithmParameters.TWOFISH",  "BlockCipherParameters");
      put("Alg.Alias.Cipher.RC4",                      "ARCFOUR");
      put("Alg.Alias.Cipher.3-DES",                    "TRIPLEDES");
      put("Alg.Alias.Cipher.3DES",                     "TRIPLEDES");
      put("Alg.Alias.Cipher.DES-EDE",                  "TRIPLEDES");
      put("Alg.Alias.Cipher.DESede",                   "TRIPLEDES");
      put("Alg.Alias.MessageDigest.SHS",               "SHA-160");
      put("Alg.Alias.MessageDigest.SHA",               "SHA-160");
      put("Alg.Alias.MessageDigest.SHA1",              "SHA-160");
      put("Alg.Alias.MessageDigest.SHA-1",             "SHA-160");
      put("Alg.Alias.MessageDigest.RIPEMD-160",        "RIPEMD160");
      put("Alg.Alias.MessageDigest.RIPEMD-128",        "RIPEMD128");
      put("Alg.Alias.Mac.HMAC-SHS",                    "HMAC-SHA160");
      put("Alg.Alias.Mac.HMAC-SHA",                    "HMAC-SHA160");
      put("Alg.Alias.Mac.HMAC-SHA1",                   "HMAC-SHA160");
      put("Alg.Alias.Mac.HMAC-SHA-160",                "HMAC-SHA160");
      put("Alg.Alias.Mac.HMAC-RIPEMD-160",             "HMAC-RIPEMD160");
      put("Alg.Alias.Mac.HMAC-RIPEMD-128",             "HMAC-RIPEMD128");
      put("Alg.Alias.SecureRandom.RC4",                "ARCFOUR");
      put("Alg.Alias.SecureRandom.SHA-1PRNG",          "SHA-160PRNG");
      put("Alg.Alias.SecureRandom.SHA1PRNG",           "SHA-160PRNG");
      put("Alg.Alias.SecureRandom.SHAPRNG",            "SHA-160PRNG");
      put("Alg.Alias.KeyPairGenerator.DSA",            "DSS");
      put("Alg.Alias.Signature.DSA",                   "DSS/RAW");
      put("Alg.Alias.Signature.SHAwithDSA",            "DSS/RAW");
      put("Alg.Alias.Signature.SHA1withDSA",           "DSS/RAW");
      put("Alg.Alias.Signature.SHA160withDSA",         "DSS/RAW");
//      put("Alg.Alias.Signature.OID.1.2.840.10040.4.3", "DSS");
//      put("Alg.Alias.Signature.1.2.840.10040.4.3",     "DSS");
//      put("Alg.Alias.Signature.1.3.14.3.2.13",         "DSS");
//      put("Alg.Alias.Signature.1.3.14.3.2.27",         "DSS");
      put("Alg.Alias.Signature.SHA/DSA",               "DSS/RAW");
      put("Alg.Alias.Signature.SHA1/DSA",              "DSS/RAW");
      put("Alg.Alias.Signature.SHA-1/DSA",             "DSS/RAW");
      put("Alg.Alias.Signature.SHA-160/DSA",           "DSS/RAW");
      put("Alg.Alias.Signature.DSAwithSHA",            "DSS/RAW");
      put("Alg.Alias.Signature.DSAwithSHA1",           "DSS/RAW");
      put("Alg.Alias.Signature.DSAwithSHA160",         "DSS/RAW");
      put("Alg.Alias.Signature.RSA-PSS",               "RSA-PSS/RAW");
      put("Alg.Alias.Signature.RSAPSS",                "RSA-PSS/RAW");
   }

   // Class methods
   // -------------------------------------------------------------------------

   /**
    * Returns a {@link Set} of names of message digest algorithms available from
    * this {@link Provider}.<p>
    *
    * @return a {@link Set} of hash names (Strings).
    */
   public static final Set getMessageDigestNames() {
      return gnu.crypto.hash.HashFactory.getNames();
   }

   /**
    * Returns a {@link Set} of names of secure random implementations available
    * from this {@link Provider}.<p>
    *
    * @return a {@link Set} of secure random names (Strings).
    */
   public static final Set getSecureRandomNames() {
      Set result = new HashSet();
      // do all the hash-based prng algorithms
      Set md = gnu.crypto.hash.HashFactory.getNames();
      for (Iterator it = md.iterator(); it.hasNext(); ) {
         result.add(((String) it.next()).toUpperCase() + "PRNG");
      }
      // add ICM and UMAC based generators
      result.add(Registry.ICM_PRNG.toUpperCase());
      result.add(Registry.UMAC_PRNG.toUpperCase());
      result.add(Registry.ARCFOUR_PRNG.toUpperCase());

      return Collections.unmodifiableSet(result);
   }

   /**
    * Returns a {@link Set} of names of keypair generator implementations
    * available from this {@link Provider}.<p>
    *
    * @return a {@link Set} of key pair generator names (Strings).
    */
   public static final Set getKeyPairGeneratorNames() {
      return gnu.crypto.sig.KeyPairGeneratorFactory.getNames();
   }

   /**
    * Returns a {@link Set} of names of signature scheme implementations
    * available from this {@link Provider}.<p>
    *
    * @return a {@link Set} of signature names (Strings).
    */
   public static final Set getSignatureNames() {
      return gnu.crypto.sig.SignatureFactory.getNames();
   }

   /**
    * Returns a {@link Set} of names of symmetric key block cipher algorithms
    * available from this {@link Provider}.<p>
    *
    * @return a {@link Set} of cipher names (Strings).
    */
   public static final Set getCipherNames() {
      HashSet s = new HashSet();
      s.addAll(gnu.crypto.cipher.CipherFactory.getNames());
      s.add(Registry.ARCFOUR_PRNG);
      return s;
   }

   /**
    * Returns a {@link Set} of names of MAC algorithms available from
    * this {@link Provider}.<p>
    *
    * @return a {@link Set} of MAC names (Strings).
    */
   public static final Set getMacNames() {
      return gnu.crypto.mac.MacFactory.getNames();
   }

   // Instance methods
   // -------------------------------------------------------------------------
}
