/*
 * Copyright © 2001 Havoc Pennington
 * Copyright © 2007, 2008 Christian Persch
 *
 * Gnome-terminal is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Gnome-terminal is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>

#include <gtk/gtk.h>

#include <X11/extensions/Xrender.h>
#include <gdk/gdkx.h>

#include <gconf/gconf.h>
#include <libgnome/gnome-util.h> /* gnome_util_user_shell */

#include "terminal-accels.h"
#include "terminal-app.h"
#include "terminal-intl.h"
#include "terminal-marshal.h"
#include "terminal-profile.h"
#include "terminal-screen-container.h"
#include "terminal-util.h"
#include "terminal-window.h"

#define HTTP_PROXY_DIR "/system/http_proxy"

#define URL_MATCH_CURSOR  (GDK_HAND2)
#define SKEY_MATCH_CURSOR (GDK_HAND2)

typedef struct
{
  int tag;
  int flavor;
} TagData;

struct _TerminalScreenPrivate
{
  TerminalWindow *window;
  TerminalProfile *profile; /* may be NULL at times */
  guint profile_changed_id;
  guint profile_forgotten_id;
  char *raw_title, *raw_icon_title;
  char *cooked_title, *cooked_icon_title;
  char *title_from_arg;
  gboolean icon_title_set;
  char **override_command;
  char *working_dir;
  int child_pid;
  double font_scale;
  guint recheck_working_dir_idle;
  gboolean user_title; /* title was manually set */
  GSList *url_tags;
  GSList *skey_tags;
};

enum
{
  PROFILE_SET,
  SHOW_POPUP_MENU,
  SKEY_CLICKED,
  URL_CLICKED,
  CLOSE_SCREEN,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_PROFILE,
  PROP_ICON_TITLE,
  PROP_ICON_TITLE_SET,
  PROP_OVERRIDE_COMMAND,
  PROP_TITLE,
};

enum
{
  TARGET_COLOR,
  TARGET_BGIMAGE,
  TARGET_RESET_BG,
  TARGET_MOZ_URL,
  TARGET_NETSCAPE_URL,
  TARGET_TAB
};

static void terminal_screen_init        (TerminalScreen      *screen);
static void terminal_screen_class_init  (TerminalScreenClass *klass);
static void terminal_screen_dispose     (GObject             *object);
static void terminal_screen_finalize    (GObject             *object);
static void terminal_screen_drag_data_received (GtkWidget        *widget,
                                                GdkDragContext   *context,
                                                gint              x,
                                                gint              y,
                                                GtkSelectionData *selection_data,
                                                guint             info,
                                                guint             time);
static void terminal_screen_system_font_notify_cb (TerminalApp *app,
                                                   GParamSpec *pspec,
                                                   TerminalScreen *screen);
static void terminal_screen_change_font (TerminalScreen *screen);
static gboolean terminal_screen_popup_menu (GtkWidget *widget);
static gboolean terminal_screen_button_press (GtkWidget *widget,
                                              GdkEventButton *event);

static void terminal_screen_window_title_changed      (VteTerminal *vte_terminal,
                                                       TerminalScreen *screen);
static void terminal_screen_icon_title_changed        (VteTerminal *vte_terminal,
                                                       TerminalScreen *screen);

static void terminal_screen_widget_child_died        (GtkWidget      *term,
                                                      TerminalScreen *screen);

static void update_color_scheme                      (TerminalScreen *screen);

static gboolean cook_title  (TerminalScreen *screen, const char *raw_title, char **old_cooked_title);

static void terminal_screen_cook_title      (TerminalScreen *screen);
static void terminal_screen_cook_icon_title (TerminalScreen *screen);

static void queue_recheck_working_dir (TerminalScreen *screen);

static void  terminal_screen_match_add         (TerminalScreen            *screen,
                                                const char           *regexp,
                                                int                   flavor);
static void  terminal_screen_skey_match_add    (TerminalScreen            *screen,
                                                const char           *regexp,
                                                int                   flavor);
static char* terminal_screen_check_match       (TerminalScreen            *screen,
                                                int                   column,
                                                int                   row,
                                                int                  *flavor);
static char* terminal_screen_skey_check_match  (TerminalScreen            *screen,
                                                int                   column,
                                                int                   row,
                                                int                  *flavor);
static void  terminal_screen_skey_match_remove (TerminalScreen            *screen);

static guint signals[LAST_SIGNAL];

G_DEFINE_TYPE (TerminalScreen, terminal_screen, VTE_TYPE_TERMINAL)

static void
free_tag_data (TagData *tagdata)
{
  g_slice_free (TagData, tagdata);
}

#ifdef DEBUG_GEOMETRY
static void
parent_size_request (GtkWidget *scrolled_window, GtkRequisition *req, GtkWidget *screen)
{
  g_print ("screen %p scrolled-window size req %d : %d\n", screen, req->width, req->height);
}
#endif

static void
parent_parent_set_cb (GtkWidget *widget,
                      GtkWidget *old_parent,
                      TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  GtkWidget *toplevel;

  if (widget->parent)
    {
      g_return_if_fail (GTK_IS_NOTEBOOK (widget->parent));

      toplevel = gtk_widget_get_toplevel (widget);
      g_return_if_fail (GTK_WIDGET_TOPLEVEL (toplevel));

      priv->window = TERMINAL_WINDOW (toplevel);
    }
  else
    priv->window = NULL;
}

static void
parent_set_callback (GtkWidget *widget,
                     GtkWidget *old_parent)
{
  if (old_parent)
    g_signal_handlers_disconnect_by_func (old_parent, G_CALLBACK (parent_parent_set_cb), widget);

  if (widget->parent)
    g_signal_connect (widget->parent, "parent-set", G_CALLBACK (parent_parent_set_cb), widget);

#ifdef DEBUG_GEOMETRY
  if (old_parent)
    g_signal_handlers_disconnect_by_func (old_parent, G_CALLBACK (parent_size_request), widget);
  if (widget->parent)
    g_signal_connect (widget->parent, "size-request", G_CALLBACK (parent_size_request), widget);
#endif
}

static void
set_background_image_file (VteTerminal *terminal,
                           const char *fname)
{
  if (fname && fname[0])
    vte_terminal_set_background_image_file (terminal,fname);
  else
    vte_terminal_set_background_image (terminal, NULL);
}

static void
terminal_screen_update_cursor_blink (TerminalScreen *screen,
                                     GtkSettings *settings)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalCursorBlinkMode mode;
  gboolean blink;

  mode = terminal_profile_get_property_enum (priv->profile, TERMINAL_PROFILE_CURSOR_BLINK_MODE);
  if (mode == TERMINAL_CURSOR_BLINK_SYSTEM)
    g_object_get (settings, "gtk-cursor-blink", &blink, NULL);
  else
    blink = (mode == TERMINAL_CURSOR_BLINK_ON);

  vte_terminal_set_cursor_blinks (VTE_TERMINAL (screen), blink);
}

static void
terminal_screen_sync_settings (GtkSettings *settings,
                               GParamSpec *pspec,
                               TerminalScreen *screen)
{
  terminal_screen_update_cursor_blink (screen, settings);
}

static void
terminal_screen_screen_changed (GtkWidget *widget, GdkScreen *previous_screen)
{
  GdkScreen *screen;
  GtkSettings *settings;

  screen = gtk_widget_get_screen (widget);
  if (previous_screen != NULL &&
      screen != previous_screen) {
    settings = gtk_settings_get_for_screen (previous_screen);
    g_signal_handlers_disconnect_matched (settings, G_SIGNAL_MATCH_DATA,
                                          0, 0, NULL, NULL,
                                          widget);
  }

  if (GTK_WIDGET_CLASS (terminal_screen_parent_class)->screen_changed) {
    GTK_WIDGET_CLASS (terminal_screen_parent_class)->screen_changed (widget, previous_screen);
  }

  if (screen == previous_screen || screen == NULL)
    return;

  settings = gtk_widget_get_settings (widget);
  terminal_screen_sync_settings (settings, NULL, TERMINAL_SCREEN (widget));
  g_signal_connect (settings, "notify::gtk-cursor-blink",
                    G_CALLBACK (terminal_screen_sync_settings), widget);
}

static void
terminal_screen_realize (GtkWidget *widget)
{
  TerminalScreen *screen = TERMINAL_SCREEN (widget);
  TerminalScreenPrivate *priv = screen->priv;
  TerminalBackgroundType bg_type;

  GTK_WIDGET_CLASS (terminal_screen_parent_class)->realize (widget);

  g_assert (priv->window != NULL);

  /* FIXME: Don't enable this if we have a compmgr. */
  bg_type = terminal_profile_get_property_enum (priv->profile, TERMINAL_PROFILE_BACKGROUND_TYPE);
  vte_terminal_set_background_transparent (VTE_TERMINAL (screen),
                                           bg_type == TERMINAL_BACKGROUND_TRANSPARENT &&
                                           !terminal_window_uses_argb_visual (priv->window));

  /* FIXME: why do this on realize? */
  terminal_window_set_size (priv->window, screen, TRUE);
}

static void
terminal_screen_style_set (GtkWidget *widget,
                           GtkStyle *previous_style)
{
  TerminalScreen *screen = TERMINAL_SCREEN (widget);
  void (* style_set) (GtkWidget*, GtkStyle*) = GTK_WIDGET_CLASS (terminal_screen_parent_class)->style_set;

  if (style_set)
    style_set (widget, previous_style);

  update_color_scheme (screen);

  if (GTK_WIDGET_REALIZED (widget))
    terminal_screen_change_font (screen);
}

#ifdef DEBUG_GEOMETRY

static void size_request (GtkWidget *widget, GtkRequisition *req)
{
  g_print ("Screen %p size-request %d : %d\n", widget, req->width, req->height);
}

static void size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
  g_print ("Screen %p size-alloc   %d : %d at (%d, %d)\n", widget, allocation->width, allocation->height, allocation->x, allocation->y);
}

#endif

static void
terminal_screen_init (TerminalScreen *screen)
{
  const GtkTargetEntry target_table[] = {
    { "GTK_NOTEBOOK_TAB", GTK_TARGET_SAME_APP, TARGET_TAB },
    { "application/x-color", 0, TARGET_COLOR },
    { "property/bgimage",    0, TARGET_BGIMAGE },
    { "x-special/gnome-reset-background", 0, TARGET_RESET_BG },
    { "text/x-moz-url",  0, TARGET_MOZ_URL },
    { "_NETSCAPE_URL", 0, TARGET_NETSCAPE_URL }
  };
  TerminalScreenPrivate *priv;
  GtkTargetList *target_list;
  GtkTargetEntry *targets;
  int n_targets;

  priv = screen->priv = G_TYPE_INSTANCE_GET_PRIVATE (screen, TERMINAL_TYPE_SCREEN, TerminalScreenPrivate);

  vte_terminal_set_mouse_autohide (VTE_TERMINAL (screen), TRUE);

  priv->working_dir = g_get_current_dir ();
  if (priv->working_dir == NULL) /* shouldn't ever happen */
    priv->working_dir = g_strdup (g_get_home_dir ());
  priv->child_pid = -1;

  priv->recheck_working_dir_idle = 0;

  priv->font_scale = PANGO_SCALE_MEDIUM;

#define USERCHARS "-A-Za-z0-9"
#define PASSCHARS "-A-Za-z0-9,?;.:/!%$^*&~\"#'"
#define HOSTCHARS "-A-Za-z0-9"
#define HOST      "[" HOSTCHARS "]+(\\.[" HOSTCHARS "]+)*"
#define PORT      "(:[0-9]{1,5})?"
#define PATHCHARS "-A-Za-z0-9_$.+!*(),;:@&=?/~#%"
#define SCHEME    "(news:|telnet:|nntp:|file:/|https?:|ftps?:|webcal:)"
#define USER      "[" USERCHARS "]+(:["PASSCHARS "]+)?"
#define URLPATH   "/[" PATHCHARS "]*[^]'.}>) \t\r\n,\\\"]"

  terminal_screen_match_add (screen,
                             "\\<" SCHEME "//(" USER "@)?" HOST
                             PORT "(" URLPATH ")?\\>/?",
                             FLAVOR_AS_IS);

  terminal_screen_match_add (screen,
                             "\\<(www|ftp)[" HOSTCHARS "]*\\." HOST
                             PORT "(" URLPATH ")?\\>/?",
                             FLAVOR_DEFAULT_TO_HTTP);

  terminal_screen_match_add (screen,
                             "\\<(callto|h323|sip):[" USERCHARS "]"
                             "[" USERCHARS ".]*(" PORT "/[a-z0-9]+)?"
                             "@" HOST "\\>",
                             FLAVOR_VOIP_CALL);

  terminal_screen_match_add (screen,
                             "\\<(mailto:)?[" USERCHARS "][" USERCHARS ".]*@"
                             "[" HOSTCHARS "]+\\." HOST "\\>",
                             FLAVOR_EMAIL);

  terminal_screen_match_add (screen,
                             "\\<news:[-A-Z\\^_a-z{|}~!\"#$%&'()*+,./0-9;:=?`]+"
                             HOST PORT "\\>",
                             FLAVOR_AS_IS);

  /* Setup DND */
  target_list = gtk_target_list_new (NULL, 0);
  gtk_target_list_add_uri_targets (target_list, 0);
  gtk_target_list_add_text_targets (target_list, 0);
  gtk_target_list_add_table (target_list, target_table, G_N_ELEMENTS (target_table));

  targets = gtk_target_table_new_from_list (target_list, &n_targets);

  gtk_drag_dest_set (GTK_WIDGET (screen),
                     GTK_DEST_DEFAULT_MOTION |
                     GTK_DEST_DEFAULT_HIGHLIGHT |
                     GTK_DEST_DEFAULT_DROP,
                     targets, n_targets,
                     GDK_ACTION_COPY | GDK_ACTION_MOVE);

  gtk_target_table_free (targets, n_targets);
  gtk_target_list_unref (target_list);

  priv->title_from_arg = NULL;
  priv->user_title = FALSE;
  
  g_signal_connect (screen, "window-title-changed",
                    G_CALLBACK (terminal_screen_window_title_changed),
                    screen);
  g_signal_connect (screen, "icon-title-changed",
                    G_CALLBACK (terminal_screen_icon_title_changed),
                    screen);

  g_signal_connect (screen, "child-exited",
                    G_CALLBACK (terminal_screen_widget_child_died),
                    screen);

  g_signal_connect (terminal_app_get (), "notify::system-font",
                    G_CALLBACK (terminal_screen_system_font_notify_cb), screen);

  g_signal_connect (screen, "parent-set", G_CALLBACK (parent_set_callback), NULL);

#ifdef DEBUG_GEOMETRY
  g_signal_connect_after (screen, "size-request", G_CALLBACK (size_request), NULL);
  g_signal_connect_after (screen, "size-allocate", G_CALLBACK (size_allocate), NULL);
#endif
}

static void
terminal_screen_get_property (GObject *object,
                              guint prop_id,
                              GValue *value,
                              GParamSpec *pspec)
{
  TerminalScreen *screen = TERMINAL_SCREEN (object);

  switch (prop_id)
    {
      case PROP_PROFILE:
        g_value_set_object (value, terminal_screen_get_profile (screen));
        break;
      case PROP_ICON_TITLE:
        g_value_set_string (value, terminal_screen_get_icon_title (screen));
        break;
      case PROP_ICON_TITLE_SET:
        g_value_set_boolean (value, terminal_screen_get_icon_title_set (screen));
        break;
      case PROP_OVERRIDE_COMMAND:
        g_value_set_boxed (value, terminal_screen_get_override_command (screen));
        break;
      case PROP_TITLE:
        g_value_set_string (value, terminal_screen_get_title (screen));
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
terminal_screen_set_property (GObject *object,
                              guint prop_id,
                              const GValue *value,
                              GParamSpec *pspec)
{
  TerminalScreen *screen = TERMINAL_SCREEN (object);

  switch (prop_id)
    {
      case PROP_PROFILE: {
        TerminalProfile *profile;

        profile = g_value_get_object (value);
        g_assert (profile != NULL);
        terminal_screen_set_profile (screen, profile);
        break;
      }
      case PROP_OVERRIDE_COMMAND:
        terminal_screen_set_override_command (screen, g_value_get_boxed (value));
        break;
      case PROP_ICON_TITLE:
      case PROP_ICON_TITLE_SET:
      case PROP_TITLE:
        /* not writable */
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
terminal_screen_class_init (TerminalScreenClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  object_class->dispose = terminal_screen_dispose;
  object_class->finalize = terminal_screen_finalize;
  object_class->get_property = terminal_screen_get_property;
  object_class->set_property = terminal_screen_set_property;

  widget_class->screen_changed = terminal_screen_screen_changed;
  widget_class->realize = terminal_screen_realize;
  widget_class->style_set = terminal_screen_style_set;
  widget_class->drag_data_received = terminal_screen_drag_data_received;
  widget_class->button_press_event = terminal_screen_button_press;
  widget_class->popup_menu = terminal_screen_popup_menu;

  signals[PROFILE_SET] =
    g_signal_new (I_("profile-set"),
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (TerminalScreenClass, profile_set),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__OBJECT,
                  G_TYPE_NONE,
                  1, TERMINAL_TYPE_PROFILE);
  
  signals[SHOW_POPUP_MENU] =
    g_signal_new (I_("show-popup-menu"),
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (TerminalScreenClass, show_popup_menu),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE,
                  1,
                  G_TYPE_POINTER);

  signals[SKEY_CLICKED] =
    g_signal_new (I_("skey-clicked"),
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (TerminalScreenClass, skey_clicked),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE,
                  1, G_TYPE_STRING);
  
  signals[URL_CLICKED] =
    g_signal_new (I_("url-clicked"),
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (TerminalScreenClass, url_clicked),
                  NULL, NULL,
                  _terminal_marshal_VOID__STRING_INT,
                  G_TYPE_NONE,
                  2, G_TYPE_STRING, G_TYPE_INT);
  
  signals[CLOSE_SCREEN] =
    g_signal_new (I_("close-screen"),
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (TerminalScreenClass, close_screen),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE,
                  0);

  g_object_class_install_property
    (object_class,
     PROP_PROFILE,
     g_param_spec_string ("profile", NULL, NULL,
                          NULL,
                          G_PARAM_READWRITE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

  g_object_class_install_property
    (object_class,
     PROP_ICON_TITLE,
     g_param_spec_string ("icon-title", NULL, NULL,
                          NULL,
                          G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

  g_object_class_install_property
    (object_class,
     PROP_ICON_TITLE_SET,
     g_param_spec_boolean ("icon-title-set", NULL, NULL,
                           FALSE,
                           G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

  g_object_class_install_property
    (object_class,
     PROP_ICON_TITLE,
     g_param_spec_boxed ("override-command", NULL, NULL,
                         G_TYPE_STRV,
                         G_PARAM_READWRITE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

  g_object_class_install_property
    (object_class,
     PROP_TITLE,
     g_param_spec_string ("title", NULL, NULL,
                          NULL,
                          G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

  g_type_class_add_private (object_class, sizeof (TerminalScreenPrivate));
}

static void
terminal_screen_dispose (GObject *object)
{
  TerminalScreen *screen = TERMINAL_SCREEN (object);
  GtkSettings *settings;

  settings = gtk_widget_get_settings (GTK_WIDGET (screen));
  g_signal_handlers_disconnect_matched (settings, G_SIGNAL_MATCH_DATA,
                                        0, 0, NULL, NULL,
                                        screen);

  G_OBJECT_CLASS (terminal_screen_parent_class)->dispose (object);
}

static void
terminal_screen_finalize (GObject *object)
{
  TerminalScreen *screen = TERMINAL_SCREEN (object);
  TerminalScreenPrivate *priv = screen->priv;

  g_signal_handlers_disconnect_by_func (terminal_app_get (),
                                        G_CALLBACK (terminal_screen_system_font_notify_cb),
                                        screen);

  terminal_screen_set_profile (screen, NULL);

  if (priv->recheck_working_dir_idle)
    g_source_remove (priv->recheck_working_dir_idle);
  
  g_free (priv->raw_title);
  g_free (priv->cooked_title);
  g_free (priv->title_from_arg);
  g_free (priv->raw_icon_title);
  g_free (priv->cooked_icon_title);
  g_strfreev (priv->override_command);
  g_free (priv->working_dir);

  g_slist_foreach (priv->url_tags, (GFunc) free_tag_data, NULL);
  g_slist_free (priv->url_tags);

  g_slist_foreach (priv->skey_tags, (GFunc) free_tag_data, NULL);
  g_slist_free (priv->skey_tags);

  G_OBJECT_CLASS (terminal_screen_parent_class)->finalize (object);
}

TerminalScreen*
terminal_screen_new (void)
{
  return g_object_new (TERMINAL_TYPE_SCREEN, NULL);
}

const char*
terminal_screen_get_raw_title (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  if (priv->raw_title)
    return priv->raw_title;

  return "";
}

const char*
terminal_screen_get_title (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  if (priv->cooked_title == NULL)
    terminal_screen_cook_title (screen);

  /* cooked_title may still be NULL */
  if (priv->cooked_title != NULL)
    return priv->cooked_title;
  else
    return "";
}

const char*
terminal_screen_get_icon_title (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  if (priv->cooked_icon_title == NULL)
    terminal_screen_cook_icon_title (screen);

  /* cooked_icon_title may still be NULL */
  if (priv->cooked_icon_title != NULL)
    return priv->cooked_icon_title;
  else
    return "";
}

gboolean
terminal_screen_get_icon_title_set (TerminalScreen *screen)
{
  return screen->priv->icon_title_set;
}

static void
terminal_screen_profile_notify_cb (TerminalProfile *profile,
                                   GParamSpec *pspec,
                                   TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  GObject *object = G_OBJECT (screen);
  VteTerminal *vte_terminal = VTE_TERMINAL (screen);
  const char *prop_name;
  TerminalBackgroundType bg_type;

  if (pspec)
    prop_name = pspec->name;
  else
    prop_name = NULL;

  g_object_freeze_notify (object);

  if (priv->window)
    {
      /* We need these in line for the set_size in
       * update_on_realize
       */
      terminal_window_update_geometry (priv->window);
    }
  
  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_SCROLLBAR_POSITION))
    _terminal_screen_update_scrollbar (screen);

  if (!prop_name ||
      prop_name == I_(TERMINAL_PROFILE_TITLE_MODE) ||
      prop_name == I_(TERMINAL_PROFILE_TITLE))
    {
      terminal_screen_cook_title (screen);
      terminal_screen_cook_icon_title (screen);
    }

  if (GTK_WIDGET_REALIZED (screen) &&
      (!prop_name ||
       prop_name == I_(TERMINAL_PROFILE_USE_SYSTEM_FONT) ||
       prop_name == I_(TERMINAL_PROFILE_FONT) ||
       prop_name == I_(TERMINAL_PROFILE_NO_AA_WITHOUT_RENDER)))
    terminal_screen_change_font (screen);

  if (!prop_name ||
      prop_name == I_(TERMINAL_PROFILE_USE_THEME_COLORS) ||
      prop_name == I_(TERMINAL_PROFILE_FOREGROUND_COLOR) ||
      prop_name == I_(TERMINAL_PROFILE_BACKGROUND_COLOR) ||
      prop_name == I_(TERMINAL_PROFILE_PALETTE))
    update_color_scheme (screen);

  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_SILENT_BELL))
      vte_terminal_set_audible_bell (vte_terminal, !terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_SILENT_BELL));

  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_WORD_CHARS))
    vte_terminal_set_word_chars (vte_terminal,
                                 terminal_profile_get_property_string (profile, TERMINAL_PROFILE_WORD_CHARS));
  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_SCROLL_ON_KEYSTROKE))
    vte_terminal_set_scroll_on_keystroke (vte_terminal,
                                          terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_SCROLL_ON_KEYSTROKE));
  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_SCROLL_ON_OUTPUT))
    vte_terminal_set_scroll_on_output (vte_terminal,
                                       terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_SCROLL_ON_OUTPUT));
  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_SCROLLBACK_LINES))
    vte_terminal_set_scrollback_lines (vte_terminal,
                                       terminal_profile_get_property_int (profile, TERMINAL_PROFILE_SCROLLBACK_LINES));

  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_USE_SKEY))
    {
      if (terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_USE_SKEY))
        {
          terminal_screen_skey_match_add (screen,
                                          "s/key [0-9]* [-A-Za-z0-9]*",
                                          FLAVOR_AS_IS);

          terminal_screen_skey_match_add (screen,
                                          "otp-[a-z0-9]* [0-9]* [-A-Za-z0-9]*",
                                          FLAVOR_AS_IS);
        }
      else
        {
          terminal_screen_skey_match_remove (screen);
        }
    }

  if (!prop_name ||
      prop_name == I_(TERMINAL_PROFILE_BACKGROUND_TYPE) ||
      prop_name == I_(TERMINAL_PROFILE_BACKGROUND_IMAGE_FILE) ||
      prop_name == I_(TERMINAL_PROFILE_BACKGROUND_DARKNESS) ||
      prop_name == I_(TERMINAL_PROFILE_SCROLL_BACKGROUND))
    {
      bg_type = terminal_profile_get_property_enum (profile, TERMINAL_PROFILE_BACKGROUND_TYPE);
      
      if (bg_type == TERMINAL_BACKGROUND_IMAGE)
        {
          /* FIXME: use the BACKGROUND property intead */
          set_background_image_file (vte_terminal,
                                    terminal_profile_get_property_string (profile, TERMINAL_PROFILE_BACKGROUND_IMAGE_FILE));
          vte_terminal_set_scroll_background (vte_terminal,
                                              terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_SCROLL_BACKGROUND));
        }
      else
        {
          set_background_image_file (vte_terminal, NULL);
          vte_terminal_set_scroll_background (vte_terminal, FALSE);
        }

      if (bg_type == TERMINAL_BACKGROUND_IMAGE ||
          bg_type == TERMINAL_BACKGROUND_TRANSPARENT)
        {
          vte_terminal_set_background_saturation (vte_terminal,
                                                  1.0 - terminal_profile_get_property_double (profile, TERMINAL_PROFILE_BACKGROUND_DARKNESS));
          vte_terminal_set_opacity (vte_terminal,
                                    0xffff * terminal_profile_get_property_double (profile, TERMINAL_PROFILE_BACKGROUND_DARKNESS));
        }
      else
        {
          vte_terminal_set_background_saturation (vte_terminal, 1.0); /* normal color */
          vte_terminal_set_opacity (vte_terminal, 0xffff);
        }
      
      /* FIXME: Don't enable this if we have a compmgr. */
      vte_terminal_set_background_transparent (vte_terminal,
                                               bg_type == TERMINAL_BACKGROUND_TRANSPARENT &&
                                               (!priv->window || !terminal_window_uses_argb_visual (priv->window)));
    }

  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_BACKSPACE_BINDING))
  vte_terminal_set_backspace_binding (vte_terminal,
                                      terminal_profile_get_property_enum (profile, TERMINAL_PROFILE_BACKSPACE_BINDING));
  
  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_DELETE_BINDING))
  vte_terminal_set_delete_binding (vte_terminal,
                                   terminal_profile_get_property_enum (profile, TERMINAL_PROFILE_DELETE_BINDING));

  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_ALLOW_BOLD))
    vte_terminal_set_allow_bold (vte_terminal,
                                 terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_ALLOW_BOLD));

  if (!prop_name || prop_name == I_(TERMINAL_PROFILE_CURSOR_BLINK_MODE))
    terminal_screen_update_cursor_blink (screen, gtk_widget_get_settings (GTK_WIDGET (screen)));

  /* Some changes require a redraw, but vte doesn't always schedule one */
  if (GTK_WIDGET_REALIZED (screen))
    gtk_widget_queue_draw (GTK_WIDGET (screen));

  g_object_thaw_notify (object);
}

/**
 * cook_title:
 * @screen:
 * @raw_title: main ingredient
 * @old_cooked_title: pointer of the current cooked_title
 * 
 * Cook title according to the profile of @screen. If the result is different from
 * <literal>*old_cooked_title</literal>, store it there.
 * Returns: %TRUE or %FALSE according to whether the cooked_title was changed.
 */

static gboolean
cook_title  (TerminalScreen *screen, const char *raw_title, char **old_cooked_title)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalTitleMode mode;
  char *new_cooked_title = NULL;
  const char *static_title_piece = NULL;

  g_return_val_if_fail (old_cooked_title != NULL, FALSE);

  mode = terminal_profile_get_property_enum (priv->profile, TERMINAL_PROFILE_TITLE_MODE);

  /* use --title argument if one was supplied, otherwise ask the profile */
  if (priv->title_from_arg)
    static_title_piece = priv->title_from_arg;
  else
    static_title_piece = terminal_profile_get_property_string (priv->profile, TERMINAL_PROFILE_TITLE);

  switch (mode)
    {
    case TERMINAL_TITLE_AFTER:
      new_cooked_title =
        g_strconcat (static_title_piece,
                     (raw_title && *raw_title) ? " - " : "",
                     raw_title,
                     NULL);
      break;
    case TERMINAL_TITLE_BEFORE:
      new_cooked_title =
        g_strconcat (raw_title ? raw_title : "",
                     (raw_title && *raw_title) ? " - " : "",
                     static_title_piece,
                     NULL);
      break;
    case TERMINAL_TITLE_REPLACE:
      if (raw_title)
        new_cooked_title = g_strdup (raw_title);
      else
        new_cooked_title = g_strdup (static_title_piece);
      break;
    case TERMINAL_TITLE_IGNORE:
      new_cooked_title = g_strdup (static_title_piece);
    /* no default so we get missing cases statically */
    }

  if (*old_cooked_title == NULL || strcmp (new_cooked_title, *old_cooked_title) != 0)
    {
      g_free (*old_cooked_title);
      *old_cooked_title = new_cooked_title;
      return TRUE;
    }
  else
    {
      g_free (new_cooked_title);
      return FALSE;
    }
}

static void 
terminal_screen_cook_title (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  if (cook_title (screen, priv->raw_title, &priv->cooked_title))
    g_object_notify (G_OBJECT (screen), "title");
}

static void 
terminal_screen_cook_icon_title (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;

  if (cook_title (screen, priv->raw_icon_title, &priv->cooked_icon_title))
    g_object_notify (G_OBJECT (screen), "icon-title");
}

static void
update_color_scheme (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalProfile *profile = priv->profile;
  GtkStyle *style;
  GdkColor colors[TERMINAL_PALETTE_SIZE];
  const GdkColor *fg_color, *bg_color;
  GdkColor fg, bg;
  guint n_colors;

  style = gtk_widget_get_style (GTK_WIDGET (screen));
  if (!style)
    return;

  fg = style->text[GTK_STATE_NORMAL];
  bg = style->base[GTK_STATE_NORMAL];

  fg_color = terminal_profile_get_property_boxed (profile, TERMINAL_PROFILE_FOREGROUND_COLOR);
  bg_color = terminal_profile_get_property_boxed (profile, TERMINAL_PROFILE_BACKGROUND_COLOR);

  if (!terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_USE_THEME_COLORS))
    {
      if (fg_color)
        fg = *fg_color;
      if (bg_color)
        bg = *bg_color;
    }

  n_colors = G_N_ELEMENTS (colors);
  terminal_profile_get_palette (priv->profile, colors, &n_colors);
  vte_terminal_set_colors (VTE_TERMINAL (screen), &fg, &bg,
                           colors, n_colors);
  vte_terminal_set_background_tint_color (VTE_TERMINAL (screen), &bg);
}

void
terminal_screen_set_font (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalProfile *profile;
  PangoFontDescription *desc;
  gboolean no_aa_without_render;

  profile = priv->profile;
  
  if (terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_USE_SYSTEM_FONT))
    g_object_get (terminal_app_get (), "system-font", &desc, NULL);
  else
    g_object_get (profile, TERMINAL_PROFILE_FONT, &desc, NULL);
  g_assert (desc);

  pango_font_description_set_size (desc,
				   priv->font_scale *
				   pango_font_description_get_size (desc));

  no_aa_without_render = terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_NO_AA_WITHOUT_RENDER);
  if (!no_aa_without_render)
    {
      vte_terminal_set_font (VTE_TERMINAL (screen), desc);
    }
  else
    {
      Display *dpy;
      gboolean has_render;
      gint event_base, error_base;

      /* FIXME multi-head/mult-screen! */
      dpy = gdk_x11_display_get_xdisplay (gtk_widget_get_display (GTK_WIDGET (screen)));
      has_render = (XRenderQueryExtension (dpy, &event_base, &error_base) &&
		    (XRenderFindVisualFormat (dpy, DefaultVisual (dpy, DefaultScreen (dpy))) != NULL));

      if (has_render)
	vte_terminal_set_font (VTE_TERMINAL (screen), desc);
      else 
	vte_terminal_set_font_full (VTE_TERMINAL (screen),
				    desc,
				    VTE_ANTI_ALIAS_FORCE_DISABLE);
    }

  pango_font_description_free (desc);
}

static void
terminal_screen_system_font_notify_cb (TerminalApp *app,
                                       GParamSpec *pspec,
                                       TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;

  if (!GTK_WIDGET_REALIZED (screen))
    return;

  if (!terminal_profile_get_property_boolean (priv->profile, TERMINAL_PROFILE_USE_SYSTEM_FONT))
    return;

  terminal_screen_change_font (screen);
}

static void
terminal_screen_change_font (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;

  terminal_screen_set_font (screen);

  terminal_window_set_size (priv->window, screen, TRUE);
}

static void
profile_forgotten_callback (TerminalProfile *profile,
                            TerminalScreen  *screen)
{
  TerminalProfile *new_profile;

  new_profile = terminal_app_get_profile_for_new_term (terminal_app_get ());
  g_assert (new_profile != NULL);
  terminal_screen_set_profile (screen, new_profile);
}

void
terminal_screen_set_profile (TerminalScreen *screen,
                             TerminalProfile *profile)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalProfile *old_profile;

  old_profile = priv->profile;
  if (profile == old_profile)
    return;

  if (priv->profile_changed_id)
    {
      g_signal_handler_disconnect (G_OBJECT (priv->profile),
                                   priv->profile_changed_id);
      priv->profile_changed_id = 0;
    }

  if (priv->profile_forgotten_id)
    {
      g_signal_handler_disconnect (G_OBJECT (priv->profile),
                                   priv->profile_forgotten_id);
      priv->profile_forgotten_id = 0;
    }
  
  priv->profile = profile;
  if (profile)
    {
      g_object_ref (profile);
      priv->profile_changed_id =
        g_signal_connect (profile, "notify",
                          G_CALLBACK (terminal_screen_profile_notify_cb),
                          screen);
      priv->profile_forgotten_id =
        g_signal_connect (G_OBJECT (profile),
                          "forgotten",
                          G_CALLBACK (profile_forgotten_callback),
                          screen);

      terminal_screen_profile_notify_cb (profile, NULL, screen);

      g_signal_emit (G_OBJECT (screen), signals[PROFILE_SET], 0, old_profile);
    }

  if (old_profile)
    g_object_unref (old_profile);

  g_object_notify (G_OBJECT (screen), "profile");
}

TerminalProfile*
terminal_screen_get_profile (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;

  g_assert (priv->profile != NULL);
  return priv->profile;
}

void
terminal_screen_set_override_command (TerminalScreen *screen,
                                      char          **argv)
{
  TerminalScreenPrivate *priv;

  g_return_if_fail (TERMINAL_IS_SCREEN (screen));

  priv = screen->priv;
  g_strfreev (priv->override_command);
  priv->override_command = g_strdupv (argv);
}

const char**
terminal_screen_get_override_command (TerminalScreen *screen)
{
  g_return_val_if_fail (TERMINAL_IS_SCREEN (screen), NULL);

  return (const char**) screen->priv->override_command;
}

static void
show_command_error_dialog (TerminalScreen *screen,
                           GError         *error)
{
  g_assert (error != NULL);
  
  terminal_util_show_error_dialog ((GtkWindow*) gtk_widget_get_ancestor (GTK_WIDGET (screen), GTK_TYPE_WINDOW), NULL,
                                   _("There was a problem with the command for this terminal: %s"), error->message);
}

static gboolean
get_child_command (TerminalScreen *screen,
                   char          **file_p,
                   char         ***argv_p,
                   GError        **err)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalProfile *profile;
  char  *file;
  char **argv;

  profile = priv->profile;

  file = NULL;
  argv = NULL;
  
  if (file_p)
    *file_p = NULL;
  if (argv_p)
    *argv_p = NULL;

  if (priv->override_command)
    {
      file = g_strdup (priv->override_command[0]);
      argv = g_strdupv (priv->override_command);
    }
  else if (terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_USE_CUSTOM_COMMAND))
    {
      if (!g_shell_parse_argv (terminal_profile_get_property_string (profile, TERMINAL_PROFILE_CUSTOM_COMMAND),
                               NULL, &argv,
                               err))
        return FALSE;

      file = g_strdup (argv[0]);
    }
  else
    {
      const char *only_name;
      char *shell;

      shell = gnome_util_user_shell ();

      file = g_strdup (shell);
      
      only_name = strrchr (shell, '/');
      if (only_name != NULL)
        only_name++;
      else
        only_name = shell;

      argv = g_new (char*, 2);

      if (terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_LOGIN_SHELL))
        argv[0] = g_strconcat ("-", only_name, NULL);
      else
        argv[0] = g_strdup (only_name);

      argv[1] = NULL;

      g_free (shell);
    }

  if (file_p)
    *file_p = file;
  else
    g_free (file);

  if (argv_p)
    *argv_p = argv;
  else
    g_strfreev (argv);

  return TRUE;
}

static char**
get_child_environment (TerminalScreen *screen)
{
  GtkWidget *term;
  char **env, **p, **retval;
  char *proxymode, *proxyhost;
  gboolean use_proxy;
  int i;
  GConfClient *conf;
#define EXTRA_ENV_VARS 8

  term = GTK_WIDGET (screen);

  env = g_listenv ();
  retval = g_new (char *, g_strv_length (env) + 1 + EXTRA_ENV_VARS);

  for (i = 0, p = env; *p; p++)
    {
      /* Strip all these out, we'll replace some of them */
      if ((strncmp (*p, "COLUMNS=", 8) == 0) ||
          (strncmp (*p, "LINES=", 6) == 0)   ||
          (strncmp (*p, "WINDOWID=", 9) == 0) ||
          (strncmp (*p, "TERM=", 5) == 0)    ||
          (strncmp (*p, "GNOME_DESKTOP_ICON=", 19) == 0) ||
          (strncmp (*p, "COLORTERM=", 10) == 0) ||
	  (strncmp (*p, "DISPLAY=", 8) == 0))
        {
          /* nothing: do not copy */
        }
      else
        {
          retval[i] = g_strdup_printf ("%s=%s", *p, g_getenv (*p));
          ++i;
        }
    }

  retval[i] = g_strdup ("COLORTERM="EXECUTABLE_NAME);
  ++i;

  retval[i] = g_strdup ("TERM=xterm"); /* FIXME configurable later? */
  ++i;

  /* FIXME: moving the tab between windows, or the window between displays will make this invalid... */
  retval[i] = g_strdup_printf ("WINDOWID=%ld",
                               GDK_WINDOW_XWINDOW (term->window));
  ++i;

  /* FIXME: moving the window between displays will make this invalid... */
  retval[i] = g_strdup_printf ("DISPLAY=%s", 
			       gdk_display_get_name(gtk_widget_get_display(term)));
  ++i;
  
  conf = gconf_client_get_default ();

  /* Series of conditions under which we don't set http_proxy */
  use_proxy = gconf_client_get_bool (conf, HTTP_PROXY_DIR "/use_http_proxy", NULL);

  /* Is the mode unset or not equal to "manual"? */
  proxymode = gconf_client_get_string (conf, "/system/proxy/mode", NULL);
  if (!proxymode || strcmp (proxymode, "manual") != 0)
    use_proxy = FALSE;
  g_free (proxymode);

  /* Do we already have a proxy setting? */
  if (getenv ("http_proxy"))
    use_proxy = FALSE;

  /* Do we have no proxy host or an empty string? */
  proxyhost = gconf_client_get_string (conf, HTTP_PROXY_DIR "/host", NULL);
  if (!proxyhost || proxyhost[0] == '\0')
    use_proxy = FALSE;
  g_free (proxyhost);

  /* Set up proxy environment variables if we passed all of the above */
  if (use_proxy)
    {
      gint port;
      GSList *ignore;
      gchar *host, *auth = NULL;

      host = gconf_client_get_string (conf, HTTP_PROXY_DIR "/host", NULL);
      port = gconf_client_get_int (conf, HTTP_PROXY_DIR "/port", NULL);
      ignore = gconf_client_get_list (conf, HTTP_PROXY_DIR "/ignore_hosts",
				      GCONF_VALUE_STRING, NULL);

      if (gconf_client_get_bool (conf, HTTP_PROXY_DIR "/use_authentication", NULL))
	{
	  char *user, *password;

	  user = gconf_client_get_string (conf,
					  HTTP_PROXY_DIR "/authentication_user",
					  NULL);

	  password = gconf_client_get_string (conf,
					      HTTP_PROXY_DIR
					      "/authentication_password",
					      NULL);

	  if (user && user != '\0')
            {
              if (password)
                auth = g_strdup_printf ("%s:%s", user, password);
              else
                auth = g_strdup (user);
            }

	  g_free (user);
	  g_free (password);
	}

      g_object_unref (conf);

      if (port && host && host != '\0')
	{
	  if (auth)
	    retval[i] = g_strdup_printf ("http_proxy=http://%s@%s:%d/",
					 auth, host, port);
	  else
	    retval[i] = g_strdup_printf ("http_proxy=http://%s:%d/",
					 host, port);

	  ++i;
	}

      if (auth)
	g_free (auth);

      if (host)
	g_free (host);

      if (ignore)
	{
	  /* code distantly based on gconf's */
	  gchar *buf = NULL;
	  guint bufsize = 64;
	  guint cur = 0;

	  buf = g_malloc (bufsize + 3);

	  while (ignore != NULL)
	    {
	      guint len = strlen (ignore->data);

	      if ((cur + len + 2) >= bufsize) /* +2 for '\0' and comma */
		{
		  bufsize = MAX(bufsize * 2, bufsize + len + 4); 
		  buf = g_realloc (buf, bufsize + 3);
		}

	      g_assert (cur < bufsize);

	      strcpy (&buf[cur], ignore->data);
	      cur += len;

	      g_assert(cur < bufsize);

	      buf[cur] = ',';
	      ++cur;

	      g_assert(cur < bufsize);

	      ignore = g_slist_next (ignore);
	    }

	  buf[cur-1] = '\0'; /* overwrites last comma */

	  retval[i] = g_strdup_printf ("no_proxy=%s", buf);
	  g_free (buf);
	  ++i;
	}
    }

  retval[i] = NULL;

  g_strfreev (env);

  return retval;
}

void
terminal_screen_launch_child (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalProfile *profile;
  char **env;
  char  *path;
  char **argv;
  GError *err;
  gboolean update_records;
  
  
  profile = priv->profile;

  err = NULL;
  if (!get_child_command (screen, &path, &argv, &err))
    {
      show_command_error_dialog (screen, err);
      g_error_free (err);
      return;
    }
  
  env = get_child_environment (screen);

  update_records = terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_UPDATE_RECORDS);

  priv->child_pid = vte_terminal_fork_command (VTE_TERMINAL (screen),
                                               path,
                                               argv,
                                               env,
                                               terminal_screen_get_working_dir (screen),
                                               terminal_profile_get_property_boolean (profile, TERMINAL_PROFILE_LOGIN_SHELL),
                                               update_records,
                                               update_records);

  if (priv->child_pid == -1)
    {

      terminal_util_show_error_dialog ((GtkWindow*) gtk_widget_get_ancestor (GTK_WIDGET (screen), GTK_TYPE_WINDOW), NULL,
                                       "%s", _("There was an error creating the child process for this terminal"));
    }
  
  g_free (path);
  g_strfreev (argv);
  g_strfreev (env);
}

static TerminalScreenPopupInfo *
terminal_screen_popup_info_new (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalScreenPopupInfo *info;

  info = g_slice_new0 (TerminalScreenPopupInfo);
  info->ref_count = 1;
  info->screen = g_object_ref (screen);
  info->window = priv->window;

  return info;
}

TerminalScreenPopupInfo *
terminal_screen_popup_info_ref (TerminalScreenPopupInfo *info)
{
  g_return_val_if_fail (info != NULL, NULL);

  info->ref_count++;
  return info;
}

void
terminal_screen_popup_info_unref (TerminalScreenPopupInfo *info)
{
  g_return_if_fail (info != NULL);

  if (--info->ref_count > 0)
    return;

  g_object_unref (info->screen);
  g_free (info->string);
  g_slice_free (TerminalScreenPopupInfo, info);
}

static gboolean
terminal_screen_popup_menu (GtkWidget *widget)
{
  TerminalScreen *screen = TERMINAL_SCREEN (widget);
  TerminalScreenPopupInfo *info;

  info = terminal_screen_popup_info_new (screen);
  info->button = 0;
  info->timestamp = gtk_get_current_event_time ();

  g_signal_emit (screen, signals[SHOW_POPUP_MENU], 0, info);
  terminal_screen_popup_info_unref (info);

  return TRUE;
}

static gboolean
terminal_screen_button_press (GtkWidget      *widget,
                              GdkEventButton *event)
{
  TerminalScreen *screen = TERMINAL_SCREEN (widget);
  TerminalScreenPrivate *priv = screen->priv;
  int char_width, char_height;
  gboolean dingus_button;
  char *matched_string;
  int matched_flavor = 0;
  guint state;

  state = event->state & gtk_accelerator_get_default_mod_mask ();

  terminal_screen_get_cell_size (screen, &char_width, &char_height);

  matched_string =
    terminal_screen_check_match (screen,
                                 event->x / char_width,
                                 event->y / char_height,
                                 &matched_flavor);
  dingus_button = ((event->button == 1) || (event->button == 2));

  if (dingus_button &&
      (state & GDK_CONTROL_MASK) &&
      terminal_profile_get_property_boolean (priv->profile, TERMINAL_PROFILE_USE_SKEY))
    {
      gchar *skey_match;

      skey_match = terminal_screen_skey_check_match (screen,
						     event->x / char_width,
						     event->y / char_height,
                                                     NULL);
      if (skey_match != NULL)
	{
          g_signal_emit (screen, signals[SKEY_CLICKED], 0, skey_match);
	  g_free (skey_match);
          g_free (matched_string);

	  return TRUE;
	}
    }

  if (dingus_button &&
      (state & GDK_CONTROL_MASK) != 0 &&
      matched_string != NULL)
    {
      gtk_widget_grab_focus (widget);

      g_signal_emit (screen, signals[URL_CLICKED], 0, matched_string, matched_flavor);
      g_free (matched_string);

      return TRUE; /* don't do anything else such as select with the click */
    }
      
  if (event->button == 3 &&
      (state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK)) == 0)
    {
      TerminalScreenPopupInfo *info;

      info = terminal_screen_popup_info_new (screen);
      info->button = event->button;
      info->timestamp = event->time;
      info->string = matched_string; /* adopted */
      info->flavour = matched_flavor;

      g_signal_emit (screen, signals[SHOW_POPUP_MENU], 0, info);
      terminal_screen_popup_info_unref (info);

      return TRUE;
    }

  /* default behavior is to let the terminal widget deal with it */
  if (GTK_WIDGET_CLASS (terminal_screen_parent_class)->button_press_event)
    return GTK_WIDGET_CLASS (terminal_screen_parent_class)->button_press_event (widget, event);

  return FALSE;
}

void
terminal_screen_set_dynamic_title (TerminalScreen *screen,
                                   const char     *title,
				   gboolean	  userset)
{
  TerminalScreenPrivate *priv = screen->priv;

  g_assert (TERMINAL_IS_SCREEN (screen));
  
  if ((priv->user_title && !userset) ||
      (priv->raw_title && title &&
       strcmp (priv->raw_title, title) == 0))
    return;

  g_free (priv->raw_title);
  priv->raw_title = g_strdup (title);
  terminal_screen_cook_title (screen);
}

void
terminal_screen_set_dynamic_icon_title (TerminalScreen *screen,
                                        const char     *icon_title,
					gboolean       userset)
{
  TerminalScreenPrivate *priv = screen->priv;
  GObject *object = G_OBJECT (screen);
  
  g_assert (TERMINAL_IS_SCREEN (screen));

  if ((priv->user_title && !userset) ||  
      (priv->icon_title_set &&
       priv->raw_icon_title &&
       icon_title &&
       strcmp (priv->raw_icon_title, icon_title) == 0))
    return;

  g_object_freeze_notify (object);

  g_free (priv->raw_icon_title);
  priv->raw_icon_title = g_strdup (icon_title);
  priv->icon_title_set = TRUE;

  g_object_notify (object, "icon-title-set");
  terminal_screen_cook_icon_title (screen);

  g_object_thaw_notify (object);
}

void
terminal_screen_set_title (TerminalScreen *screen,
			   const char     *title)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  if (priv->title_from_arg)
    g_free (priv->title_from_arg);
  priv->title_from_arg = g_strdup (title);
}

const char*
terminal_screen_get_dynamic_title (TerminalScreen *screen)
{
  g_return_val_if_fail (TERMINAL_IS_SCREEN (screen), NULL);
  
  return screen->priv->raw_title;
}

const char*
terminal_screen_get_dynamic_icon_title (TerminalScreen *screen)
{
  g_return_val_if_fail (TERMINAL_IS_SCREEN (screen), NULL);
  
  return screen->priv->raw_icon_title;
}

void
terminal_screen_set_working_dir (TerminalScreen *screen,
                                 const char     *dirname)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  g_return_if_fail (TERMINAL_IS_SCREEN (screen));

  g_free (priv->working_dir);
  priv->working_dir = g_strdup (dirname);
}

const char*
terminal_screen_get_working_dir (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  g_return_val_if_fail (TERMINAL_IS_SCREEN (screen), NULL);

  /* Try to update the working dir using various OS-specific mechanisms */
  if (priv->child_pid >= 0)
    {
      char *file;
      char buf[PATH_MAX+1];
      int len;

      /* readlink (/proc/pid/cwd) will work on Linux */
      file = g_strdup_printf ("/proc/%d/cwd", priv->child_pid);

      /* Silently ignore failure here, since we may not be on Linux */
      len = readlink (file, buf, sizeof (buf) - 1);

      if (len > 0 && buf[0] == '/')
        {
          buf[len] = '\0';
          
          g_free (priv->working_dir);
          priv->working_dir = g_strdup (buf);
        }
      else if (len == 0)
        {
          /* On Solaris, readlink returns an empty string, but the
           * link can be used as a directory, including as a target
           * of chdir().
           */
          char *cwd;

          cwd = g_get_current_dir ();
          if (cwd != NULL)
            {
              if (chdir (file) == 0)
                {
                  g_free (priv->working_dir);
                  priv->working_dir = g_get_current_dir ();
                  chdir (cwd);
                }
              g_free (cwd);
            }
        }
      
      g_free (file);
    }
  
  return priv->working_dir;
}

static gboolean
recheck_dir (void *data)
{
  TerminalScreen *screen = data;
  TerminalScreenPrivate *priv = screen->priv;

  priv->recheck_working_dir_idle = 0;
  
  /* called just for side effect */
  terminal_screen_get_working_dir (screen);

  /* remove idle */
  return FALSE;
}

static void
queue_recheck_working_dir (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  if (priv->recheck_working_dir_idle == 0)
    {
      priv->recheck_working_dir_idle =
        g_idle_add_full (G_PRIORITY_LOW + 50,
                         recheck_dir,
                         screen,
                         NULL);
    }
}

void
terminal_screen_set_font_scale (TerminalScreen *screen,
                                double          factor)
{
  TerminalScreenPrivate *priv = screen->priv;
  
  g_return_if_fail (TERMINAL_IS_SCREEN (screen));

  if (factor < TERMINAL_SCALE_MINIMUM)
    factor = TERMINAL_SCALE_MINIMUM;
  if (factor > TERMINAL_SCALE_MAXIMUM)
    factor = TERMINAL_SCALE_MAXIMUM;
  
  priv->font_scale = factor;
  
  if (GTK_WIDGET_REALIZED (screen))
    {
      /* Update the font */
      terminal_screen_change_font (screen);
    }
}

double
terminal_screen_get_font_scale (TerminalScreen *screen)
{
  g_return_val_if_fail (TERMINAL_IS_SCREEN (screen), 1.0);
  
  return screen->priv->font_scale;
}

static void
terminal_screen_window_title_changed (VteTerminal *vte_terminal,
                                      TerminalScreen *screen)
{
  terminal_screen_set_dynamic_title (screen,
                                     vte_terminal_get_window_title (vte_terminal),
				     FALSE);

  queue_recheck_working_dir (screen);
}

static void
terminal_screen_icon_title_changed (VteTerminal *vte_terminal,
                                    TerminalScreen *screen)
{
  terminal_screen_set_dynamic_icon_title (screen,
                                          vte_terminal_get_icon_title (vte_terminal),
					  FALSE);  

  queue_recheck_working_dir (screen);
}


static void
terminal_screen_widget_child_died (GtkWidget      *term,
                                   TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  TerminalExitAction action;

  priv->child_pid = -1;
  
  action = terminal_profile_get_property_enum (priv->profile, TERMINAL_PROFILE_EXIT_ACTION);
  
  switch (action)
    {
    case TERMINAL_EXIT_CLOSE:
      g_signal_emit (screen, signals[CLOSE_SCREEN], 0);
      break;
    case TERMINAL_EXIT_RESTART:
      terminal_screen_launch_child (screen);
      break;
    case TERMINAL_EXIT_HOLD:
      break;
    }
}

void
terminal_screen_set_user_title (TerminalScreen *screen,
                                const char *text)
{
  TerminalScreenPrivate *priv = screen->priv;

  /* The user set the title to nothing, let's understand that as a
     request to revert to dynamically setting the title again. */
  if (!text || !text[0])
    priv->user_title = FALSE;
  else
    {
      priv->user_title = TRUE;
      terminal_screen_set_dynamic_title (screen, text, TRUE);
      terminal_screen_set_dynamic_icon_title (screen, text, TRUE);
    }
}

static void
terminal_screen_drag_data_received (GtkWidget        *widget,
                                    GdkDragContext   *context,
                                    gint              x,
                                    gint              y,
                                    GtkSelectionData *selection_data,
                                    guint             info,
                                    guint             time)
{
  TerminalScreen *screen = TERMINAL_SCREEN (widget);
  TerminalScreenPrivate *priv = screen->priv;

#if 0
  {
    GList *tmp;

    g_print ("info: %d\n", info);
    tmp = context->targets;
    while (tmp != NULL)
      {
        GdkAtom atom = GDK_POINTER_TO_ATOM (tmp->data);

        g_print ("Target: %s\n", gdk_atom_name (atom));        
        
        tmp = tmp->next;
      }

    g_print ("Chosen target: %s\n", gdk_atom_name (selection_data->target));
  }
#endif

  if (gtk_targets_include_uri (&selection_data->target, 1))
    {
      char **uris;
      char *text;

      uris = gtk_selection_data_get_uris (selection_data);
      if (!uris)
        return;

      terminal_util_transform_uris_to_quoted_fuse_paths (uris);

      text = g_strjoinv (" ", uris);
      vte_terminal_feed_child (VTE_TERMINAL (screen), text, strlen (text));
      g_free (text);

      g_strfreev (uris);
    }
  else if (gtk_targets_include_text (&selection_data->target, 1))
    {
      char *text;

      text = (char *) gtk_selection_data_get_text (selection_data);
      if (text && text[0])
        vte_terminal_feed_child (VTE_TERMINAL (screen), text, strlen (text));
      g_free (text);
    }
  else switch (info)
    {
    case TARGET_COLOR:
      {
        guint16 *data = (guint16 *)selection_data->data;
        GdkColor color;

        /* We accept drops with the wrong format, since the KDE color
         * chooser incorrectly drops application/x-color with format 8.
         */
        if (selection_data->length != 8)
          return;

        color.red = data[0];
        color.green = data[1];
        color.blue = data[2];
        /* FIXME: use opacity from data[3] */

        g_object_set (priv->profile,
                      TERMINAL_PROFILE_BACKGROUND_TYPE, TERMINAL_BACKGROUND_SOLID,
                      TERMINAL_PROFILE_USE_THEME_COLORS, FALSE,
                      TERMINAL_PROFILE_BACKGROUND_COLOR, &color,
                      NULL);
      }
      break;

    case TARGET_MOZ_URL:
      {
        char *utf8_data, *newline;
        char *uris[2];
        
        /* MOZ_URL is in UCS-2 but in format 8. BROKEN!
         *
         * The data contains the URL, a \n, then the
         * title of the web page.
         */
        if (selection_data->format != 8 ||
            selection_data->length == 0 ||
            (selection_data->length % 2) != 0)
          return;

        utf8_data = g_utf16_to_utf8 ((const gunichar2*) selection_data->data,
                                     selection_data->length / 2,
                                     NULL, NULL, NULL);
        if (!utf8_data)
          return;

        newline = strchr (utf8_data, '\n');
        if (newline)
          *newline = '\0';

        uris[0] = utf8_data;
        uris[1] = NULL;
        terminal_util_transform_uris_to_quoted_fuse_paths (uris); /* This may replace uris[0] */

        vte_terminal_feed_child (VTE_TERMINAL (screen), uris[0], strlen (uris[0]));
        g_free (uris[0]);
      }
      break;

    case TARGET_NETSCAPE_URL:
      {
        char *utf8_data, *newline;
        char *uris[2];
        
        /* The data contains the URL, a \n, then the
         * title of the web page.
         */
        if (selection_data->length < 0 || selection_data->format != 8)
          return;

        utf8_data = g_strndup ((char *) selection_data->data, selection_data->length);
        newline = strchr (utf8_data, '\n');
        if (newline)
          *newline = '\0';

        uris[0] = utf8_data;
        uris[1] = NULL;
        terminal_util_transform_uris_to_quoted_fuse_paths (uris); /* This may replace uris[0] */

        vte_terminal_feed_child (VTE_TERMINAL (screen), uris[0], strlen (uris[0]));
        g_free (uris[0]);
      }
      break;
       
    case TARGET_BGIMAGE:
      {
        char *utf8_data;
        char **uris;
        
        if (selection_data->length < 0 || selection_data->format != 8)
          return;
        
        utf8_data = g_strndup ((char *) selection_data->data, selection_data->length);
        uris = g_uri_list_extract_uris (utf8_data);
        g_free (utf8_data);

        /* FIXME: use terminal_util_transform_uris_to_quoted_fuse_paths? */

        if (uris && uris[0])
          {
            char *filename;

            filename = g_filename_from_uri (uris[0], NULL, NULL);
            if (filename)
              {
                g_object_set (priv->profile,
                              TERMINAL_PROFILE_BACKGROUND_TYPE, TERMINAL_BACKGROUND_IMAGE,
                              TERMINAL_PROFILE_BACKGROUND_IMAGE_FILE, filename,
                              NULL);
              }

            g_free (filename);
          }

        g_strfreev (uris);
      }
      break;

    case TARGET_RESET_BG:
      g_object_set (priv->profile,
                    TERMINAL_PROFILE_BACKGROUND_TYPE, TERMINAL_BACKGROUND_SOLID,
                    NULL);
      break;

    case TARGET_TAB:
      {
        GtkWidget *container;
        TerminalScreen *moving_screen;
        TerminalWindow *source_window;
        TerminalWindow *dest_window;
        GtkWidget *dest_notebook;
        int page_num;

        container = *(GtkWidget**) selection_data->data;
        if (!GTK_IS_WIDGET (container))
          return;

        moving_screen = terminal_screen_container_get_screen (container);
        g_return_if_fail (TERMINAL_IS_SCREEN (moving_screen));
        if (!TERMINAL_IS_SCREEN (moving_screen))
          return;

        source_window = moving_screen->priv->window;
        dest_window = screen->priv->window;
        dest_notebook = terminal_window_get_notebook (dest_window);
        page_num = gtk_notebook_page_num (GTK_NOTEBOOK (dest_notebook), 
                                          GTK_WIDGET (screen));
        terminal_window_move_screen (source_window, dest_window, moving_screen, page_num + 1);

        gtk_drag_finish (context, TRUE, TRUE, time);
      }
      break;

    default:
      g_assert_not_reached ();
    }
}

void
_terminal_screen_update_scrollbar (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  GtkWidget *parent;
  GtkPolicyType policy = GTK_POLICY_ALWAYS;
  GtkCornerType corner = GTK_CORNER_TOP_LEFT;

  parent = GTK_WIDGET (screen)->parent;
  if (!parent)
    return;

  switch (terminal_profile_get_property_enum (priv->profile, TERMINAL_PROFILE_SCROLLBAR_POSITION))
    {
    case TERMINAL_SCROLLBAR_HIDDEN:
      policy = GTK_POLICY_NEVER;
      break;
    case TERMINAL_SCROLLBAR_RIGHT:
      policy = GTK_POLICY_ALWAYS;
      corner = GTK_CORNER_TOP_LEFT;
      break;
    case TERMINAL_SCROLLBAR_LEFT:
      policy = GTK_POLICY_ALWAYS;
      corner = GTK_CORNER_TOP_RIGHT;
      break;
    default:
      g_assert_not_reached ();
      break;
    }

  terminal_screen_container_set_placement (parent, corner);
  terminal_screen_container_set_policy (parent, GTK_POLICY_NEVER, policy);
}

void
terminal_screen_get_size (TerminalScreen *screen,
			  int       *width_chars,
			  int       *height_chars)
{
  VteTerminal *terminal = VTE_TERMINAL (screen);

  *width_chars = terminal->column_count;
  *height_chars = terminal->row_count;
}

void
terminal_screen_get_cell_size (TerminalScreen *screen,
			       int                  *cell_width_pixels,
			       int                  *cell_height_pixels)
{
  VteTerminal *terminal = VTE_TERMINAL (screen);

  *cell_width_pixels = terminal->char_width;
  *cell_height_pixels = terminal->char_height;
}

static void
terminal_screen_match_add (TerminalScreen            *screen,
                           const char           *regexp,
                           int                   flavor)
{
  TerminalScreenPrivate *priv = screen->priv;
  VteTerminal *terminal = VTE_TERMINAL (screen);
  TagData *tag_data;
  int tag;
  
  tag = vte_terminal_match_add (terminal, regexp);
  vte_terminal_match_set_cursor_type (terminal, tag, URL_MATCH_CURSOR);

  tag_data = g_slice_new (TagData);
  tag_data->tag = tag;
  tag_data->flavor = flavor;

  priv->url_tags = g_slist_append (priv->url_tags, tag_data);
}

static void
terminal_screen_skey_match_add (TerminalScreen *screen,
                                const char *regexp,
                                int flavor)
{
  TerminalScreenPrivate *priv = screen->priv;
  VteTerminal *terminal = VTE_TERMINAL (screen);
  TagData *tag_data;
  int tag;
  
  tag = vte_terminal_match_add (terminal, regexp);
  vte_terminal_match_set_cursor_type (terminal, tag, SKEY_MATCH_CURSOR);

  tag_data = g_slice_new (TagData);
  tag_data->tag = tag;
  tag_data->flavor = flavor;

  priv->skey_tags = g_slist_append (priv->skey_tags, tag_data);
}

static void
terminal_screen_skey_match_remove (TerminalScreen *screen)
{
  TerminalScreenPrivate *priv = screen->priv;
  GSList *tags;
  
  for (tags = priv->skey_tags; tags != NULL; tags = tags->next)
    vte_terminal_match_remove (VTE_TERMINAL (screen),
                               GPOINTER_TO_INT(((TagData*)tags->data)->tag));

  g_slist_foreach (priv->skey_tags, (GFunc) free_tag_data, NULL);
  g_slist_free (priv->skey_tags);
  priv->skey_tags = NULL;
}

static char*
terminal_screen_check_match (TerminalScreen *screen,
			     int        column,
			     int        row,
                             int       *flavor)
{
  TerminalScreenPrivate *priv = screen->priv;
  GSList *tags;
  gint tag;
  char *match;

  match = vte_terminal_match_check (VTE_TERMINAL (screen), column, row, &tag);
  for (tags = priv->url_tags; tags != NULL; tags = tags->next)
    {
      TagData *tag_data = (TagData*) tags->data;
      if (GPOINTER_TO_INT(tag_data->tag) == tag)
	{
	  if (flavor)
	    *flavor = tag_data->flavor;
	  return match;
	}
    }

  g_free (match);
  return NULL;
}

static char*
terminal_screen_skey_check_match (TerminalScreen *screen,
				  int        column,
				  int        row,
                                  int       *flavor)
{
  TerminalScreenPrivate *priv = screen->priv;
  GSList *tags;
  gint tag;
  char *match;

  match = vte_terminal_match_check (VTE_TERMINAL (screen), column, row, &tag);
  for (tags = priv->skey_tags; tags != NULL; tags = tags->next)
    {
      TagData *tag_data = (TagData*) tags->data;
      if (GPOINTER_TO_INT(tag_data->tag) == tag)
	{
	  if (flavor)
	    *flavor = tag_data->flavor;
	  return match;
	}
    }

  g_free (match);
  return NULL;
}
