/* 
   Copyright (C) 2004 Fernando Herrera <fherrera@onirica.com>
   Copyright (C) 2004 Mariano Suárez-Alvarez <mariano@gnome.org>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-ui-init.h>

#include <stdlib.h>
#include <popt.h>

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"
#include "gnome-keyring-manager-keyring-manager.h"
#include "gnome-keyring-manager-keyring-editor.h"

#define GNOME_KEYRING_ICON "stock_keyring"

/********************************************************************
 * The application
 */

typedef struct _GKMApplication GKMApplication;

struct _GKMApplication
{
  GHashTable *editors;
  GtkWidget *about;
  GtkWidget *manager;
};

enum
{
  MANAGER_WINDOW,
  EDITOR_WINDOW,
  ABOUT_WINDOW
};

static GKMApplication *application = NULL;

static void gkm_application_window_destroyed_callback (GtkWidget *window, gpointer user_data);

void
gkm_application_init (void)
{
  g_assert (application == NULL);

  application = g_new0 (GKMApplication, 1);

  application->editors = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
  application->about = NULL;
  application->manager = NULL;
}

void
gkm_application_open_keyring_editor_for (const char *keyring, GtkWindow *transient_parent)
{
  GtkWidget *editor;

  g_assert (application != NULL);

  editor = g_hash_table_lookup (application->editors, keyring);

  if (editor == NULL)
    {
      editor = gkm_keyring_editor_new (keyring, transient_parent);
      g_object_set_data_full (G_OBJECT (editor), "keyring-name", g_strdup (keyring), g_free);
      g_signal_connect (G_OBJECT (editor), "destroy", 
                        G_CALLBACK (gkm_application_window_destroyed_callback), GINT_TO_POINTER (EDITOR_WINDOW));
      g_hash_table_insert (application->editors, g_strdup (keyring), editor);
    }
  else
    {
      gtk_window_set_transient_for (GTK_WINDOW (editor), transient_parent);
    }
      
  gtk_window_present (GTK_WINDOW (editor));
}

void
gkm_application_open_about_dialog (GtkWindow *transient_parent)
{
  g_assert (application != NULL);

  if (application->about == NULL)
    {
      application->about = gkm_about_dialog_new (transient_parent);
      g_signal_connect (G_OBJECT (application->about), "destroy", 
                        G_CALLBACK (gkm_application_window_destroyed_callback), GINT_TO_POINTER (ABOUT_WINDOW));
    }
  else
    {
      gtk_window_set_transient_for (GTK_WINDOW (application->about), transient_parent);
    }

  gtk_window_present (GTK_WINDOW (application->about));
}

void
gkm_application_open_keyring_manager (void)
{
  g_assert (application != NULL);

  if (application->manager == NULL)
    {
      application->manager = gkm_keyring_manager_new ();
      g_signal_connect (G_OBJECT (application->manager), "destroy", 
                        G_CALLBACK (gkm_application_window_destroyed_callback), GINT_TO_POINTER (MANAGER_WINDOW));
    }

  gtk_window_present (GTK_WINDOW (application->manager));
}

void
close_editor (gpointer key G_GNUC_UNUSED, gpointer value, gpointer user_data G_GNUC_UNUSED)
{
  g_return_if_fail (GKM_IS_KEYRING_EDITOR (value));

  gtk_widget_destroy (GTK_WIDGET (value));
}

void
gkm_application_quit (void)
{
  /*FIXME: is this all? */
  if (application->about != NULL)
    {
      gtk_widget_destroy (application->about);
    }
  if (application->manager != NULL)
    {
      gtk_widget_destroy (application->manager);
    }
  g_hash_table_foreach (application->editors, (GHFunc) close_editor, NULL);

  g_free (application);
  application = NULL;

  gtk_main_quit ();
}

static void
gkm_application_window_destroyed_callback (GtkWidget *window, gpointer user_data)
{
  const char *name;

  switch (GPOINTER_TO_INT (user_data))
    {
    case MANAGER_WINDOW:
      application->manager = NULL;
      break;
    case ABOUT_WINDOW:
      application->about = NULL;
      break;
    case EDITOR_WINDOW:
      name = g_object_get_data (G_OBJECT (window), "keyring-name");
      g_hash_table_remove (application->editors, name);
      break;
    default:
      g_assert_not_reached ();
    }

  if (g_hash_table_size (application->editors) == 0 && application->manager == NULL)
    {
      gkm_application_quit ();
    }
}

/********************************************************************
 * Getting things going
 */

static void
start_keyring_editor_for_default (GnomeKeyringResult  result,
                                  const char         *default_keyring,
                                  gpointer            user_data G_GNUC_UNUSED)
{
  if (result != GNOME_KEYRING_RESULT_OK)
    {
      g_warning (_("Failed get the default keyring."));

      exit (1);
    }

  if (default_keyring == NULL)
    {
      gkm_application_open_keyring_manager ();
    }
  else
    {
      gkm_application_open_keyring_editor_for (default_keyring, NULL);
    }
}

enum
{ 
  OPTION_DEFAULT_KEYRING,
  OPTION_MANAGER,
  OPTION_KEYRING
};

typedef struct _CommandLine CommandLine;
struct _CommandLine
{
  GList *keyrings;
  guint default_keyring : 1;
  guint manager : 1;
};

static CommandLine *
command_line_new (void)
{
  CommandLine *cmdline;

  cmdline = g_new0 (CommandLine, 1);

  cmdline->keyrings = NULL;
  cmdline->default_keyring = 0;
  cmdline->manager = 0;

  return cmdline;
}

static void
command_line_free (CommandLine *cmdline)
{
  g_list_foreach (cmdline->keyrings, (GFunc) g_free, NULL);
  g_list_free (cmdline->keyrings);
  g_free (cmdline);
}

static void
command_line_exec (CommandLine *cmdline)
{
  GList *keyring;
  gboolean something;

  gkm_application_init ();

  something = FALSE;

  if (cmdline->manager)
    {
      gkm_application_open_keyring_manager ();
      something = TRUE;
    }
  for (keyring = cmdline->keyrings; keyring; keyring = keyring->next)
    {
      gkm_application_open_keyring_editor_for ((const char *) keyring->data, NULL);
      something = TRUE;
    }
  if (cmdline->default_keyring || !something)
    {
      gnome_keyring_get_default_keyring ((GnomeKeyringOperationGetStringCallback) start_keyring_editor_for_default , NULL, NULL);
    }

  gtk_main();
}

static void
parse_option_callback (poptContext ctxt G_GNUC_UNUSED, 
                       enum poptCallbackReason reason,
                       const struct poptOption *opt, 
                       const char *arg, 
                       CommandLine *cmdline)
{
  if (reason != POPT_CALLBACK_REASON_OPTION)
    {
      return;
    }

  switch (opt->val & POPT_ARG_MASK)
    {
      case OPTION_DEFAULT_KEYRING:
        cmdline->default_keyring = 1; 
        break;
      case OPTION_MANAGER:
        cmdline->manager = 1;
        break;
      case OPTION_KEYRING:
        cmdline->keyrings = g_list_append (cmdline->keyrings, g_strdup (arg));
        break;
      default:
        g_assert_not_reached ();
    }
}

struct poptOption options[] =
{
    { NULL, '\0', POPT_ARG_CALLBACK | POPT_CBFLAG_POST, (poptCallbackType) parse_option_callback, 0, NULL, NULL },
    { "default-keyring", 'd', POPT_ARG_NONE, NULL, OPTION_DEFAULT_KEYRING, 
      N_("Open the keyring editor for the default keyring (default)"), NULL },
    { "manager", 'm', POPT_ARG_NONE, NULL, OPTION_MANAGER, 
      N_("Open the keyring manager"), NULL },
    { "keyring", 'k', POPT_ARG_STRING, NULL, OPTION_KEYRING, 
      N_("Open the keyring editor for KEYRING"), N_("KEYRING") }
};

int
main (int argc, char **argv)
{
  GtkIconInfo *icon_info;
  CommandLine *cmdline;
  
  if (setlocale (LC_ALL, "") == NULL)
    {
      g_warning ("locale not understood by C library, internationalization will not work.");
    }

  bindtextdomain (GETTEXT_PACKAGE, GNOME_KEYRING_MANAGER_LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
  
  g_set_application_name (_("Keyring Manager"));

  cmdline = command_line_new ();
  options[0].descrip = (void*) cmdline;

  gnome_program_init (PACKAGE, PACKAGE,
                      LIBGNOMEUI_MODULE, argc, argv,
                      GNOME_PARAM_POPT_TABLE, options,
                      NULL);
  
  icon_info = gtk_icon_theme_lookup_icon (gtk_icon_theme_get_default (), "stock_keyring", 48, 0);
  if (icon_info != NULL)
    {
      gtk_window_set_default_icon_from_file (gtk_icon_info_get_filename (icon_info), NULL);
      gtk_icon_info_free (icon_info);
    }
  
  command_line_exec (cmdline);
  command_line_free (cmdline);

  return 0;
}
