/* gok-word-complete.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
* Copyright 2001,2002 University Of Toronto
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*/

/*
* To use this thing:
* - Call "gok_wordcomplete_open". If it returns TRUE then you're ready to go.
* - Call "gok_wordcomplete_predict" to make the word predictions.
* - Call "gok_wordcomplete_close" when you're done. 
* - To add a word, call "gok_wordcomplete_add_new_word".
*
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "gok-word-complete.h"
#include "gok-keyslotter.h"
#include "main.h"
#include "gok-log.h"
#include "gok-modifier.h"
#include "gok-data.h"
#include "gok-keyboard.h"

#include "word-complete.h" /* TODO: remove this when we make this class abstract */

static void gok_wordcomplete_finalize (GObject *obj);
static void gok_wordcomplete_real_reset (GokWordComplete *complete);
static gchar *gok_wordcomplete_real_process_unichar (GokWordComplete *complete, gunichar letter);
static gchar *gok_wordcomplete_real_unichar_push (GokWordComplete *complete, gunichar letter);
static gchar *gok_wordcomplete_real_unichar_pop (GokWordComplete *complete);
static GokWordCompletionCharacterCategory gok_wordcomplete_real_unichar_get_category (
	                                         GokWordComplete *complete, gunichar letter);
static gchar **gok_wordcomplete_real_predict (GokWordComplete *complete, gint num_predictions);
static gchar *gok_wordcomplete_real_get_word_part (GokWordComplete *complete);


struct _GokWordCompletePrivate 
{
        /* the word, or part word, that is getting completed */
	gchar     *word_part;
};

static GokWordComplete *default_wordcomplete_engine = NULL;

GNOME_CLASS_BOILERPLATE (GokWordComplete, gok_wordcomplete, 
			 GObject, G_TYPE_OBJECT)

static void
gok_wordcomplete_instance_init (GokWordComplete *complete)
{
	complete->priv = g_new0 (GokWordCompletePrivate, 1);
	complete->priv->word_part = NULL;
}

static void
gok_wordcomplete_class_init (GokWordCompleteClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	/* public methods */
	gobject_class->finalize = gok_wordcomplete_finalize;
	klass->reset = gok_wordcomplete_real_reset;
	klass->process_unichar = gok_wordcomplete_real_process_unichar;
	klass->predict = gok_wordcomplete_real_predict;

	/* 'protected' methods, for use by subclasses/implementors only */
	klass->unichar_get_category = gok_wordcomplete_real_unichar_get_category;
	klass->unichar_push = gok_wordcomplete_real_unichar_push;
	klass->unichar_pop = gok_wordcomplete_real_unichar_pop;
	klass->get_word_part = gok_wordcomplete_real_get_word_part;
}

static void
gok_wordcomplete_finalize (GObject *obj)
{
	GokWordComplete *complete = GOK_WORDCOMPLETE (obj);

	if (complete && complete->priv)
		g_free (complete->priv);
	G_OBJECT_CLASS (GOK_WORDCOMPLETE_GET_CLASS (obj))->finalize (obj);
}

/**
* gok_wordcomplete_open
* 
* Opens and initializes the word completor engine.
*
* returns: TRUE if it was opend OK, FALSE if not.
**/
gboolean gok_wordcomplete_open (GokWordComplete *complete, gchar *directory)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	
	if (klass->open)
		return (* klass->open)(complete, directory);
	else
		return FALSE;
}

/**
* gok_wordcomplete_close
* 
* Closes the word completor engine.
*
* returns: void
**/
void gok_wordcomplete_close (GokWordComplete *complete)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);

	if (klass->close)
		(*klass->close) (complete);
}

/**
 * gok_wordcomplete_unichar_push:
 *
 **/
gchar *
gok_wordcomplete_unichar_push (GokWordComplete *complete, gunichar letter)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);

	if (klass->unichar_push)
		return (*klass->unichar_push) (complete, letter);
	else
		return NULL;
}

/**
 * gok_wordcomplete_unichar_pop:
 *
 **/
gchar *
gok_wordcomplete_unichar_pop (GokWordComplete *complete)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);

	if (klass->unichar_pop)
		return (*klass->unichar_pop) (complete);
	else
		return NULL;
}



/**
 * gok_wordcomplete_unichar_get_category
 *
 **/
GokWordCompletionCharacterCategory
gok_wordcomplete_unichar_get_category (GokWordComplete *complete, gunichar letter)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	
	if (klass->unichar_get_category)
		return (* klass->unichar_get_category)(complete, letter);
	else
		return GOK_WORDCOMPLETE_CHAR_TYPE_NORMAL;
}

/**
 * gok_wordcomplete_process_unichar:
 *
 **/
gchar *
gok_wordcomplete_process_unichar (GokWordComplete *complete, gunichar letter)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (klass->process_unichar)
		(* klass->process_unichar) (complete, letter);
}

/**
 *
 **/
gchar *
gok_wordcomplete_get_word_part (GokWordComplete *complete)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (klass->get_word_part)
		return (* klass->get_word_part) (complete);
	else
		return NULL;
}

/**
* gok_wordcomplete_process_and_predict:
* 
* Makes a prediction based on the effect of @letter on the current prediction state machine.
*
* returns: An 
array of strings representing the predicted completions.
**/
gchar **
gok_wordcomplete_process_and_predict (GokWordComplete *complete, const gunichar letter, gint num_predictions)
{
	return gok_wordcomplete_predict_string (complete, 
						gok_wordcomplete_process_unichar (complete, letter),
						num_predictions);
}

/**
* gok_wordcomplete_predict_string:
* 
* Makes a prediction. If the currently displayed keyboard is showing prediction
* keys then they are filled in with the predictions.
*
* returns: An 
array of strings representing the predicted completions.
**/
gchar **
gok_wordcomplete_predict_string (GokWordComplete *complete, const gchar *string, gint num_predictions)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (klass->predict_string)
		return (* klass->predict_string) (complete, string, num_predictions);
	else
		return NULL;
}

/**
* gok_wordcomplete_predict
* 
* Makes a prediction. If the currently displayed keyboard is showing prediction
* keys then they are filled in with the predictions.
*
* returns: An 
array of strings representing the predicted completions.
**/
gchar **
gok_wordcomplete_predict (GokWordComplete *complete, gint num_predictions)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (klass->predict)
		(* klass->predict) (complete, num_predictions);
}

/**
* gok_wordcomplete_reset
* 
* Resets the part word buffer.
*
* returns: void
**/
void 
gok_wordcomplete_reset (GokWordComplete *complete)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (klass->reset)
		(* klass->reset) (complete);
}

/**
* gok_wordcomplete_add_new_word
* 
* Adds a new word to the predictor dictionary.
*
* returns: TRUE if the word was added to the dictionary, FALSE if not.
**/
gboolean gok_wordcomplete_add_new_word (GokWordComplete *complete, const gchar* pWord)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (*klass->add_new_word)
		return (* klass->add_new_word)(complete, pWord);
	else
		return FALSE;
}

/**
* gok_wordcomplete_increment_word_frequency
* 
* Increments the frequency of a word in the dictionary.
*
* returns: TRUE if the word's frequency was incremented, FALSE if not.
**/
gboolean 
gok_wordcomplete_increment_word_frequency (GokWordComplete *complete, const gchar* pWord)
{
	GokWordCompleteClass *klass = GOK_WORDCOMPLETE_GET_CLASS (complete);
	if (*klass->increment_word_frequency)
		return (* klass->increment_word_frequency)(complete, pWord);
	else
		return FALSE;
}

/**
 * gok_wordcomplete_get_default:
 *
 * Returns the current default word completion engine.
 **/
GokWordComplete*
gok_wordcomplete_get_default (void)
{
	if (default_wordcomplete_engine == NULL)
		default_wordcomplete_engine = g_object_new (GOK_TYPE_TRIEWORDCOMPLETE, NULL);  
        /* TODO: establish default subtype and create type - making the base class abstract */
	return default_wordcomplete_engine;
}

/* private methods */
static gboolean 
gok_wordcomplete_unichar_is_backspace (GokWordComplete *complete, gunichar letter) 
{
	return ((letter == 0x08) || (letter == 0x7f) || (letter == 0xfffd));
}

static gboolean
gok_wordcomplete_unichar_is_delimiter (GokWordComplete *complete, gunichar letter)
{		
/* 
 * Is this whitespace, empty, punctuation, or nonprintable? Also catches NULL/0 
 * we make exception for U+00a0 (nonbreak space) so 'words' containing 
 * space can be handled. 
 */
	return (!g_unichar_validate (letter) ||  /* if invalid, */
		((g_unichar_ispunct (letter) || !g_unichar_isgraph (letter)) /* or punct/non-graphical, */
		 && (letter != 0x00a0)) || /* but not nbspace */
		(letter == 0x2f)); /* or if destructive backspace */
}

/* implementations of GokWordCompleteClass methods */
static void 
gok_wordcomplete_real_reset (GokWordComplete *complete)
{
	g_free (complete->priv->word_part);
	complete->priv->word_part = NULL;
}

static gchar *
gok_wordcomplete_real_unichar_push (GokWordComplete *complete, gunichar letter)
{
	gchar *tmp, utf8char[7]; /* min 6, plus NULL */
	gchar *word_part = NULL;

	if (complete && complete->priv) 
	{
		word_part = gok_wordcomplete_get_word_part (complete);
		utf8char [g_unichar_to_utf8 (letter, utf8char)] = '\0';
		if (word_part) 
		{
			tmp = g_strconcat (word_part, utf8char, NULL);
			g_free (word_part);
			word_part = tmp;
		}
		else
		{
			word_part = g_strdup (utf8char);
		}
		complete->priv->word_part = word_part;
	}
	return word_part;
}

static gchar *
gok_wordcomplete_real_unichar_pop (GokWordComplete *complete)
{
	gint length;
	if (complete->priv && complete->priv->word_part) 
	{
		length = strlen (complete->priv->word_part);
		if (length)
		{
			*g_utf8_offset_to_pointer (complete->priv->word_part, --length) = '\0';
		}
	}
	return (complete->priv) ? complete->priv->word_part : NULL;
}

/**
 * gok_wordcomplete_real_unichar_get_category
 *
 **/
static GokWordCompletionCharacterCategory
gok_wordcomplete_real_unichar_get_category (GokWordComplete *complete, gunichar letter)
{
	if (gok_wordcomplete_unichar_is_backspace (complete, letter))
		return GOK_WORDCOMPLETE_CHAR_TYPE_BACKSPACE;
	else if (gok_wordcomplete_unichar_is_delimiter (complete, letter))
		return GOK_WORDCOMPLETE_CHAR_TYPE_DELIMITER;
	else
		return GOK_WORDCOMPLETE_CHAR_TYPE_NORMAL;
}

static gchar *
gok_wordcomplete_real_process_unichar (GokWordComplete *complete, gunichar letter)
{
	gchar *prefix = NULL;
	switch (gok_wordcomplete_unichar_get_category (complete, letter))
	{
	case GOK_WORDCOMPLETE_CHAR_TYPE_BACKSPACE:
		prefix = gok_wordcomplete_unichar_pop (complete);
		break;
	case GOK_WORDCOMPLETE_CHAR_TYPE_DELIMITER:
		gok_wordcomplete_reset (complete);
		break;
	case GOK_WORDCOMPLETE_CHAR_TYPE_NORMAL:
	default:
		prefix = gok_wordcomplete_unichar_push (complete, letter);
		break;
	}
	return prefix;
}

/**
 **/
gchar *
gok_wordcomplete_real_get_word_part (GokWordComplete *complete)
{
	if (complete && complete->priv)
		return complete->priv->word_part;
	else
		return NULL;
}

/**
* gok_wordcomplete_real_predict
* 
* Makes a prediction. If the currently displayed keyboard is showing prediction
* keys then they are filled in with the predictions.
*
* returns: An array of strings representing the predicted completions.
**/
gchar **
gok_wordcomplete_real_predict (GokWordComplete *complete, gint num_predictions)
{
	return gok_wordcomplete_predict_string (complete, 
						gok_wordcomplete_get_word_part (complete),
						num_predictions);
}
