/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gkr-pk-util.c - miscellaneous utilities for dealing with PKCS#11

   Copyright (C) 2007 Stefan Walter

   The Gnome Keyring Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Keyring Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Author: Stef Walter <stef@memberwebs.com>
*/

#include "config.h"

#include "gkr-pk-util.h"

#include "pkcs11/pkcs11.h"
#include "pkcs11/pkcs11g.h"
#include "pkcs11/pkcs11n.h"

#include <glib.h>

#include <stdio.h>

GkrPkDataType
gkr_pk_attribute_data_type (CK_ATTRIBUTE_TYPE type)
{
	switch(type)
	{
	/* CK_ULONG attribute types */
	case CKA_CLASS:
	case CKA_CERTIFICATE_TYPE:
	case CKA_CERTIFICATE_CATEGORY:
	case CKA_KEY_TYPE:
	case CKA_MODULUS_BITS:
	case CKA_PRIME_BITS:
	/* case CKA_SUBPRIME_BITS: */
	case CKA_SUB_PRIME_BITS: 
	case CKA_VALUE_BITS:
	case CKA_VALUE_LEN:
	case CKA_KEY_GEN_MECHANISM:
	case CKA_HW_FEATURE_TYPE:
	case CKA_PIXEL_X:
	case CKA_PIXEL_Y:
	case CKA_RESOLUTION:
	case CKA_CHAR_ROWS:
	case CKA_CHAR_COLUMNS:
	case CKA_BITS_PER_PIXEL:
	case CKA_MECHANISM_TYPE:
	case CKA_JAVA_MIDP_SECURITY_DOMAIN:
	case CKA_TRUST_DIGITAL_SIGNATURE:
	case CKA_TRUST_NON_REPUDIATION:
	case CKA_TRUST_KEY_ENCIPHERMENT:
	case CKA_TRUST_DATA_ENCIPHERMENT:
	case CKA_TRUST_KEY_AGREEMENT:
	case CKA_TRUST_KEY_CERT_SIGN:
	case CKA_TRUST_CRL_SIGN:
	case CKA_TRUST_SERVER_AUTH:
	case CKA_TRUST_CLIENT_AUTH:
	case CKA_TRUST_CODE_SIGNING:
	case CKA_TRUST_EMAIL_PROTECTION:
	case CKA_TRUST_IPSEC_END_SYSTEM:
	case CKA_TRUST_IPSEC_TUNNEL:
	case CKA_TRUST_IPSEC_USER:
	case CKA_TRUST_TIME_STAMPING:
	case CKA_GNOME_USER_TRUST:
		return GKR_PK_DATA_ULONG;

	/* CK_BBOOL attribute types */
	case CKA_TOKEN:
	case CKA_PRIVATE:
	case CKA_MODIFIABLE:
	case CKA_TRUSTED:
	case CKA_SENSITIVE:
	case CKA_DECRYPT:
	case CKA_SIGN:
	case CKA_SIGN_RECOVER:
	case CKA_UNWRAP:
	case CKA_EXTRACTABLE:
	case CKA_NEVER_EXTRACTABLE:
	case CKA_ALWAYS_SENSITIVE:
	case CKA_WRAP_WITH_TRUSTED:
	case CKA_ALWAYS_AUTHENTICATE:
	case CKA_ENCRYPT:
	case CKA_WRAP:
	case CKA_VERIFY:
	case CKA_VERIFY_RECOVER:
	case CKA_DERIVE:
	case CKA_LOCAL:
	case CKA_RESET_ON_INIT:
	case CKA_HAS_RESET:
	case CKA_COLOR:
	case CKA_TRUST_STEP_UP_APPROVED:
	case CKA_GNOME_PURPOSE_RESTRICTED:
	case CKA_GNOME_PURPOSE_SSH_AUTH:
	case CKA_GNOME_PURPOSE_SERVER_AUTH:
	case CKA_GNOME_PURPOSE_CLIENT_AUTH:
	case CKA_GNOME_PURPOSE_CODE_SIGNING:
	case CKA_GNOME_PURPOSE_EMAIL_PROTECTION:
	case CKA_GNOME_PURPOSE_IPSEC_END_SYSTEM:
	case CKA_GNOME_PURPOSE_IPSEC_TUNNEL:
	case CKA_GNOME_PURPOSE_IPSEC_USER:
	case CKA_GNOME_PURPOSE_TIME_STAMPING:
		return GKR_PK_DATA_BOOL;

	/* Raw or string data */
	case CKA_LABEL:
	case CKA_APPLICATION:
	case CKA_VALUE:
	case CKA_OBJECT_ID:
	case CKA_CHECK_VALUE:
	case CKA_ISSUER:
	case CKA_SERIAL_NUMBER:
	case CKA_SUBJECT:
	case CKA_ID:
	case CKA_URL:
	case CKA_HASH_OF_SUBJECT_PUBLIC_KEY:
	case CKA_HASH_OF_ISSUER_PUBLIC_KEY:
	case CKA_AC_ISSUER:
	case CKA_OWNER:
	case CKA_ATTR_TYPES:
	case CKA_MODULUS:
	case CKA_PUBLIC_EXPONENT:
	case CKA_PRIVATE_EXPONENT:
	case CKA_PRIME_1:
	case CKA_PRIME_2:
	case CKA_EXPONENT_1:
	case CKA_EXPONENT_2:
	case CKA_COEFFICIENT:
	case CKA_PRIME:
	case CKA_SUBPRIME:
	case CKA_BASE:
	case CKA_ECDSA_PARAMS:
	/* case CKA_EC_PARAMS: */
	case CKA_EC_POINT:
	case CKA_CHAR_SETS:
	case CKA_ENCODING_METHODS:
	case CKA_MIME_TYPES:
	case CKA_REQUIRED_CMS_ATTRIBUTES:
	case CKA_DEFAULT_CMS_ATTRIBUTES:
	case CKA_SUPPORTED_CMS_ATTRIBUTES:
	case CKA_CERT_SHA1_HASH:
	case CKA_CERT_MD5_HASH:
	case CKA_GNOME_PURPOSE_OIDS: 
	case CKA_START_DATE:
	case CKA_END_DATE:
		return GKR_PK_DATA_BYTES;

	/* Arrays are nasty */
	case CKA_WRAP_TEMPLATE:
	case CKA_ALLOWED_MECHANISMS:
	case CKA_UNWRAP_TEMPLATE:
	default:
		return GKR_PK_DATA_UNKNOWN;
	};
}

CK_ATTRIBUTE_PTR
gkr_pk_attribute_new (CK_ATTRIBUTE_TYPE type)
{
	CK_ATTRIBUTE_PTR attr;
	
	attr = g_slice_new0 (CK_ATTRIBUTE);
	attr->type = type;
	
	return attr;
} 

CK_ATTRIBUTE_PTR
gkr_pk_attribute_dup (const CK_ATTRIBUTE_PTR attr)
{
	CK_ATTRIBUTE_PTR nattr = gkr_pk_attribute_new (attr->type);
	gkr_pk_attribute_copy (nattr, attr);
	return nattr;
}

gboolean
gkr_pk_attribute_equal (const CK_ATTRIBUTE_PTR one, const CK_ATTRIBUTE_PTR two)
{
	if (one->type != two->type)
		return FALSE;

	if (one->ulValueLen != two->ulValueLen)
		return FALSE;
	if (one->pValue == two->pValue)
		return TRUE;
	if (!one->pValue || !two->pValue)
		return FALSE;
	return memcmp (one->pValue, two->pValue, one->ulValueLen) == 0;	
}

void
gkr_pk_attribute_copy (CK_ATTRIBUTE_PTR dest, const CK_ATTRIBUTE_PTR attr)
{
	g_assert (dest && attr);
	gkr_pk_attribute_set_data (dest, attr->pValue, attr->ulValueLen);
	dest->type = attr->type;
}

void
gkr_pk_attribute_set_invalid (CK_ATTRIBUTE_PTR attr)
{
	g_assert (attr);
	gkr_pk_attribute_clear (attr);	
	attr->ulValueLen = (CK_ULONG)-1;
}

void
gkr_pk_attribute_take_data (CK_ATTRIBUTE_PTR attr, gpointer value, gsize n_value)
{
	g_assert (attr);

	gkr_pk_attribute_clear (attr);

	attr->ulValueLen = n_value;
	if (n_value > 0) {
		g_assert (value);
		attr->pValue = value;
	}	
}

void
gkr_pk_attribute_set_data (CK_ATTRIBUTE_PTR attr, gconstpointer value, gsize n_value)
{
	g_assert (attr);

	gkr_pk_attribute_clear (attr);

	attr->ulValueLen = n_value;
	if (n_value > 0) {
		g_assert (value);
		attr->pValue = g_malloc (n_value);
		memcpy (attr->pValue, value, n_value);
	}
}

void
gkr_pk_attribute_set_string (CK_ATTRIBUTE_PTR attr, const gchar *str)
{
	g_assert (attr);
	g_assert (str);
	
	gkr_pk_attribute_set_data (attr, str, strlen (str) + 1);
}

void
gkr_pk_attribute_set_boolean (CK_ATTRIBUTE_PTR attr, CK_BBOOL value)
{
	g_assert (attr);
	
	gkr_pk_attribute_clear (attr);
	attr->pValue = g_new (CK_BBOOL, 1);
	*((CK_BBOOL*)attr->pValue) = value;
	attr->ulValueLen = sizeof (CK_BBOOL);
}

void
gkr_pk_attribute_set_date (CK_ATTRIBUTE_PTR attr, time_t time)
{
	CK_DATE *date;
	struct tm tm;
	gchar buf[16];
	
	/* 'Empty' date as defined in PKCS#11 */
	if (time == (time_t)-1) {
		gkr_pk_attribute_set_data (attr, NULL, 0);
		return;
	}
	
	gkr_pk_attribute_clear (attr);
	
	if (!gmtime_r (&time, &tm))
		g_return_if_reached ();
		
	date = g_new0 (CK_DATE, 1);

	g_assert (sizeof (date->year) == 4);
	snprintf ((char*)buf, 5, "%04d", 1900 + tm.tm_year);
	memcpy (date->year, buf, 4);
	 
	g_assert (sizeof (date->month) == 2);
	snprintf ((char*)buf, 3, "%02d", tm.tm_mon + 1);
	memcpy (date->month, buf, 2);
	
	g_assert (sizeof (date->day) == 2);
	snprintf ((char*)buf, 3, "%02d", tm.tm_mday);
	memcpy (date->day, buf, 2);
		
	attr->pValue = date;
	attr->ulValueLen = sizeof (CK_DATE);
}

void
gkr_pk_attribute_set_id (CK_ATTRIBUTE_PTR attr, gkrconstid id)
{
	const guchar* data;
	gsize n_data;
	
	g_assert (attr);
	
	data = gkr_id_get_raw (id, &n_data);
	g_return_if_fail (data && n_data);
	
	gkr_pk_attribute_set_data (attr, data, n_data);
}

void
gkr_pk_attribute_set_ulong (CK_ATTRIBUTE_PTR attr, CK_ULONG value)
{
	g_assert (attr);
	
	gkr_pk_attribute_clear (attr);
	attr->pValue = g_new (CK_ULONG, 1);
	*((CK_ULONG*)attr->pValue) = value;
	attr->ulValueLen = sizeof (CK_ULONG);	
}

void
gkr_pk_attribute_set_mpi (CK_ATTRIBUTE_PTR attr, gcry_mpi_t mpi)
{
	gsize len;
  	gcry_error_t gcry;

	g_assert (attr);
	g_assert (mpi);
	
	gkr_pk_attribute_clear (attr);
	
	/* Get the size */
	gcry = gcry_mpi_print (GCRYMPI_FMT_USG, NULL, 0, &len, mpi);
	g_return_if_fail (gcry == 0);
	
	if (!len)
		return;

	attr->pValue = g_malloc (len);
	attr->ulValueLen = len;
	
	/* Write in directly to attribute */
	gcry = gcry_mpi_print (GCRYMPI_FMT_USG, attr->pValue, len, &len, mpi);	
	g_return_if_fail (gcry == 0);
}

gboolean
gkr_pk_attribute_get_boolean (const CK_ATTRIBUTE_PTR attr, CK_BBOOL *value)
{
	if (attr->ulValueLen != sizeof (CK_BBOOL))
		return FALSE;
	*value = *((const CK_BBOOL*)attr->pValue);
	return TRUE;
}

gboolean
gkr_pk_attribute_get_ulong (const CK_ATTRIBUTE_PTR attr, CK_ULONG *value)
{
	if (attr->ulValueLen != sizeof (CK_ULONG))
		return FALSE;
	*value = *((const CK_ULONG*)attr->pValue);
	return TRUE;
}

void
gkr_pk_attribute_clear (CK_ATTRIBUTE_PTR attr)
{
	if (attr->pValue) {
		g_assert (attr->ulValueLen > 0);
		g_assert (attr->ulValueLen != (CK_ULONG)-1);
		g_free (attr->pValue);
		attr->pValue = NULL;
	}
	attr->ulValueLen = 0;
}

void
gkr_pk_attribute_free (gpointer v)
{
	if (v) {
		CK_ATTRIBUTE_PTR attr = (CK_ATTRIBUTE_PTR)v;
		gkr_pk_attribute_clear (attr);
		g_slice_free (CK_ATTRIBUTE, attr);
	}
}

CK_ATTRIBUTE_PTR
gkr_pk_attributes_find (const GArray* attrs, CK_ATTRIBUTE_TYPE type)
{
	CK_ATTRIBUTE_PTR attr;
	guint i;

	for (i = 0; i < attrs->len; ++i) {
		attr = &(g_array_index (attrs, CK_ATTRIBUTE, i));
		if (attr->type == type)
			return attr;
	}

	return NULL;
}

gboolean
gkr_pk_attributes_ulong (const GArray* attrs, CK_ATTRIBUTE_TYPE type, CK_ULONG *value)
{
	CK_ATTRIBUTE_PTR attr = gkr_pk_attributes_find (attrs, type);
	if (attr && attr->pValue && attr->ulValueLen == sizeof (CK_ULONG)) {
		*value = *((CK_ULONG*)attr->pValue);
		return TRUE;
	}
	return FALSE;
}

gboolean
gkr_pk_attributes_boolean (const GArray* attrs, CK_ATTRIBUTE_TYPE type, CK_BBOOL *value)
{
	CK_ATTRIBUTE_PTR attr = gkr_pk_attributes_find (attrs, type);
	if (attr && attr->pValue && attr->ulValueLen == sizeof (CK_BBOOL)) {
		*value = *((CK_BBOOL*)attr->pValue);
		return TRUE;
	}
	return FALSE;
}

gboolean
gkr_pk_attributes_mpi (const GArray* attrs, CK_ATTRIBUTE_TYPE type, gcry_mpi_t *mpi)
{
	CK_ATTRIBUTE_PTR attr;
	gcry_error_t gcry;
	
	attr = gkr_pk_attributes_find (attrs, type);
	if (attr && attr->pValue && attr->ulValueLen) {
		gcry = gcry_mpi_scan (mpi, GCRYMPI_FMT_USG, attr->pValue, attr->ulValueLen, NULL);
		return gcry == 0;
	}
	return FALSE;
}

void
gkr_pk_attributes_append (GArray *attrs, CK_ATTRIBUTE_PTR attr)
{
	g_array_append_vals (attrs, attr, 1);
	memset (attr, 0, sizeof (*attr));
}

void
gkr_pk_attributes_free (GArray *attrs)
{
	CK_ATTRIBUTE_PTR attr;
	guint i;

	if (!attrs)
		return;
		
	for (i = 0; i < attrs->len; ++i) {
		attr = &(g_array_index (attrs, CK_ATTRIBUTE, i));
		gkr_pk_attribute_clear (attr);
	}

	g_array_free (attrs, TRUE);
}

gboolean
gkr_pk_attribute_is_consumed (CK_ATTRIBUTE_PTR attr)
{
	return attr->type == (CK_ULONG)-1;
}

void
gkr_pk_attribute_consume (CK_ATTRIBUTE_PTR attr)
{
	attr->type = (CK_ULONG)-1;
} 

void
gkr_pk_attributes_consume (GArray *attrs, ...)
{
	CK_ATTRIBUTE_TYPE type;
	CK_ATTRIBUTE_PTR attr;
	GArray *types;
	guint i, j;
	va_list va;

	/* Convert the var args into an array */
	types = g_array_new (FALSE, TRUE, sizeof (CK_ATTRIBUTE_TYPE));
	va_start (va, attrs);
	while ((type = va_arg (va, CK_ATTRIBUTE_TYPE)) != (CK_ULONG)-1)
		 g_array_append_val (types, type);
	va_end (va);
	
	/* Consume each attribute whose type was in the var args */
	for (i = 0; i < attrs->len; ++i) {
		attr = &(g_array_index (attrs, CK_ATTRIBUTE, i));
		if (gkr_pk_attribute_is_consumed (attr))
			continue;
		for (j = 0; j < types->len; ++j) {
			if (attr->type == g_array_index (types, CK_ATTRIBUTE_TYPE, j)) {
				gkr_pk_attribute_consume (attr);
				break;
			}
		}
	}
	
	g_array_free (types, TRUE);
}

gboolean
gkc_pk_class_is_private (CK_OBJECT_CLASS cls)
{
	switch (cls) {
	case CKO_PRIVATE_KEY:
	case CKO_SECRET_KEY:
		return TRUE;
	default:
		return FALSE;
	}
}
