/*
 * Copyright (C) 2003 Ross Burton <ross@burtonini.com>
 *
 * Sound Juicer - sj-play.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Ronald S. Bultje <rbultje@ronald.bitfreak.net>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkrange.h>
#include <gtk/gtkstatusbar.h>
#include <gtk/gtkmessagedialog.h>

#include "bacon-volume.h"
#include "sound-juicer.h"

static GstElement *pipeline = NULL;
static guint id = 0, button_change_id = 0;
static gint seek_to_track = -1, current_track = -1;
static gboolean seeking = FALSE, internal_update = FALSE;
static guint64 slen = GST_CLOCK_TIME_NONE;
static gfloat vol = 1.0;

static GtkTreeIter current_iter;

static GtkWidget *play_button, *next_menuitem, *prev_menuitem, *reread_menuitem, *seek_scale, *volume_button, *statusbar;

/**
 * Select track number.
 */
static gboolean
select_track (void)
{
  GstElement *cd;

  if (seek_to_track == -1) {
    gint tracks =
        gtk_tree_model_iter_n_children (GTK_TREE_MODEL (track_store), NULL);

    seek_to_track = 0;
    while (seek_to_track < tracks) {
      gboolean do_play;
      GtkTreeIter next_iter;

      if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (track_store),
                                          &next_iter, NULL, seek_to_track)) {
        g_warning (G_STRLOC ": cannot get nth child");
        return FALSE;
      }
      gtk_tree_model_get (GTK_TREE_MODEL (track_store), &next_iter,
          COLUMN_EXTRACT, &do_play, -1);
      if (do_play)
        break;
      seek_to_track++;
    }
    if (seek_to_track == tracks) {
      seek_to_track = -1;
      return FALSE;
    }
  }
  if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (track_store),
                                     &current_iter, NULL, seek_to_track)) {
    g_warning (G_STRLOC ": cannot get nth child");
    return FALSE;
  }

  gst_element_set_state (pipeline, GST_STATE_PAUSED);
  cd = gst_bin_get_by_name_recurse_up (GST_BIN (pipeline), "cd-source");
  gst_element_seek (pipeline, 1.0, gst_format_get_by_nick ("track"), GST_SEEK_FLAG_FLUSH, GST_SEEK_TYPE_SET, seek_to_track, GST_SEEK_TYPE_NONE, -1);
  /* get_state neccessary because of bug #326311 */
  gst_element_get_state (pipeline, NULL, NULL, -1);
  current_track = seek_to_track;
  seek_to_track = -1;

  return TRUE;
}

/**
 * Start playing.
 */
static void
play (void)
{
  gst_element_set_state (pipeline, GST_STATE_PLAYING);
  
  gtk_widget_set_sensitive (next_menuitem, TRUE);
  gtk_widget_set_sensitive (prev_menuitem, TRUE);
  gtk_widget_set_sensitive (reread_menuitem, FALSE);
}

/**
 * Pause
 */
static void
pause (void)
{
  gst_element_set_state (pipeline, GST_STATE_PAUSED);
}

/**
 * Stop.
 */
static void
stop (void)
{
  if (pipeline != NULL)
    gst_element_set_state (pipeline, GST_STATE_NULL);

  gtk_widget_set_sensitive (next_menuitem, FALSE);
  gtk_widget_set_sensitive (prev_menuitem, FALSE);
  gtk_widget_set_sensitive (reread_menuitem, TRUE);
  /* Call this to ensure that the UI is reset */
  sj_main_set_title (NULL);
}

/**
 * Are we playing?
 */
static gboolean
is_playing (void)
{
  return (pipeline != NULL && GST_STATE (pipeline) == GST_STATE_PLAYING);
}

/*
 * callbacks.
 */

static void
cb_hop_track (GstBus *bus, GstMessage *message, gpointer user_data)
{
  GtkTreeModel *model;
  gint tracks, next_track = current_track + 1;
  GtkTreeIter next_iter;

  model = GTK_TREE_MODEL (track_store);
  tracks = gtk_tree_model_iter_n_children (model, NULL);

  while (next_track < tracks) {
    gboolean do_play;

    if (!gtk_tree_model_iter_nth_child (model, &next_iter, NULL, next_track))
      return;
    gtk_tree_model_get (GTK_TREE_MODEL (track_store), &next_iter,
        COLUMN_EXTRACT, &do_play, -1);
    if (do_play)
      break;
    next_track++;
  }

  if (next_track >= tracks) {
    stop ();
    seek_to_track = 0;
  } else {
    char *title;
    seek_to_track = next_track;
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_IDLE, -1);
    select_track ();
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_PLAYING, -1);
    gtk_tree_model_get (model,
        &current_iter, COLUMN_TITLE, &title, -1);
    sj_main_set_title (title);
    g_free (title);
    play ();
  }
}

static void
cb_error (GstBus *bus, GstMessage *message, gpointer user_data)
{
  GtkWidget *dialog;
  GError *error;

  gst_message_parse_error (message, &error, NULL);
  dialog = gtk_message_dialog_new (GTK_WINDOW (main_window), 0,
				   GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				   _("Error playing CD.\n\nReason: %s"),
				   error->message);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  
  g_error_free (error);
}

static gchar *
get_label_for_time (gint sec)
{
  return g_strdup_printf ("%d:%02d", sec / 60, sec % 60);
}

static void
set_statusbar_pos (gint pos, gint len)
{
  static gint prev_pos = 0, prev_len = 0;
  gchar *x, *y, *r;

  if (prev_pos == pos && prev_len == len)
    return;
  prev_pos = pos;
  prev_len = len;

  x = get_label_for_time (pos);
  y = get_label_for_time (len);
  r = g_strdup_printf ("%s / %s", x, y);
  g_free (x);
  g_free (y);

  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 0, r);
  g_free (r);
}

static gboolean
cb_set_time (gpointer data)
{
  GstElement *cd;
  GstFormat fmt = GST_FORMAT_TIME;
  gint64 pos, len;

  if (seeking)
    return TRUE;

  cd = gst_bin_get_by_name_recurse_up (GST_BIN (pipeline), "cd-source");

  if (gst_element_query_duration (cd, &fmt, &len) &&
      gst_element_query_position (cd, &fmt, &pos)) {
    internal_update = TRUE;
    gtk_range_set_value (GTK_RANGE (seek_scale), (gdouble) pos / len);
    set_statusbar_pos (pos / GST_SECOND, len / GST_SECOND);
    slen = len;
    internal_update = FALSE;
  }

  /* repeat */
  return TRUE;
}

static gboolean
cb_change_button (gpointer data)
{
  button_change_id = 0;

  gtk_button_set_label (GTK_BUTTON (data), GTK_STOCK_MEDIA_PLAY);

  /* once */
  return FALSE;
}

static void
cb_state (GstBus *bus, GstMessage *message, gpointer user_data)
{
  GstState old_state, new_state;
  GstStateChange transition;

  gst_message_parse_state_changed (message, &old_state, &new_state, NULL);
  transition = GST_STATE_TRANSITION (old_state, new_state);

  /* all pipe elements will receive state transition messages,
   * so we filter those out. This won't be neccessary after
   * the playbin migration */
  if ((GstElement*)GST_MESSAGE_SRC(message) != pipeline)
    return;

  if (transition == GST_STATE_CHANGE_READY_TO_PAUSED) {
    char *title;
    gtk_widget_show (seek_scale);
    gtk_widget_show (volume_button);
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_PLAYING, -1);
    gtk_tree_model_get (GTK_TREE_MODEL (track_store),
        &current_iter, COLUMN_TITLE, &title, -1);
    sj_main_set_title (title);
    g_free (title);
  } else if (transition == GST_STATE_CHANGE_PAUSED_TO_READY) {
    gtk_widget_hide (seek_scale);
    gtk_widget_hide (volume_button);
    gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 0);
    slen = GST_CLOCK_TIME_NONE;
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_IDLE, -1);
    sj_main_set_title (NULL);
    gtk_statusbar_push (GTK_STATUSBAR (statusbar), 0, "");
    current_track = -1;
  } else if (transition == GST_STATE_CHANGE_PAUSED_TO_PLAYING) {
    gtk_button_set_label (GTK_BUTTON (play_button), GTK_STOCK_MEDIA_PAUSE);
    if (id)
      g_source_remove (id);
    id = g_timeout_add (100, (GSourceFunc) cb_set_time, NULL);
    if (button_change_id) {
      g_source_remove (button_change_id);
      button_change_id = 0;
    }
  } else if (transition == GST_STATE_CHANGE_PLAYING_TO_PAUSED) {
    if (id) {
      g_source_remove (id);
      id = 0;
    }
    /* otherwise button flickers on track-switch */
    if (button_change_id)
      g_source_remove (button_change_id);
    button_change_id =
        g_timeout_add (500, (GSourceFunc) cb_change_button, play_button);
  }
}

/**
 * Create and update pipeline for playback of a particular track.
 */

static gboolean
setup (GError **err)
{
  if (!pipeline) {
    GstBus *bus;
    GstElement *out, *queue, *conv, *resample, *cdp, *volume;

    /* TODO:
     * replace with playbin.  Tim says:
     * 
     * just create playbin and your audio sink, then do g_object_set (playbin,
     * "audio-sink", audiosink, NULL); and set cdda://<track-number> as URI on
     * playbin and set it to PLAYING.  and then go
     * gst_element_query_duration/position(playbin) and
     * gst_element_seek(playbin, .... track_format, track-1, ...) for changing
     * track
     */

    pipeline = gst_pipeline_new ("playback");

    bus = gst_element_get_bus (pipeline);
    gst_bus_add_signal_watch (bus);

    g_signal_connect (bus, "message::eos", G_CALLBACK (cb_hop_track), NULL);
    g_signal_connect (bus, "message::error", G_CALLBACK (cb_error), NULL);
    g_signal_connect (bus, "message::state-changed", G_CALLBACK (cb_state), NULL);

    cdp = gst_element_factory_make (CD_SRC, "cd-source");
    if (!cdp) {
      gst_object_unref (GST_OBJECT (pipeline));
      pipeline = NULL;
      g_set_error (err, 0, 0, _("Failed to create CD source element"));
      return FALSE;
    }
    /* do not set to 1, that messes up buffering. 2 is enough to keep
     * noise from the drive down. */
    /* TODO: will not notice drive changes, should monitor */
    g_object_set (G_OBJECT (cdp),
                  "read-speed", 2,
                  "device", nautilus_burn_drive_get_device (drive),
                  NULL);

    queue = gst_element_factory_make ("queue", "queue"); g_assert (queue);
    /* Set the buffer size to protect against underflow on busy systems */
    g_object_set (queue,
                  "min-threshold-time", (guint64) 200 * GST_MSECOND,
                  "max-size-time", (guint64) 2 * GST_SECOND,
                  NULL);
    conv = gst_element_factory_make ("audioconvert", "conv"); g_assert (conv);
    resample = gst_element_factory_make ("audioresample", "resample"); g_assert (resample);
    volume = gst_element_factory_make ("volume", "vol"); g_assert (volume);
    g_object_set (G_OBJECT (volume), "volume", vol, NULL);
    out = gst_element_factory_make ("gconfaudiosink", "out"); g_assert (out);

    gst_bin_add_many (GST_BIN (pipeline), cdp, queue, conv, resample, volume, out, NULL);
    if (!gst_element_link_many (cdp, queue, conv, resample, volume, out, NULL))
      g_warning (_("Failed to link pipeline"));

    /* if something went wrong, cleanup here is easier... */
    if (!out) {
      gst_object_unref (GST_OBJECT (pipeline));
      pipeline = NULL;
      g_set_error (err, 0, 0, _("Failed to create audio output"));
      return FALSE;
    }

    if (!gtk_tree_model_get_iter_first (GTK_TREE_MODEL (track_store), &current_iter))
      g_warning ("Cannot get first iter");
  }
  
  return TRUE;
}

/*
 * Public function to release device.
 */

void
stop_playback (void)
{
  stop ();
}

/*
 * Interface entry point.
 */

void
on_play_activate (GtkWidget *button, gpointer user_data)
{
  GError *err = NULL;

  if (is_playing ()) {
    pause ();
  } else if (pipeline && GST_STATE (pipeline) == GST_STATE_PAUSED && 
                (current_track == seek_to_track || seek_to_track == -1)) {
    play ();
  } else if (setup (&err)) {
    char *title;
    if (current_track != -1) {
      if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (track_store),
                                          &current_iter, NULL, current_track))
        return;
    }
    gtk_list_store_set (track_store, &current_iter,
                        COLUMN_STATE, STATE_IDLE, -1);
    if (select_track ()) {
      gtk_list_store_set (track_store, &current_iter,
          COLUMN_STATE, STATE_PLAYING, -1);
      gtk_tree_model_get (GTK_TREE_MODEL (track_store),
          &current_iter, COLUMN_TITLE, &title, -1);
      sj_main_set_title (title);
      g_free (title);

      play ();
    } else {
      g_warning (G_STRLOC ": failed to select track");
      stop ();
    }
  } else {
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new (GTK_WINDOW (main_window), 0,
				     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				     _("Error playing CD.\n\nReason: %s"),
				     err->message);
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    g_error_free (err);
  }
}

void
on_tracklist_row_activate (GtkTreeView * treeview, GtkTreePath * path,
    GtkTreeViewColumn * col, gpointer user_data)
{
  GError *err = NULL;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gint track;

  model = gtk_tree_view_get_model (treeview);
  if (!gtk_tree_model_get_iter (model, &iter, path))
    return;
  gtk_tree_model_get (model, &iter, COLUMN_NUMBER, &track, -1);
  if (gtk_list_store_iter_is_valid (GTK_LIST_STORE (model), &current_iter))
      gtk_list_store_set (track_store, &current_iter, COLUMN_STATE, STATE_IDLE, -1);

  if (setup (&err)) {
    char *title;
    seek_to_track = track - 1;

    if (current_track != -1) {
      if (!gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (track_store),
                                          &current_iter, NULL, current_track))
        return;
    }

    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_IDLE, -1);
    select_track ();
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_PLAYING, -1);
    gtk_tree_model_get (GTK_TREE_MODEL (track_store),
        &current_iter, COLUMN_TITLE, &title, -1);
    sj_main_set_title (title);
    g_free (title);
    current_iter = iter;
    play ();
  } else {
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new (GTK_WINDOW (main_window), 0,
				     GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				     _("Error playing CD.\n\nReason: %s"),
				     err->message);
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    g_error_free (err);
  }
}

void
on_next_track_activate(GtkWidget *button, gpointer data)
{
  cb_hop_track (NULL, NULL, NULL);
}

void
on_previous_track_activate(GtkWidget *button, gpointer data)
{
  GtkTreeModel *model;
  gint tracks, prev_track = current_track - 1;
  GtkTreeIter prev_iter;

  model = GTK_TREE_MODEL (track_store);
  tracks = gtk_tree_model_iter_n_children (model, NULL);

  while (prev_track >= 0) {
    gboolean do_play;

    if (!gtk_tree_model_iter_nth_child (model, &prev_iter, NULL, prev_track))
      return;
    gtk_tree_model_get (GTK_TREE_MODEL (track_store), &prev_iter,
        COLUMN_EXTRACT, &do_play, -1);
    if (do_play)
      break;
    prev_track--;
  }

  if (prev_track < 0) {
    stop ();
    seek_to_track = 0;
  } else {
    char *title;
    seek_to_track = prev_track;
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_IDLE, -1);
    select_track ();
    gtk_list_store_set (track_store, &current_iter,
        COLUMN_STATE, STATE_PLAYING, -1);
    gtk_tree_model_get (model,
        &current_iter, COLUMN_TITLE, &title, -1);
    sj_main_set_title (title);
    g_free (title);
    play ();
  }
}

void
on_tracklist_row_selected (GtkTreeView *treeview,
               gpointer user_data)
{
  gint track;
  GtkTreeSelection *selection = gtk_tree_view_get_selection (treeview);

  if (is_playing ())
    return;

  if (gtk_tree_selection_get_selected (selection, NULL, &current_iter)) {
    gtk_tree_model_get (GTK_TREE_MODEL (track_store),
        &current_iter, COLUMN_NUMBER, &track, -1);
    seek_to_track = track - 1;
  }
}

/*
 * Volume.
 */

void
on_volume_changed (GtkWidget * volb, gpointer data)
{
  vol = bacon_volume_button_get_value (BACON_VOLUME_BUTTON (volb));
  if (pipeline) {
    GstElement *volume;

    volume = gst_bin_get_by_name_recurse_up (GST_BIN (pipeline), "vol");
    g_object_set (G_OBJECT (volume), "volume", vol, NULL);
  }  
  gconf_client_set_float (gconf_client, GCONF_AUDIO_VOLUME, vol, NULL);
}

/*
 * Seeking.
 */

gboolean
on_seek_press (GtkWidget * scale, GdkEventButton * event, gpointer user_data)
{
  seeking = TRUE;

  return FALSE;
}

void
on_seek_moved (GtkWidget * scale, gpointer user_data)
{
  gdouble val = gtk_range_get_value (GTK_RANGE (scale));
  gchar *x, *m;

  if (internal_update || !GST_CLOCK_TIME_IS_VALID (slen))
    return;

  x = get_label_for_time (slen * val / GST_SECOND);
  m = g_strdup_printf (_("Seeking to %s"), x);
  g_free (x);
  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 0, m);
  g_free (m);
}

gboolean
on_seek_release (GtkWidget * scale, GdkEventButton * event, gpointer user_data)
{
  gdouble val = gtk_range_get_value (GTK_RANGE (scale));
  GstElement *cd;

  cd = gst_bin_get_by_name_recurse_up (GST_BIN (pipeline), "cd-source");
  seeking = FALSE;

  /* set_state/get_state neccessary because of bug #326311 */
  gst_element_set_state (pipeline, GST_STATE_PAUSED);
  gst_element_seek (pipeline, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH, GST_SEEK_TYPE_SET, slen * val, GST_SEEK_TYPE_NONE, -1);
  gst_element_get_state (pipeline, NULL, NULL, -1);
  gst_element_set_state (pipeline, GST_STATE_PLAYING);

  return FALSE;
}

/*
 * Grim. Very Grim.
 */
void
stop_ui_hack (void)
{
  gtk_button_set_label (GTK_BUTTON (play_button), GTK_STOCK_MEDIA_PLAY);
  gtk_widget_hide (seek_scale);
  gtk_widget_hide (volume_button);
  gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 0);
  slen = GST_CLOCK_TIME_NONE;
  if (gtk_list_store_iter_is_valid (track_store, &current_iter))
    gtk_list_store_set (track_store, &current_iter, COLUMN_STATE, STATE_IDLE, -1);
  sj_main_set_title (NULL);
  gtk_statusbar_push (GTK_STATUSBAR (statusbar), 0, "");
  current_track = -1;
}

/**
 * Init
 */
void
sj_play_init (void)
{
  play_button = glade_xml_get_widget (glade, "play_button");
  next_menuitem = glade_xml_get_widget (glade, "next_track_menuitem");
  prev_menuitem = glade_xml_get_widget (glade, "previous_track_menuitem");
  reread_menuitem = glade_xml_get_widget (glade, "re-read");
  seek_scale = glade_xml_get_widget (glade, "seek_scale");
  volume_button = glade_xml_get_widget (glade, "volume_button");
  statusbar = glade_xml_get_widget (glade, "status_bar");
}
