/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "rtlsdr.h"
#include "shared.h"

/* rtlsdr device handle */
static rtlsdr_dev_t *dev = NULL;

static int
  num_gains,      /* Number of tuner gain values */
  *gains = NULL;  /* Tuner gains array */

/*-------------------------------------------------------------------*/

// void SimFM( double *i, double *q );

/* Rtl_Cb()
 *
 * Callback function for rtlsdr_read_async()
 */
  void
Rtl_Cb( unsigned char *buf, uint32_t len, void *ctx )
{
  int ids = 0, idx = 0;

  /* Convert sample values to range of +/- 127 */
  while( idx < (int)len )
  {
    filter_data_i.samples_buf[ids] = (double)( buf[idx] - 127 );
    idx++;
    filter_data_q.samples_buf[ids] = (double)( buf[idx] - 127 );
    idx++;
    ids++;
  }

  // SimFM( filter_data_i.samples_buf, filter_data_q.samples_buf );

  /* Low pass filter I and Q data */
  DSP_Filter( &filter_data_i );
  DSP_Filter( &filter_data_q );

  /* Save samples for carrier ifft from RTLSDR */
  if( isFlagSet(CARRIER_SPECTRUM) )
  {
    int
      decim_cnt = 0,
      data_idx  = 0,
      buf_idx   = 0;

    double
      sum_i = 0.0,
      sum_q = 0.0;

    for( idx = 0; idx < ifft_data_length; idx++ )
    {
      sum_i += filter_data_i.samples_buf[buf_idx];
      sum_q += filter_data_q.samples_buf[buf_idx];
      buf_idx++;

      decim_cnt++;
      if( decim_cnt >= rc_data.ifft_decimate )
      {
        ifft_data[data_idx++] = (int16_t)sum_i;
        ifft_data[data_idx++] = (int16_t)sum_q;
        decim_cnt = 0;
        sum_i     = 0.0;
        sum_q     = 0.0;
      }
    } /* for( idx = 0; idx < ifft_data_length; idx++ ) */

    Display_Waterfall();

  } /* if( isFlagSet(CARRIER_SPECTRUM) ) */

  /* Post to semaphore that DSP data is ready */
  int sval;
  sem_getvalue( &pback_semaphore, &sval );
  if( !sval ) sem_post( &pback_semaphore );

} /* Rtl_Cb() */

/*-----------------------------------------------------------------------*/

/* RtlSdr_Read_Async()
 *
 * Pthread function for async reading of RTL I/Q samples
 */
  void *
RtlSdr_Read_Async( void *pid )
{
  rtlsdr_read_async(
      dev, Rtl_Cb, pid, NUM_ASYNC_BUF, ASYNC_BUF_LEN );

  return( NULL );
} /* RtlSdr_Read_Async() */

/*-----------------------------------------------------------------------*/

/* Rtlsdr_Set_Center_Freq()
 *
 * Sets the Center Frequency of the RTL-SDR Tuner
 */
  gboolean
Rtlsdr_Set_Center_Freq( uint32_t center_freq )
{
  char text[40];
  int ret;


  /* Set the Center Frequency of the RTL_SDR Device */
  ret = rtlsdr_set_center_freq( dev, center_freq );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set Center Frequency"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Get the Center Frequency of the RTL_SDR Device */
  ret = (int)rtlsdr_get_center_freq( dev );
  if( (ret != (int)center_freq) || (ret == 0) )
  {
    Show_Message( _("Failed to set Center Frequency"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Display center frequency in messages */
  if( isFlagClear(ENABLE_DOPPLER_CORR) )
  {
    snprintf( text, sizeof(text),
        _("Setting Center Frequency to %0.1fkHz"),
        (double)center_freq / 1000.0 );
    Show_Message( text, "black" );
  }

  return( TRUE );
} /* Rtlsdr_Set_Center_Freq() */

/*-----------------------------------------------------------------------*/

/* Get_Tuner_Gain()
 *
 * Gets the Gain setting from the RTL-SDR Tuner
 * and displays it in the relevant entry widget
 */
  gboolean
Get_Tuner_Gain( void )
{
  char text[6];
  GtkLabel *label = GTK_LABEL(
      Builder_Get_Object(main_window_builder, "gain_label") );


  /* Get the Tuner Gain */
  int ret = rtlsdr_get_tuner_gain( dev );
  snprintf( text, sizeof(text), "%4.1f", (double)ret / 10.0 );
  gtk_label_set_text( label, text );

  return( TRUE );
} /* Get_Tuner_Gain() */

/*-----------------------------------------------------------------------*/

/* Set_Tuner_Gain_Mode()
 *
 * Sets the Tunaer Gain mode to Auto or Manual
 */
  gboolean
Set_Tuner_Gain_Mode( int mode )
{
  /* Set Tuner Gain Mode */
  gboolean ret = rtlsdr_set_tuner_gain_mode( dev, mode );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set Tuner Gain Mode"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Get and display Tuner gain also */
  if( !Get_Tuner_Gain() ) return( FALSE );

  return( TRUE );
} /* Set_Tuner_Gain_Mode() */

/*-----------------------------------------------------------------------*/

/* Set_Tuner_Gain()
 *
 * Set the Tuner Gain if in Manual mode
 */
  gboolean
Set_Tuner_Gain( double gain )
{
  int
    idx,
    min,
    diff,
    igain,
    igx;

  if( isFlagClear(RTLSDR_INIT) ) return( TRUE );

  if( gains == NULL )
  {
    /* Get the available Tuner Gains */
    num_gains = rtlsdr_get_tuner_gains( dev, NULL );
    if( num_gains <= 0 )
    {
      Show_Message( _("Failed to get Tuner Number of Gains"), "red" );
      Error_Dialog();
      return( FALSE );
    }

    /* Get the Gains List from the Device */
    gboolean ret = (int)mem_alloc(
        (void *)&gains, (size_t)num_gains * sizeof(int) );
    if( !ret )
    {
      Show_Message( _("Failed to allocate Tuner Gains Array"), "red" );
      Error_Dialog();
      return( FALSE );
    }

    num_gains = rtlsdr_get_tuner_gains( dev, gains );
    if( num_gains <= 0 )
    {
      Show_Message( _("Failed to get Tuner Gains Array"), "red" );
      Error_Dialog();
      return( FALSE );
    }

  } /* if( gains == NULL ) */

  /* Find nearest available gain */
  igain = (int)( gain * 10.0 );
  min = 1000; igx = 0; /* Prime */
  for( idx = 0; idx < num_gains; idx++ )
  {
    diff = abs( gains[idx] - igain );
    if( diff < min )
    {
      min = diff;
      igx = idx;
    }
  }
  if( igx >= num_gains )
    igx = num_gains - 1;

  /* Set the Tuner Gain */
  int ret = rtlsdr_set_tuner_gain( dev, gains[igx] );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set Tuner Gain"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Get and display Tuner gain also */
  if( !Get_Tuner_Gain() ) return( FALSE );

  return( TRUE );
} /* Set_Tuner_Gain() */

/*-----------------------------------------------------------------------*/

/* Rtlsdr_Set_Agc_Mode()
 *
 * Sets the RTL-SDR Digital AGC Mode
 */
  gboolean
Rtlsdr_Set_Agc_Mode( int mode )
{
  /* Set the internal digital AGC mode of RTL2832 */
  int ret = rtlsdr_set_agc_mode( dev, mode );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set RTL2832 AGC mode"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Get and display Tuner gain also */
  if( !Get_Tuner_Gain() ) return( FALSE );

  return( TRUE );
} /* Rtlsdr_Set_Agc_Mode() */

/*-----------------------------------------------------------------------*/

/* Rtlsdr_Init()
 *
 * Initialize rtlsdr device status
 */
  gboolean
Rtlsdr_Init( void )
{
  /* rtlsdr device sample rate */
  uint32_t sample_rate;

  /* Check function return values */
  int ret = -1;

  /* Device USB strings */
  char
    manufact[256],
    product[256],
    serial[256],
    mesg[512];

  const char *tuner_types[] = { TUNER_TYPES };

  /* rtlsdr tuner handle */
  enum rtlsdr_tuner tuner;

  /* Device name */
  const char *dev_name = NULL;

  /* Thread ID for the newly created thread */
  pthread_t pthread_id;


  /* Abort if already init */
  if( isFlagSet(RTLSDR_INIT) ) return( TRUE );

  /* Get RTL Device Name */
  Show_Message( _("Initializing RTLSDR Device "), "black" );
  Show_Message( _("RTLSDR Device Information:"), "black" );
  dev_name = rtlsdr_get_device_name( rc_data.rtlsdr_dev_index );

  /* Get device USB strings */
  ret = rtlsdr_get_device_usb_strings(
      rc_data.rtlsdr_dev_index, manufact, product, serial );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to get device usb strings"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Display device name and USB strings */
  snprintf( mesg, sizeof(mesg),
      _("Device Index: %d\nName: %s\n"\
        "Manufacturer: %s\nProduct: %s\nSerial: %s"),
      rc_data.rtlsdr_dev_index, dev_name, manufact, product, serial );
  Show_Message( mesg, "black" );

  /* Open RTL-SDR Device */
  ret = rtlsdr_open( &dev, rc_data.rtlsdr_dev_index);
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to open RTL-SDR device"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Set the Center Frequency of the RTL_SDR Device */
  ret = Rtlsdr_Set_Center_Freq( rc_data.rtlsdr_center_freq );
  if( !ret ) return( FALSE );

  /* Set the Frequency Correction factor for the device */
  if( rc_data.rtlsdr_freq_corr )
  {
    ret = rtlsdr_set_freq_correction( dev, rc_data.rtlsdr_freq_corr );
    if( ret != SUCCESS )
    {
      Show_Message( _("Failed to set Frequency Correction factor"), "red" );
      Error_Dialog();
      return( FALSE );
    }
  }

  /* Get the Frequency Correction factor from the device */
  rc_data.rtlsdr_freq_corr = rtlsdr_get_freq_correction( dev );
  snprintf( mesg, sizeof(mesg),
      _("Frequency Correction: %d ppm"), rc_data.rtlsdr_freq_corr );
  Show_Message( mesg, "black" );

  /* Get the RTL_SDR Tuner type */
  tuner = rtlsdr_get_tuner_type( dev );
  if( tuner == RTLSDR_TUNER_UNKNOWN )
  {
    Show_Message( _("Failed to get Tuner type"), "red" );
    Error_Dialog();
    return( FALSE );
  }
  snprintf( mesg, sizeof(mesg), _("Tuner Type: %s"), tuner_types[tuner] );
  Show_Message( mesg, "black" );
  GtkLabel *label = GTK_LABEL(
      Builder_Get_Object(main_window_builder, "tuner_label") );
  snprintf( mesg, sizeof(mesg), "%s", tuner_types[tuner] );
  gtk_label_set_text( label, mesg );

  /* Set Tuner Gain Mode to auto */
  ret = rtlsdr_set_tuner_gain_mode( dev, TUNER_GAIN_AUTO );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to set Tuner Gain Mode"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Enable RTLSDR Digital AGC */
  if( !Rtlsdr_Set_Agc_Mode(RTL_DAGC_ON) )
    return( FALSE );

  /* Set RTL Sample Rate */
  ret = rtlsdr_set_sample_rate( dev, RTL_DSP_RATE );
  if( (ret != SUCCESS) || (ret == -EINVAL) )
  {
    Show_Message( _("Failed to set ADC Sample Rate"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Get RTL Sample Rate */
  sample_rate = rtlsdr_get_sample_rate( dev );
  if( sample_rate == 0 )
  {
    Show_Message( _("Failed to get ADC Sample Rate"), "red" );
    Error_Dialog();
    return( FALSE );
  }
  snprintf( mesg, sizeof(mesg),
      _("ADC Sample Rate: %7d S/s"), sample_rate );
  Show_Message( mesg, "black" );

  /* Get the Direct Sampling mode status */
  ret = rtlsdr_get_direct_sampling( dev );
  if( ret < 0 )
  {
    Show_Message( _("Failed to get Direct Sampling mode"), "red" );
    Error_Dialog();
    return( FALSE );
  }
  snprintf( mesg, sizeof(mesg), _("Direct Sampling mode: %d"), ret );
  Show_Message( mesg, "black" );

  /* Get offset tuning status */
  ret = rtlsdr_get_offset_tuning( dev );
  if( ret < 0 )
  {
    Show_Message( _("Failed to get Offset Tuning mode"), "red" );
    Error_Dialog();
    return( FALSE );
  }
  snprintf( mesg, sizeof(mesg), _("Offset Tuning mode: %d"), ret );
  Show_Message( mesg, "black" );

  /* Reset RTL data buffer */
  ret = rtlsdr_reset_buffer( dev );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to Reset sampling Buffer"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  /* Init Chebyshev Low Pass Filters I/Q */
  size_t req = (size_t)IQ_BUF_LEN * sizeof(double);
  ret = mem_alloc( (void *)&(filter_data_i.samples_buf), req );
  if( !ret )
  {
    Show_Message( _("Failed to Allocate Filter Buffer"), "red" );
    Error_Dialog();
    return( FALSE );
  }
  ret = mem_alloc( (void *)&(filter_data_q.samples_buf), req );
  if( !ret )
  {
    Show_Message( _("Failed to Allocate Filter Buffer"), "red" );
    Error_Dialog();
    return( FALSE );
  }

  filter_data_i.cutoff   = (double)(rc_data.rtlsdr_lpf_bw / 2);
  filter_data_i.cutoff  /= (double)RTL_DSP_RATE;
  filter_data_i.ripple   = FILTER_RIPPLE;
  filter_data_i.npoles   = FILTER_POLES;
  filter_data_i.type     = FILTER_LOWPASS;
  filter_data_i.ring_idx = 0;
  filter_data_i.samples_buf_len = IQ_BUF_LEN;
  Init_Chebyshev_Filter( &filter_data_i );

  filter_data_q.cutoff   = (double)(rc_data.rtlsdr_lpf_bw / 2);
  filter_data_q.cutoff  /= (double)RTL_DSP_RATE;
  filter_data_q.ripple   = FILTER_RIPPLE;
  filter_data_q.npoles   = FILTER_POLES;
  filter_data_q.type     = FILTER_LOWPASS;
  filter_data_q.ring_idx = 0;
  filter_data_q.samples_buf_len = IQ_BUF_LEN;
  Init_Chebyshev_Filter( &filter_data_q );

  /* Create a thread for async read from RTL device */
  ret = pthread_create( &pthread_id, NULL, RtlSdr_Read_Async, NULL );
  if( ret != SUCCESS )
  {
    Show_Message( _("Failed to create async read thread"), "red" );
    Error_Dialog();
    return( FALSE );
  }
  sleep( 1 );

  SetFlag( RTLSDR_INIT );
  Show_Message( _("RTLSDR Device Initialized OK"), "green" );

  return( TRUE );
} /* Rtlsdr_Init() */

/*-----------------------------------------------------------------------*/

/* Read_RTL_Buffer()
 *
 * Reads the RTLSDR buffer
 */
  gboolean
Read_RTL_Buffer( short *buffer, int buff_size )
{
  int cnt, idx, i, q;

  double
    phi1 = 0.0, /* Phase angle of sampled IF signal */
    phi2 = 0.0, /* Phase angle of sampled IF signal */
    dphi = 0.0, /* Change in Phase angle of sampled IF signal */
    freq = 0.0, /* Measured frequency of sampled IF signal */
    frqi = 0.0, /* Instantaneous frequency of IF signal */
    i1 = 0.0,   /* In-phase samples in float */
    q1 = 0.0,   /* Quadrature samples in float */
    i2 = 0.0,   /* In-phase samples in float */
    q2 = 0.0;   /* Quadrature samples in float */

  /* Offset between Satellite freq and Tuner center freq */
  static double freq_offset = 0.0;
  static int offset_cnt = 0;

  /* This ring buffer saves some rtlsdr I/Q samples so
   * that the estimate of instantaneous frequency is done
   * using I/Q samples spaced FMDET_RINGBUF_LEN apart */
  static char *ring_buf = NULL;
  static int
    ring_buf_idx = 0,
    ringbuf_len2 = (int)( (double)FMDET_RINGBUF_LEN / 2.0 );

  /* IFFT samples input buffer idx */
  static int iq_idx = 0;

  /* Current and Previous Tuner center frequency */
  static int tuner_freq, tuner_xfreq;


  /* Allocate and clear ring buffer */
  if( ring_buf == NULL )
  {
    if( !mem_alloc((void *)&ring_buf, FMDET_RINGBUF_LEN) )
      return( FALSE );
    bzero( ring_buf, FMDET_RINGBUF_LEN );
    tuner_freq  = (int)rc_data.rtlsdr_center_freq;
    tuner_xfreq = (int)rc_data.rtlsdr_center_freq;
  }

  /* Transfer data from RTL buffer to buffer to be processed */
  if( buff_size >= BLOCK_BUFFER_SIZE )
    buff_size -= BLOCK_BUFFER_SIZE;

  /* Fill buffer from RTLSDR device. The 2.4kHz
   * sub-carrier amplitude is proportional to
   * the measured VHF carrier's freq. deviation
   */
  for( idx = 0; idx < buff_size; idx++ )
  {
    /* Use RTLSDR_OVERSAMPLING samples to produce
     * one subcarrier amplitude sample */
    freq = 0.0;
    for( cnt = 0; cnt < RTLSDR_OVERSAMPLING; cnt++ )
    {
      /* Wait for rtlsdr buffer to refill */
      if( iq_idx >= IQ_BUF_LEN )
      {
        /* Wait on DSP data to be ready for processing */
        sem_wait( &pback_semaphore );
        iq_idx = 0;
      }

      /* Past In-phase and Quadrature signal samples */
      i  = (int)ring_buf[ring_buf_idx];
      i1 = (double)i;
      q  = (int)ring_buf[ring_buf_idx+1];
      q1 = (double)q;

      /* Current In-phase and Quadrature signal samples */
      i  = (int)filter_data_i.samples_buf[iq_idx];
      i2 = (double)i;
      ring_buf[ring_buf_idx] = (char)i;
      ring_buf_idx++;

      q  = (int)filter_data_q.samples_buf[iq_idx];
      q2 = (double)q;
      ring_buf[ring_buf_idx] = (char)q;
      ring_buf_idx++;
      iq_idx++;

      if( ring_buf_idx >= FMDET_RINGBUF_LEN )
        ring_buf_idx = 0;

      /* Phase angle of Past sampled IF signal */
      phi1 = atan2( q1, i1 );

      /* Phase angle of Current sampled IF signal */
      phi2 = atan2( q2, i2 );

      /* Phase difference from past to current I/Q sample */
      dphi = phi2 - phi1;

      /* Correct for discontinuity at +-180 deg phase angle */
      if( dphi > M_PI )        dphi -= M_2PI;
      else if(  dphi < -M_PI ) dphi += M_2PI;

      /* Measured frequency deviation */
      dphi /= ringbuf_len2;
      frqi  = (double)RTL_DSP_RATE * dphi / M_2PI;
      freq += frqi;

      /* Measure and compenste freq offset */
      if( isFlagSet(ENABLE_DOPPLER_CORR) )
      {
        /* Doppler shift is averaged over
         * 1 sec, e.g RTL_DSP_RATE counts */
        freq_offset += frqi;
        offset_cnt++;
        if( (offset_cnt >= RTL_DSP_RATE) )
        {
          char text[8];
          GtkLabel *label = GTK_LABEL(
              Builder_Get_Object(main_window_builder, "offset_label") );

          /* Reset on new center freq */
          if( tuner_xfreq != (int)rc_data.rtlsdr_center_freq )
          {
            tuner_freq = (int)rc_data.rtlsdr_center_freq;
            tuner_xfreq = tuner_freq;
          }

          /* Average frequency offset */
          freq_offset /= (double)offset_cnt;
          offset_cnt = 0;

          /* Display frequency offset, usually caused by Doppler shift*/
          tuner_freq += (int)( 4.0 * freq_offset / 5.0 );

          /* Limit max frequency offset correction to +/- 5kHz */
          if( (uint32_t)tuner_freq > (rc_data.rtlsdr_center_freq + MAX_DOPPLER_CORR) )
            tuner_freq = (int)(rc_data.rtlsdr_center_freq + MAX_DOPPLER_CORR);
          if( (uint32_t)tuner_freq < (rc_data.rtlsdr_center_freq - MAX_DOPPLER_CORR) )
            tuner_freq = (int)(rc_data.rtlsdr_center_freq - MAX_DOPPLER_CORR);
          snprintf( text, sizeof(text), "%d",
              (unsigned int)tuner_freq - rc_data.rtlsdr_center_freq );
          gtk_label_set_text( label, text );

          /* Correct for frequency offset if enabled */
          Rtlsdr_Set_Center_Freq( (uint32_t)tuner_freq );
          freq_offset = 0.0;

        } /* if( offset_cnt >= RTL_DSP_RATE ) */
      } /* if(  isFlagSet(ENABLE_DOPPLER_CORR) ) */
      else if( tuner_freq != (int)rc_data.rtlsdr_center_freq )
      {
        GtkLabel *label = GTK_LABEL(
            Builder_Get_Object(main_window_builder, "offset_label") );
        tuner_freq = (int)rc_data.rtlsdr_center_freq;
        freq_offset = 0;
        Rtlsdr_Set_Center_Freq( (uint32_t)tuner_freq );
        gtk_label_set_text( label, " - - - " );
      }
    } /* for( cnt = 0; cnt < RTLSDR_OVERSAMPLING; cnt++ ) */

    /* Fill the buffer used to produce image */
    freq /= (double)RTLSDR_OVERSAMPLING;
    buffer[idx] = (short)( freq * RTL_SCALE_FACTOR );

    /* Display 2400 Hz subcarrier spectrum in waterfall */
    if( isFlagClear(CARRIER_SPECTRUM) )
      IFFT_Data( buffer[idx] );

  } /* for( idx = 0; idx < buff_size; idx++ ) */

  /* Display selected waterfall
  if( isFlagSet(CARRIER_SPECTRUM) )
    Display_Waterfall();
  else
  {
    det_ifft_idx = 0;
    Display_Waterfall();
  } */

  return( TRUE );
} /* Read_RTL_Buffer() */

/*-----------------------------------------------------------------------*/

/* Close_RTL_Device()
 *
 * Closes thr RTLSDR device, if open
 */
  void
Close_RTL_Device( void )
{
  rtlsdr_cancel_async( dev );
  if( dev != NULL )
  {
    rtlsdr_close( dev );
    dev = NULL;
  }

  free_ptr( (void *)&gains );
  free_ptr( (void *)&(filter_data_i.samples_buf) );
  free_ptr( (void *)&(filter_data_q.samples_buf) );

  ClearFlag( RTLSDR_INIT );
} /* Close_RTL_Device() */

/*-----------------------------------------------------------------------*/

/* Simulates an FM carrier with 2.4 KHz FM modulation
  void
SimFM( double *i, double *q )
{
  double
    fdev = 17000.0,
    mfrq = 2400.0,
    dphi = 0.0,
    dmph = 0.0,
    cfrq = 0.0,
    dtim = 1.0 / (double)RTL_DSP_RATE;

  static double phas = 0.0, mphs = 0.0;

  int scnt;

  dmph = M_2PI * mfrq * dtim;
  for( scnt = 0; scnt < ASYNC_BUF_LEN/2; scnt++ )
  {
    i[scnt] = (double)( 100.0 * cos(phas) );
    q[scnt] = (double)( 100.0 * sin(phas) );

    cfrq = fdev * sin( mphs );
    mphs += dmph;
    if( mphs >=  M_2PI ) mphs -= M_2PI;

    dphi = M_2PI * cfrq * dtim;
    phas += dphi;
    if( phas >=  M_2PI ) phas -= M_2PI;
    if( phas <= -M_2PI ) phas += M_2PI;
  }
} */

/*-----------------------------------------------------------------------*/

