/**
 * Copyright (c) 2010-2015, Abel Hegedus, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.addon.querybasedfeatures.tooling;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.viatra.addon.querybasedfeatures.tooling.QueryBasedFeatureParameters;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Annotation;
import org.eclipse.viatra.query.patternlanguage.emf.vql.AnnotationParameter;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ClassType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.StringValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Type;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ValueReference;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.runtime.extensibility.PQueryExtensionFactory;
import org.eclipse.viatra.query.runtime.extensibility.ViatraQueryRuntimeConstants;
import org.eclipse.viatra.query.tooling.core.generator.ExtensionData;
import org.eclipse.viatra.query.tooling.core.generator.ExtensionGenerator;
import org.eclipse.viatra.query.tooling.core.generator.fragments.IGenerationFragment;
import org.eclipse.viatra.query.tooling.core.generator.genmodel.IVQGenmodelProvider;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.generator.IFileSystemAccess;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.w3c.dom.Element;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class SurrogateGeneratorFragment implements IGenerationFragment {
  @Inject
  protected IVQGenmodelProvider provider;
  
  @Inject
  protected Logger logger;
  
  @Inject
  protected IErrorFeedback errorFeedback;
  
  @Inject
  @Extension
  protected ExtensionGenerator exGen;
  
  @Inject
  @Extension
  protected EMFPatternLanguageJvmModelInferrerUtil inferrerUtil;
  
  protected static String SURROGATE_ERROR_CODE = "org.eclipse.viatra.query.runtime.surrogatequeryemf.error";
  
  protected static String SURROGATE_EXTENSION_PREFIX = "extension.surrogate.";
  
  @Override
  public Collection<Pair<String, String>> getRemovableExtensions() {
    return CollectionLiterals.<Pair<String, String>>newArrayList(
      Pair.<String, String>of(SurrogateGeneratorFragment.SURROGATE_EXTENSION_PREFIX, ViatraQueryRuntimeConstants.SURROGATE_QUERY_EXTENSIONID));
  }
  
  @Override
  public Iterable<Pair<String, String>> removeExtension(final Pattern pattern) {
    return CollectionLiterals.<Pair<String, String>>newArrayList(
      Pair.<String, String>of(this.derivedContributionId(pattern), ViatraQueryRuntimeConstants.SURROGATE_QUERY_EXTENSIONID));
  }
  
  @Override
  public Iterable<ExtensionData> extensionContribution(final Pattern pattern) {
    final Procedure1<Element> _function = (Element it) -> {
      final Consumer<QueryBasedFeatureParameters> _function_1 = (QueryBasedFeatureParameters parameters) -> {
        final Procedure1<Element> _function_2 = (Element it_1) -> {
          this.exGen.contribAttribute(it_1, "package-nsUri", parameters.ePackage.getNsURI());
          this.exGen.contribAttribute(it_1, "class-name", parameters.source.getName());
          this.exGen.contribAttribute(it_1, "feature-name", parameters.feature.getName());
          this.exGen.contribAttribute(it_1, "query-fqn", PatternLanguageHelper.getFullyQualifiedName(pattern));
          String _canonicalName = PQueryExtensionFactory.class.getCanonicalName();
          String _plus = (_canonicalName + ":");
          String _qualifiedName = this.inferrerUtil.findInferredSpecification(pattern).getQualifiedName();
          String _plus_1 = (_plus + _qualifiedName);
          this.exGen.contribAttribute(it_1, "surrogate-query", _plus_1);
        };
        this.exGen.contribElement(it, "surrogate-query-emf", _function_2);
      };
      this.gatherSurrogateParameters(pattern).forEach(_function_1);
    };
    final ArrayList<ExtensionData> surrogateExtension = CollectionLiterals.<ExtensionData>newArrayList(
      this.exGen.contribExtension(this.derivedContributionId(pattern), ViatraQueryRuntimeConstants.SURROGATE_QUERY_EXTENSIONID, _function));
    return surrogateExtension;
  }
  
  protected ArrayList<QueryBasedFeatureParameters> gatherSurrogateParameters(final Pattern pattern) {
    final ArrayList<QueryBasedFeatureParameters> parameterList = CollectionLiterals.<QueryBasedFeatureParameters>newArrayList();
    Collection<Annotation> _annotationsByName = PatternLanguageHelper.getAnnotationsByName(pattern, "Surrogate");
    for (final Annotation annotation : _annotationsByName) {
      try {
        QueryBasedFeatureParameters _processAnnotation = this.processAnnotation(pattern, annotation, false);
        parameterList.add(_processAnnotation);
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          this.logger.error(e.getMessage());
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    boolean _isEmpty = parameterList.isEmpty();
    if (_isEmpty) {
      return CollectionLiterals.<QueryBasedFeatureParameters>newArrayList();
    }
    return parameterList;
  }
  
  protected String derivedContributionId(final Pattern pattern) {
    String _fullyQualifiedName = PatternLanguageHelper.getFullyQualifiedName(pattern);
    return (SurrogateGeneratorFragment.SURROGATE_EXTENSION_PREFIX + _fullyQualifiedName);
  }
  
  @Override
  public void generateFiles(final Pattern pattern, final IFileSystemAccess fsa) {
    final Consumer<QueryBasedFeatureParameters> _function = (QueryBasedFeatureParameters parameters) -> {
    };
    this.gatherSurrogateParameters(pattern).forEach(_function);
  }
  
  @Override
  public void cleanUp(final Pattern pattern, final IFileSystemAccess fsa) {
    final Consumer<QueryBasedFeatureParameters> _function = (QueryBasedFeatureParameters parameters) -> {
    };
    this.gatherSurrogateParameters(pattern).forEach(_function);
  }
  
  @Override
  public IPath[] getAdditionalBinIncludes() {
    return ((IPath[])Conversions.unwrapArray(CollectionLiterals.<IPath>newArrayList(), IPath.class));
  }
  
  @Override
  public String[] getProjectDependencies() {
    return ((String[])Conversions.unwrapArray(CollectionLiterals.<String>newArrayList(), String.class));
  }
  
  @Override
  public String getProjectPostfix() {
    return null;
  }
  
  protected QueryBasedFeatureParameters processAnnotation(final Pattern pattern, final Annotation annotation, final boolean feedback) {
    final QueryBasedFeatureParameters parameters = new QueryBasedFeatureParameters();
    parameters.pattern = pattern;
    parameters.annotation = annotation;
    String featureTmp = "";
    int _size = pattern.getParameters().size();
    boolean _notEquals = (_size != 2);
    if (_notEquals) {
      if (feedback) {
        this.errorFeedback.reportError(pattern, "Pattern must have exactly 2 parameters!", SurrogateGeneratorFragment.SURROGATE_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus = ("Surrogate pattern " + _fullyQualifiedName);
      String _plus_1 = (_plus + " must have 2 parameters!");
      throw new IllegalArgumentException(_plus_1);
    }
    EList<AnnotationParameter> _parameters = annotation.getParameters();
    for (final AnnotationParameter ap : _parameters) {
      boolean _matches = ap.getName().matches("feature");
      if (_matches) {
        ValueReference _value = ap.getValue();
        featureTmp = ((StringValue) _value).getValue();
      }
    }
    boolean _equals = Objects.equal(featureTmp, "");
    if (_equals) {
      featureTmp = pattern.getName();
    }
    final Variable sourcevar = pattern.getParameters().get(0);
    final Type sourceType = sourcevar.getType();
    if (((!(sourceType instanceof ClassType)) || (!(((ClassType) sourceType).getClassname() instanceof EClass)))) {
      if (feedback) {
        String _name = sourcevar.getName();
        String _plus_2 = ("Source " + _name);
        String _plus_3 = (_plus_2 + " is not EClass!");
        this.errorFeedback.reportError(sourcevar, _plus_3, SurrogateGeneratorFragment.SURROGATE_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_1 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_4 = ("Surrogate pattern " + _fullyQualifiedName_1);
      String _plus_5 = (_plus_4 + ": Source ");
      String _name_1 = sourcevar.getName();
      String _plus_6 = (_plus_5 + _name_1);
      String _plus_7 = (_plus_6 + " is not EClass!");
      throw new IllegalArgumentException(_plus_7);
    }
    EClassifier _classname = ((ClassType) sourceType).getClassname();
    EClass source = ((EClass) _classname);
    parameters.source = source;
    if (((source == null) || (source.getEPackage() == null))) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, "Source EClass or EPackage not found!", SurrogateGeneratorFragment.SURROGATE_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_2 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_8 = ("Surrogate pattern " + _fullyQualifiedName_2);
      String _plus_9 = (_plus_8 + ": Source EClass or EPackage not found!");
      throw new IllegalArgumentException(_plus_9);
    }
    final EPackage pckg = source.getEPackage();
    if ((pckg == null)) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, "EPackage not found!", SurrogateGeneratorFragment.SURROGATE_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_3 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_10 = ("Surrogate pattern " + _fullyQualifiedName_3);
      String _plus_11 = (_plus_10 + ": EPackage not found!");
      throw new IllegalArgumentException(_plus_11);
    }
    parameters.ePackage = pckg;
    final String featureString = featureTmp;
    final Function1<EStructuralFeature, Boolean> _function = (EStructuralFeature it) -> {
      String _name_2 = it.getName();
      return Boolean.valueOf(Objects.equal(_name_2, featureString));
    };
    final Iterable<EStructuralFeature> features = IterableExtensions.<EStructuralFeature>filter(source.getEAllStructuralFeatures(), _function);
    int _size_1 = IterableExtensions.size(features);
    boolean _notEquals_1 = (_size_1 != 1);
    if (_notEquals_1) {
      if (feedback) {
        String _name_2 = source.getName();
        String _plus_12 = ((("Feature " + featureTmp) + " not found in class ") + _name_2);
        String _plus_13 = (_plus_12 + "!");
        this.errorFeedback.reportError(annotation, _plus_13, SurrogateGeneratorFragment.SURROGATE_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_4 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_14 = ("Surrogate pattern " + _fullyQualifiedName_4);
      String _plus_15 = (_plus_14 + ": Feature ");
      String _plus_16 = (_plus_15 + featureTmp);
      String _plus_17 = (_plus_16 + " not found in class ");
      String _name_3 = source.getName();
      String _plus_18 = (_plus_17 + _name_3);
      String _plus_19 = (_plus_18 + "!");
      throw new IllegalArgumentException(_plus_19);
    }
    final EStructuralFeature feature = features.iterator().next();
    parameters.feature = feature;
    parameters.targetVar = pattern.getParameters().get(1).getName();
    return parameters;
  }
}
