/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.ide.processors;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.epp.internal.logging.aeri.ide.IProcessorDescriptor;
import org.eclipse.epp.logging.aeri.core.IReport;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

public abstract class AbstractCachingStringProcessor implements IEditableReportProcessor {

    private class EditDialog extends Dialog {

        private static final int SIZE_HINT_WIDTH = 500;
        private static final int SIZE_HINT_HEIGHT = 260;
        private StyledText editText;
        private String value;

        protected EditDialog(String value, Shell parentShell) {
            super(parentShell);
            setShellStyle(SWT.MODELESS | SWT.DIALOG_TRIM | SWT.RESIZE | SWT.MAX | SWT.APPLICATION_MODAL);
            this.value = value;
        }

        @Override
        protected void configureShell(Shell shell) {
            super.configureShell(shell);
            shell.setText("Edit '" + AbstractCachingStringProcessor.this.name + "'");
        }

        @Override
        protected Control createDialogArea(Composite parent) {
            GridLayoutFactory.fillDefaults().margins(10, 10).applyTo(parent);
            Label label = new Label(parent, SWT.NONE);
            label.setText(editLabel);

            GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.FILL).grab(true, false).applyTo(label);
            editText = new StyledText(parent, SWT.BORDER | SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
            editText.setFont(JFaceResources.getFont(JFaceResources.TEXT_FONT));
            editText.setAlwaysShowScrollBars(false);
            editText.setText(value);
            GridDataFactory.fillDefaults().align(SWT.FILL, SWT.FILL).hint(SIZE_HINT_WIDTH, SIZE_HINT_HEIGHT).grab(true, true)
                    .applyTo(editText);

            return parent;
        }

        @Override
        protected void okPressed() {
            value = editText.getText();
            super.okPressed();
        }

        public String getValue() {
            return value;
        }

    }

    private final String directive;
    private final String name;
    private final String editLabel;

    public AbstractCachingStringProcessor(String directive, String name, String editLabel) {
        this.directive = directive;
        this.name = name;
        this.editLabel = editLabel;
    }

    public AbstractCachingStringProcessor(IProcessorDescriptor descriptor) {
        this(descriptor.getDirective(), descriptor.getName(), "Please remove any irrelevant or sensitive data.");
    }

    @Override
    public final boolean canContribute(IStatus status, IEclipseContext context) {
        String contextKey = getContextKey() + ".canContribute";
        if (context.containsKey(contextKey)) {
            return (Boolean) context.get(contextKey);
        } else {
            boolean canContribute = canContribute(status);
            context.set(contextKey, canContribute);
            return canContribute;
        }
    }

    protected abstract boolean canContribute(IStatus status);

    @Override
    public final void process(IReport report, IStatus status, IEclipseContext context) {
        String value = (String) context.get(getContextKey());
        if (value == null) {
            value = process(status);
            context.set(getContextKey(), value);
        }
        report.getAuxiliaryInformation().put(directive, value);
    }

    protected abstract String process(IStatus status);

    protected String getContextKey() {
        return "processor." + directive + ".value";
    }

    @Override
    public EditResult edit(IStatus status, IEclipseContext context, Shell parent) {
        String information = (String) context.get(getContextKey());
        EditDialog dialog = new EditDialog(information, parent);
        if (dialog.open() != Window.OK) {
            return EditResult.CANCELED;
        }

        String newInformation = dialog.getValue();
        if (information.equals(newInformation)) {
            return EditResult.UNMODIFIED;
        }

        context.set(getContextKey(), newInformation);
        return EditResult.MODIFIED;
    }

    @Override
    public void reset(IStatus status, IEclipseContext context) {
        context.remove(getContextKey());
    }
}
