#!/bin/sh
# create site.xml and/or site-ganymede.xml from a build type label, a target dir, and a set of directories to merge

# REQUIREMENTS 
# vm (/path/to/java if not /usr/bin/java)
# targetDir (/path/to/output/the/site.xml file)
# basebuilderDir (/path/to/org.eclipse.releng.basebuilder)
# trackstats (true if we're generating site.xml w/ http:// urls for tracking download stats)

norm="\033[0;39m";
grey="\033[1;30m";
green="\033[1;32m";
brown="\033[0;33m";
yellow="\033[1;33m";
blue="\033[1;34m";
cyan="\033[1;36m";
red="\033[1;31m";

vm=/usr/bin/java
debug=0;
trackstats=false;

##########################################################################################

function usage ()
{
	echo " "
	echo "usage: ${0##*/}"
	echo "-targetDir /path/to/\$label/      (required: path to dir containing site.xml)"
	echo "-basebuilderDir /path/to/org.eclipse.releng.basebuilder (optional: will try to guess)"
	echo "-debug [1|2]                      (optional)"
	echo "-vm /path/to/java                 (optional; default: $vm)" 
	echo "-trackstats                       (optional; default: false)"
	echo " "
	echo "examples:"
	echo "	${0##*/} -basebuilderDir /home/www-data/build/org.eclipse.releng.basebuilder -targetDir \$localUpdatesWebDir"
	echo "	${0##*/} -basebuilderDir /opt/public/modeling/build/org.eclipse.releng.basebuilder -targetDir \$updatesDir"
	exit 1;
}

if [ $# -lt 2 ]; then
	usage;
fi

while [ "$#" -gt 0 ]; do
	case $1 in
		'-debug')		debug=$2;				shift 1;;
		'-vm')			vm=$2;					shift 1;;
		'-targetDir')	targetDir=$2;			shift 1;;
		'-basebuilderDir') basebuilderDir=$2;	shift 1;;
		'-trackstats')	trackstats=$2;			shift 1;;
	esac
	shift 1
done

if [[ ! $basebuilderDir ]] || [[ ! -d $basebuilderDir ]]; then
	for d in /home/www-data/build/org.eclipse.releng.basebuilder /opt/public/modeling/build/org.eclipse.releng.basebuilder; do
		if [[ -d $d ]]; then
			basebuilderDir=$d;
			break;
		fi
	done
fi
if [[ ! $basebuilderDir ]] || [[ ! -d $basebuilderDir ]]; then
	usage;
fi

##########################################################################################

# create a site digest for the composite site
echo -e '[compose] ['`date +%H:%M:%S`'] Run siteOptimizer to create '$blue$targetDir$norm/${blue}digest.zip${norm}' ...'
if [[ $trackstats == "true" ]]; then
	# temporarily remove any download tracking urls in <feature>s
	pushd ${targetDir} >/dev/null;
	mv -f site.xml site-full.xml;
	cat site-full.xml | \
		perl -pe "s#http://www.eclipse.org/downloads/download.php\?r=1\&amp;file=.+features/#features/#g" \
		> site.xml;
fi
pushd $basebuilderDir >/dev/null;
command="$vm -cp $basebuilderDir/plugins/org.eclipse.equinox.launcher.jar org.eclipse.equinox.launcher.Main";
command=$command" -application org.eclipse.update.core.siteOptimizer";
command=$command" -digestBuilder -digestOutputDir=${targetDir}";
command=$command" -siteXML=${targetDir}/site.xml"; # -verbose
command=$command" -vmargs -Xmx128M";

tmpout=`mktemp`;
if [[ $debug -gt 0 ]]; then 
	echo "$command" | perl -pe "s/ -/\n  -/g" ; # pretty printing
fi
$command 1>$tmpout 2>/dev/null;
if [[ $debug -gt 0 ]]; then 
	cat $tmpout;
	echo "";
fi

if [[ $(egrep "Error" $tmpout) ]]; then
	grep "Error" $tmpout | perl -pe "s#$targetDir/#$yellow#g" | perl -pe "s#(.+)#[update] ${red}*!* \1$norm ${red}*!*${norm}#";
	returnCode=1;
else
	returnCode=0;
fi

rm -fr $tmpout;
popd >/dev/null;

if [[ $trackstats == "true" ]]; then
	# undo temporary changes to site.xml
	mv -f site-full.xml site.xml;
	popd >/dev/null;
fi
	
if [[ ! -f $targetDir/digest.zip ]]; then
	echo -e "[update] ${red}*!* Warning: no digest.zip created. ${red}*!*${norm}";
fi

exit $returnCode;