/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.validation;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import fr.inria.diverse.melange.metamodel.melange.MelangePackage;
import fr.inria.diverse.melange.validation.MelangeValidator;
import java.util.HashMap;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.BasicDiagnostic;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EValidator;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.util.IAcceptor;
import org.eclipse.xtext.util.internal.Stopwatches;
import org.eclipse.xtext.validation.CheckMode;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.validation.impl.ConcreteSyntaxEValidator;
import org.eclipse.xtext.xbase.annotations.validation.DerivedStateAwareResourceValidator;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Custom efficient validation process:
 *   - Don't check derived state (it should be correct by construction)
 *   - Don't check external Aspects/Ecore resources
 *   - Directly dispatch on the MelangeValidator, Diagnostician is too slow
 */
@SuppressWarnings("all")
public class MelangeLocalResourceValidator extends DerivedStateAwareResourceValidator {
  @Inject
  private MelangeValidator validator;
  
  @Override
  public void validate(final Resource resource, final CheckMode mode, final CancelIndicator monitor, final IAcceptor<Issue> acceptor) {
    boolean _isCanceled = monitor.isCanceled();
    if (_isCanceled) {
      return;
    }
    String _fileExtension = resource.getURI().fileExtension();
    boolean _equals = Objects.equal(_fileExtension, "melange");
    if (_equals) {
      URI _uRI = resource.getURI();
      String _plus = ("MelangeLocalResourceValidator.validate(" + _uRI);
      String _plus_1 = (_plus + ")");
      final Stopwatches.StoppedTask task = Stopwatches.forTask(_plus_1);
      task.start();
      this.validate(resource, IterableExtensions.<EObject>head(resource.getContents()), mode, monitor, acceptor);
      task.stop();
    } else {
      super.validate(resource, mode, monitor, acceptor);
    }
  }
  
  @Override
  public void validate(final Resource resource, final EObject element, final CheckMode mode, final CancelIndicator monitor, final IAcceptor<Issue> acceptor) {
    boolean _isCanceled = monitor.isCanceled();
    if (_isCanceled) {
      return;
    }
    String _fileExtension = resource.getURI().fileExtension();
    boolean _equals = Objects.equal(_fileExtension, "melange");
    if (_equals) {
      final BasicDiagnostic diagChain = new BasicDiagnostic();
      HashMap<Object, Object> _newHashMap = CollectionLiterals.<Object, Object>newHashMap();
      final Procedure1<HashMap<Object, Object>> _function = new Procedure1<HashMap<Object, Object>>() {
        @Override
        public void apply(final HashMap<Object, Object> it) {
          it.put(CheckMode.KEY, mode);
          it.put(ConcreteSyntaxEValidator.DISABLE_CONCRETE_SYNTAX_EVALIDATOR, 
            Boolean.TRUE);
          it.put(EValidator.class, MelangeLocalResourceValidator.this.validator);
        }
      };
      final HashMap<Object, Object> context = ObjectExtensions.<HashMap<Object, Object>>operator_doubleArrow(_newHashMap, _function);
      final TreeIterator<EObject> i = element.eAllContents();
      while (i.hasNext()) {
        {
          final EObject obj = i.next();
          EPackage _ePackage = obj.eClass().getEPackage();
          boolean _equals_1 = Objects.equal(_ePackage, MelangePackage.eINSTANCE);
          if (_equals_1) {
            this.validator.validate(obj, diagChain, context);
          }
        }
      }
      boolean _isEmpty = diagChain.getChildren().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final Consumer<Diagnostic> _function_1 = new Consumer<Diagnostic>() {
          @Override
          public void accept(final Diagnostic diag) {
            MelangeLocalResourceValidator.this.issueFromEValidatorDiagnostic(diag, acceptor);
          }
        };
        diagChain.getChildren().forEach(_function_1);
      } else {
        this.issueFromEValidatorDiagnostic(diagChain, acceptor);
      }
    } else {
      super.validate(resource, mode, monitor, acceptor);
    }
  }
}
