"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.encodeNamedCurves = void 0;
const bn_js_1 = __importDefault(require("bn.js"));
const needs_1 = require("./needs");
const prime256v1 = eccEncodeCompressedPoint(32);
const secp384r1 = eccEncodeCompressedPoint(48);
exports.encodeNamedCurves = Object.freeze({
    // NodeJS/OpenSSL names
    prime256v1,
    secp384r1,
    // WebCrypto/Browser names
    'P-256': prime256v1,
    'P-384': secp384r1,
});
/*
 * 1. This only works for prime curves
 * 2. This will not handle the point at infinity
 */
function eccEncodeCompressedPoint(keyLength) {
    return function encode(publicKey) {
        /* Precondition: publicKey must be the right length.
         * The format for the public key is [type, ...keyLength, ...keyLength]
         */
        (0, needs_1.needs)(publicKey.byteLength === 1 + keyLength * 2, 'Malformed public key.');
        // const type = publicKey[0]
        const x = publicKey.slice(1, keyLength + 1);
        const y = publicKey.slice(keyLength + 1, keyLength * 2 + 1);
        const yOrder = new bn_js_1.default([...y]).mod(new bn_js_1.default(2)).toNumber() + 2;
        const compressPoint = new Uint8Array(1 + x.length);
        compressPoint.set([yOrder], 0);
        compressPoint.set(x, 1);
        return compressPoint;
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWNjX2VuY29kZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9lY2NfZW5jb2RlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7Ozs7QUFFdEMsa0RBQXNCO0FBRXRCLG1DQUErQjtBQUUvQixNQUFNLFVBQVUsR0FBRyx3QkFBd0IsQ0FBQyxFQUFFLENBQUMsQ0FBQTtBQUMvQyxNQUFNLFNBQVMsR0FBRyx3QkFBd0IsQ0FBQyxFQUFFLENBQUMsQ0FBQTtBQVFqQyxRQUFBLGlCQUFpQixHQUFnQyxNQUFNLENBQUMsTUFBTSxDQUFDO0lBQzFFLHVCQUF1QjtJQUN2QixVQUFVO0lBQ1YsU0FBUztJQUNULDBCQUEwQjtJQUMxQixPQUFPLEVBQUUsVUFBVTtJQUNuQixPQUFPLEVBQUUsU0FBUztDQUNuQixDQUFDLENBQUE7QUFFRjs7O0dBR0c7QUFDSCxTQUFTLHdCQUF3QixDQUFDLFNBQWlCO0lBQ2pELE9BQU8sU0FBUyxNQUFNLENBQUMsU0FBcUI7UUFDMUM7O1dBRUc7UUFDSCxJQUFBLGFBQUssRUFBQyxTQUFTLENBQUMsVUFBVSxLQUFLLENBQUMsR0FBRyxTQUFTLEdBQUcsQ0FBQyxFQUFFLHVCQUF1QixDQUFDLENBQUE7UUFFMUUsNEJBQTRCO1FBQzVCLE1BQU0sQ0FBQyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLFNBQVMsR0FBRyxDQUFDLENBQUMsQ0FBQTtRQUMzQyxNQUFNLENBQUMsR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLFNBQVMsR0FBRyxDQUFDLEVBQUUsU0FBUyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQTtRQUUzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLGVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxlQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUE7UUFFM0QsTUFBTSxhQUFhLEdBQUcsSUFBSSxVQUFVLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQTtRQUNsRCxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUE7UUFDOUIsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUE7UUFFdkIsT0FBTyxhQUFhLENBQUE7SUFDdEIsQ0FBQyxDQUFBO0FBQ0gsQ0FBQyJ9