﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/synthetics/Synthetics_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Synthetics {
namespace Model {

/**
 * <p>If this canary is to test an endpoint in a VPC, this structure contains
 * information about the subnets and security groups of the VPC endpoint. For more
 * information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_VPC.html">
 * Running a Canary in a VPC</a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/synthetics-2017-10-11/VpcConfigInput">AWS
 * API Reference</a></p>
 */
class VpcConfigInput {
 public:
  AWS_SYNTHETICS_API VpcConfigInput() = default;
  AWS_SYNTHETICS_API VpcConfigInput(Aws::Utils::Json::JsonView jsonValue);
  AWS_SYNTHETICS_API VpcConfigInput& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SYNTHETICS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The IDs of the subnets where this canary is to run.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnetIds() const { return m_subnetIds; }
  inline bool SubnetIdsHasBeenSet() const { return m_subnetIdsHasBeenSet; }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  void SetSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds = std::forward<SubnetIdsT>(value);
  }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  VpcConfigInput& WithSubnetIds(SubnetIdsT&& value) {
    SetSubnetIds(std::forward<SubnetIdsT>(value));
    return *this;
  }
  template <typename SubnetIdsT = Aws::String>
  VpcConfigInput& AddSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds.emplace_back(std::forward<SubnetIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the security groups for this canary.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  VpcConfigInput& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  VpcConfigInput& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set this to <code>true</code> to allow outbound IPv6 traffic on VPC canaries
   * that are connected to dual-stack subnets. The default is <code>false</code> </p>
   */
  inline bool GetIpv6AllowedForDualStack() const { return m_ipv6AllowedForDualStack; }
  inline bool Ipv6AllowedForDualStackHasBeenSet() const { return m_ipv6AllowedForDualStackHasBeenSet; }
  inline void SetIpv6AllowedForDualStack(bool value) {
    m_ipv6AllowedForDualStackHasBeenSet = true;
    m_ipv6AllowedForDualStack = value;
  }
  inline VpcConfigInput& WithIpv6AllowedForDualStack(bool value) {
    SetIpv6AllowedForDualStack(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_subnetIds;
  bool m_subnetIdsHasBeenSet = false;

  Aws::Vector<Aws::String> m_securityGroupIds;
  bool m_securityGroupIdsHasBeenSet = false;

  bool m_ipv6AllowedForDualStack{false};
  bool m_ipv6AllowedForDualStackHasBeenSet = false;
};

}  // namespace Model
}  // namespace Synthetics
}  // namespace Aws
