﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/gamelift/GameLiftRequest.h>
#include <aws/gamelift/GameLift_EXPORTS.h>
#include <aws/gamelift/model/Tag.h>

#include <utility>

namespace Aws {
namespace GameLift {
namespace Model {

/**
 */
class CreateLocationRequest : public GameLiftRequest {
 public:
  AWS_GAMELIFT_API CreateLocationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateLocation"; }

  AWS_GAMELIFT_API Aws::String SerializePayload() const override;

  AWS_GAMELIFT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A descriptive name for the custom location.</p>
   */
  inline const Aws::String& GetLocationName() const { return m_locationName; }
  inline bool LocationNameHasBeenSet() const { return m_locationNameHasBeenSet; }
  template <typename LocationNameT = Aws::String>
  void SetLocationName(LocationNameT&& value) {
    m_locationNameHasBeenSet = true;
    m_locationName = std::forward<LocationNameT>(value);
  }
  template <typename LocationNameT = Aws::String>
  CreateLocationRequest& WithLocationName(LocationNameT&& value) {
    SetLocationName(std::forward<LocationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of labels to assign to the new resource. Tags are developer-defined
   * key-value pairs. Tagging Amazon Web Services resources are useful for resource
   * management, access management, and cost allocation. For more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
   * Amazon Web Services Resources</a> in the <i>Amazon Web Services General
   * Rareference</i>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateLocationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateLocationRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_locationName;
  bool m_locationNameHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
