/**
 * kcfgcreator.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "kcfgcreator.h"

#include "applicationtoken.h"
#include "grouptoken.h"
#include "entrytoken.h"

#include <qdom.h>

namespace KCfgCreator
{

static QDomElement
createEntry( QDomDocument& doc, const EntryToken* entryToken )
{
    QDomElement entry = doc.createElement( "entry" );

    entry.setAttribute( "type", entryToken->typeName() );
    entry.setAttribute( "key", entryToken->key() );

    QString n = entryToken->key().remove( ' ' );
    if ( !entryToken->name().isEmpty() &&
         ( n != entryToken->name() ) )
        entry.setAttribute( "name", entryToken->name() );

    if ( entryToken->hidden() )
        entry.setAttribute( "hidden", "true" );

    if ( !entryToken->label().isEmpty() ) {
        QDomElement sub = doc.createElement( "label" );
        QDomText t = doc.createTextNode( entryToken->label() );
        sub.appendChild( t );
        entry.appendChild( sub );
    }
    if ( !entryToken->whatsThis().isEmpty() ) {
        QDomElement sub = doc.createElement( "whatsthis" );
        QDomText t = doc.createTextNode( entryToken->whatsThis() );
        sub.appendChild( t );
        entry.appendChild( sub );
    }

    if ( !entryToken->values().empty() ) {
        QDomElement sub = doc.createElement( "values" );
        QStringList values = entryToken->values();
        for ( QStringList::Iterator itr = values.begin();
              itr != values.end(); ++itr ) {
            QDomElement sub2 = doc.createElement( "value" );
            QDomText t = doc.createTextNode( *itr );
            sub2.appendChild( t );
            sub.appendChild( sub2 );
        }
        entry.appendChild( sub );
    }

    if ( !entryToken->defaultValue().value.isEmpty() ) {
        QDomElement sub = doc.createElement( "default" );
        if ( entryToken->defaultValue().isCode )
            sub.setAttribute( "code", "true" );
        if ( !entryToken->defaultValue().param.isEmpty() )
            sub.setAttribute( "param", entryToken->defaultValue().param );
        QDomText t = doc.createTextNode( entryToken->defaultValue().value );
        sub.appendChild( t );
        entry.appendChild( sub );
    }

    if ( !entryToken->choices().empty() ) {
        QDomElement choices = doc.createElement( "choices" );

        QValueList<EntryToken::Choice> choicesList = entryToken->choices();
        for ( QValueList<EntryToken::Choice>::Iterator itr = choicesList.begin();
              itr != choicesList.end(); ++itr ) {
            EntryToken::Choice ch = *itr;

            QDomElement choice = doc.createElement( "choice" );
            choice.setAttribute( "name", ch.name );

            if ( !ch.label.isEmpty() ) {
                QDomElement label = doc.createElement( "label" );
                QDomText t = doc.createTextNode( ch.label );
                label.appendChild( t );
                choice.appendChild( label );
            }

            if ( !ch.whatsThis.isEmpty() ) {
                QDomElement whatsThis = doc.createElement( "whatsthis" );
                QDomText t = doc.createTextNode( ch.whatsThis );
                whatsThis.appendChild( t );
                choice.appendChild( whatsThis );
            }

            choices.appendChild( choice );
        }

        entry.appendChild( choices );
    }

    if ( !entryToken->min().value.isEmpty() ) {
        QDomElement min = doc.createElement( "min" );
        if ( entryToken->min().isCode )
            min.setAttribute( "code", "true" );
        QDomText t = doc.createTextNode( entryToken->min().value );
        min.appendChild( t );
        entry.appendChild( min );
    }

    if ( !entryToken->max().value.isEmpty() ) {
        QDomElement max = doc.createElement( "max" );
        if ( entryToken->max().isCode )
            max.setAttribute( "code", "true" );
        QDomText t = doc.createTextNode( entryToken->max().value );
        max.appendChild( t );
        entry.appendChild( max );
    }

    if ( !entryToken->parameter().name.isEmpty() ) {
        QDomElement param = doc.createElement( "parameter" );
        if ( !entryToken->parameter().type.isEmpty() )
            param.setAttribute( "type", entryToken->parameter().type );
        if ( !entryToken->parameter().max != -1 )
            param.setAttribute( "max", entryToken->parameter().max );

        QStringList valueList = entryToken->parameter().values;
        if ( !valueList.empty() ) {
            QDomElement values = doc.createElement( "values" );
            for ( QStringList::Iterator itr = valueList.begin();
                  itr != valueList.end(); ++itr ) {
                QDomElement value = doc.createElement( "value" );
                QDomText t = doc.createTextNode( *itr );
                value.appendChild( t );
                values.appendChild( value );
            }
            param.appendChild( values );
        }

        entry.appendChild( param );
    }

    return entry;
}

QDomElement
CreatorKCfg::createGroup( QDomDocument& doc, const GroupToken* group )
{
    QDomElement tag = doc.createElement( "group" );
    tag.setAttribute( "name", group->name() );

    QPtrList<EntryToken> elist = group->entryList();
    QPtrListIterator<EntryToken> itr( elist );
    EntryToken *entryToken = 0;
    while ( (entryToken = itr.current()) ) {
        QDomElement entry = createEntry( doc, entryToken );
        tag.appendChild( entry );
        ++itr;
    }
    return tag;

}

CreatorKCfg::CreatorKCfg( QObject* parent )
    : CreatorInterface( parent )
{
}

void
CreatorKCfg::createConfig( ApplicationToken* token )
{
    emit done( token, doWork( token ).toString().utf8() );
}

QString
CreatorKCfg::configAsString( ApplicationToken* token )
{
    return doWork( token ).toString();
}

QDomDocument
CreatorKCfg::doWork(  ApplicationToken* token )
{
    QDomImplementation impl;
    QDomElement tag;

    QDomDocumentType docType = impl.createDocumentType(
        "kcfg",
        QString::null,
        "http://www.kde.org/standards/kcfg/1.0/kcfg.dtd" );

    QDomDocument doc( docType );
    QDomElement root = doc.createElement( "kcfg" );
    doc.appendChild( root );

    QStringList includes = token->includeFiles();
    for ( QStringList::Iterator itr = includes.begin();
          itr != includes.end(); ++itr ) {
        tag = doc.createElement( "include" );
        QDomText inc = doc.createTextNode( *itr );
        tag.appendChild( inc );
        root.appendChild( tag );
    }

    tag = doc.createElement( "kcfgfile" );
    tag.setAttribute( "name", token->name() );
    QStringList parameters = token->parameters();
    for ( QStringList::Iterator itr = parameters.begin();
          itr != parameters.end(); ++itr ) {
        QDomElement param = doc.createElement( "parameter" );
        param.setAttribute( "name", *itr );
        tag.appendChild( param );
    }
    root.appendChild( tag );

    QPtrList<GroupToken> l = token->groupList();
    QPtrListIterator<GroupToken> gitr( l );
    while ( gitr.current() ) {
        tag = createGroup( doc, gitr.current() );
        root.appendChild( tag );
        ++gitr;
    }
    return doc;
}

}//end namespace

#include "kcfgcreator.moc"
