/**
 * entries.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "entries.h"

using namespace KCfgCreator;

StringEntry::StringEntry( GroupToken* parent, const QString& key,
                          const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant StringEntry::variant() const
{
    return QVariant( m_value );
}

void StringEntry::setVariant( const QVariant& var )
{
    m_value = var.toString();
}

QString  StringEntry::valueAsString() const
{
    return m_value;
}

void StringEntry::setValueFromString( const QString& value )
{
    m_value = value;
}

StringListEntry::StringListEntry( GroupToken* parent, const QString& key,
                                  const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant StringListEntry::variant() const
{
    return QVariant( m_value );
}

void StringListEntry::setVariant( const QVariant& var )
{
    m_value = var.toStringList();
}

QString  StringListEntry::valueAsString() const
{
    return m_value.join( ", " );
}

void StringListEntry::setValueFromString( const QString& value )
{
    m_value = QStringList::split( value,  ", " );
}

FontEntry::FontEntry( GroupToken* parent, const QString& key,
                      const QString& name , bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant FontEntry::variant() const
{
    return QVariant( m_value );
}

void FontEntry::setVariant( const QVariant& var )
{
    m_value = var.toFont();
}

QString  FontEntry::valueAsString() const
{
    return m_value.toString();
}

void FontEntry::setValueFromString( const QString& val )
{
    m_value.fromString( val );
}

RectEntry::RectEntry( GroupToken* parent, const QString& key,
                      const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant RectEntry::variant() const
{
    return QVariant( m_value );
}

void RectEntry::setVariant( const QVariant& var )
{
    m_value = var.toRect();
}

QString  RectEntry::valueAsString() const
{
    return QString( "%1,%2,%3,%4" )
        .arg( m_value.left() )
        .arg( m_value.top() )
        .arg( m_value.width() )
        .arg( m_value.height() );
}

void RectEntry::setValueFromString( const QString& val )
{
    if ( !val.isEmpty() ) {
        int left, top, width, height;
        if ( sscanf( val.latin1(), "%d,%d,%d,%d", &left, &top,
                     &width, &height ) == 4 ) {
            m_value = QRect( left, top, width, height );
            return;
        }
    }

    //rectangle coordinates are not in the passed
    //string (val)
    m_value = QRect();
}

SizeEntry::SizeEntry( GroupToken* parent, const QString& key,
                      const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant SizeEntry::variant() const
{
    return QVariant( m_value );
}

void SizeEntry::setVariant( const QVariant& var )
{
    m_value = var.toSize();
}

QString  SizeEntry::valueAsString() const
{
    return QString( "%1,%2" )
        .arg( m_value.width() )
        .arg( m_value.height() );
}

void SizeEntry::setValueFromString( const QString& val )
{
    if ( !val.isEmpty() )
    {
        int width, height;
        if ( sscanf( val.latin1(), "%d,%d", &width, &height ) == 2 )
        {
            m_value = QSize( width, height );
            return;
        }
    }

    m_value = QSize();
}


PointEntry::PointEntry( GroupToken* parent, const QString& key,
                        const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant PointEntry::variant() const
{
    return QVariant( m_value );
}

void PointEntry::setVariant( const QVariant& var )
{
    m_value = var.toPoint();
}

QString  PointEntry::valueAsString() const
{
    return QString( "%1,%2" )
        .arg( m_value.x() )
        .arg( m_value.y() );
}

void PointEntry::setValueFromString( const QString& val )
{
    if ( !val.isEmpty() )
    {
        int x, y;
        if ( sscanf( val.latin1(), "%d,%d", &x, &y ) == 2 )
        {
            m_value = QPoint( x, y );
            return;
        }
    }
    //if got here val wasn't a valid point
    m_value = QPoint();
}

ColorEntry::ColorEntry( GroupToken* parent, const QString& key,
            const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant ColorEntry::variant() const
{
    return QVariant( m_value );
}

void ColorEntry::setVariant( const QVariant& var )
{
    m_value = var.toColor();
}

QString  ColorEntry::valueAsString() const
{
    return m_value.name();
}

void ColorEntry::setValueFromString( const QString& val )
{
    m_value.setNamedColor( val );
}

IntEntry::IntEntry( GroupToken* parent, const QString& key,
          const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant IntEntry::variant() const
{
    return QVariant( m_value );
}

void IntEntry::setVariant( const QVariant& var )
{
    m_value = var.toInt();
}

QString  IntEntry::valueAsString() const
{
    return QString::number( m_value );
}

void IntEntry::setValueFromString( const QString& val )
{
    m_value = val.toInt();
}

UIntEntry::UIntEntry( GroupToken* parent, const QString& key,
                      const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant UIntEntry::variant() const
{
    return QVariant( m_value );
}

void UIntEntry::setVariant( const QVariant& var )
{
    m_value = var.toUInt();
}

QString  UIntEntry::valueAsString() const
{
    return QString::number( m_value );
}

void UIntEntry::setValueFromString( const QString& val )
{
    m_value = val.toUInt();
}

BoolEntry::BoolEntry( GroupToken* parent, const QString& key,
                      const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant BoolEntry::variant() const
{
    return QVariant( m_value );
}

void BoolEntry::setVariant( const QVariant& var )
{
    m_value = var.toBool();
}

QString  BoolEntry::valueAsString() const
{
    return m_value ? "true" : "false";
}

void BoolEntry::setValueFromString( const QString& val )
{
    if ( val == "true" ) m_value = true;
    else m_value = false;
}

DoubleEntry::DoubleEntry( GroupToken* parent, const QString& key,
             const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant DoubleEntry::variant() const
{
    return QVariant( m_value );
}

void DoubleEntry::setVariant( const QVariant& var )
{
    m_value = var.toDouble();
}

QString  DoubleEntry::valueAsString() const
{
    return QString::number( m_value );
}

void DoubleEntry::setValueFromString( const QString& val )
{
    m_value = val.toDouble();
}

DateTimeEntry::DateTimeEntry( GroupToken* parent, const QString& key,
               const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant DateTimeEntry::variant() const
{
    return QVariant( m_value );
}

void DateTimeEntry::setVariant( const QVariant& var )
{
    m_value = var.toDateTime();
}

QString  DateTimeEntry::valueAsString() const
{
    return m_value.toString();
}

void DateTimeEntry::setValueFromString( const QString& val )
{
    m_value = QDateTime::fromString( val );
}

Int64Entry::Int64Entry( GroupToken* parent, const QString& key,
                        const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant Int64Entry::variant() const
{
    return QVariant( m_value );
}

void Int64Entry::setVariant( const QVariant& var )
{
    m_value = var.toLongLong();
}

QString  Int64Entry::valueAsString() const
{
    return QString::number( m_value );
}

void Int64Entry::setValueFromString( const QString& val )
{
    m_value = val.toLongLong();
}

UInt64Entry::UInt64Entry( GroupToken* parent, const QString& key,
                          const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant UInt64Entry::variant() const
{
    return QVariant( m_value );
}

void UInt64Entry::setVariant( const QVariant& var )
{
    m_value = var.toULongLong();
}

QString  UInt64Entry::valueAsString() const
{
    return QString::number( m_value );
}

void UInt64Entry::setValueFromString( const QString& val )
{
    m_value = val.toULongLong();
}

IntListEntry::IntListEntry( GroupToken* parent, const QString& key,
                            const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant IntListEntry::variant() const
{
    return QVariant( "FIXME" );
}

void IntListEntry::setVariant( const QVariant&  )
{
    //FIXME implement
    //m_value = var.toList();
}

QString  IntListEntry::valueAsString() const
{
    QStringList list;
    for ( unsigned int i=0; i<m_value.count(); ++i )
        list.append( QString( "%1" ).arg( m_value[ i ] ) );
    return list.join( "," );
}

void IntListEntry::setValueFromString( const QString& val )
{
    QStringList list = QStringList::split( ',', val );
    m_value.clear();
    for ( unsigned int i=0; i<list.count(); ++i )
        m_value.append( list[ i ].toInt() );
}

EnumEntry::EnumEntry( GroupToken* parent, const QString& key,
                      const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant EnumEntry::variant() const
{
    return QVariant( "FIXME" );
}

void EnumEntry::setVariant( const QVariant& )
{
    //FIXME
}

QString  EnumEntry::valueAsString() const
{
    return "FIXME";
}

void EnumEntry::setValueFromString( const QString& )
{
    //FIXME
}

PathEntry::PathEntry( GroupToken* parent, const QString& key,
                      const QString& name, bool hidden )
    : EntryToken( parent, key, name, hidden )
{
}

QVariant PathEntry::variant() const
{
    return QVariant( m_value.prettyURL() );
}

void PathEntry::setVariant( const QVariant& var )
{
    m_value = KURL( var.toString() );
}

QString  PathEntry::valueAsString() const
{
    return m_value.prettyURL();
}

void PathEntry::setValueFromString( const QString& val )
{
    m_value = KURL();
    m_value.setPath( val );
}
