/* func.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This module represents QueryField of type QUERY_FIELD_FUNCTION:
 * a function
 */

#include <config.h>
#include "../query.h"
#include "../query-field-private.h"
#include "../database.h"
#include "../object-selector.h"

static void        q_init            (QueryField *qf);
static void        q_finalize         (QueryField *qf);
static void        q_deactivate      (QueryField *qf);
static void        q_activate        (QueryField *qf);
static GtkWidget * q_get_edit_widget (QueryField *qf);
static GtkWidget * q_get_sel_widget  (QueryField *qf, GCallback callback, gpointer data);
static gchar     * q_render_as_sql   (QueryField *qf, GSList * missing_values);
static xmlNodePtr  q_render_as_xml   (QueryField *qf, GSList * missing_values);
static gchar     * q_render_as_string(QueryField *qf, GSList * missing_values);
static void        q_save_to_xml     (QueryField *qf, xmlNodePtr node);
static void        q_load_from_xml   (QueryField *qf, xmlNodePtr node);
static void        q_copy_other_field(QueryField *qf, QueryField *other);
static gboolean    q_is_equal_to     (QueryField *qf, QueryField *other);
static GSList    * q_get_monitored_objects (QueryField *qf);
static void        q_replace_comp    (QueryField *qf, gint ref, GObject   *old, GObject   *new);

/* Weak ref from the refering object */
static void        q_obj_destroyed_cb (QueryField *qf, GObject   *obj);
static void        arg_destroyed_cb (QueryField *qf, QueryField *arg); 

typedef struct {
	ServerFunction *func;

	/* XML field name if we did not find it in the first place (like TVxxx:FIyyy) */
	gchar          *func_name;

	GSList         *weak_ref_objects;

	/* list of references to arguments (struct Arg)  */
	GSList         *args;
} private_data;


/* structure to hold an argument */
typedef struct {
	QueryField *qf;
	gchar      *xml;
} Arg;


#define QF_PRIVATE_DATA(qf) ((private_data *) qf->private_data)
#define QF_FUNC_ARG(arg) ((Arg *) (arg))

QueryFieldIface * 
query_field_function_get_iface()
{
	QueryFieldIface *iface;

	iface = g_new0 (QueryFieldIface, 1);
	iface->field_type = QUERY_FIELD_FUNCTION;
	iface->name = "function";
	iface->pretty_name = _("Function");
	iface->init = q_init;
	iface->destroy = q_finalize;
	iface->deactivate = q_deactivate;
	iface->activate = q_activate;
	iface->get_edit_widget = q_get_edit_widget;
	iface->get_sel_widget = q_get_sel_widget;
	iface->render_as_sql = q_render_as_sql;
	iface->render_as_xml = q_render_as_xml;
	iface->render_as_string = q_render_as_string;
	iface->save_to_xml = q_save_to_xml;
	iface->load_from_xml = q_load_from_xml;
	iface->copy_other_field = q_copy_other_field;
	iface->is_equal_to = q_is_equal_to;
	iface->get_monitored_objects = q_get_monitored_objects;
	iface->replace_comp = q_replace_comp;

	return iface;
}

#ifdef debug
static void
func_dump (QueryField *qf)
{
	GSList *list;
	g_print ("DUMPING FUNCTION QF %p:\n", qf);
	list = QF_PRIVATE_DATA(qf)->args;
	while (list) {
		Arg *arg = QF_FUNC_ARG(list->data);
		g_print ("\tARG %p\n", arg);
		g_print ("\t\tqf=%p, xml=%s\n", arg->qf, arg->xml);
		list = g_slist_next (list);
	}
}
#endif

static void        
q_init            (QueryField *qf)
{
	private_data *data;
	data = g_new0 (private_data, 1);
	qf->private_data = (gpointer) data;
	QF_PRIVATE_DATA(qf)->func = NULL;
	QF_PRIVATE_DATA(qf)->func_name = NULL;
	QF_PRIVATE_DATA(qf)->args = NULL;
	QF_PRIVATE_DATA(qf)->weak_ref_objects = NULL;
}

static void free_args (QueryField *qf);

static void        
q_finalize         (QueryField *qf)
{
	q_deactivate (qf);
	if (qf->private_data) {
		if (QF_PRIVATE_DATA(qf)->func_name)
			g_free (QF_PRIVATE_DATA(qf)->func_name);

		free_args (qf);

		g_free (qf->private_data);
		qf->private_data = NULL;
	}
}

static void 
free_args (QueryField *qf)
{
	GSList *list;

	list = QF_PRIVATE_DATA(qf)->args;
	while (list) {
		if (QF_FUNC_ARG(list->data)->xml)
			g_free (QF_FUNC_ARG(list->data)->xml);
		
		if (QF_FUNC_ARG(list->data)->qf &&
		    g_slist_find (QF_PRIVATE_DATA(qf)->weak_ref_objects, QF_FUNC_ARG(list->data)->qf)) {
			g_object_weak_unref (G_OBJECT (QF_FUNC_ARG(list->data)->qf), 
					     (GWeakNotify) arg_destroyed_cb, qf);
			QF_PRIVATE_DATA(qf)->weak_ref_objects = g_slist_remove (QF_PRIVATE_DATA(qf)->weak_ref_objects,
										QF_FUNC_ARG(list->data)->qf);	
		}
		g_free (list->data);
		
		list = g_slist_next (list);
	}
	g_slist_free (QF_PRIVATE_DATA(qf)->args);
	QF_PRIVATE_DATA(qf)->args = NULL;
}

static void qf_set_func_ptr (QueryField *qf, ServerFunction *func);
static void sub_qf_status_changed_cb (QueryField *qf, QueryField *pqf);
static void
q_deactivate      (QueryField *qf)
{
	GSList *list;
	g_print ("q_deactivate (%p)\n", qf);

	/* This function disconnects any event handler from any object
	   this QueryField wants to receive events from.
	   Here we disconnect from the function if we are connected */

	if (!qf->activated) {
		g_print ("Not activated, returning!\n");
		return;
	}

	/* Function part */
	if (QF_PRIVATE_DATA(qf)->func) {
		if (QF_PRIVATE_DATA(qf)->func_name) {
			g_free (QF_PRIVATE_DATA(qf)->func_name);
			QF_PRIVATE_DATA(qf)->func_name = NULL;
		}
		
		QF_PRIVATE_DATA(qf)->func_name = server_function_get_xml_id (QF_PRIVATE_DATA(qf)->func);
		qf_set_func_ptr (qf, NULL);
	}		


	/* Taking care of the arguments */
	list = QF_PRIVATE_DATA(qf)->args;
	while (list) {
		/* unreference that QueryField argument */
		/* g_print ("Now calls query_field_free_ref (%p)\n", QF_FUNC_ARG(list->data)->qf); */
/* 		query_field_free_ref (QF_FUNC_ARG(list->data)->qf); */		
		
		if (QF_FUNC_ARG(list->data)->xml &&
		    QF_FUNC_ARG(list->data)->qf) {
			g_free (QF_FUNC_ARG(list->data)->xml);
			QF_FUNC_ARG(list->data)->xml = NULL;
		}

		if (QF_FUNC_ARG(list->data)->qf) {
			if (g_slist_find (QF_PRIVATE_DATA(qf)->weak_ref_objects, QF_FUNC_ARG(list->data)->qf)) {
				g_object_weak_unref (G_OBJECT (QF_FUNC_ARG(list->data)->qf), 
						     (GWeakNotify) arg_destroyed_cb, qf);
				QF_PRIVATE_DATA(qf)->weak_ref_objects = 
					g_slist_remove (QF_PRIVATE_DATA(qf)->weak_ref_objects,
							QF_FUNC_ARG(list->data)->qf);
			}

			QF_FUNC_ARG(list->data)->xml = query_field_get_xml_id (QF_FUNC_ARG(list->data)->qf);
			g_signal_handlers_disconnect_by_func (G_OBJECT (QF_FUNC_ARG(list->data)->qf),
							      G_CALLBACK (sub_qf_status_changed_cb), qf);
		}
		QF_FUNC_ARG(list->data)->qf = NULL;

		list = g_slist_next (list);
	}

	query_field_set_activated (qf, FALSE);
}

static QueryField *get_field_from_xml (GSList *list, gchar *xmlid);
static void
q_activate        (QueryField *qf)
{
	/* this function gets references to any object this QueryField wants to 
	   receive events from. */
	ServerFunction *func;

	g_print ("q_activate (%p)\n", qf);
	
	if (qf->activated) {
		g_print ("Already activated, returning!\n");
		return;
	}

	func = QF_PRIVATE_DATA(qf)->func;
	
	if (!func && QF_PRIVATE_DATA(qf)->func_name) {
		func = server_function_get_from_xml_id (qf->query->conf->srv, 
							QF_PRIVATE_DATA(qf)->func_name);
		if (func) {
			g_free (QF_PRIVATE_DATA(qf)->func_name);
			QF_PRIVATE_DATA(qf)->func_name = NULL;	
		}
	}


	if (func) {
		GSList *list;
		gboolean allfound = TRUE;

		/* set QF_PRIVATE_DATA(qf)->func and weak ref */
		qf_set_func_ptr (qf, func);

		/* Taking care of the arguments */
		list = QF_PRIVATE_DATA(qf)->args;
		while (list) {
			if (! QF_FUNC_ARG(list->data)->qf) {
				if (QF_FUNC_ARG(list->data)->xml) {
					QueryField *aqf;
					GSList **qfs;
					
					/* fetch the list of QueryFields to work with */
					qfs = g_object_get_data (G_OBJECT (qf), "qf_list");

					if (!qfs)
						qfs = &(qf->query->fields);
					
					/* fetch the argument QueryField */
					aqf = get_field_from_xml (*qfs, QF_FUNC_ARG(list->data)->xml);

					if (aqf) {
						QF_FUNC_ARG(list->data)->qf = aqf;
						g_object_weak_ref (G_OBJECT (aqf), 
								   (GWeakNotify) arg_destroyed_cb, qf);
						QF_PRIVATE_DATA(qf)->weak_ref_objects = 
							g_slist_append (QF_PRIVATE_DATA(qf)->weak_ref_objects,
									aqf);

						if (! aqf->activated) {
							query_field_activate (aqf);
							if (! aqf->activated)
								allfound = FALSE;
						}
						g_signal_connect (G_OBJECT (aqf), "status_changed",
								  G_CALLBACK (sub_qf_status_changed_cb), qf);
					}
					else 
						allfound = FALSE;
				}
				else
					allfound = FALSE;
			}
			else {
				if (! QF_FUNC_ARG(list->data)->qf->activated) {
					query_field_activate (QF_FUNC_ARG(list->data)->qf);
					if (! QF_FUNC_ARG(list->data)->qf->activated)
						allfound = FALSE;
				}
			}
			
			list = g_slist_next (list);
		}


		/* make sure we have the right number of arguments */
		if (g_slist_length (QF_PRIVATE_DATA(qf)->args) != 
		    g_slist_length (QF_PRIVATE_DATA(qf)->func->args)) {
			g_warning ("Wrong number of arguments for Function\n");
			allfound = FALSE;
		}

		/* pronounce the activation if OK */
		if (allfound) {
			GSList *list;

			/* add a reference to the QueryField object as arguments */
			list = QF_PRIVATE_DATA(qf)->args;
			while (list) {
				query_field_use_ref (QF_FUNC_ARG(list->data)->qf);
				list = g_slist_next (list);
			}

			/* now it's activated */
			query_field_set_activated (qf, TRUE);
		}
	}
}


/* we want to have either:
   - QF_PRIVATE_DATA(qf)->func = NULL and no weak ref, or
   - QF_PRIVATE_DATA(qf)->func != NULL and a weak ref on it
*/
static void 
qf_set_func_ptr (QueryField *qf, ServerFunction *func)
{
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));
	g_return_if_fail ((func && IS_SERVER_FUNCTION (func)) || !func);

	if (QF_PRIVATE_DATA(qf)->func == func)
		return;

	if (QF_PRIVATE_DATA(qf)->func) {
		if (g_slist_find (QF_PRIVATE_DATA(qf)->weak_ref_objects, QF_PRIVATE_DATA(qf)->func)) {
			g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->func), 
					     (GWeakNotify) q_obj_destroyed_cb, qf);
			QF_PRIVATE_DATA(qf)->weak_ref_objects = g_slist_remove (QF_PRIVATE_DATA(qf)->weak_ref_objects,
										QF_PRIVATE_DATA(qf)->func);
		}

		QF_PRIVATE_DATA(qf)->func = NULL;
	}

	if (func) {
		QF_PRIVATE_DATA(qf)->func = func;
		
		if (!g_slist_find (QF_PRIVATE_DATA(qf)->weak_ref_objects, QF_PRIVATE_DATA(qf)->func)) {
			g_object_weak_ref (G_OBJECT (QF_PRIVATE_DATA(qf)->func), 
					   (GWeakNotify) q_obj_destroyed_cb, qf);
			QF_PRIVATE_DATA(qf)->weak_ref_objects = g_slist_append (QF_PRIVATE_DATA(qf)->weak_ref_objects,
										QF_PRIVATE_DATA(qf)->func);	
		}
	}
}

static void 
sub_qf_status_changed_cb (QueryField *qf, QueryField *pqf)
{
	if (qf->activated)
		query_field_activate (pqf);
	else 
		query_field_set_activated (pqf, FALSE);
}

static QueryField *
get_field_from_xml (GSList *list, gchar *xmlid)
{
	QueryField *qf = NULL;
	gchar *str, *ptr;
	guint id;
	GSList *token;

	g_return_val_if_fail (xmlid, NULL);

	str = g_strdup (xmlid);
	ptr = strtok (str, ":");
	ptr += 2;
	
	ptr = strtok (NULL, ":");
	ptr +=2;
	id = atoi (ptr);
	g_free(str);

	token = list;
	while (token && !qf) {
		if (QUERY_FIELD (token->data)->id == id)
			qf = QUERY_FIELD (token->data);
		token = g_slist_next (token);
	}

	return qf;
}


static void        
q_obj_destroyed_cb (QueryField *qf, GObject   *obj)
{
	QF_PRIVATE_DATA(qf)->weak_ref_objects = g_slist_remove (QF_PRIVATE_DATA(qf)->weak_ref_objects,
								QF_PRIVATE_DATA(qf)->func);	
	q_deactivate (qf);
	
	/* if the field or table disappear, then destroy is the result */
	g_object_unref (G_OBJECT (qf));
}

static void
arg_destroyed_cb (QueryField *qf, QueryField *arg)
{
	/* the steps are to deactivate the function, and empyt the arg */
	GSList *list;
	Arg *found = NULL;
	
	list = QF_PRIVATE_DATA(qf)->args;
	while (list && !found) {
		if (QF_FUNC_ARG(list->data)->qf == arg)
			found = QF_FUNC_ARG(list->data);
		list = g_slist_next (list);
	}

	if (!found) {
		g_warning ("Can't find argument destroyed!");
		return;
	}

	QF_PRIVATE_DATA(qf)->weak_ref_objects = g_slist_remove (QF_PRIVATE_DATA(qf)->weak_ref_objects,
								arg);
	query_field_deactivate (qf);
	found->qf = NULL;
	if (found->xml) { 
		g_free (found->xml);
		found->xml = NULL;
	}

#ifdef debug_signal
	g_print (">> 'FIELD_MODIFIED' from query-fields/arg_destroyed_cb\n");
#endif
	g_signal_emit_by_name (G_OBJECT (qf), "field_modified");
#ifdef debug_signal
	g_print ("<< 'FIELD_MODIFIED' from query-fields/arg_destroyed_cb\n");
#endif	
}


static void function_selected_cb (ObjectSelector *os, GObject *sel, QueryField *qf);

static GtkWidget * 
q_get_edit_widget (QueryField *qf)
{
	GtkWidget *vbox, *label, *os;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD/2.);

	os = object_selector_new (qf->query->conf, OBJECT_SELECTOR_PROCS, -1);
	object_selector_set_column_label (OBJECT_SELECTOR (os), 0, _("Function"));
	if (QF_PRIVATE_DATA(qf)->func) 
		object_selector_set_selection (OBJECT_SELECTOR (os), G_OBJECT (QF_PRIVATE_DATA(qf)->func));
	gtk_box_pack_start (GTK_BOX (vbox), os, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (os);

	g_signal_connect (G_OBJECT (os), "selection_changed",
			  G_CALLBACK (function_selected_cb), qf);

	return vbox;
}

static void 
function_selected_cb (ObjectSelector *os, GObject *sel, QueryField *qf)
{
	if (IS_SERVER_FUNCTION (sel)) {
		ServerFunction *func;
		func = SERVER_FUNCTION (sel);

		if (QF_PRIVATE_DATA(qf)->func != func) {
			GSList *list;
			
			query_field_deactivate (qf);
			
			/* Function's name */
			QF_PRIVATE_DATA(qf)->func = func;
			
			/* Arguments */
			free_args (qf);
			list = func->args;
			while (list) {
				QF_PRIVATE_DATA(qf)->args = g_slist_append (QF_PRIVATE_DATA(qf)->args, 
									    g_new0 (Arg, 1));
				list = g_slist_next (list);
			}
			query_field_activate (qf);
#ifdef debug_signal
			g_print (">> 'FIELD_MODIFIED' from function_selected_cb\n");
#endif
			g_signal_emit_by_name (G_OBJECT (qf), "field_modified");
#ifdef debug_signal
			g_print ("<< 'FIELD_MODIFIED' from function_selected_cb\n");
#endif	
		}
	}
}

static GtkWidget * 
q_get_sel_widget  (QueryField *qf, GCallback callback, gpointer data)
{
	GtkWidget *hbox;
	GtkWidget *button, *label;
	gchar *str;
	GSList *list;
	gboolean firstarg = TRUE;
	guint i;

	hbox = gtk_hbox_new (FALSE, GNOME_PAD/2.);
	
	/* button for the function's name */
	if (QF_PRIVATE_DATA(qf)->func)
		str = g_strdup_printf ("%s", QF_PRIVATE_DATA(qf)->func->sqlname);
	else
		str = g_strdup (_("Function"));
	button = gtk_button_new_with_label (str);	
	g_free (str);
	g_signal_connect (G_OBJECT (button), "clicked", callback, data);
	g_object_set_data (G_OBJECT (button), "qf", qf);
	gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
	gtk_widget_show (button);	

	label = gtk_label_new (" (");
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
	gtk_widget_show (label);

	/* Set the "QF_obj_emit_sig" attribute so that we can attach attributes to that button
	   which will be transmitted when the user clicks on it */
	g_object_set_data (G_OBJECT (hbox), "QF_obj_emit_sig", button);

	/* arguments */
	i = 0;
	list = QF_PRIVATE_DATA(qf)->args;
	while (list) {
		GObject   *obj;
		GtkWidget *wid;
		
		/* separator */
		if (firstarg) 
			firstarg = FALSE;
		else {
			label = gtk_label_new (", ");
			gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
			gtk_widget_show (label);
		}

		/* sub widget */
		wid = query_field_get_select_widget (QF_FUNC_ARG(list->data)->qf, callback, data);
		gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);

		obj = g_object_get_data (G_OBJECT (wid), "QF_obj_emit_sig");
		g_object_set_data (G_OBJECT (obj), "pqf", qf);
		g_object_set_data (G_OBJECT (obj), "ref", GUINT_TO_POINTER (i));

		list = g_slist_next (list);
		i++;
	}

	label = gtk_label_new (")");
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
	gtk_widget_show (label);

	return hbox;
}

static gchar     * 
q_render_as_sql   (QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;

	if (qf->activated) {
		GString *string;
		GSList *list;
		gboolean firstarg = TRUE;

		string = g_string_new ("");
		g_string_sprintfa (string, "%s (", QF_PRIVATE_DATA(qf)->func->sqlname);
		
		list = QF_PRIVATE_DATA(qf)->args;
		while (list) {
			gchar *str2;
			if (firstarg) 
				firstarg = FALSE;
			else
				g_string_append (string, ", ");

			str2 = query_field_render_as_sql (QF_FUNC_ARG(list->data)->qf, NULL);
			g_string_append (string, str2);
			g_free (str2);
			list = g_slist_next (list);
		}
		
		g_string_append (string, ")");
		str = string->str;
		g_string_free (string, FALSE);
	}

	return str;
}

static xmlNodePtr  
q_render_as_xml   (QueryField *qf, GSList * missing_values)
{
	return NULL;
}

static gchar * 
q_render_as_string(QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;

	if (qf->activated) {
		GString *string;
		GSList *list;
		gboolean firstarg = TRUE;
		
		string = g_string_new ("");
		g_string_sprintfa (string, "%s (", QF_PRIVATE_DATA(qf)->func->sqlname);
		
		list = QF_PRIVATE_DATA(qf)->args;
		while (list) {
			gchar *str2;
			if (firstarg) 
				firstarg = FALSE;
			else
				g_string_append (string, ", ");
			
			str2 = query_field_render_as_string (QF_FUNC_ARG(list->data)->qf, NULL);
			g_string_append (string, str2);
			g_free (str2);
			
			list = g_slist_next (list);
		}
		
		g_string_append (string, ")");
		str = string->str;
		g_string_free (string, FALSE);
	}
	else 
		str = g_strdup (_("--Arguments Missing--"));

	return str;
}

static void  
q_save_to_xml     (QueryField *qf, xmlNodePtr node)
{
	if (qf->activated) {
		GSList *list;
		gchar *str;
		xmlNodePtr arg;

		g_print("Saving field to node %p...", node);	

		/* node object ref */
		str = server_function_get_xml_id (QF_PRIVATE_DATA(qf)->func);
		xmlSetProp (node, "object", str);
		g_free (str);

		xmlSetProp (node, "type", "function");
		list = QF_PRIVATE_DATA(qf)->args;
		while (list) {
			arg = xmlNewChild (node, NULL, "QueryFieldArg", NULL);
			str = query_field_get_xml_id (QF_FUNC_ARG(list->data)->qf);
			xmlSetProp (arg, "ref", str);
			g_free (str);

			list = g_slist_next (list);
		}

		g_print("done\n");
	}
	else
		g_warning ("QueryField not activated; can't save\n");
}

static void        
q_load_from_xml   (QueryField *qf, xmlNodePtr node)
{
	query_field_deactivate (qf);
	
	/* check we have a QueryField */
	if (!strcmp (node->name, "QueryField")) {
		gchar *str;
		xmlNodePtr arg;

		str = xmlGetProp (node, "type");
		if (!str || (str && strcmp (str, "function"))) {
			if (str) g_free (str);
			return;
		}

		str = xmlGetProp (node, "object");
		/* check we have a query field relating to a function */
		if ((*str == 'P') && (*(str + 1) == 'R')) {
			QF_PRIVATE_DATA(qf)->func_name = g_strdup (str);
		}
		g_free (str);

		/* Getting the arguments */
		arg = node->xmlChildrenNode;
		while (arg) {
			if (*arg->name == 'Q') {
				str = xmlGetProp (arg, "ref");
				if (str) {
					Arg *arg;
					
					arg = g_new0 (Arg, 1);
					arg->xml = str;
					QF_PRIVATE_DATA(qf)->args = g_slist_append (QF_PRIVATE_DATA(qf)->args,
										    arg);
				}
			}
			arg = arg->next;
		}

		query_field_activate (qf);
	}
}

static void        
q_copy_other_field(QueryField *qf, QueryField *other)
{
	GSList *list;
	
	/* we can't call q_destroy(qf) because we don't know what the type
	   of QueryField it was before. This is normally done by the
	   QueryField object before the copy */

	if (QF_PRIVATE_DATA(other)->func) 
		QF_PRIVATE_DATA(qf)->func = QF_PRIVATE_DATA(other)->func;
	else {
		if (QF_PRIVATE_DATA(other)->func_name) 
			QF_PRIVATE_DATA(qf)->func_name = g_strdup (QF_PRIVATE_DATA(other)->func_name);
	}

	/* arguments of the function */
	list = QF_PRIVATE_DATA(other)->args;
	while (list) {
		Arg *arg;

		arg = g_new0 (Arg, 1);
		arg->xml = g_strdup (QF_FUNC_ARG(list->data)->xml);
		QF_PRIVATE_DATA(qf)->args = g_slist_append (QF_PRIVATE_DATA(qf)->args, arg);
		list = g_slist_next (list);
	}
	query_field_activate (qf);
}

static gboolean
q_is_equal_to (QueryField *qf, QueryField *other)
{
	gboolean retval = FALSE;

	if (qf->activated && other->activated) {
		if (QF_PRIVATE_DATA(qf)->func == QF_PRIVATE_DATA(other)->func)
			retval = TRUE;
	}

	return retval;
}



static GSList *
q_get_monitored_objects (QueryField *qf)
{
	GSList *list = NULL, *args;

	g_assert (qf && IS_QUERY_FIELD (qf));

	if (qf->activated)
		list = g_slist_prepend (NULL, QF_PRIVATE_DATA(qf)->func);

	g_print ("QF_PRIVATE_DATA(qf) = %p\n", QF_PRIVATE_DATA(qf));
	args = QF_PRIVATE_DATA(qf)->args;
	while (args) {
		if (QF_FUNC_ARG(args->data)->qf)
			list = g_slist_prepend (list, QF_FUNC_ARG(args->data)->qf);
		args = g_slist_next (args);
	}

	return list;
}


static void
q_replace_comp (QueryField *qf, gint ref, GObject   *old, GObject   *new)
{
	gboolean replaced = FALSE;

	g_assert (qf && IS_QUERY_FIELD (qf));

	if ((!old || (old && IS_QUERY_FIELD (old)) ) &&
	    (new && IS_QUERY_FIELD (new))) {
#ifdef debug
		g_print ("For QF %p, replacing (old=%p, ref=%d) with %p\n", qf, old, ref, new);
		func_dump (qf); /* REMOVE */
#endif

		if (old) {
			Arg *arg = NULL;
			GSList *list = QF_PRIVATE_DATA(qf)->args;

			while (list && !arg) {
				if (QF_FUNC_ARG(list->data)->qf == QUERY_FIELD (old))
					arg = QF_FUNC_ARG(list->data);
				list = g_slist_next (list);
			}
			if (arg) {
				query_field_deactivate (qf);
				if (arg->xml)
					g_free (arg->xml);
				arg->xml = query_field_get_xml_id (QUERY_FIELD (new));
				query_field_activate (qf);
				replaced = TRUE;
			}
		}
		else {
			Arg *arg;
		
			arg = g_slist_nth_data (QF_PRIVATE_DATA(qf)->args, ref);
			if (arg) {
				query_field_deactivate (qf);
				if (arg->xml)
					g_free (arg->xml);
				arg->xml = NULL;
				arg->xml = query_field_get_xml_id (QUERY_FIELD (new));
				query_field_activate (qf);
				replaced = TRUE;
			}
		}

		if (!replaced)
			g_warning ("Reference to be replaced is NULL\n");
		else {
#ifdef debug_signal
			g_print (">> 'FIELD_MODIFIED' from function_selected_cb\n");
#endif
			g_signal_emit_by_name (G_OBJECT (qf), "field_modified");
#ifdef debug_signal
			g_print ("<< 'FIELD_MODIFIED' from function_selected_cb\n");
#endif			
		}
	}
}


/* 
 * 
 * QueryField object's different implementations
 * 
 *
 */

