/* object-selector-ext.c
 *
 * Copyright (C) 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "object-selector.h"
#include "object-selector-ext.h"
#include "object-selector-priv.h"
#include "database.h"
#include "query.h"
#include <libgnomedb/libgnomedb.h>

/*
 *  
 * Model for the tables and views 
 * 
 */
static GdkPixbuf *get_db_icon      (Database *db);
static gchar     *get_db_name      (Database *db);
static GdkPixbuf *get_dbtable_icon (DbTable *table);
static gchar     *get_dbtable_name (DbTable *table);
static GSList    *get_dbtable_cols (DbTable *table, guint nbcols);
static gchar     *get_dbfield_name (DbField *field);
static GSList    *get_dbfield_cols (DbField *field, guint nbcols);

void 
object_selector_create_model_tables_views (ObjectSelector *os, gint n_top_levels, guint *max_cols)
{
	GtkTreeIter  iter, *iter_ptr = NULL;
	
	ObjectInfo db_info = {DATABASE_TYPE, 
			      NULL,
			      NULL,
			      NULL,
			      NULL};

	ObjectInfo table_info = {DB_TABLE_TYPE, 
				 GET_ICON_FUNC (get_dbtable_icon),
				 GET_INFO_FUNC (get_dbtable_name),
				 GET_INFO_FUNC (get_dbtable_name),
				 GET_COLS_FUNC (get_dbtable_cols)};

	ObjectInfo field_info = {DB_FIELD_TYPE, 
				 NULL,
				 GET_INFO_FUNC (get_dbfield_name),
				 GET_INFO_FUNC (get_dbfield_name),
				 GET_COLS_FUNC (get_dbfield_cols)};
	
	/* adding the manager(s) */
	if (os->priv->mode & OBJECT_SELECTOR_TABLES ||
	    os->priv->mode & OBJECT_SELECTOR_VIEWS) {
		GSList *list;

		if (n_top_levels > 1) {
			db_info.get_icon = GET_ICON_FUNC (get_db_icon);
			db_info.get_name = GET_INFO_FUNC (get_db_name);
		}

		object_selector_add_manager_class (os, OBJECT_SELECTOR_TABLES | OBJECT_SELECTOR_VIEWS,
						   &db_info, "table_created", "table_dropped",
						   &table_info);

		object_selector_add_manager_class (os, OBJECT_SELECTOR_DBFIELDS,
						   &table_info, "field_created", "field_dropped",
						   &field_info);

		object_selector_add_manager (os, G_OBJECT (os->priv->conf->db), &db_info);
		if (n_top_levels > 1) {
			object_selector_add_object (os, NULL, NULL, 
						    G_OBJECT (os->priv->conf->db), &db_info, &iter, FALSE);
			iter_ptr = &iter;
		}
		
		/* existing tables */
		list = os->priv->conf->db->tables;
		while (list) {
			GtkTreeIter iter2;
			/* add the table */
			object_selector_add_manager (os, G_OBJECT (list->data), &table_info);
			object_selector_add_object (os, iter_ptr, G_OBJECT (os->priv->conf->db),
						    G_OBJECT (list->data), &table_info, &iter2, TRUE);

			if (os->priv->mode & OBJECT_SELECTOR_DBFIELDS) { 
				GSList *list2;
					
				/* existing fields in that table */
				list2 = DB_TABLE (list->data)->fields;
				while (list2) {
					object_selector_add_object (os, &iter2, G_OBJECT (list->data),
								    G_OBJECT (list2->data), &field_info, NULL,
								    FALSE);

					list2 = g_slist_next (list2);
				}
			}
			
			list = g_slist_next (list);
		}
	}
}

static GdkPixbuf *
get_db_icon (Database *db)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (db && IS_DATABASE (db), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_TABLES);
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}

static gchar *
get_db_name (Database *db)
{
	gchar *str;

	g_return_val_if_fail (db && IS_DATABASE (db), NULL);
	str = g_strdup (_("Tables & views"));
	return str;
}

static GdkPixbuf *
get_dbtable_icon (DbTable *table)
{
	static GdkPixbuf *table_pixbuf = NULL, *view_pixbuf = NULL;
	

	g_return_val_if_fail (table && IS_DB_TABLE (table), NULL);

	if (!table_pixbuf) {
		table_pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-table_16x16.png");
		if (table_pixbuf)
			g_object_ref (G_OBJECT (table_pixbuf));
	}
	if (!view_pixbuf) {
		view_pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-views_16x16.png");
		if (view_pixbuf)
			g_object_ref (G_OBJECT (view_pixbuf));
	}

	if (table->is_view)
		return view_pixbuf;
	else
		return table_pixbuf;
}


static gchar *
get_dbtable_name    (DbTable *table)
{
	gchar *str;
	g_return_val_if_fail (table && IS_DB_TABLE (table), NULL);
	str = g_strdup (table->name);

	return str;
}

static GSList *
get_dbtable_cols (DbTable *table, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (table && IS_DB_TABLE (table), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (table->comments)
		str = g_strdup (table->comments);
	else 
		str = g_strdup ("");

	return g_slist_append (NULL, str);
}

static gchar *
get_dbfield_name    (DbField *field)
{
	gchar *str;
	g_return_val_if_fail (field && IS_DB_FIELD (field), NULL);
	str = g_strdup (field->name);

	return str;
}

static GSList *
get_dbfield_cols (DbField *field, guint nbcols)
{
	gchar *str;
	g_return_val_if_fail (field && IS_DB_FIELD (field), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (field->length >= 0)
		str = g_strdup_printf ("%s (%d)", field->type->sqlname, field->length);
	else
		str = g_strdup (field->type->sqlname);

	return g_slist_append (NULL, str);
}









/*
 *  
 * Model for the sequences
 * 
 */
static GdkPixbuf *get_db_icon_seq     (Database *db);
static gchar     *get_db_name_seq     (Database *db);
static GdkPixbuf *get_dbsequence_icon (DbSequence *sequence);
static gchar     *get_dbsequence_name (DbSequence *sequence);
static GSList    *get_dbsequence_cols (DbSequence *sequence, guint nbcols);

void 
object_selector_create_model_sequences    (ObjectSelector *os, gint n_top_levels, guint *max_cols)
{
	GtkTreeIter iter, *iter_ptr = NULL;
	
	ObjectInfo db_info = {DATABASE_TYPE, 
			      NULL,
			      NULL,
			      NULL,
			      NULL};

	ObjectInfo seq_info = {DB_SEQUENCE_TYPE, 
			       GET_ICON_FUNC (get_dbsequence_icon),
			       GET_INFO_FUNC (get_dbsequence_name),
			       GET_INFO_FUNC (get_dbsequence_name),
			       GET_COLS_FUNC (get_dbsequence_cols)};

	/* adding the manager */
	if (os->priv->mode & OBJECT_SELECTOR_SEQUENCES) {
		GSList *list;

		if (n_top_levels > 1) {
			db_info.get_icon = GET_ICON_FUNC (get_db_icon_seq);
			db_info.get_name = GET_INFO_FUNC (get_db_name_seq);
		}

		object_selector_add_manager_class (os, OBJECT_SELECTOR_SEQUENCES,
						   &db_info, "seq_created", "seq_dropped",
						   &seq_info);

		object_selector_add_manager (os, G_OBJECT (os->priv->conf->db), &db_info);

		if (n_top_levels > 1) {
			object_selector_add_object (os, NULL, NULL, 
						    G_OBJECT (os->priv->conf->db), &db_info, &iter, FALSE);
			iter_ptr = &iter;
		}

		/* existing sequences */
		list = os->priv->conf->db->sequences;
		while (list) {
			object_selector_add_object (os, iter_ptr, G_OBJECT (os->priv->conf->db),
						    G_OBJECT (list->data), &seq_info, NULL, TRUE);
			
			list = g_slist_next (list);
		}
	}
}


static GdkPixbuf *
get_db_icon_seq (Database *db)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (db && IS_DATABASE (db), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_SEQUENCES);
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}

static gchar *
get_db_name_seq (Database *db)
{
	gchar *str;

	g_return_val_if_fail (db && IS_DATABASE (db), NULL);
	str = g_strdup (_("Sequences"));
	return str;
}

static GdkPixbuf *
get_dbsequence_icon (DbSequence *sequence)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (sequence && IS_DB_SEQUENCE (sequence), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-sequences_16x16.png");
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}


static gchar *
get_dbsequence_name    (DbSequence *sequence)
{
	g_return_val_if_fail (sequence && IS_DB_SEQUENCE (sequence), NULL);
	return g_strdup (sequence->name);
}

static GSList *
get_dbsequence_cols (DbSequence *sequence, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (sequence && IS_DB_SEQUENCE (sequence), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (sequence->comments)
		str = g_strdup (sequence->comments);
	else 
		str = g_strdup ("");

	return g_slist_append (NULL, str);
}





/*
 *  
 * Model for the data types
 * 
 */

static GdkPixbuf  *get_serveraccess_icon_dt  (ServerAccess *srv);
static gchar      *get_serveraccess_name_dt  (ServerAccess *srv);
static GdkPixbuf  *get_datatype_icon         (ServerDataType *type);
static gchar      *get_datatype_name         (ServerDataType *type);
static gchar      *get_datatype_cmp_name     (ServerDataType *type);
static GSList     *get_datatype_cols         (ServerDataType *type, guint nbcols);

void 
object_selector_create_model_datatypes    (ObjectSelector *os, gint n_top_levels, guint *max_cols)
{
	GtkTreeIter iter, *iter_ptr = NULL;
	
	ObjectInfo srv_info = {SERVER_ACCESS_TYPE, 
			       NULL,
			       NULL,
			       NULL,
			       NULL};

	ObjectInfo dt_info = {SERVER_DATA_TYPE_TYPE,
			      GET_ICON_FUNC (get_datatype_icon),
			      GET_INFO_FUNC (get_datatype_name),
			      GET_INFO_FUNC (get_datatype_cmp_name),
			      GET_COLS_FUNC (get_datatype_cols)};

	/* adding the manager */
	if (os->priv->mode & OBJECT_SELECTOR_DATA_TYPES) {
		GSList *list;

		if (n_top_levels > 1) {
			srv_info.get_icon = GET_ICON_FUNC (get_serveraccess_icon_dt);
			srv_info.get_name = GET_INFO_FUNC (get_serveraccess_name_dt);
		}

		object_selector_add_manager_class (os, OBJECT_SELECTOR_DATA_TYPES,
						   &srv_info, "data_type_added", "data_type_removed",
						   &dt_info);
		object_selector_add_manager (os, G_OBJECT (os->priv->conf->srv), &srv_info);

		if (n_top_levels > 1) {
			object_selector_add_object (os, NULL, NULL, 
						    G_OBJECT (os->priv->conf->srv), &srv_info, &iter, FALSE);
			iter_ptr = &iter;
		}


		/* existing data types */
		list = os->priv->conf->srv->data_types;
		while (list) {
			object_selector_add_object (os, iter_ptr, G_OBJECT (os->priv->conf->srv),
						    G_OBJECT (list->data), &dt_info, NULL, FALSE);
			
			list = g_slist_next (list);
		}
	}	
}

static GdkPixbuf  *
get_serveraccess_icon_dt (ServerAccess *srv)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (srv && IS_SERVER_ACCESS (srv), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_TYPES);
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}

static gchar  *
get_serveraccess_name_dt (ServerAccess *srv)
{
	return g_strdup (_("Data types"));
}


static GdkPixbuf *
get_datatype_icon (ServerDataType *type)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (type && IS_SERVER_DATA_TYPE (type), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-types_16x16.png");
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}


static gchar *
get_datatype_name    (ServerDataType *type)
{
	gchar *str;
	g_return_val_if_fail (type && IS_SERVER_DATA_TYPE (type), NULL);

	if (type->sqlname)
		str = g_strdup (type->sqlname);
	else
		str = g_strdup ("???");

	return str;
}

static GSList *
get_datatype_cols (ServerDataType *type, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (type && IS_SERVER_DATA_TYPE (type), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (type->descr)
		str = g_strdup (type->descr);
	else 
		str = g_strdup ("");

	return g_slist_append (NULL, str);
}

static gchar *
get_datatype_cmp_name    (ServerDataType *type)
{
	g_return_val_if_fail (type && IS_SERVER_DATA_TYPE (type), NULL);
	return g_strdup (type->sqlname);
}








/*
 *  
 * Model for the functions
 * 
 */

static GdkPixbuf  *get_serveraccess_icon (ServerAccess *srv);
static gchar      *get_serveraccess_name (ServerAccess *srv);
static GdkPixbuf  *get_function_icon     (ServerFunction *function);
static gchar      *get_function_name     (ServerFunction *function);
static gchar      *get_function_cmp_name (ServerFunction *function);
static GSList     *get_function_cols     (ServerFunction *function, guint nbcols);

void 
object_selector_create_model_functions    (ObjectSelector *os, gint n_top_levels, guint *max_cols)
{
	GtkTreeIter iter, *iter_ptr = NULL;
	
	ObjectInfo srv_info = {SERVER_ACCESS_TYPE, 
			       NULL,
			       NULL,
			       NULL,
			       NULL};

	ObjectInfo func_info = {SERVER_FUNCTION_TYPE, 
				GET_ICON_FUNC (get_function_icon),
				GET_INFO_FUNC (get_function_name),
				GET_INFO_FUNC (get_function_cmp_name),
				GET_COLS_FUNC (get_function_cols)};

	/* adding the manager */
	if (os->priv->mode & OBJECT_SELECTOR_PROCS) {
		GSList *list;

		if (n_top_levels > 1) {
			srv_info.get_icon = GET_ICON_FUNC (get_serveraccess_icon);
			srv_info.get_name = GET_INFO_FUNC (get_serveraccess_name);
		}

		object_selector_add_manager_class (os, OBJECT_SELECTOR_PROCS,
						   &srv_info, "data_function_added", "data_function_removed",
						   &func_info);
		object_selector_add_manager (os, G_OBJECT (os->priv->conf->srv), &srv_info);

		if (n_top_levels > 1) {
			object_selector_add_object (os, NULL, NULL, 
						    G_OBJECT (os->priv->conf->srv), &srv_info, &iter, FALSE);
			iter_ptr = &iter;
		}


		/* existing functions */
		list = os->priv->conf->srv->data_functions;
		while (list) {
			object_selector_add_object (os, iter_ptr, G_OBJECT (os->priv->conf->srv),
						    G_OBJECT (list->data), &func_info, NULL, FALSE);
			
			list = g_slist_next (list);
		}
	}	
}

static GdkPixbuf  *
get_serveraccess_icon (ServerAccess *srv)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (srv && IS_SERVER_ACCESS (srv), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_PROCEDURES);
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}

static gchar  *
get_serveraccess_name (ServerAccess *srv)
{
	return g_strdup (_("Functions"));
}

static GdkPixbuf *
get_function_icon (ServerFunction *function)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (function && IS_SERVER_FUNCTION (function), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-procedures_16x16.png");
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}


static gchar *
get_function_name    (ServerFunction *function)
{
	GString *string;
	gboolean start = TRUE;
	GSList *list;
	gchar *str;

	g_return_val_if_fail (function && IS_SERVER_FUNCTION (function), NULL);
	string = g_string_new (function->sqlname);
	list = function->args;

	while (list) {
		if (start) {
			start = FALSE;
			g_string_append (string, " (");
		}
		else
			g_string_append (string, ", ");
		g_string_append (string, SERVER_DATA_TYPE (list->data)->sqlname);
			list = g_slist_next (list);
	}
	g_string_append (string, ")");
	str = string->str;
	g_string_free (string, FALSE);

	return str;
}

static GSList *
get_function_cols (ServerFunction *function, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (function && IS_SERVER_FUNCTION (function), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (function->descr)
		str = g_strdup (function->descr);
	else 
		str = g_strdup ("");

	return g_slist_append (NULL, str);
}

static gchar *
get_function_cmp_name    (ServerFunction *function)
{
	g_return_val_if_fail (function && IS_SERVER_FUNCTION (function), NULL);
	return g_strdup (function->sqlname);
}








/*
 *  
 * Model for the aggregates
 * 
 */

static GdkPixbuf  *get_serveraccess_icon_agg (ServerAccess *srv);
static gchar      *get_serveraccess_name_agg (ServerAccess *srv);
static GdkPixbuf  *get_aggregate_icon        (ServerAggregate *aggregate);
static gchar      *get_aggregate_name        (ServerAggregate *aggregate);
static gchar      *get_aggregate_cmp_name    (ServerAggregate *aggregate);
static GSList     *get_aggregate_cols        (ServerAggregate *aggregate, guint nbcols);

void 
object_selector_create_model_aggregates    (ObjectSelector *os, gint n_top_levels, guint *max_cols)
{
	GtkTreeIter iter, *iter_ptr = NULL;
	
	ObjectInfo srv_info = {SERVER_ACCESS_TYPE, 
			       NULL,
			       NULL,
			       NULL,
			       NULL};

	ObjectInfo agg_info = {SERVER_AGGREGATE_TYPE, 
			       GET_ICON_FUNC (get_aggregate_icon),
			       GET_INFO_FUNC (get_aggregate_name),
			       GET_INFO_FUNC (get_aggregate_cmp_name),
			       GET_COLS_FUNC (get_aggregate_cols)};

	/* adding the manager */
	if (os->priv->mode & OBJECT_SELECTOR_AGGS) {
		GSList *list;

		if (n_top_levels > 1) {
			srv_info.get_icon = GET_ICON_FUNC (get_serveraccess_icon_agg);
			srv_info.get_name = GET_INFO_FUNC (get_serveraccess_name_agg);
		}

		object_selector_add_manager_class (os, OBJECT_SELECTOR_AGGS,
						   &srv_info, "data_aggregate_added", "data_aggregate_removed",
						   &agg_info);
		object_selector_add_manager (os, G_OBJECT (os->priv->conf->srv), &srv_info);

		if (n_top_levels > 1) {
			object_selector_add_object (os, NULL, NULL, 
						    G_OBJECT (os->priv->conf->srv), &srv_info, &iter, FALSE);
			iter_ptr = &iter;
		}


		/* existing aggregates */
		list = os->priv->conf->srv->data_aggregates;
		while (list) {
			object_selector_add_object (os, iter_ptr, G_OBJECT (os->priv->conf->srv),
						    G_OBJECT (list->data), &agg_info, NULL, FALSE);
			
			list = g_slist_next (list);
		}
	}	
}

static GdkPixbuf  *
get_serveraccess_icon_agg (ServerAccess *srv)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (srv && IS_SERVER_ACCESS (srv), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_AGGREGATES);
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}

static gchar  *
get_serveraccess_name_agg (ServerAccess *srv)
{
	return g_strdup (_("Aggregates"));
}


static GdkPixbuf *
get_aggregate_icon (ServerAggregate *aggregate)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (aggregate && IS_SERVER_AGGREGATE (aggregate), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-aggregates_16x16.png");
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}


static gchar *
get_aggregate_name    (ServerAggregate *aggregate)
{
	gchar *str;
	g_return_val_if_fail (aggregate && IS_SERVER_AGGREGATE (aggregate), NULL);

	if (aggregate->arg_type)
		str = g_strdup_printf ("%s (%s)", aggregate->sqlname,  aggregate->arg_type->sqlname);
	else
		str = g_strdup_printf (_("%s (any argument type)"), aggregate->sqlname);

	return str;
}

static GSList *
get_aggregate_cols (ServerAggregate *aggregate, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (aggregate && IS_SERVER_AGGREGATE (aggregate), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (aggregate->descr)
		str = g_strdup (aggregate->descr);
	else 
		str = g_strdup ("");

	return g_slist_append (NULL, str);
}

static gchar *
get_aggregate_cmp_name    (ServerAggregate *aggregate)
{
	g_return_val_if_fail (aggregate && IS_SERVER_AGGREGATE (aggregate), NULL);
	return g_strdup (aggregate->sqlname);
}









/*
 *  
 * Model for the queries
 * 
 */

static GdkPixbuf  *get_top_query_icon (Query *query);
static gchar      *get_top_query_name (Query *query);
static GdkPixbuf  *get_query_icon (Query *query);
static gchar      *get_query_name (Query *query);
static gchar      *get_query_cmp_name (Query *query);
static GSList     *get_query_cols (Query *query, guint nbcols);
static GdkPixbuf  *get_queryview_icon (QueryView *qv);
static gchar      *get_queryview_name (QueryView *qv);
static gchar      *get_queryview_cmp_name (QueryView *qv);
static GSList     *get_queryview_cols (QueryView *qv, guint nbcols);


static void real_create_model_query (ObjectSelector *os, GtkTreeIter *parent_iter, Query *parent_query,
				     gboolean sub_queries);
void 
object_selector_create_model_queries (ObjectSelector *os, gint n_top_levels, guint *max_cols,
				      Query *top_query, gboolean sub_queries)
{
	GtkTreeIter iter, *iter_ptr = NULL;
	
	ObjectInfo topq_info = {QUERY_TYPE, 
				NULL,
				NULL,
				NULL,
				NULL};

	ObjectInfo q_info = {QUERY_TYPE, 
			     GET_ICON_FUNC (get_query_icon),
			     GET_INFO_FUNC (get_query_name),
			     GET_INFO_FUNC (get_query_cmp_name),
			     GET_COLS_FUNC (get_query_cols)};

	ObjectInfo qview_info = {QUERY_VIEW_TYPE, 
				 GET_ICON_FUNC (get_queryview_icon),
				 GET_INFO_FUNC (get_queryview_name),
				 GET_INFO_FUNC (get_queryview_cmp_name),
				 GET_COLS_FUNC (get_queryview_cols)};

	
	/* adding the manager */
	if (os->priv->mode & OBJECT_SELECTOR_QUERIES) {
		if (n_top_levels > 1) {
			topq_info.get_icon = GET_ICON_FUNC (get_top_query_icon);
			topq_info.get_name = GET_INFO_FUNC (get_top_query_name);
		}
		
		object_selector_add_manager_class (os, OBJECT_SELECTOR_QUERIES,
						   &q_info, "child_query_created", "child_query_dropped",
						   &q_info);
		if (!top_query) 
			object_selector_add_manager_class (os, OBJECT_SELECTOR_QUERIES,
							   &topq_info, "child_query_created", "child_query_dropped",
							   &q_info);

		if (os->priv->mode & OBJECT_SELECTOR_QVIEWS)  /* QueryView as well */
			object_selector_add_manager_class (os, OBJECT_SELECTOR_QUERIES,
							   &q_info, "query_view_added", "query_view_removed",
							   &qview_info);
		
		if (!top_query ||
		    (top_query && (top_query == QUERY (os->priv->conf->top_query)))) 
			object_selector_add_manager (os, G_OBJECT (os->priv->conf->top_query), &topq_info);
		else
			object_selector_add_manager (os, G_OBJECT (top_query), &q_info);

		if (n_top_levels > 1) {
			object_selector_add_object (os, NULL, NULL, 
						    G_OBJECT (os->priv->conf->top_query), 
						    &topq_info, &iter, FALSE);
			iter_ptr = &iter;
			if (top_query && (top_query != QUERY (os->priv->conf->top_query))) {
				object_selector_add_object (os, iter_ptr, NULL, 
							    G_OBJECT (top_query), &q_info, &iter, FALSE);
				iter_ptr = &iter;
			}
		}

		/* existing queries */
		if (!top_query ||
		    (top_query && (top_query == QUERY (os->priv->conf->top_query)))) 
			real_create_model_query (os, iter_ptr, QUERY (os->priv->conf->top_query), sub_queries);
		else
			real_create_model_query (os, iter_ptr, top_query, sub_queries);
	}	
}

static void 
real_create_model_query (ObjectSelector *os, GtkTreeIter *parent_iter, Query *parent_query, gboolean sub_queries)
{
	ObjectInfo q_info = {QUERY_TYPE, 
			     GET_ICON_FUNC (get_query_icon),
			     GET_INFO_FUNC (get_query_name),
			     GET_INFO_FUNC (get_query_cmp_name),
			     GET_COLS_FUNC (get_query_cols)};
	ObjectInfo qview_info = {QUERY_VIEW_TYPE, 
				 GET_ICON_FUNC (get_queryview_icon),
				 GET_INFO_FUNC (get_queryview_name),
				 GET_INFO_FUNC (get_queryview_cmp_name),
				 GET_COLS_FUNC (get_queryview_cols)};
	GSList *list;

	if (os->priv->mode & OBJECT_SELECTOR_QVIEWS && (parent_query != QUERY (os->priv->conf->top_query))) {
		list = parent_query->views;
		while (list) {
			object_selector_add_object (os, parent_iter, G_OBJECT (parent_query),
						    G_OBJECT (list->data), &qview_info, NULL, FALSE);
			list = g_slist_next (list);
		}
	}

	if (sub_queries) {
		list = parent_query->sub_queries;
		while (list) {
			GtkTreeIter iter;
			
			object_selector_add_manager (os, G_OBJECT (list->data), &q_info);
			object_selector_add_object (os, parent_iter, G_OBJECT (parent_query),
						    G_OBJECT (list->data), &q_info, &iter, FALSE);
			real_create_model_query (os, &iter, QUERY (list->data), sub_queries);
			
			list = g_slist_next (list);
		}
	}
}


static GdkPixbuf  *
get_top_query_icon (Query *query)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_QUERY);
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}

static gchar  *
get_top_query_name (Query *query)
{
	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	return g_strdup (_("Queries"));
}


static GdkPixbuf *
get_query_icon (Query *query)
{
	static GdkPixbuf *pixbuf = NULL;

	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	if (!pixbuf) {
		pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-query_16x16.png");
		if (pixbuf)
			g_object_ref (G_OBJECT (pixbuf));
	}

	return pixbuf;
}


static gchar *
get_query_name    (Query *query)
{
	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	return g_strdup (query->name);
}

static GSList *
get_query_cols (Query *query, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);
	if (query->descr)
		str = g_strdup (query->descr);
	else 
		str = g_strdup ("");

	return g_slist_append (NULL, str);
}

static gchar *
get_query_cmp_name    (Query *query)
{
	g_return_val_if_fail (query && IS_QUERY (query), NULL);
	return g_strdup_printf ("%s%d", query->name, query->id);
}

static GdkPixbuf  *
get_queryview_icon (QueryView *qv)
{
	static GdkPixbuf *table_pixbuf = NULL, *view_pixbuf = NULL, *query_pixbuf = NULL;

	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);

	if (!table_pixbuf) {
		table_pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-table_16x16.png");
		if (table_pixbuf)
			g_object_ref (G_OBJECT (table_pixbuf));
	}
	if (!view_pixbuf) {
		view_pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-views_16x16.png");
		if (view_pixbuf)
			g_object_ref (G_OBJECT (view_pixbuf));
	}

	if (!query_pixbuf) {
		query_pixbuf = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-query_16x16.png");
		if (query_pixbuf)
			g_object_ref (G_OBJECT (query_pixbuf));
	}


	if (IS_DB_TABLE (qv->obj)) {
		if (DB_TABLE (qv->obj)->is_view)
			return view_pixbuf;
		else
			return table_pixbuf;
	}
	else
		return query_pixbuf;
}

static gchar      *
get_queryview_name (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	return g_strdup (query_view_get_name (qv));
}

static gchar      *
get_queryview_cmp_name (QueryView *qv)
{
	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	return query_view_get_xml_id (qv);
}

static GSList     *
get_queryview_cols (QueryView *qv, guint nbcols)
{
	gchar *str;

	g_return_val_if_fail (qv && IS_QUERY_VIEW (qv), NULL);
	g_return_val_if_fail (nbcols == 1, NULL);

	str = g_strdup_printf (_("Represents %s"), query_view_get_real_name (qv));
	return g_slist_append (NULL, str);
}
