/* show infoscreen
   show helpscreen

   Written by Matthias Hensler
   Copyright WSPse 1999-2004
   eMail: matthias@wspse.de

Created: 1999/06/13
Updated: 2004/04/15, MH: fixed non ASCII characters
         2006/03/08, MH: version 0.30
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ncurses.h>
#include "mp3creat.h"

extern WINDOW *c_newwin(int h, int l, int y, int x, void *proc, int arg1, int arg2);
extern void c_delwin(WINDOW *win);
extern void free_char_array(char **array);
extern void store_win_poi(WINDOW *win, void *pointer);
extern void *pop_win_poi(WINDOW *win);
extern void wuuush(int code);
extern char external_version_str[];

int is_curr_line;
int is_max_len;
int is_like_len, is_like_height;

/* Global pointers to textes */
char **is_online_help = NULL;

int is_count_lines(const char **text)
{
  int lines;

  if(! text) return 0;

  lines = 0;
  while(text[lines]) lines++;

  return(lines == 0 ? lines : lines-1);
}

void is_stat(WINDOW *win, int ys, int ye, int x, int abs, int tot)
{
  int i, y1, y2, yb;

  yb = (ye-ys) + 1;
  if(yb >= tot) return;
  ys--;

  if(! tot) return;

  y1 = (abs * yb) / tot;
  y2 = y1 + ((yb*yb) / tot) + 1;
  y1++;
  if(y2 > yb) y1 = (yb - (y2-y1));

  for(i=1;i<=yb;i++) {
    if((i>=y1) && (i<=y2)) mvwaddch(win, ys+i, x, ACS_BLOCK);
    else                   mvwaddch(win, ys+i, x, ACS_VLINE);
  }
}

void is_line(WINDOW *win, int h, int l, const char **text)
{
  int x, y;

  for(y=1;y<h;y++) {
    mvwhline(win, y, 1, ' ', l-2);
    if((y-1) + is_curr_line <= is_max_len) {
      if(! text[(y-1) + is_curr_line]) break;
      x = l - strlen(text[(y-1) + is_curr_line]);
      if(x < 2) x = 2;
      x = (x>>1);
      mvwaddnstr(win, y, x, text[(y-1) + is_curr_line], l-2);
    }
  }

  is_stat(win, 1, h-1, l-1, is_curr_line, is_max_len);
  wrefresh(win);
}

void is_up(WINDOW *win, const char **text)
{
  int maxy, maxx;
  int l, h;

  if(is_curr_line == 0) return;
  is_curr_line--;
  getmaxyx(win, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;

  is_line(win, h-1, l, text);
}

void is_down(WINDOW *win, const char **text)
{
  int maxy, maxx;
  int l, h;

  getmaxyx(win, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;

  if(is_curr_line + (h-2) <= is_max_len) {
    is_curr_line++;
    is_line(win, h-1, l, text);
  }
}

void is_pup(WINDOW *win, const char **text)
{
  int maxy, maxx;
  int l, h;

  if(is_curr_line == 0) return;
  getmaxyx(win, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;

  is_curr_line -= h-4;
  if(is_curr_line < 0) is_curr_line = 0;
  is_line(win, h-1, l, text);
}

void is_pdown(WINDOW *win, const char **text)
{
  int maxy, maxx;
  int l, h;

  getmaxyx(win, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;

  if(is_curr_line + (h-2) <= is_max_len) {
    is_curr_line += h-4;
    if(is_curr_line + (h-3) > is_max_len) is_curr_line = is_max_len - (h-3);
    is_line(win, h-1, l, text);
  }
}

void is_home(WINDOW *win, const char **text)
{
  int maxy, maxx;
  int l, h;

  if(is_curr_line == 0) return;
  getmaxyx(win, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;

  is_curr_line = 0;
  is_line(win, h-1, l, text);
}

void is_end(WINDOW *win, const char **text)
{
  int maxy, maxx;
  int l, h;

  getmaxyx(win, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;

  if(is_curr_line + (h-2) <= is_max_len) {
    is_curr_line = is_max_len - (h-3);
    is_line(win, h-1, l, text);
  }
}

void is_win_rebuild(WINDOW *win, int arg1, int arg2)
{
  int maxy, maxx;
  int l, h, x, y;

  getmaxyx(stdscr, maxy, maxx);
  l = is_like_len;
  h = is_like_height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;
  x = (maxx-l)>>1;
  y = (maxy-h)>>1;
  wresize(win, h, l);
  mvwin(win, y, x);

  wbkgd(win, COLOR_PAIR(1) | A_BOLD);
  wclear(win);
  box(win, 0, 0);

  is_line(win, h-1, l, (const char **) pop_win_poi(win));

  cbreak();
  noecho();
  keypad(win, TRUE);
  nodelay(win, FALSE);
  curs_set(0);
}
  
/* popup a textwindow, displaying "text" with len "len" and
   height "height" */
void is_popup_text(const char **text, int len, int height)
{
  WINDOW *win;
  int maxy, maxx;
  int l, h, x, y;
  int inp_char;

  if((! text) || (! text[0])) return;

  is_max_len = is_count_lines(text);

  getmaxyx(stdscr, maxy, maxx);
  l = len;
  h = height;
  if(l>maxx) l = maxx;
  if(h>maxy) h = maxy;
  x = (maxx-l)>>1;
  y = (maxy-h)>>1;
  is_like_len    = len;
  is_like_height = height;

  win = c_newwin(h, l, y, x, is_win_rebuild, maxy, maxx);
  store_win_poi(win, (void *) text);
  is_curr_line = 0;
  is_win_rebuild(win, maxy, maxx);

  while(1) {
    inp_char = wgetch(win);
    switch(inp_char) {
      case KEY_UP:
	is_up(win, text);
	break;
	
      case KEY_DOWN:
	is_down(win, text);
	break;
	
      case KEY_PPAGE:
	is_pup(win, text);
	break;
	
      case KEY_NPAGE:
	is_pdown(win, text);
	break;
	
      case KEY_HOME:
      case 362:
	is_home(win, text);
	break;
	
      case KEY_END:
      case 385:
	is_end(win, text);
	break;
	
      case '':
	is_win_rebuild(win, maxy, maxx);
	break;
	
      case ' ':
      case KEY_ENTER:
      case KEY_F(12):
      case '\'':
      case '\n':
      case '\r':
      case 'q':
      case 'Q':
	c_delwin(win);
	return;
	break;
    }
  }
}

/* build up char array from string (use \n to split string)
   int-pointer will show resulting lines in char array */
char **is_build_array(char *str, int *lines)
{
  char **ret_str;       /* use this as return value */
  int line, pos;        /* count lines and position */
  int i;

  ret_str = (char **) malloc(sizeof(char *));
  if(! ret_str) {
    wuuush(1);
  }
  ret_str[0] = NULL;

  i    = 0;
  line = 0;
  pos  = 0;

  while(1) {
    if((*(str + pos) == 0) || (*(str + pos) == '\n')) {
      ret_str = (char **) realloc(ret_str, sizeof(char *) * (line+2));
      if(! ret_str) {
	wuuush(1);
      }
      ret_str[line+1] = NULL;
      ret_str[line] = (char *) malloc(sizeof(char) * ((pos-i) + 1));
      if(! ret_str[line]) {
	wuuush(1);
      }
      if(pos-i) memcpy(ret_str[line], (str + i), pos-i);  /* only needed for non-null string */
      *(ret_str[line] + (pos - i)) = 0;
      line++;
      if(*(str + pos) == 0) break;     /* string ended */
      pos++;
      i = pos;
    } else {
      pos++;
    }
  }

  *lines = line;
  return ret_str;
}

void pop_up_info()
{
  char *temp_buf;
  char **info_text;
  int  len;

  /* Initialize INFO-text */
  temp_buf = (char *) malloc(sizeof(char) * 1024);
  if(! temp_buf) {
    wuuush(1);
  }
  sprintf(temp_buf, _("WSPse - MP3Creator\n"
		      "\n"
		      "written by Matthias Hensler 1999-2006\n"
		      "licensed under GPL\n"
		      "\n"
		      "http://mp3c.wspse.de/\n"
		      "eMail: matthias@wspse.de\n"
		      "\n"
		      "contains CDDB functions\n"
		      "\nprogram version:\n%s")
	  , external_version_str);
  info_text = is_build_array(temp_buf, &len);
  free(temp_buf);

  is_popup_text((const char **) info_text, 65, len+2);

  /* free info-text array */
  free_char_array(info_text);
}

void pop_up_help()
{
  int len;
  
  /* initialize online-help text if needed */
  if(! is_online_help) {
    is_online_help = is_build_array(
	_("Onlinehelp for MP3Creator\n"
	  "\n"
	  "on both sides:\n"
	  "F1: This help\n"
	  "F2: Optionsmenu\n"
	  "F12: Quit program\n"
	  "\n"
	  "on left side:\n"
	  "F3: Start encoding (marked tracks)\n"
	  "F4: Encode only actual track\n"
	  "F5: Output Batch-script\n"
	  "F6: Start ripping (marked tracks) and\n"
	  "    put them into right window       \n"
	  "F7: Rip only actual track\n"
	  "\n"
	  "on right side:\n"
	  "F3+F6: Start encoding (marked tracks)\n"
	  "F4+F7: Encode only actual track\n"
	  "F8: Delete marked tracks\n"
	  "F9: Delete actual track\n"
	  "\n"
	  "Note for F-keys:\n"
	  "Since F-keys won't work in all enviroments,\n"
	  "you can use number-keys instead. Press '1'\n"
	  "for F1, '2' for F2, and so on\n"
	  "\n"
	  "CRSR-keys: move in window up/down\n"
	  "Page-keys: fast up and down\n"
	  "Pos1/End: jump to first/last track on\n"
	  "          screen/list                \n"
	  "Del/Space: swap encoding flag         \n"
	  "           (if file should be encoded)\n"
	  "'*': swap encoding flag for all tracks\n"
	  "TAB: switch between left and right window\n"
	  "\n"
	  "Enter: open window to change information for track(s)\n"
	  "Here you can make changes for one or all tracks at once,   \n"
	  "without knowing all the shortcuts that follow later.       \n"
	  "You can see there fields with \"[ ]\" which shows information\n"
	  "which shouldn't change and fields with \"[*]\" for data you  \n"
	  "want to change.                                            \n"
	  "Every time you enter new data a field is converted to \"[*]\"\n"
	  "You can swap these flags by pressing SPACE.                \n"
	  "\n"
	  "a (#): input artist name for current track        \n"
	  "t (#): input title for current track              \n"
	  "b (#): input albumname for current track          \n"
	  "y (#): input year for current track               \n"
	  "c (#): input comment for current track            \n"
	  "g (#): select genre for current track             \n"
	  "o (#): swap \"on the fly\" flag for current track   \n"
	  "s (#): swap artist and interpret for current track\n"
	  "ALT+j: input year for all tracks (all selected)   \n"
	  "\n"
	  "w    : export actual track (only on right side)   \n"
	  "W    : export marked tracks                       \n"
	  "r    : import tracklist                           \n"
	  "       (see HINTS about tracklist export)         \n"
	  "d    : remove dead files from list                \n"
	  "\n"
	  "p    : play actual track from cd                  \n"
	  "x    : stop playing                               \n"
	  "e    : eject tray of cdrom-device                 \n"
	  "E    : close tray of cdrom-device                 \n"
	  "\n"
	  "v    : reinit cd-volume (necessary after new cd)  \n"
	  "V    : force CDDB interpretation for sampler cds  \n"
	  "F    : force loading of CDDB (even if there is an \n"
	  "       local entry)                               \n"
	  "l    : lock/unlock remote CDDB access             \n"
	  "L    : toggle sampler flag for actual track       \n"
	  "m    : write CDDB entry to local file             \n"
	  "M    : transfer CDDB entry to remote database     \n"
	  "\n"
	  "q    : used to cancel and leave most windows      \n"
	  "\n"
	  "Note: if a key is marked with (#) you can perform\n"
	  "      the operation on all selected tracks, if   \n"
	  "      you press down ALT-key too. (e.g. press    \n"
	  "      ALT+s to swap artist and title for all     \n"
	  "      marked tracks, or ALT+g to select genre for\n"
	  "      marked tracks)                             \n"
	  "Note2: some (#) keys let you input information   \n"
	  "       for each track seperately (e.g. year or   \n"
	  "       title). Here you can press SHIFT down too,\n"
	  "       to input information for all selected     \n"
	  "       tracks at once. (e.g. press ALT+SHIFT+y to\n"
	  "       input year once for all selected tracks). \n"
	  "       This isn't really useful for tracktitles, \n"
	  "       I think, but maybe..... ;-)               \n"
	  "\n"
	  "\n"
	  "HINTS\n"
	  "-------\n"
	  "\n"
	  "On-the fly encoding\n"
	  "MP3-file will be created without using a temporary-\n"
	  "file. Must be supported by your ripper (output wav \n"
	  "or cda to stdout) and your encoder (use wav or cda \n"
	  "input from stdin). This sometimes causes \"klicks\"  \n"
	  "on mp3-files. Streams will be buffered over a FIFO,\n"
	  "set size of FIFO in option-menu.                   \n"
	  "\n"
	  "Ripping all tracks before encoding\n"
	  "If you want to rip all tracks before encoding just \n"
	  "activate the flag in option-menu. Remember that you\n"
	  "should have enough space on your harddisk, since   \n"
	  "wav-data uses much space.                          \n"
	  "Ripped data is stored in the same temp-directory   \n"
	  "like normally, you don't have to change your old   \n"
	  "setting. An unique number is inserted without      \n"
	  "destroying any \".wav\"-ending.                      \n"
	  "\n"
	  "Replace slash character\n"
	  "You can set a character which is inserted instead  \n"
	  "of slashs in option-menu. This prevents sensless   \n"
	  "directory creation if a songtitle contains a slash.\n"
	  "\n"
	  "Character set\n"
	  "You cannot input all characters or some characters \n"
	  "are missing in CDBB-entries (eg Umlauts, etc)? Just\n"
	  "set character-handling to \"non-strict\" in option-  \n"
	  "menu.                                              \n"
	  "\n"
	  "CDDB\n"
	  "No internet-connection? Just input \"0\" as CDDB-    \n"
	  "server, and only local database is used.           \n"
	  "\n"
	  "Filename-Creation\n"
	  "You have lots of options for filename-creation.    \n"
	  "The first thing is to set the root-directory for   \n"
	  "your mp3-files. Then create a pattern-mask for the \n"
	  "filenames (here you can use several patterns for   \n"
	  "inserting artistname, songtitle, etc. You can even \n"
	  "use slashes to split the files into different      \n"
	  "directorys). Furthermore you can set if you accept \n"
	  "spaces in filenames or not and you can input a list\n"
	  "with all characters you don't like in filenames.   \n"
	  "Each character which is killed is then replaced by \n"
	  "an defined char or left out. You can select what   \n"
	  "you like by setting \"Patternmode\" in option-menu.  \n"
	  "A nice additional feature is to convert characters \n"
	  "toupper if the previous was killed. This is really \n"
	  "nice if you kill every space without converting    \n"
	  "them to underscores. Just try option-menu for all  \n"
	  "the features.                                      \n"
	  "\n"
	  "Ripping several cds before encoding\n"
	  "As introduced with version 0.21 you can rip several\n"
	  "tracks, without doing encoding. Each ripped track  \n"
	  "is placed on the right window. Of course the old   \n"
	  "behavior is still possible (F3 and F4), but you can\n"
	  "also use F6 to rip all marked tracks (or F7 to rip \n"
	  "the actual track) without encoding. (If you use F3 \n"
	  "or F4 and encoding fails, the already ripped track \n"
	  "is saved).                                         \n"
	  "On the right side you can do the same like on the  \n"
	  "left one: mark tracks, encode one or all and open a\n"
	  "setup window with ENTER-key. Furthermore you can   \n"
	  "delete one or all marked tracks by using F8 and F9.\n"
	  "Maybe you will notice that there is no \"on-the-fly\"\n"
	  "flag on the right side, but a \"delete\"-flag. This  \n"
	  "flag shows if a track should be deleted after en-  \n"
	  "coding. Its setting is also checked on exit if the \n"
	  "\"Delete Tempfiles on Exit\" option is set to \"1\".   \n"
	  "Later you will be able to import new wav-files.    \n"
	  "\n"
	  "Tracklist export and import\n"
	  "Since version 0.21 it is possible to rip several cds  \n"
	  "without doing the encoding directly after it. But what\n"
	  "if you have no time to encode and want to leave MP3c? \n"
	  "Well, no problem. Just export the ripped tracks and do\n"
	  "the encoding in the next session. You can select a    \n"
	  "file to store the datas in option-menu. Then you can  \n"
	  "press \"W\" to save all marked tracks in the right win- \n"
	  "dow, or \"w\" to save only the actual track. If a file  \n"
	  "is already existing you can overwrite it, or add the  \n"
	  "new tracks. Don't worry if you export some tracks     \n"
	  "double, MP3c will allow only one unique track in list.\n"
	  "But be careful, watch the deleteflag of the tracks and\n"
	  "the setting \"delete tempfiles on exit\" in optionmenu, \n"
	  "otherwise you could lose your ripped files if you quit\n"
	  "MP3c. Note: you can set an option to clear the delete-\n"
	  "flag on export in option-menu.                        \n"
	  "You can read export tracks just by pressing \"r\".      \n"
	  "By pressing \"d\" you can remove every dead track.      \n"
	  "\n"
	  "Configuration\n"
	  "MP3c saves his configuration in \"~/.mp3crc\". You can \n"
	  "also put a global configuration file \"mp3crc\" (no    \n"
	  "dot) into \"$prefix/etc/\", where $prefix is the global\n"
	  "path MP3c was stored in (\"/usr/etc\" for distributed  \n"
	  "versions and \"/usr/local/etc\" if you self installed  \n"
	  "MP3c). You can set global-path to \"/etc\" by changing \n"
	  "\"mp3creat.h\" before compiling it.                    \n"
	  "To dump actual configuration just use \"Save Config\"  \n"
	  "in option-menu, or enable \"Auto Save\" to save before \n"
	  "leaving MP3c.                                        \n"
	  "The configurationfile is an normal ascii-text file.  \n"
	  "You can edit it manually without any risks, each key \n"
	  "stored in it has a lot of comments, so you should    \n"
	  "understand the file easily.                          \n"
	  "\n"
	  "\n"
	  "Hope you find this program useful. If you found a\n"
	  "bug, write to: matthias@wspse.de\n"
	  "see file FAQ for reporting bugs or start MP3c like this:\n"
	  "\"mp3c -v\"\n"
	  "\n"
	  "look on http://mp3c.wspse.de/\n"
	  "for new versions\n"
	  "\n"
	  "WSPse 1999-2006")
	  , &len);
  }
  
  is_popup_text((const char **) is_online_help, 76, 20);
} 
