/* gpgme.c -  My GnuPG Made Easy
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <windows.h>
#include <process.h>

#include "util.h"
#include "context.h"
#include "ops.h"

char *_gpgme_gpg_program;
char *_gpgme_gpg_optfile;
char *_gpgme_gpg_pubring;
char *_gpgme_gpg_secring;

/**
 * gpgme_new:
 * @r_ctx: Returns the new context
 * 
 * Create a new context to be used with most of the other GPGME
 * functions.  Use gpgme_release_contect() to release all resources
 *
 * Return value: An error code 
 **/
gpgme_error_t
gpgme_new (gpgme_ctx_t * r_ctx)
{
    gpgme_ctx_t ctx;

    if (!r_ctx)
	return mk_error (Invalid_Value);
    *r_ctx = NULL;
    ctx = calloc (1, sizeof *ctx);
    if (!ctx)
        return mk_error (Out_Of_Core);
    ctx->cipher_algo = GPGME_CIPHER_CAST5;
    ctx->s2k.digest_algo = GPGME_MD_SHA1;
    ctx->s2k.mode = GPGME_S2K_ITERSALTED;
    *r_ctx = ctx;
    return 0;
}

/**
 * gpgme_release:
 * @c: Context to be released. 
 * 
 * Release all resources associated with the given context.
 **/
void
gpgme_release (gpgme_ctx_t ctx)
{
    if (!ctx)
        return;
    _gpgme_gpg_release (&ctx->gpg); 
    _gpgme_release_result (ctx);
    gpgme_key_release (ctx->tmp_key);
    gpgme_data_release (ctx->help_data_1);
    gpgme_data_release (ctx->notation);
    gpgme_data_release (ctx->logging);
    gpgme_signers_clear (ctx);
    safe_free (ctx->signers);
    safe_free (ctx->keygen_fpr);
    safe_free (ctx->locusr);
    safe_free (ctx->comment);
    /* fixme: release the key_queue */
    safe_free (ctx);
} /* gpgme_release */


void
_gpgme_release_result (gpgme_ctx_t c)
{
    assert (c);
    switch (c->result_type) {
    case RESULT_TYPE_NONE:
	break;

    case RESULT_TYPE_EDITKEY:
	_gpgme_release_editkey_result (c->result.editk);
	break;

    case RESULT_TYPE_ENCRYPT:		  
	_gpgme_release_encrypt_result( c->result.encrypt );
	break;

    case RESULT_TYPE_VERIFY:
	  _gpgme_release_verify_result ( c->result.verify );
	  break;

    case RESULT_TYPE_DECRYPT:
	_gpgme_release_decrypt_result ( c->result.decrypt );
	break;

    case RESULT_TYPE_SIGN:
	_gpgme_release_sign_result ( c->result.sign );
	break;

    case RESULT_TYPE_SIGN_ENCRYPT:
	_gpgme_release_sign_encrypt_result( c->result.sign_enc );
	break;

    case RESULT_TYPE_IMPORT:
	_gpgme_release_import_result( c->result.import );
	break;

    case RESULT_TYPE_SYMENC:
	_gpgme_release_symenc_result( c->result.symenc );
	break;
    }	

    c->result.verify = NULL;
    c->result_type = RESULT_TYPE_NONE;
} /* _gpgme_release_result */


/**
 * gpgme_cancel:
 * @c: the context
 * 
 * Cancel the current operation.  It is not guaranteed that it will work for
 * all kinds of operations.  It is especially useful in a passphrase callback
 * to stop the system from asking another time for the passphrase.
 **/
void
gpgme_cancel (gpgme_ctx_t c)
{
    return_if_fail (c);
    c->cancel = 1;
    c->pending = 0;
} /* gpgme_cancel */


/**
 * gpgme_get_notation:
 * @c: the context 
 * 
 * If there is notation data available from the last signature check, this
 * function may be used to return this notation data as a string.  The string
 * is a represantaton of that data embedded in a %<notation> container.
 * 
 * Return value: An string or NULL if no notation data is available.
 **/
char *
gpgme_get_notation (gpgme_ctx_t c)
{
    if (!c  || !c->notation)
	return NULL;
    return _gpgme_data_get_as_string (c->notation);
} /* gpgme_get_notation */


char *
gpgme_get_logging (gpgme_ctx_t c)
{
    if (!c || !c->logging)
	return NULL;
    return _gpgme_data_get_as_string (c->logging);
}


void
gpgme_set_comment (gpgme_ctx_t c, const char * text)
{
    if (!c)
	return;
    safe_free (c->comment);
    c->comment = strdup (text);
    if (!c->comment)
	c->out_of_core = 1;
}


void
gpgme_set_homedir (gpgme_ctx_t c, const char *homedir)
{
    if (!c)
	return;
    safe_free (c->homedir);
    c->homedir = strdup (homedir);
    if (!c->homedir)
	c->out_of_core = 1;
}


void *
gpgme_global_control( gpgme_global_t cmd, void * val )
{
   void *retval = NULL;

    switch( cmd ) {
        case GPGME_GLOBAL_GPGPROGRAM:
            if (val)
                _gpgme_gpg_program = strdup((char *)val);
            else
                retval = (void *)_gpgme_gpg_program;
            break;

        case GPGME_GLOBAL_OPTFILE:
            if (val)
                _gpgme_gpg_optfile = strdup((char *)val);
            else
                retval = (void *)_gpgme_gpg_optfile;
            break;

        case GPGME_GLOBAL_PUBRING:
            if (val)
                _gpgme_gpg_pubring = strdup((char *)val);
            else
                retval = (void *)_gpgme_gpg_pubring;
            break;

        case GPGME_GLOBAL_SECRING:
            if (val)
                _gpgme_gpg_secring = strdup((char *)val);
            else
                retval = (void *)_gpgme_gpg_secring;
            break;
    }  // switch( cmd )
    return retval;
}


/**
 * gpgme_control:
 * @c: the context
 * @cmd: command to set
 * @val: the value for the command.
 *
 * This is a replace for gpgme_set_xxx because in a DLL it's better to
 * reduce the amount of functions.
 **/
void *
gpgme_control( gpgme_ctx_t c, int cmd, int val )
{
    if( !c )
	return 0;

    if( val == -1 ) {
	switch( cmd ) {
	case GPGME_CTRL_ARMOR: return (int *)c->use_armor;
	case GPGME_CTRL_FPR:   return c->keygen_fpr;
	default:               return NULL;
	}
    }

    switch( cmd ) {

    case GPGME_CTRL_S2K:
	/* Pass S2K options down to GPG (no=0, yes=1) */
	c->s2k.used = val;
	break;

    case GPGME_CTRL_S2K_MODE:
	c->s2k.mode = val;
	break;

    case GPGME_CTRL_S2K_HASH:
	c->s2k.digest_algo = val;
	break;

    case GPGME_CTRL_TEXTMODE:
	/* Enable or disable the use of the special textmode.  Textmode is for example
	    used for MIME (RFC2015) signatures. */
	c->use_textmode = val;
	break;
	
    case GPGME_CTRL_LISTMODE:
	/* This function changes the default behaviour of the keylisting functions.
	Defines values for @mode are: %0 = normal, %1 = fast listing without
	information about key validity. */
	c->keylist_mode |= val;
	break;

    case GPGME_CTRL_NO_COMPR:
	c->no_compress = val;	    
	c->force_mdc = val == 1? 1 : 0;
	break;

    case GPGME_CTRL_CIPHER:
	if (val == -1) { /* disable it */
	    c->cipher_algo = -1;
	    break;
	}
	if (val < 1 || val > 7) {
	    c->cipher_algo =  GPGME_CIPHER_CAST5;
	    break;
	}
	c->cipher_algo = val;
	break;

    case GPGME_CTRL_TMPFILES:        c->use_tmpfiles = val; break;
    case GPGME_CTRL_WITH_SECRET_KEY: c->with_secret_key = val; break;
    case GPGME_CTRL_FORCETRUST:      c->force_trust = val; break;
    case GPGME_CTRL_FORCEOPT:        c->force_opt = val; break;
    case GPGME_CTRL_FILE:            c->use_file = val; break;
    case GPGME_CTRL_THROWKEYID:      c->use_throwkeyid = val; break;
    case GPGME_CTRL_ARMOR:           c->use_armor = val; break;	
    case GPGME_CTRL_INTERACTIVE:     c->interactive = val; break;
    case GPGME_CTRL_PIPEMODE:        c->pipemode = val; break;
    case GPGME_CTRL_LOGGING:         c->use_logging = val; break;
    case GPGME_CTRL_CB_VAL:	     c->cb.progress_value_int = val; break;
    }

    return NULL;
}

/**
 * gpgme_set_passphrase_cb:
 * @c: the context 
 * @cb: A callback function
 * @cb_value: The value passed to the callback function
 * 
 * This function sets a callback function to be used to pass a passphrase
 * to gpg. The preferred way to handle this is by using the gpg-agent, but
 * because that beast is not ready for real use, you can use this passphrase
 * thing.
 *
 * The callback function is defined as:
 * <literal>
 * typedef const char *(*gpgme_passphrase_cb_t)(void*cb_value,
 *                                          const char *desc,
 *                                          void *r_hd);
 * </literal>
 * and called whenever gpgme needs a passphrase. DESC will have a nice
 * text, to be used to prompt for the passphrase and R_HD is just a parameter
 * to be used by the callback it self.  Becuase the callback returns a const
 * string, the callback might want to know when it can release resources
 * assocated with that returned string; gpgme helps here by calling this
 * passphrase callback with an DESC of %NULL as soon as it does not need
 * the returned string anymore.  The callback function might then choose
 * to release resources depending on R_HD.
 *
 **/
void
gpgme_set_passphrase_cb (gpgme_ctx_t ctx, gpgme_passphrase_cb_t cb, 
			 void * cb_value)
{
    if (!ctx)
	return;
    ctx->cb.passphrase = cb;
    ctx->cb.passphrase_value = cb_value;
} /* gpgme_set_passphrase_cb */


void
gpgme_set_interactive_cb (gpgme_ctx_t ctx, gpgme_interactive_cb_t cb, 
			  void * cb_value)
{
    if (!ctx)
	return;
    ctx->cb.interactiv = cb;
    ctx->cb.interactiv_value = cb_value;
} /* gpgme_set_inteactive_cb */


/**
 * gpgme_set_pprogress_cb:
 * @c: the context 
 * @cb: A callback function
 * @cb_value: The value passed to the callback function
 * 
 * This function sets a callback function to be used as a progress indicator.
 *
 * The callback function is defined as:
 * <literal>
 * typedef void (*gpgme_progress_cb_t) (void * cb_value,
 *                                  const char * what, int type,
 *                                  int curretn, int total);
 * </literal>
 * For details on the progress events, see the entry for the PROGRESS
 * status in the file doc/DETAILS of the GnuPG distribution.
 **/
void
gpgme_set_progress_cb (gpgme_ctx_t ctx, gpgme_progress_cb_t cb, 
		       void * cb_value)
{	
    if (!ctx)
	return;
    ctx->cb.progress = cb;
    ctx->cb.progress_value = cb_value;	
} /* gpgme_set_progress_cb */


void
gpgme_set_edit_ctx( gpgme_ctx_t ctx, void * edit_ctx, int edit_id )
{
    if( !ctx )
	return;
    ctx->edit_opaque = edit_ctx;
    ctx->edit_cmd = edit_id;
} /* gpgme_set_edit_ctx */


void
gpgme_set_list_options (gpgme_ctx_t ctx, int optid)
{
    if (!ctx)
	return;
    ctx->list_opts |= optid;
}


void
gpgme_set_cache_ctx (gpgme_ctx_t ctx, void * cache_ctx)
{	
    if( ctx )
	ctx->key_lookup = cache_ctx;
} /* gpgme_set_cache_ctx */


void
gpgme_set_passphrase( gpgme_ctx_t ctx, const char * passphrase )
{	
    if( !ctx )
	return;
    ctx->passphrase_value = passphrase;
    ctx->use_pass_fd = 1;
} /* gpgme_set_passphrase */


gpgme_error_t
_gpgme_add_passphrase( gpgme_ctx_t ctx )
{	
    gpgme_error_t err;
    gpgme_data_t passphrase_fd;

    if( !ctx )
	return mk_error( Invalid_Value );
    if( !ctx->passphrase_value || !ctx->use_pass_fd )
	return mk_error( General_Error );
    
    err = gpgme_data_new_from_mem( &passphrase_fd, ctx->passphrase_value,
				    strlen( ctx->passphrase_value ), 0 );
    if( err )
	return err;
    _gpgme_data_set_mode( passphrase_fd, GPGME_DATA_MODE_OUT );
    _gpgme_gpg_add_arg( ctx->gpg, "--passphrase-fd" );
    _gpgme_gpg_add_data( ctx->gpg, passphrase_fd, -2 );

    return 0;
} /* _gpgme_add_passphrase */


void
gpgme_set_debug_mode( int val )
{	
    if( val )
	putenv( "GPGME_DEBUG=5:gpgme.dbg" );
    else
	putenv( "GPGME_DEBUG=" );
} /* gpgme_set_debug_mode */


const char*
_gpgme_get_tmpfile( int input )
{
    static char tmpfile[384];
    char path[256];

    GetTempPath( sizeof path -1, path );
    _snprintf( tmpfile, sizeof tmpfile -1, 
	"%sgpgme%s.%u", path, input? "IN" : "OUT", getpid() );
    return tmpfile;
} /* _gpgme_get_tmpfile */


static int
_gpgme_unlink( const char * file, int arg )
{
    return unlink( file );
} /* _gpgme_unlink */


void
_gpgme_del_tmpfiles( gpgme_wipe_t wipe_fnc )
{
    gpgme_wipe_t erase = wipe_fnc? wipe_fnc: (gpgme_wipe_t)_gpgme_unlink;

    erase( _gpgme_get_tmpfile( 0 ), 1 );	
    erase( _gpgme_get_tmpfile( 1 ), 1 );
} /* _gpgme_del_tmpfiles */

void
gpgme_set_wipe_fnc( gpgme_ctx_t ctx, gpgme_wipe_t fnc )
{
    if( ctx )
	ctx->wipe_fnc = fnc;
} /* gpgme_set_wipe_fnc */


void
gpgme_set_local_user( gpgme_ctx_t ctx, const char * name )
{
    if( ctx && name ) {
	safe_free (ctx->locusr);
	ctx->locusr = strdup( name );
    }
} /* gpgme_set_local_user */


int
gpgme_get_process_rc (gpgme_ctx_t ctx)
{
    if (!ctx)
	return -1;
    return (int)ctx->proc_rc;
} /* gpgme_get_process_rc */


void
_gpgme_progress_handler (gpgme_ctx_t ctx, char * args)
{
    const char * name = args;
    char * buf, * p;
    char * what = NULL;
    unsigned curr, total, i=0;

    /* what */
    p = strchr (args, '?');
    if (!p)
	return;
    what = malloc (p-args+2);
    if (!what) {
	ctx->out_of_core = 1;
	return;
    }
    while (name && *name != '?')
	what[i++] = *name++;
    what[i] = '\0';
    /* XXX remove space at the end */
    i = 0;
    buf = args + (p-args);
    while ((p = strsep (&buf, " ")) != NULL) {
	switch (i) {
	case 0: 
	    break; /* type */
	case 1: 
	    curr = strtoul (p, NULL, 10); 
	    break; /* current offset */
	case 2: 
	    total = strtoul (p, NULL, 10); 
	    break; /* total */
	}
	i++;
    }
    if (what) {
	ctx->cb.progress (ctx->cb.progress_value, what, 0, curr, total);
	safe_free (what);
    }
} /* _gpgme_progress_handler */


void
_gpgme_add_comment (gpgme_ctx_t ctx)
{
    char * p;

    if (ctx->comment) 
    {
	p = calloc (1, strlen (ctx->comment)+2);
	if (!p)
	{
	    ctx->out_of_core = 1;
	    return;
	}
	_gpgme_gpg_add_arg (ctx->gpg, "--comment");
	p[0] = '"';
	strcpy (p+1, ctx->comment);
	strcat (p, "\"");
	_gpgme_gpg_add_arg (ctx->gpg, p);
	safe_free (p);
    }
}


gpgme_error_t
gpgme_check_logging (gpgme_ctx_t ctx)
{
    char * buf;

    if (!ctx)
	return mk_error (Invalid_Value);
    buf = gpgme_get_logging (ctx);
    if (!buf)
	return mk_error (No_Error);

    /* XXX: this only works for English GPG output!!! */
    if (strstr (buf, "gpg.conf:") && strstr (buf, "invalid option"))
	return mk_error (Conf_InvOption);
    
    free (buf);
    return 0;
}


gpgme_error_t
gpgme_lib_init (void)
{
    _gpgme_gpg_program = NULL;
    _gpgme_gpg_optfile = NULL;
    _gpgme_gpg_pubring = NULL;
    _gpgme_gpg_secring = NULL;
    return 0;
}


void
gpgme_lib_cleanup (void)
{
    if (_gpgme_gpg_program)
        free(_gpgme_gpg_program);
    if (_gpgme_gpg_optfile)
        free(_gpgme_gpg_optfile);
    if (_gpgme_gpg_pubring)
        free(_gpgme_gpg_pubring);
    if (_gpgme_gpg_secring)
        free(_gpgme_gpg_secring);

    debug_cleanup ();
    rungpg_cleanup ();
    io_cleanup ();
    wait_cleanup ();
    util_cleanup ();
}
