/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core.internal.ddl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.core.util.RdbUtils;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLSchemaFactory;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.schema.impl.SQLSchemaFactoryImpl;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesFactory;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;

public class DdlGenerator {
    private EntityModel model;
    private Database database;
    private Schema defaultSchema;
    private TableComparator tableComparator = new TableComparator();
    private DatabaseDefinition dbDef;
    
    private HashMap schemaMap = new HashMap();
    private HashMap tableMap = new HashMap();
    private IEntityModelManager modelManager;
    
    
    public DdlGenerator(EntityModel model, Database database, Schema defaultSchema) {
        this.model = model;
        this.database = database;
        this.defaultSchema = defaultSchema;
        this.dbDef = RdbUtils.getDatabaseDefinitionRegistry().getDefinition(database);
    }
    
    public DdlGenerator(IEntityModelManager modelManager, DatabaseDefinition dbDef) {
        this.modelManager = modelManager;
        this.model = modelManager.getEntityModel();
        this.dbDef = dbDef;
        this.database = SQLSchemaFactoryImpl.eINSTANCE.createDatabase();
        database.setVendor(dbDef.getProduct());
        database.setVersion(dbDef.getVersion());
        this.defaultSchema = SQLSchemaFactoryImpl.eINSTANCE.createSchema();
        database.getSchemas().add(defaultSchema);
    }

    public List getCreateDDL(){
        loadFromModel();
        ArrayList tables = new ArrayList(tableMap.values());
        Collections.sort(tables, tableComparator);
        return tables;
    }

    private List getMigrateDDL(){
        loadFromDatabase();
        return getCreateDDL();
    }

    private void loadFromModel() {
        if(modelManager != null){
            Set sqlObjects = modelManager.getExtraSqlObjects();
            if(sqlObjects != null){
                for(Iterator sqlObjs = sqlObjects.iterator(); sqlObjs.hasNext();){
                    Object sqlObj =  sqlObjs.next();
                    if(sqlObj instanceof OrmTable){
                        addOrmTable((OrmTable)sqlObj);
                    }
                }
            }
        }
        if(model == null){
            return;
        }
        Set sqlObjects = model.getTopLevelSqlObjects();
        for(Iterator sqlObjs = sqlObjects.iterator(); sqlObjs.hasNext();){
            Object sqlObj =  sqlObjs.next();
            if(sqlObj instanceof OrmTable){
                addOrmTable((OrmTable)sqlObj);
            }
        }
    }

    private void addOrmTable(OrmTable ormTable) {
        String catalogName = ormTable.getCatalog();
        String schemaName = ormTable.getSchema();
        Schema schema;
        if(schemaName == null || schemaName.trim().length() == 0){
            schema = defaultSchema;
        }else{
            schema = (Schema) schemaMap.get(schemaName);
        }
        if(schema == null){
            schema = SQLSchemaFactory.eINSTANCE.createSchema();
            schema.setName(schemaName);
            schemaMap.put(schemaName, schema);
            database.getSchemas().add(schema);
        }
        String tableName = ormTable.getName();
        Table table = (Table) tableMap.get(tableName);
        if(table == null){
            table = (Table) dbDef.getDataModelElementFactory().create(
                    SQLTablesPackage.eINSTANCE.getPersistentTable());
//            table = SQLTablesFactory.eINSTANCE.createPersistentTable();
            table.setName(tableName);
            tableMap.put(tableName, table);
            table.setSchema(schema);
        }
        for(Iterator cols = ormTable.getColumnList().iterator(); cols.hasNext();){
            OrmColumn ormCol = (OrmColumn) cols.next();
            Column rdbCol = getRdbColumn(table, ormCol);
            RdbUtils.copyToRdb(ormCol, rdbCol, dbDef);
        }
        RdbUtils.copyToRdb(ormTable, (BaseTable)table, dbDef);
    }

    private Column getRdbColumn(Table table, OrmColumn ormCol) {
        for(Iterator cols = table.getColumns().iterator(); cols.hasNext();){
            Column rdbCol = (Column) cols.next();
            if(rdbCol.getName().equals(ormCol.getName())){
                return rdbCol;
            }
        }
        Column rdbCol = SQLTablesFactory.eINSTANCE.createColumn();
        table.getColumns().add(rdbCol);
        return rdbCol;
    }

    private void loadFromDatabase() {
        // TODO Auto-generated method stub
        
    }
}
